#ifndef _SG_GEODESY_HXX
#define _SG_GEODESY_HXX

#include <simgear/math/point3d.hxx>

/** 
 * Convert from geocentric coordinates to geodetic coordinates
 * @param lat_geoc (in) Geocentric latitude, radians, + = North
 * @param radius (in) C.G. radius to earth center (meters)
 * @param lat_geod (out) Geodetic latitude, radians, + = North
 * @param alt (out) C.G. altitude above mean sea level (meters)
 * @param sea_level_r (out) radius from earth center to sea level at
 *        local vertical (surface normal) of C.G. (meters)
 */
void sgGeocToGeod(double lat_geoc, double radius,
                  double *lat_geod, double *alt, double *sea_level_r);


/**
 * Convert from geodetic coordinates to geocentric coordinates.
 * WARNING: this function is non-reversible.  Due to the fact that
 * "up" is a different direction for geocentric and geodetic frames,
 * you can not simply add your "alt" parameter to the "sl_radius"
 * result and get back (via sgGeodToGeoc()) to the coordinates you
 * started with.  The error under normal conditions will be of
 * centimeter order; whether that is important or not is application
 * dependent. Consider using sgGeodToCart() instead.
 *
 * @param lat_geod (in) Geodetic latitude, radians, + = North
 * @param alt (in) C.G. altitude above mean sea level (meters)
 * @param sl_radius (out) SEA LEVEL radius to earth center (meters)
 * @param lat_geoc (out) Geocentric latitude, radians, + = North
 */
void sgGeodToGeoc(double lat_geod, double alt,
                  double *sl_radius, double *lat_geoc );

/**
 * Convert a cartesian point to a geodetic lat/lon/altitude.
 *
 * @param xyz (in) Pointer to cartesian point.
 * @param lat (out) Latitude, in radians
 * @param lon (out) Longitude, in radians
 * @param alt (out) Altitude, in meters above the WGS84 ellipsoid
 */
void sgCartToGeod(double* xyz, double* lat, double* lon, double* alt);

/**
 * Convert a cartesian point to a geodetic lat/lon/altitude.
 * Alternate form using Point3D objects.
 *
 * @param cartesian point
 * @return geodetic point
 */
inline Point3D sgCartToGeod(const Point3D& p)
{
    double lat, lon, alt, xyz[3];
    xyz[0] = p.x(); xyz[1] = p.y(); xyz[2] = p.z(); 
    sgCartToGeod(xyz, &lat, &lon, &alt);
    return Point3D(lon, lat, alt);
}


/**
 * Convert a geodetic lat/lon/altitude to a cartesian point.
 *
 * @param lat (in) Latitude, in radians
 * @param lon (in) Longitude, in radians
 * @param alt (in) Altitude, in meters above the WGS84 ellipsoid
 * @param xyz (out) Pointer to cartesian point.
 */
void sgGeodToCart(double lat, double lon, double alt, double* xyz);

/**
 * Convert a geodetic lat/lon/altitude to a cartesian point.
 * Alternate form using Point3D objects.
 *
 * @param geodetic point
 * @return cartesian point
 */
inline Point3D sgGeodToCart(const Point3D& geod)
{
    double xyz[3];
    sgGeodToCart(geod.lat(), geod.lon(), geod.elev(), xyz);
    return Point3D(xyz[0], xyz[1], xyz[2]);
}

/**
 * Given a starting position and an offset radial and distance,
 * calculate an ending positon on a wgs84 ellipsoid.
 * @param alt (in) meters
 * @param lat1 (in) degrees
 * @param lon1 (in) degrees
 * @param az1 (in) degrees
 * @param s (in) distance in meters
 * @param lat2 (out) degrees
 * @param lon2 (out) degrees
 * @param az2 (out) return course in degrees
 */
int geo_direct_wgs_84 ( double alt, double lat1,
                        double lon1, double az1, 
			double s, double *lat2, double *lon2,
                        double *az2 );


/**
 * Given an altitude and two sets of (lat, lon) calculate great circle
 * distance between them as well as the starting and ending azimuths.
 * @param alt (in) meters
 * @param lat1 (in) degrees
 * @param lon1 (in) degrees
 * @param lat2 (in) degrees
 * @param lon2 (in) degrees
 * @param az1 (out) start heading degrees
 * @param az2 (out) end heading degrees
 * @param s (out) distance meters
 */
int geo_inverse_wgs_84( double alt, double lat1,
                        double lon1, double lat2,
			double lon2, double *az1, double *az2,
                        double *s );

#endif // _SG_GEODESY_HXX
