/*
 * $Id: usb.c,v 1.1 2004/01/02 04:03:21 troth Exp $
 *
 ****************************************************************************
 *
 * simulavr - A simulator for the Atmel AVR family of microcontrollers.
 * Copyright (C) 2003  Keith Gudger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 ****************************************************************************
 */

/**
 * \file usb.c
 * \brief Module to simulate the AVR's USB module.
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>

#include "avrerror.h"
#include "avrmalloc.h"
#include "avrclass.h"
#include "utils.h"
#include "callback.h"
#include "op_names.h"

#include "storage.h"
#include "flash.h"

#include "vdevs.h"
#include "memory.h"
#include "stack.h"
#include "register.h"
#include "sram.h"
#include "usb.h"

#include "intvects.h"

void usb_port_wr (char *name, uint8_t val);
uint8_t usb_port_rd (char *name);

/*****************************************************************************\
 *
 * USB Interrupts 
 *
\*****************************************************************************/

static uint8_t usb_intr_read (VDevice *dev, int addr);
static void usb_intr_write (VDevice *dev, int addr, uint8_t val);
static void usb_intr_reset (VDevice *dev);
static char *usb_intr_reg_name (VDevice *dev, int addr);

/** \brief Allocate a new USB interrupt */

USBInter_T *
usb_intr_new (uint8_t func_mask)
{
    USBInter_T *usb;

    usb = avr_new (USBInter_T, 1);
    usb_intr_construct (usb, func_mask);
    class_overload_destroy ((AvrClass *)usb, usb_intr_destroy);

    return usb;
}

/** \brief Constructor for usb interrupt object. */

void
usb_intr_construct (USBInter_T *usb, uint8_t func_mask)
{
    char *name = "USBIntr";

    if (usb == NULL)
        avr_error ("passed null ptr");

    vdev_construct ((VDevice *)usb, name, USB_INTR_BASE, USB_INTR_SIZE,
                    usb_intr_read, usb_intr_write, usb_intr_reset,
                    usb_intr_reg_name);

    usb->func_mask = func_mask;

    usb_intr_reset ((VDevice *)usb);
}

/** \brief Destructor for usb interrupt object. */

void
usb_intr_destroy (void *usb)
{
    if (usb == NULL)
        return;

    vdev_destroy (usb);
}

static uint8_t
usb_intr_read (VDevice *dev, int addr)
{
    USBInter_T *usb = (USBInter_T *)dev;

    switch (addr - vdev_get_base (dev))
    {
        case USB_INTR_ENABLE_ADDR:
            return (usb->uier);
        case USB_INTR_MASK_ADDR:
            return (usb->uimsk);
        case USB_INTR_STATUS_ADDR:
            return (usb->uisr =
                    usb_port_rd (usb_intr_reg_name ((VDevice *)usb, addr)));
        case USB_INTR_SPRSIE_ADDR:
            return (usb->sprsie);

        case USB_INTR_SPRSR_ADDR:
            return (usb->sprsr =
                    usb_port_rd (usb_intr_reg_name ((VDevice *)usb, addr)));
        case USB_GLB_STATE_ADDR:
            return (usb->glb_state);
        case USB_FRM_NUM_L_ADDR:
            return (usb->frm_num_l);
        case USB_FRM_NUM_H_ADDR:
            return (usb->frm_num_h);
        case USB_INTR_SPRSMSK_ADDR:
            return (usb->sprsmsk);
        case USB_INTR_ACK_ADDR:
        case USB_INTR_UNUSED_ADDR:
        default:
            avr_error ("Bad address: 0x%04x", addr);
    }
    return 0;                   /* will never get here */
}

static void
usb_intr_write (VDevice *dev, int addr, uint8_t val)
{
    USBInter_T *usb = (USBInter_T *)dev;

    switch (addr - vdev_get_base (dev))
    {
        case USB_INTR_ENABLE_ADDR:
            (usb->uier = val);
            break;
        case USB_INTR_MASK_ADDR:
            (usb->uimsk = val);
            break;
        case USB_INTR_SPRSMSK_ADDR:
            (usb->sprsmsk = val);
            break;
        case USB_INTR_SPRSIE_ADDR:
            (usb->sprsie = val);
            break;
        case USB_INTR_ACK_ADDR:
            (usb->uiar = val);
            break;
        case USB_GLB_STATE_ADDR:
            (usb->glb_state = val);
            break;
        case USB_FRM_NUM_L_ADDR:
            (usb->frm_num_l = val);
            break;
        case USB_FRM_NUM_H_ADDR:
            (usb->frm_num_h = val);
            break;
        case USB_INTR_STATUS_ADDR:
        case USB_INTR_SPRSR_ADDR:
        case USB_INTR_UNUSED_ADDR:
        default:
            avr_error ("Bad address: 0x%04x", addr);
    }
}

static void
usb_intr_reset (VDevice *dev)
{
    USBInter_T *usb = (USBInter_T *)dev;

    usb->sprsr = 0;
    usb->uisr = 0;
}

static char *
usb_intr_reg_name (VDevice *dev, int addr)
{
    switch (addr - vdev_get_base (dev))
    {
        case USB_INTR_ENABLE_ADDR:
            return ("UIER");
        case USB_INTR_MASK_ADDR:
            return ("UIMSK");
        case USB_INTR_STATUS_ADDR:
            return ("UISR");
        case USB_INTR_SPRSIE_ADDR:
            return ("SPRSIE");
        case USB_INTR_SPRSR_ADDR:
            return ("SPRSR");
        case USB_GLB_STATE_ADDR:
            return ("GLB_STATE");
        case USB_FRM_NUM_L_ADDR:
            return ("FRM_NUM_L");
        case USB_FRM_NUM_H_ADDR:
            return ("FRM_NUM_H");
        case USB_INTR_SPRSMSK_ADDR:
            return ("SPRSMSK");
        case USB_INTR_ACK_ADDR:
            return ("UIAR");
        case USB_INTR_UNUSED_ADDR:
            return ("UNUSED");
        default:
            avr_error ("Bad address: 0x%04x", addr);
    }
    return NULL;                /* will never get here */
}

/*****************************************************************************\
 *
 * USB  
 *
\*****************************************************************************/

static uint8_t usb_read (VDevice *dev, int addr);
static void usb_write (VDevice *dev, int addr, uint8_t val);
static void usb_reset (VDevice *dev);
static char *usb_reg_name (VDevice *dev, int addr);

/** \brief Allocate a new USB structure. */

USB_T *
usb_new (void)
{
    USB_T *usb;

    usb = avr_new (USB_T, 1);
    usb_construct (usb);
    class_overload_destroy ((AvrClass *)usb, usb_destroy);

    return usb;
}

/** \brief Constructor for new USB object. */

void
usb_construct (USB_T *usb)
{
    char *name = "USB";

    if (usb == NULL)
        avr_error ("passed null ptr");

    vdev_construct ((VDevice *)usb, name, USB_BASE, USB_SIZE, usb_read,
                    usb_write, usb_reset, usb_reg_name);

    usb_reset ((VDevice *)usb);
}

/** \brief Destructor for USB object. */

void
usb_destroy (void *usb)
{
    if (usb == NULL)
        return;

    vdev_destroy (usb);
}

static uint8_t
usb_read (VDevice *dev, int addr)
{
    USB_T *usb = (USB_T *)dev;

    switch (addr - vdev_get_base (dev))
    {
        case USB_UOVCER_ADDR:
            return usb->uovcer;
        case USB_HADDR_ADDR:
            return usb->haddr;
        case USB_FADDR_ADDR:
            return usb->faddr;
        case USB_HSTR_ADDR:
            return usb->hstr;
        case USB_HPCON_ADDR:
            return usb->hpcon;
        case USB_ICSR_ADDR:
            return usb->icsr;
        case USB_FENDP5_CNTR_ADDR:
            return usb->fendp5_cntr;
        case USB_FENDP4_CNTR_ADDR:
            return usb->fendp4_cntr;
        case USB_FENDP3_CNTR_ADDR:
            return usb->fendp3_cntr;
        case USB_FENDP2_CNTR_ADDR:
            return usb->fendp2_cntr;
        case USB_FENDP1_CNTR_ADDR:
            return usb->fendp1_cntr;
        case USB_FENDP0_CNTR_ADDR:
            return usb->fendp0_cntr;
        case USB_HENDP1_CNTR_ADDR:
            return usb->hendp1_cntr;
        case USB_HENDP0_CNTR_ADDR:
            return usb->hendp0_cntr;
        case USB_FCSR5_ADDR:
            return usb->fcsr5 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_FCSR4_ADDR:
            return usb->fcsr4 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_FCSR3_ADDR:
            return usb->fcsr3 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_FCSR2_ADDR:
            return usb->fcsr2 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_FCSR1_ADDR:
            return usb->fcsr1 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_FCSR0_ADDR:
            usb->fcsr0 = usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
            if (usb->fcsr0 & RX_SETUP)
                usb->fbyte_cnt0 = 10;
            return usb->fcsr0;
        case USB_HCSR0_ADDR:
            usb->hcsr0 = usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
            if (usb->hcsr0 & RX_SETUP)
                usb->hbyte_cnt0 = 10;
            return usb->hcsr0;
        case USB_FCAR5_ADDR:
            return usb->fcar5;
        case USB_FCAR4_ADDR:
            return usb->fcar4;
        case USB_FCAR3_ADDR:
            return usb->fcar3;
        case USB_FCAR2_ADDR:
            return usb->fcar2;
        case USB_FCAR1_ADDR:
            return usb->fcar1;
        case USB_FCAR0_ADDR:
            return usb->fcar0;
        case USB_HCAR0_ADDR:
            return usb->hcar0;
        case USB_HPSTAT1_ADDR:
            return usb->hpstat1;
        case USB_HPSTAT2_ADDR:
            return usb->hpstat2;
        case USB_HPSTAT3_ADDR:
            return usb->hpstat3;
        case USB_HPSTAT4_ADDR:
            return usb->hpstat4;
        case USB_HPSTAT5_ADDR:
            return usb->hpstat5;
        case USB_HPSTAT6_ADDR:
            return usb->hpstat6;
        case USB_HPSTAT7_ADDR:
            return usb->hpstat7;
        case USB_HPSTAT8_ADDR:
            return usb->hpstat8;
        case USB_PSTATE1_ADDR:
            return usb->pstate1;
        case USB_PSTATE2_ADDR:
            return usb->pstate2;
        case USB_PSTATE3_ADDR:
            return usb->pstate3;
        case USB_PSTATE4_ADDR:
            return usb->pstate4;
        case USB_PSTATE5_ADDR:
            return usb->pstate5;
        case USB_PSTATE6_ADDR:
            return usb->pstate6;
        case USB_PSTATE7_ADDR:
            return usb->pstate7;
        case USB_PSTATE8_ADDR:
            return usb->pstate8;
        case USB_HPSCR1_ADDR:
            return usb->hpscr1;
        case USB_HPSCR2_ADDR:
            return usb->hpscr2;
        case USB_HPSCR3_ADDR:
            return usb->hpscr3;
        case USB_HPSCR4_ADDR:
            return usb->hpscr4;
        case USB_HPSCR5_ADDR:
            return usb->hpscr5;
        case USB_HPSCR6_ADDR:
            return usb->hpscr6;
        case USB_HPSCR7_ADDR:
            return usb->hpscr7;
        case USB_HPSCR8_ADDR:
            return usb->hpscr8;
        case USB_FBYTE_CNT5_ADDR:
            return usb->fbyte_cnt5;
        case USB_FBYTE_CNT4_ADDR:
            return usb->fbyte_cnt4;
        case USB_FBYTE_CNT3_ADDR:
            return usb->fbyte_cnt3;
        case USB_FBYTE_CNT2_ADDR:
            return usb->fbyte_cnt2;
        case USB_FBYTE_CNT1_ADDR:
            return usb->fbyte_cnt1;
        case USB_FBYTE_CNT0_ADDR:
            return usb->fbyte_cnt0;
        case USB_HBYTE_CNT0_ADDR:
            return usb->hbyte_cnt0;
        case USB_FDR5_ADDR:
            return usb->fdr5 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_FDR4_ADDR:
            return usb->fdr4 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_FDR3_ADDR:
            return usb->fdr3 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_FDR2_ADDR:
            return usb->fdr2 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_FDR1_ADDR:
            return usb->fdr1 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_FDR0_ADDR:
            return usb->fdr0 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        case USB_HDR0_ADDR:
            return usb->hdr0 =
                usb_port_rd (usb_reg_name ((VDevice *)usb, addr));
        default:
            avr_error ("Bad address: 0x%04x", addr);
    }
    return 0;                   /* will never get here */
}

static void
usb_write (VDevice *dev, int addr, uint8_t val)
{
    USB_T *usb = (USB_T *)dev;

    switch (addr - vdev_get_base (dev))
    {
        case USB_UOVCER_ADDR:
            (usb->uovcer = val);
            break;
        case USB_HADDR_ADDR:
            (usb->haddr = val);
            break;
        case USB_FADDR_ADDR:
            (usb->faddr = val);
            break;
        case USB_HSTR_ADDR:
            (usb->hstr = val);
            break;
        case USB_HPCON_ADDR:
            (usb->hpcon = val);
            break;
        case USB_FENDP5_CNTR_ADDR:
            usb->fendp5_cntr = val;
            break;
        case USB_FENDP4_CNTR_ADDR:
            usb->fendp4_cntr = val;
            break;
        case USB_FENDP3_CNTR_ADDR:
            usb->fendp3_cntr = val;
            break;
        case USB_FENDP2_CNTR_ADDR:
            usb->fendp2_cntr = val;
            break;
        case USB_FENDP1_CNTR_ADDR:
            usb->fendp1_cntr = val;
            break;
        case USB_FENDP0_CNTR_ADDR:
            usb->fendp0_cntr = val;
            break;
        case USB_HENDP1_CNTR_ADDR:
            usb->hendp1_cntr = val;
            break;
        case USB_HENDP0_CNTR_ADDR:
            usb->hendp0_cntr = val;
            break;
        case USB_FCAR5_ADDR:
            usb->fcar5 = val;
            usb->fcsr5 &= ~val;
            (usb->fbyte_cnt5) = 0;
            break;
        case USB_FCAR4_ADDR:
            usb->fcar4 = val;
            usb->fcsr4 &= ~val;
            (usb->fbyte_cnt4) = 0;
            break;
        case USB_FCAR3_ADDR:
            usb->fcar3 = val;
            usb->fcsr3 &= ~val;
            (usb->fbyte_cnt3) = 0;
            break;
        case USB_FCAR2_ADDR:
            usb->fcar2 = val;
            usb->fcsr2 &= ~val;
            (usb->fbyte_cnt2) = 0;
            break;
        case USB_FCAR1_ADDR:
            usb->fcar1 = val;
            usb->fcsr1 &= ~val;
            (usb->fbyte_cnt1) = 0;
            break;
        case USB_FCAR0_ADDR:
            usb->fcar0 = val;
            usb->fcsr0 &= ~val;
            (usb->fbyte_cnt0) = 0;
            break;
        case USB_HCAR0_ADDR:
            usb->hcar0 = val;
            usb->hcsr0 &= ~val;
            (usb->hbyte_cnt0) = 0;
            break;
        case USB_FDR5_ADDR:
            usb->fdr5 = val;
            (usb->fbyte_cnt5)++;
            usb_port_wr (usb_reg_name ((VDevice *)usb, addr), val);
            break;
        case USB_FDR4_ADDR:
            usb->fdr4 = val;
            (usb->fbyte_cnt4)++;
            usb_port_wr (usb_reg_name ((VDevice *)usb, addr), val);
            break;
        case USB_FDR3_ADDR:
            usb->fdr3 = val;
            (usb->fbyte_cnt3)++;
            usb_port_wr (usb_reg_name ((VDevice *)usb, addr), val);
            break;
        case USB_FDR2_ADDR:
            usb->fdr2 = val;
            (usb->fbyte_cnt2)++;
            usb_port_wr (usb_reg_name ((VDevice *)usb, addr), val);
            break;
        case USB_FDR1_ADDR:
            usb->fdr1 = val;
            (usb->fbyte_cnt1)++;
            usb_port_wr (usb_reg_name ((VDevice *)usb, addr), val);
            break;
        case USB_FDR0_ADDR:
            usb->fdr0 = val;
            (usb->fbyte_cnt0)++;
            usb_port_wr (usb_reg_name ((VDevice *)usb, addr), val);
            break;
        case USB_HDR0_ADDR:
            usb->hdr0 = val;
            (usb->hbyte_cnt0)++;
            usb_port_wr (usb_reg_name ((VDevice *)usb, addr), val);
            break;
        default:
            avr_error ("Bad address: 0x%04x", addr);
    }
}

static void
usb_reset (VDevice *dev)
{
    USB_T *usb = (USB_T *)dev;

    usb->haddr = 0;
    usb->faddr = 0;

    usb->hstr = 0;
    usb->hpcon = 0;

    usb->uovcer = 0;
}

static char *
usb_reg_name (VDevice *dev, int addr)
{
    switch (addr - vdev_get_base (dev))
    {
        case USB_FCAR5_ADDR:
            return "FCAR5";
        case USB_FCAR4_ADDR:
            return "FCAR4";
        case USB_FCAR3_ADDR:
            return "FCAR3";
        case USB_FCAR2_ADDR:
            return "FCAR2";
        case USB_FCAR1_ADDR:
            return "FCAR1";
        case USB_FCAR0_ADDR:
            return "FCAR0";
        case USB_HCAR0_ADDR:
            return "HCAR0";
        case USB_PSTATE1_ADDR:
            return "PSTATE1";
        case USB_PSTATE2_ADDR:
            return "PSTATE2";
        case USB_PSTATE3_ADDR:
            return "PSTATE3";
        case USB_PSTATE4_ADDR:
            return "PSTATE4";
        case USB_PSTATE5_ADDR:
            return "PSTATE5";
        case USB_PSTATE6_ADDR:
            return "PSTATE6";
        case USB_PSTATE7_ADDR:
            return "PSTATE7";
        case USB_PSTATE8_ADDR:
            return "PSTATE8";
        case USB_HPSCR1_ADDR:
            return "HPSCR1";
        case USB_HPSCR2_ADDR:
            return "HPSCR2";
        case USB_HPSCR3_ADDR:
            return "HPSCR3";
        case USB_HPSCR4_ADDR:
            return "HPSCR4";
        case USB_HPSCR5_ADDR:
            return "HPSCR5";
        case USB_HPSCR6_ADDR:
            return "HPSCR6";
        case USB_HPSCR7_ADDR:
            return "HPSCR7";
        case USB_HPSCR8_ADDR:
            return "HPSCR8";
        case USB_HPSTAT1_ADDR:
            return "HPSTAT1";
        case USB_HPSTAT2_ADDR:
            return "HPSTAT2";
        case USB_HPSTAT3_ADDR:
            return "HPSTAT3";
        case USB_HPSTAT4_ADDR:
            return "HPSTAT4";
        case USB_HPSTAT5_ADDR:
            return "HPSTAT5";
        case USB_HPSTAT6_ADDR:
            return "HPSTAT6";
        case USB_HPSTAT7_ADDR:
            return "HPSTAT7";
        case USB_HPSTAT8_ADDR:
            return "HPSTAT8";
        case USB_HPCON_ADDR:
            return "HPCON";
        case USB_HSTR_ADDR:
            return "HSTR";
        case USB_FBYTE_CNT5_ADDR:
            return "FBYTE_CNT5";
        case USB_FBYTE_CNT4_ADDR:
            return "FBYTE_CNT4";
        case USB_FBYTE_CNT3_ADDR:
            return "FBYTE_CNT3";
        case USB_FBYTE_CNT2_ADDR:
            return "FBYTE_CNT2";
        case USB_FBYTE_CNT1_ADDR:
            return "FBYTE_CNT1";
        case USB_FBYTE_CNT0_ADDR:
            return "FBYTE_CNT0";
        case USB_HBYTE_CNT0_ADDR:
            return "HBYTE_CNT0";
        case USB_FDR5_ADDR:
            return "FDR5";
        case USB_FDR4_ADDR:
            return "FDR4";
        case USB_FDR3_ADDR:
            return "FDR3";
        case USB_FDR2_ADDR:
            return "FDR2";
        case USB_FDR1_ADDR:
            return "FDR1";
        case USB_FDR0_ADDR:
            return "FDR0";
        case USB_HDR0_ADDR:
            return "HDR0";
        case USB_FCSR5_ADDR:
            return "FCSR5";
        case USB_FCSR4_ADDR:
            return "FCSR4";
        case USB_FCSR3_ADDR:
            return "FCSR3";
        case USB_FCSR2_ADDR:
            return "FCSR2";
        case USB_FCSR1_ADDR:
            return "FCSR1";
        case USB_FCSR0_ADDR:
            return "FCSR0";
        case USB_HCSR0_ADDR:
            return "HCSR0";
        case USB_FENDP5_CNTR_ADDR:
            return "FENDP5_CNTR";
        case USB_FENDP4_CNTR_ADDR:
            return "FENDP4_CNTR";
        case USB_FENDP3_CNTR_ADDR:
            return "FENDP3_CNTR";
        case USB_FENDP2_CNTR_ADDR:
            return "FENDP2_CNTR";
        case USB_FENDP1_CNTR_ADDR:
            return "FENDP1_CNTR";
        case USB_FENDP0_CNTR_ADDR:
            return "FENDP0_CNTR";
        case USB_HENDP1_CNTR_ADDR:
            return "HENDP1_CNTR";
        case USB_HENDP0_CNTR_ADDR:
            return "HENDP0_CNTR";
        case USB_FADDR_ADDR:
            return "FADDR";
        case USB_HADDR_ADDR:
            return "HADDR";
        case USB_ICSR_ADDR:
            return "ICSR";
        case USB_UOVCER_ADDR:
            return "UOVCER";
        default:
            avr_error ("Bad address: 0x%04x", addr);
    }
    return NULL;
}

uint8_t
usb_port_rd (char *name)
{
    int data;
    char line[80];

    while (1)
    {
        fprintf (stderr, "\nEnter a byte of hex data to read into %s: ",
                 name);

        /* try to read in a line of input */
        if (fgets (line, sizeof (line), stdin) == NULL)
            continue;

        /* try to parse the line for a byte of data */
        if (sscanf (line, "%x\n", &data) != 1)
            continue;

        break;
    }
    return (uint8_t) (data & 0xff);
}

void
usb_port_wr (char *name, uint8_t val)
{
    fprintf (stderr, "wrote 0x%02x to %s\n", val, name);
}
