#include <vector>
#include <boost/bind.hpp>
#include "tcpmessageserverconnection.h"
#include "messagebasictypes.h"
#include <iostream>
using namespace std;

TCPMessageServerConnection::TCPMessageServerConnection(asio::io_service& io_service,
    TCPMessageServerConnectionManager& manager, ReceiveMessageSignal & receiveMessageSignal)
    : socket_(io_service), tcpMessageServerConnectionManager(manager), receiveMessageSignal(receiveMessageSignal)
{
  sendQueueCurrentlySending=false;
}


asio::ip::tcp::socket& TCPMessageServerConnection::socket()
{
  return socket_;
}


void TCPMessageServerConnection::start()
{
  asio::async_read(socket_,
                   asio::buffer(data, sizeof(uint32)),
                   asio::transfer_at_least(sizeof(uint32)),
                   boost::bind(&TCPMessageServerConnection::handleReadMessageSize, this, asio::placeholders::error, asio::placeholders::bytes_transferred));
}


void TCPMessageServerConnection::stop()
{
  socket_.close();
}


void TCPMessageServerConnection::handleReadMessageSize(const asio::error_code& err, size_t length)
{
  if (!err)
  {
    // cout << "handleReadMessageSize length=" << length << endl;
    Message sizeMessage(length,data);
    uint32 messageSizeTmp;
    Msg::popFrontuint32(sizeMessage, messageSizeTmp);
    messageSize=messageSizeTmp;

    asio::async_read(socket_,
                     asio::buffer(data, messageSize),
                     asio::transfer_at_least(messageSize),
                     boost::bind(&TCPMessageServerConnection::handleReadMessage, this, asio::placeholders::error, asio::placeholders::bytes_transferred));
  }
  else if (err != asio::error::operation_aborted)
  {
    tcpMessageServerConnectionManager.stop(shared_from_this());
  }
}


void TCPMessageServerConnection::handleReadMessage(const asio::error_code& err, size_t length)
{
  if (!err)
  {
    Message message(length,data);
    Message returnMessage;

    // call server functions
    receiveMessageSignal(returnMessage,message);

    queueAndSendMessageSlot(returnMessage);

    // read next size
    asio::async_read(socket_,
                     asio::buffer(data, sizeof(uint32)),
                     asio::transfer_at_least(sizeof(uint32)),
                     boost::bind(&TCPMessageServerConnection::handleReadMessageSize, this, asio::placeholders::error, asio::placeholders::bytes_transferred));
  }
  else if (err != asio::error::operation_aborted)
  {
    tcpMessageServerConnectionManager.stop(shared_from_this());
  }
}


void TCPMessageServerConnection::queueAndSendMessageSlot(Message & message)
{

  if (sendQueue.size()<500) // FIXME determine queue size
  {
    sendQueue.push_back(message);
    Msg::pushFrontint32(sendQueue.back(),message.size());
  }
  startNewTransmission();
}


void TCPMessageServerConnection::startNewTransmission()
{
  if ((false==sendQueueCurrentlySending) && (sendQueue.size()>0))
  {
    Message message=sendQueue.front();
    sendQueueCurrentlySending=true;
    asio::async_write(socket_,asio::buffer(message.getDataPtr(), message.size()),
                      boost::bind(&TCPMessageServerConnection::handleWriteMessage, this, asio::placeholders::error));
  }
}


void TCPMessageServerConnection::handleWriteMessage(const asio::error_code& err)
{
  // cout << "UDPMessageClient::handleSendTo" << endl;
  // cout << "sendQueue.front().size()=" << sendQueue.front().size() << endl;

  if (!err)
  {
    sendQueue.pop_front();
    sendQueueCurrentlySending=false;
    startNewTransmission();
  }
  else
  {
    cout << "TCPMessageServerConnection::handleWriteMessage error: " << err.message() << endl;
  }
}


void TCPMessageServerConnectionManager::start(connection_ptr c)
{
  connectionPtrSet.insert(c);
  c->start();
}


void TCPMessageServerConnectionManager::stop(connection_ptr c)
{
  connectionPtrSet.erase(c);
  c->stop();
}


void TCPMessageServerConnectionManager::stopAll()
{
  std::for_each(connectionPtrSet.begin(), connectionPtrSet.end(),
                boost::bind(&TCPMessageServerConnection::stop, _1));
  connectionPtrSet.clear();
}
