#ifndef _protocoltimer_h
#define _protocoltimer_h

#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/bind.hpp>
#include <asio.hpp>


class ProtocolIO
{
public:
  ProtocolIO(asio::io_service & ioservice) : ioservice(ioservice)
  {
  }


  boost::posix_time::ptime currentTimeUTC()
  {
    return boost::posix_time::microsec_clock::universal_time();
  }


  asio::io_service & ioservice;
};


class ProtocolTimer
{
private:
  class TimerEventHandlerBase
  {
  public:
    virtual void operator()()=0;
  };


  template <class TClass>
  class TimerEventHandler : public TimerEventHandlerBase
  {
  private:
    TClass localHandler;

  public:
    TimerEventHandler(TClass handler) : localHandler(handler)
    {
    };


    virtual void operator()()
    {
      localHandler();
    };
  };




  asio::deadline_timer timer;


  void timerExpiredEvent(const asio::error_code& err)
  {
    if (asio::error::operation_aborted==err)
    {
    }
    else
    {
      if (myHandlerBase)
      {
        (*myHandlerBase)();
      }
    }
  }


  TimerEventHandlerBase * myHandlerBase;


public:
  ProtocolTimer(ProtocolIO & protocolIO) : timer(protocolIO.ioservice)
  {
    myHandlerBase=0;
  }


  ~ProtocolTimer()
  {
    if (myHandlerBase)
    {
      delete myHandlerBase;
    }
  }


  template <typename WaitHandler>
  void startAlarmAt(const boost::posix_time::ptime & time, WaitHandler handler)
  {
    if (myHandlerBase)
    {
      delete myHandlerBase;
    }
    myHandlerBase = new TimerEventHandler<WaitHandler>(handler);

    timer.expires_at(time);
    timer.async_wait(boost::bind(&ProtocolTimer::timerExpiredEvent, this, asio::placeholders::error));
  }


  template <typename WaitHandler>
  void startAlarmAfter(const boost::posix_time::time_duration & expiry_time, WaitHandler handler)
  {
    if (myHandlerBase)
    {
      delete myHandlerBase;
    }
    myHandlerBase = new TimerEventHandler<WaitHandler>(handler);

    timer.expires_from_now(expiry_time);
    timer.async_wait(boost::bind(&ProtocolTimer::timerExpiredEvent, this, asio::placeholders::error));
  }


  void stop()
  {
    timer.cancel();
  }
};


#endif
