/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
*/

#define MAIN
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#ifdef HAVE_FFMPEG
#include <libavformat/avformat.h>
#endif
#include "siril.h"
#include "proto.h"
#include "conversion.h"
#include "callbacks.h"
#include "registration.h"

/* the global variables of the whole project */
cominfo com;	// the main data struct
fits gfit;	// currently loaded image
fits wfit[5];	// used for temp files, can probably be replaced by local variables
GtkBuilder *builder;	// get widget references anywhere
void initialize_scrollbars();


char *siril_glade_sources[] = {
	"siril3.glade",
	PACKAGE_DATA_DIR"/siril3.glade",
	"/usr/share/siril/siril3.glade",
	"/usr/local/share/siril/siril3.glade",
};

void usage(const char *command) {
    printf("\nUsage:  %s [OPTIONS]... \n\n", command);
    puts("-i                      With init file name in argument. Start Siril.");
    puts("-f (or --format)        Print all supported image file formats (depending on the libraries you've installed)");
    puts("-v (or --version)       Print program name and version and exit");
    puts("-h (or --help)          This text");
}

int main(int argc, char *argv[]) {
	char c;
	int i;
	extern char *optarg;
	extern int opterr;
	
	setenv("LC_NUMERIC", "C", 1);		// avoid possible bugs using french separator ","
	setenv("LANG", "C", 1);				// on french (or other) system, it avoids the mix between languages
	opterr = 0;
	memset(&com, 0, sizeof(struct cominf));	// needed?
	com.initfile = NULL;

	while(1) {
		c=getopt(argc, argv, "i:hfv");
		if (c=='?') {
			for(i=1; i<argc; i++) {
				if(argv[i][1] == '-') {
					if(!strcmp(argv[i], "--version"))
						c='v';
					else if(!strcmp(argv[i], "--help"))
						c='h';				
					else if(!strcmp(argv[i], "--format"))
						c='f';					
					else {
						usage(argv[0]);
						exit(1);
					}
				}
			}
		}

		if (c==-1) break;
		switch(c) {
			case 'i':
				com.initfile = strdup(optarg);
				break;
			case 'v':
				fprintf(stdout, "%s %s\n", PACKAGE, VERSION);
				exit(1);
				break;
			case 'f':
				list_format_available();
				exit(1);
				break;
			default:
				fprintf(stderr, "unknown command line parameter `%c'\n", c);
			case 'h':
				usage(argv[0]);
				exit(1);
		}
	}

	gtk_init (&argc, &argv);

	/* try to load the glade file, from the sources defined above */
	builder = gtk_builder_new();
	i = 0;
	do {
		GError *err = NULL;
		if (gtk_builder_add_from_file (builder, siril_glade_sources[i], &err)) {
			fprintf(stdout, "successfully loaded %s\n", siril_glade_sources[i]);
			break;
		}
		// fprintf(stderr, "error: %s\n", err->message);
		g_error_free(err);
		i++;
	} while (i < sizeof(siril_glade_sources)/sizeof(char *));
	if (i == sizeof(siril_glade_sources) / sizeof(char *)) {
		fprintf(stderr, "siril3.glade was not found or contains errors, cannot render GUI. Exiting.\n");
		exit(1);
	}
	gtk_builder_connect_signals (builder, NULL);

	/* image on buttons: remove padding between image and button border */
	const gchar styledata[] = "* {"
		"-GtkButton-default-border : 0; "
		"-GtkButton-default-outside-border : 0; "
		"-GtkButton-inner-border: 0; "
		"-GtkWidget-focus-line-width : 0; "
		"-GtkWidget-focus-padding : 0; "
		"padding: 0; "
		"}";
	GtkCssProvider *provider = gtk_css_provider_new ();
	gtk_css_provider_load_from_data (provider, styledata, -1, NULL);
	gtk_style_context_add_provider (gtk_widget_get_style_context(lookup_widget("neg_button")),
			GTK_STYLE_PROVIDER (provider), GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
	gtk_style_context_add_provider (gtk_widget_get_style_context(lookup_widget("colormap_button")),
			GTK_STYLE_PROVIDER (provider), GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
	gtk_style_context_add_provider (gtk_widget_get_style_context(lookup_widget("mirrorx_button")),
			GTK_STYLE_PROVIDER (provider), GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
	gtk_style_context_add_provider (gtk_widget_get_style_context(lookup_widget("mirrory_button")),
			GTK_STYLE_PROVIDER (provider), GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
	gtk_style_context_add_provider (gtk_widget_get_style_context(lookup_widget("histogram_button")),
			GTK_STYLE_PROVIDER (provider), GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
	gtk_style_context_add_provider (gtk_widget_get_style_context(lookup_widget("colormap_button")),
			GTK_STYLE_PROVIDER (provider), GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
	gtk_style_context_add_provider (gtk_widget_get_style_context(lookup_widget("seqlist_button")),
			GTK_STYLE_PROVIDER (provider), GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
			
	/* Create tags associated with the buffer for the output text. */
	GtkTextBuffer *tbuf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(gtk_builder_get_object(builder, "output")));
	/* Tag with weight bold and tag name "bold" . */
	gtk_text_buffer_create_tag (tbuf, "bold", "weight", PANGO_WEIGHT_BOLD, NULL);
	/* Tag with style normal */
	gtk_text_buffer_create_tag (tbuf, "normal", "weight", PANGO_WEIGHT_NORMAL, NULL);
	/* Couleur Tags */
	gtk_text_buffer_create_tag (tbuf, "red", "foreground", "#e72828", NULL);
	gtk_text_buffer_create_tag (tbuf, "salmon", "foreground", "#ff9898", NULL);
	gtk_text_buffer_create_tag (tbuf, "green", "foreground", "#01b301", NULL);
	gtk_text_buffer_create_tag (tbuf, "blue", "foreground", "#7a7af8", NULL);
	

	/* activate accelerators (keyboard shortcut in GTK language) */
	GtkAccelGroup *accel = GTK_ACCEL_GROUP(gtk_builder_get_object(builder, "accelgroup1"));
	gtk_widget_add_accelerator(lookup_widget("exit"), "activate", accel,
			      GDK_KEY_q, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE); 
	//g_object_unref (G_OBJECT(builder));
	
	siril_log_color_message("Welcome to %s v%s\n", "bold", PACKAGE, VERSION);

	/* initialize converters (utilities used for different image types importing) */
	initialize_converters();

	/* initialize combobox of supported raw */
	initialize_combocamera();
	initialize_default_extension();

	/* initializing internal structures with widgets (drawing areas) */
	com.vport[RED_VPORT] = lookup_widget("drawingarear");
	com.vport[GREEN_VPORT] = lookup_widget("drawingareag");
	com.vport[BLUE_VPORT] = lookup_widget("drawingareab");
	com.vport[RGB_VPORT] = lookup_widget("drawingareargb");
	com.preview_area[0] = lookup_widget("drawingarea_preview1");
	com.preview_area[1] = lookup_widget("drawingarea_preview2");
	com.cvport = RED_VPORT;
	com.show_excluded = TRUE;
	com.selected_star = -1;
	com.star_is_seqdata = FALSE;
	com.stars = NULL;
	com.uniq = NULL;
	com.grad = NULL;
	com.grad_boxes_drawn = TRUE;
	com.color = NORMAL_COLOR;
	for (i=0; i<MAXVPORT; i++)
		com.buf_is_dirty[i] = TRUE;
	initialize_remap();
	memset(&com.selection, 0, sizeof(rectangle));
	memset(com.layers_hist, 0, sizeof(com.layers_hist));
	initialize_scrollbars();
	init_mouse();

	/* Fill the about window */
	gtk_label_set_text (GTK_LABEL(lookup_widget("aboutlabel")), PACKAGE " v" VERSION );
	char *markup = g_markup_printf_escaped ("<a href=\"%s\">Visit the Siril website</a>", "http://free-astro.vinvin.tf/index.php/Siril");
	gtk_label_set_markup (GTK_LABEL(lookup_widget("labelurl")), markup);
	g_free(markup);
	/* Select combo boxes that trigger some text display or other things */
	gtk_combo_box_set_active(GTK_COMBO_BOX(gtk_builder_get_object(builder, "comboboxstack_methods")), 0);
	gtk_combo_box_set_active(GTK_COMBO_BOX(gtk_builder_get_object(builder, "comboboxstacksel")), 0);

	/* initialize the com struct and zoom level */
	com.drawn = FALSE;
	com.sliders = MINMAX;
	com.zoom_value = ZOOM_DEFAULT;
	zoomcombo_update_display_for_zoom();
	
	/* initialize comboboxs of extraction background */
	GtkComboBox *order = GTK_COMBO_BOX(gtk_builder_get_object(builder, "combo_polyorder"));
	gtk_combo_box_set_active(order, POLY_4);
	GtkComboBox *grad_inter = GTK_COMBO_BOX(gtk_builder_get_object(builder, "combobox_gradient_inter"));
	gtk_combo_box_set_active(grad_inter, 0);

	/* initialize sequence-related stuff */
	initialize_sequence(&com.seq, TRUE);
	adjust_sellabel();
#ifdef HAVE_FFMPEG
	av_register_all();
#endif

	/* set default CWD and load init file */
	com.wd = malloc(PATH_MAX+1);	// PATH_MAX may not be available on all systems
	if (!getcwd(com.wd, PATH_MAX)) {
		free(com.wd);
		com.wd = NULL;
	}
	checkinitfile();
	
	/* initialize prepreocess-related stuff */
	initialize_preprocessing();
	
	/* initialize registration methods */
	initialize_registration_methods();
	
	/* initialization of the setting parameters */
	GtkComboBox *binning = GTK_COMBO_BOX(gtk_builder_get_object(builder, "combobinning"));
	gtk_combo_box_set_active(binning, 0);
#ifdef HAVE_LIBRAW
	set_GUI_LIBRAW();
#endif
	
	/* Get CPU number and set the number of threads */
	siril_log_message("Parallel processing %s: Using %d logical processor%s",
#ifdef _OPENMP
			"enabled", omp_get_num_procs(), "s.\n"
#else
			"disabled", 1, ".\n"
#endif
			);

	/* start Siril */
	update_used_memory();
	gtk_main ();
	return 0;
}

void initialize_scrollbars() {
	GtkScrolledWindow *win;
	int i;
	char *vport_names[] = { "r", "g", "b", "rgb" };
	char window_name[32];

	for (i=0; i<sizeof(vport_names)/sizeof(char *); i++) {
		sprintf(window_name, "scrolledwindow%s", vport_names[i]);
		win = GTK_SCROLLED_WINDOW(gtk_builder_get_object(builder, window_name));
		com.hadj[i] = gtk_scrolled_window_get_hadjustment(win);
		g_signal_connect(com.hadj[i], "value-changed", G_CALLBACK(scrollbars_hadjustment_changed_handler), NULL);
		com.vadj[i] = gtk_scrolled_window_get_vadjustment(win);
		g_signal_connect(com.vadj[i], "value-changed", G_CALLBACK(scrollbars_vadjustment_changed_handler), NULL);
	}
}

