=begin
 * Name: SiSU - Simple information Structuring Universe - Structured information, Serialized Units
 * Author: Ralph Amissah
   * http://www.jus.uio.no/sisu
   * http://www.jus.uio.no/sisu/SiSU/download

 * Description: system environment, resource control and configuration details

 * Copyright (C) 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006 Ralph Amissah

 * License: GPL 2 or later

  Summary of GPL 2

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2 of the License, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA

  If you have Internet connection, the latest version of the GPL should be
  available at these locations:
    http://www.fsf.org/licenses/gpl.html
    http://www.gnu.org/copyleft/gpl.html
    http://www.jus.uio.no/sisu/gpl2.fsf

  SiSU was first released to the public on January 4th 2005

  SiSU uses:
  
  *  Standard SiSU markup syntax,
  *  Standard SiSU meta-markup syntax, and the
  *  Standard SiSU object citation numbering and system
  
  © Ralph Amissah 1997, current 2006.
  All Rights Reserved.

 * Ralph Amissah: ralph@amissah.com
                  ralph.amissah@gmail.com

 * Notes: configure rc in ~/.sisu/sisurc.yaml
=end
@@cX=nil
@@current_document=Dir.pwd #nil #''
module SiSU_Env
  require 'fileutils'
  require 'singleton'
  include FileUtils::Verbose
  @@noyaml=false
  class Info_date
    require 'date'
    attr_accessor :dt,:t
    def initialize
      @dt,@t=Date.today.to_s,Time.now
    end
    def week
      w=@t.strftime("%W")
      "#{@t.year}w#{w}"
    end
    def month
      "#{@t.year}#{@t.month}"
    end
    def year
      @t.year
    end
    def weekonly
      #@t.cweek
      @t.strftime("%W")
    end
    def monthonly
      @t.month
    end
  end
  class Info_system
    require 'rbconfig'
    include Singleton
    @@user,@@home,@@hostname,@@pwd,@@sisu_etc,@@host,@@arch,@@rbver,@@dir_arch,@@dir_sitearch,@@dir_bin,@@locale,@@rc=ENV['USER'],ENV['HOME'],ENV['HOSTNAME'],ENV['PWD'],Config::CONFIG['sysconfdir'] + '/sisu',Config::CONFIG['host'],Config::CONFIG['arch'],%x{ruby -v}.strip,Config::CONFIG['archdir'],Config::CONFIG['sitearchdir'],Config::CONFIG['bindir'],%x{locale charmap}.strip,nil # %x{ruby -v}.strip # Config::CONFIG['rb_ver']
    out=Config::CONFIG['localstatedir']
    etc=Config::CONFIG['sysconfdir'] + '/sisu'
    share=Config::CONFIG['datadir'] + '/sisu'
    data=Config::CONFIG['datadir'] + '/sisu-examples/sample'
    processing='_sisu_processing'
    tmp_processing="/tmp/#{processing}"
    user=ENV['USER']
    IMAGES=:images
    SISU_ETC=:sisu_etc
    SISU_SHARE=:sisu_share
    SAMPLE_DATA_PATH=:sample_data_path
    IMAGE_STUB=:image_stub
    STYLESHEET_STUB=:stylesheet_stub
    IMAGE_LOCAL=:image_local
    WEBSERV_PATH=:webserv_path
    WEBSERV_MAN=:webserv_man
    WEBSERV_PHP=:webserv_php
    WEBSERV_CGI=:webserv_cgi
    WEBSERV_RSS=:webserv_rss
    WEBSERV_SQLITE=:webserv_sqlite
    OUTPUT_LOCAL=:output_local
    PROCESSING_PATH=:processing_path
    PROCESSING_PATH_TMP_BASE=:processing_path_tmp_base
    PROCESSING_PATH_TMP=:processing_path_tmp
    PROCESSING_METAVERSE=:processing_metaverse
    PROCESSING_TUNE=:processing_tune
    PROCESSING_LATEX=:processing_latex
    PROCESSING_TEXINFO=:processing_texinfo
    PROCESSING_LOUT=:processing_lout
    PROCESSING_ENCODING=:processing_encoding
    PAPERSIZE=:papersize
    LANGUAGE=:language
    LANGUAGE_CODE=:language_code
    MULTILINGUAL=:multilingual
    CONCORD_MAX=:concord_max
    DIGEST=:digest
    WEBRICK_PORT=:webrick_port
    POSTGRESQL_USER=:postgresql_user
    POSTGRESQL_PORT=:postgresql_port
    SQLITE_USER=:sqlite_user
    SQLITE_PATH=:sqlite_path
    SQLITE_PORT=:sqlite_port
    DEFAULT_DIR={
      IMAGES =>                '_sisu/image',
      SISU_ETC =>              etc,
      SISU_SHARE =>            share,
      SAMPLE_DATA_PATH =>      data,
      IMAGE_STUB =>            '_sisu/image',
      STYLESHEET_STUB =>       '_sisu/css',
      IMAGE_LOCAL =>           @@pwd + '/_sisu/image',
      WEBSERV_PATH =>          out + '/www',
      #WEBSERV_DIR =>          www, # uncomment for urls...
      #WEBSERV_IMAGE =>         out + '/www/_sisu/image',
      WEBSERV_MAN =>           out + '/www/man', #alter
      WEBSERV_PHP =>           out + '/www/php',
      WEBSERV_CGI =>           '/usr/lib/cgi-bin',
      WEBSERV_RSS =>           out + '/www/feed',
      WEBSERV_SQLITE =>        out + '/www/sqlite',
      OUTPUT_LOCAL =>          @@home + '/sisu_www',
      PROCESSING_PATH =>       @@pwd + "/#{processing}",
      PROCESSING_PATH_TMP_BASE => tmp_processing,
      PROCESSING_PATH_TMP =>   "#{tmp_processing}/#{user}",
      PROCESSING_METAVERSE =>  'metaverse',
      PROCESSING_TUNE =>       'tune',
      PROCESSING_LATEX =>      'tex',
      PROCESSING_TEXINFO =>    'texinfo',
      PROCESSING_LOUT =>       'lout',
      PROCESSING_ENCODING =>   'encoding',
      #TEXINFO_STUB =>          'texinfo',
      PAPERSIZE =>             'A4', #A4, US_letter, book_b5, book_a5, US_legal
      LANGUAGE =>              'English',
      LANGUAGE_CODE =>         'en', #change, unecessary duplication though currently used
      MULTILINGUAL =>          true,
      CONCORD_MAX =>           200000,
      DIGEST =>                'md5',
      WEBRICK_PORT =>          8081, #8111,8123,8081
      POSTGRESQL_USER =>       @@user, #'ralph', # change user !!!
      POSTGRESQL_PORT =>       '5432',
      #POSGRESQL_LINKS_PATH => '',
      SQLITE_USER =>           @@user,
      SQLITE_PATH =>           @@user, #??
      SQLITE_PORT =>           '**',
    }
    @@default_dir=DEFAULT_DIR
    @@rc_path = ["#@@pwd/_sisu","#@@home/.sisu",@@sisu_etc] #_sisu/config [depreciated] no longer supported
    attr_accessor :user,:home,:hostname,:pwd,:host,:arch,:rbver,:dir_arch,:dir_sitearch,:dir_bin,:locale,:webserv_path,:webrick_port,:default_dir,:rc_path
    def initialize
      @user,@home,@hostname,@pwd,@sisu_etc,@host,@arch,@rbver,@dir_arch,@dir_sitearch,@dir_bin,@locale,@default_dir,@rc_path=@@user,@@home,@@hostname,@@pwd,@@sisu_etc,@@host,@@arch,@@rbver,@@dir_arch,@@dir_sitearch,@@dir_bin,@@locale,@@default_dir,@@rc_path
      #note rbver is duplicated in Info_version
    end
  end
  class Load
    def initialize(prog,mandatory=false)
      @prog,@mandatory=prog,mandatory
    end
    def prog
      load_prog=false
      $:.each do |reqpath|
        if FileTest.exists?("#{reqpath}/#@prog.rb")
          load_prog=true
          break
        end
      end
      if load_prog and @prog=~/dbi/: require 'dbi' #revisit
      end
      if load_prog: require @prog
      else
        tell=if @mandatory: SiSU_Screen::Ansi.new(@cmd,"module required: #@prog") 
        else                SiSU_Screen::Ansi.new(@cmd,"#@prog load requested") 
        end
        tell.warn
      end
      load_prog
    end
    def prog?
      load_prog=false
      $:.each do |reqpath|
        if FileTest.exists?("#{reqpath}/#@prog.rb"): load_prog=true
          break
        end
      end
      load_prog
    end
  end
  class Get_yaml < Info_system
    include Singleton
    @@rc=nil
    @@noyaml=false
    attr_accessor :yaml
    def initialize
      super()
    end
    def yamlrc
      @rc_path.each do |v|
        if @@noyaml or FileTest.exists?("#{v}/noyaml")
          puts "WARNING - YAML loading switched off, to enable delete the file:\n\t#{v}/noyaml\n\n" unless @@noyaml
          @@noyaml=true
          break
        else
          if FileTest.exists?("#{v}/sisurc.yaml")
            unless @@rc
              require 'yaml'
              @@rc ||= YAML::load(File::open("#{v}/sisurc.yaml"))
            end
            @@rc
            break
          end
        end
      end
      @@rc
    end
  end
  class Env_call
    @@rc,@@fns,@@fnn,@@fnb,@@fnt,@@flv=nil,nil,nil,nil,nil,nil
    attr_accessor :rc,:fnn,:fnb,:fnt,:fnv
    def initialize(fns='')
      super()
      @fns=fns
      @sys=Info_system.instance
      @rc=Get_yaml.instance.yamlrc
      if @fns and @fns != '' and @fns !=@@fns
        @@fns,@@fnn,@@fnb,@@fnt,@@flv=@fns,nil,nil,nil,nil
      end
      if @fns and @fns != '' #watch
        if multilingual
          m=/((.+?)(?:\~\w{2,3})?)\.([_-]?sst|ssm)$/
          @@fnn ||=@fns[m,1]
          @@fnb ||=@fns[m,2]
          @@fnt ||=@fns[m,3]
          @@flv ||=document_language_versions_found[:f]
        else m=/(.+?)\.([_-]?sst|ssm)$/
          @@fnb ||=@fns[m,1]
          @@fnm ||=@fns[m,1]
          @@fnt ||=@fns[m,2]
        end
      end
      @fnn,@fnb,@fnt,@flv=@@fnn,@@fnb,@@fnt,@@flv
    end
    def multilingual
      if defined? @rc['default']['multilingual'] and @rc['default']['multilingual'] != nil:  @rc['default']['multilingual']
      else                                         true #defaults[:multilingual]
      end
    end
    def document_language_versions_found
      @fn=Hash.new
      unless (@fns.nil? or @fns.empty?)
        if multilingual
          m=/((.+?)(?:\~\w{2,3})?)\.([_-]?sst$)/
          @fn[:b],@fn[:m],@fn[:t]=@fns[m,1],@fns[m,2],@fns[m,3]
        else m=/(.+?)\.([_-]?sst$)/
          @fn[:b]=@fn[:m]=@fns[m,1]
          @fn[:t]=@fns[m,2]
        end
      end
      lang=SiSU_Env::Standardise_language.new
      langs=lang.codes
      x=Array.new
      if FileTest.file?("#{@fn[:m]}.#{@fn[:t]}"): x << "#{@fn[:m]}.#{@fn[:t]}"
      end
      x << @fns
      langs.each do |l|
        lng=SiSU_Env::Standardise_language.new(l)
        if FileTest.file?("#{@fn[:m]}~#{lng.code}.#{@fn[:t]}")
          x << "#{@fn[:m]}~#{lng.code}.#{@fn[:t]}"
        elsif FileTest.file?("#{@fn[:m]}~#{lng.name}.#{@fn[:t]}")
          x << "#{@fn[:m]}~#{lng.name}.#{@fn[:t]}"
        end
      end
      @fn[:f]=x.uniq!
      @fn
    end
    def published_manifests?(output_base)
      ob=output_base
      @fn=Hash.new
      @m=Array.new
      unless (@fns.nil? or @fns.empty?)
        if multilingual
          m=/((.+?)(?:\~\w{2,3})?)\.([_-]?sst$)/
          @fn[:b],@fn[:m],@fn[:t]=@fns[m,1],@fns[m,2],@fns[m,3]
          #@fl[:v]=@env.document_language_versions_found
        else m=/(.+?)\.([_-]?sst$)/
          @fn[:b]=@fn[:m]=@fns[m,1]
          @fn[:t]=@fns[m,2]
        end
      end
      lang=SiSU_Env::Standardise_language.new
      langs=lang.codes
      x=Array.new
      if FileTest.file?("#{@fn[:m]}.#{@fn[:t]}"): x << "#{@fn[:m]}.#{@fn[:t]}"
      end
      dir=SiSU_Env::Info_dir.new(@fns)
      @m << {:m => 'sisu_manifest.html', :l => 'English' } #fix later, default language
      langs.each do |l|
        lng=SiSU_Env::Standardise_language.new(l)
        fns_c="#{@fn[:m]}~#{lng.code}.#{@fn[:t]}"
        fns_l="#{@fn[:m]}~#{lng.name}.#{@fn[:t]}"
        if FileTest.file?(fns_c)
          fn_set_lang=SiSU_Env::Standardise_language.new.file_to_language(fns_c)
          lng=fn_set_lang[:l]
          fn=SiSU_Env::Env_call.new(fns_c).lang(fn_set_lang[:c])
          @m << {:m => fn[:manifest], :l => lng }
        elsif FileTest.file?(fns_l)
          fn_set_lang=SiSU_Env::Standardise_language.new.file_to_language(fns_l)
          @fnl=dir.lang_filename(fn_set_lang[:c])
          fn=SiSU_Env::Env_call.new(fns_l).lang(fn_set_lang[:c])
          @m << {:m => fn[:manifest], :l => lng }
        end
      end
      @m.uniq!
      @m
    end
    def filename(code,name,suffix)
      d=SiSU_Env::Info_dir.new(@fns)
      fnl=d.lang_filename(code)
      "#{fnl[:pre]}#{name}#{fnl[:mid]}#{suffix}#{fnl[:post]}"
    end
    def lang(code)
      @fn=Hash.new
      @fn[:html]=        filename(code,'','.html')
      @fn[:concordance]= filename(code,'concordance','.html')
      @fn[:sax]=         filename(code,'sax','.xml')
      @fn[:dom]=         filename(code,'dom','.xml')
      @fn[:docbook]=     filename(code,'docbook','.xml')
      @fn[:xhtml]=       filename(code,'scroll','.xhtml')
      @fn[:pdf_l]=       filename(code,'landscape','.pdf')
      @fn[:pdf_p]=       filename(code,'portrait','.pdf')
      @fn[:toc]=         filename(code,'toc','.html')
      @fn[:doc]=         filename(code,'doc','.html')
      @fn[:index]=       filename(code,'index','.html')
      @fn[:odf]=         filename(code,'opendocument','.odt')
      @fn[:plain]=       filename(code,'plain','.txt')
      @fn[:digest]=      filename(code,'digest','.txt')
      @fn[:metadata]=    filename(code,'metadata','.html') #chk
      @fn[:manifest]=    filename(code,'sisu_manifest','.html')
      #@fn[:kdissert]=    filename(code,'kdissert','.kdi')
      @fn
    end
  end
  class System_call
    @@locale_flag=false
    def initialize(input='',output='',cmd='')
      @input,@output,@cmd=input,output,cmd
      @prog=SiSU_Env::Info_program.new
      @sys=Info_system.instance
    end
    def program_found?(program)
      found=`whereis #{program}`
      state=if found =~/bin\/#{program}\b/: true
      else                                  false
      end
    end
    def locale                                                                   #locales utf8 or other
      unless @@locale_flag
        @@locale_flag=true
        puts @sys.locale #unless @cmd =~/q/                                       # locale info UTF8 etc. KEEP
      end
      @sys.locale
    end
    def file_encoding(filename,cmd='')                                         #file encoding
      program='file'
      if program_found?(program)
        encoding=%x{file #{filename}}.strip
        encoding.gsub!(/#{filename}:(\s+|$)/,'')
        encoding=if encoding and not encoding.empty?: encoding
        else 'UTF-8 assumed, encoding undetermined'
        end
        puts encoding unless cmd =~/q/
        encoding
      else encoding='UTF-8 assumed, file encoding check program unavailable'
      end
    end
    def wc                                                                     #word count
      program='wc'
      if program_found?(program) and locale !~/utf-?8/i: true
      else
        program_ref="(not available)" unless program_found?(program)
        program_ref="(UTF-8)" if locale =~/utf-?8/i
        #puts "\tWARN: #{program}\n\t\t using alternative to word count program #{program_ref}"
        false
      end
    end
    def rcs                                                                    #rcs for document markup data
      program='rcs'
      program_ref="\n\t\tdocument version information requested"
      if program_found?(rcs): true
      else puts "\tWARN: #{program} is not installed #{program_ref}"               #if @cmd =~/v/
        false
      end
    end
    def cvs                                                                    #cvs for document markup data
      program='cvs'
      program_ref="\n\t\tdocument version information requested"
      if program_found?(cvs): true
      else puts "\tWARN: #{program} is not installed #{program_ref}"               #if @cmd =~/v/
        false
      end
    end
    def openssl                                                                #openssl for digests
      program='openssl'
      program_ref="\n\t\tused to generate requested source document identification digest"
      if program_found?(program): true
      else puts "\tWARN: #{program} is not installed #{program_ref}"               #if @cmd =~/v/
        false
      end
    end
    def md5(filename)                                                          #md5 dgst
      program='openssl'
      program_ref="\n\t\tmd5 digest requested"
      if program_found?(program)
        dgst=%x{openssl dgst -md5 #{filename}}.strip
        dgst.scan(/\S+/)
      else puts "\tWARN: #{program} is not installed #{program_ref}"               #if @cmd =~/v/
        false
      end
    end
    def sha256(filename)                                                          #sha dgst
      program='openssl'
      program_ref="\n\t\tsha digest requested"
      if program_found?(program)
        dgst=%x{openssl dgst -sha256 #{filename}}.strip
        dgst.scan(/\S+/)
      else puts "\tWARN: #{program} is not installed #{program_ref}"               #if @cmd =~/v/
        false
      end
    end
    def create_pg_db(dbname_stub=nil)                                          #createdb
      unless dbname_stub
        @pwd ||=Dir.pwd
        m=/.+\/(\S+)/im
        dbname_stub=@pwd[m,1]
      end
      program='createdb'
      program_ref="\n\t\tcreatedb dbname SiSU_#{dbname_stub} #for postgresql database creation"
      if program_found?(program): system(%{createdb SiSU_#{dbname_stub}})
      else puts "\tWARN: #{program} is not available #{program_ref}"               #if @cmd =~/v/
      end
    end
    def relaxng(cmd='')                                                                #trang - convert between different schema languages for XML
      program='trang'
      program_ref="\n\t\tsee http://www.thaiopensource.com/relaxng/trang.html"
      if program_found?(program): system("trang #@input #@output")
      else                        puts "\tWARN: #{program} is not installed #{program_ref}" if cmd =~/V/
      end
    end
    def well_formed?                                                           #tidy - check for well formed xml xhtml etc.
      program=@prog.tidy
      program_ref="\n\t\tsee http://tidy.sourceforge.net/"
      if program_found?(program): system("#{@prog.tidy} -xml #@input > #@output")
      else                        puts "\tWARN: #{program} is not installed #{program_ref}"
      end
    end
    def latex2pdf                                                              #convert from latex to pdf
      prog=Array.new
      prog=['pdflatex','pdfetex','pdftex']
      program_ref="\n\t\tSee http://www.tug.org/applications/pdftex/\n\t\tOn Debian this is is included in tetex-extra"
      @pdfetex_flag=false
      @cmd ||=''
      tell=if @cmd =~/[MVv]/: ''
      else                  '> /dev/null'
      end
      mode='batchmode'
      #mode='nonstopmode'
      prog.each do |program|
        if program_found?(program)
          case program
          when /pdflatex/: system("#{program} -interaction=#{mode} #@input #{tell}\n")
          when /pdfetex/: system("#{program} -interaction=#{mode} -fmt=pdflatex #@input #{tell}\n") # debian specific paramters ?
            #system("#{program} -interaction=batchmode -progname=pdflatex #@input\n")
          when /pdftex/: system("#{program} -interaction=#{mode} -fmt=pdflatex #@input #{tell}\n")
          end
          @pdfetex_flag=true
          break
        end
        unless @pdfetex_flag: puts "\tWARN: none of the following programs are installed: #{program[0]}, #{program[1]}, #{program[2]} is installed. #{program_ref}"
        end
      end
    end
    def makeinfo                                                               #texinfo
      program='makeinfo'
      program_ref="\n\t\tsee http://www.gnu.org/software/texinfo/"
      if program_found?(program): system("makeinfo #@input\n")
      else                        puts "\tWARN: #{program} is not installed #{program_ref}"
      end
    end
    def scp
      program='scp'
      if program_found?(program): system("scp -Cr #@input #@output")
      else                        puts "\tWARN: #{program} not found"
      end
    end
    def rsync(action='')
      program='rsync'
      if program_found?(program)
        vb=if @cmd =~/q/: 'q'
        elsif @cmd =~/v/: 'v'
        else             ''
        end
        msg=''
        msg=" && echo 'OK: #@input -> #@output'" unless @cmd =~/q/
        system("rsync -az#{vb} #{action} #@input #@output #{msg}")
      else puts "\tWARN: #{program} not found"
      end
    end
    def rm
      if @cmd =~/^-Z[mMvVq]*$/:      system("rm -rv #@input")
      elsif @cmd =~/V/:              system("rm -vi #@input")
      elsif @cmd !~/q/:              system("rm -v #@input")
      elsif @cmd =~/q/:              system("rm #{action} #@input")
      else                          puts "\tWARN: operation ignored"
      end
    end
  end
  class Standardise_language
    def initialize(l='')
      @language=l
      if @language.empty?
        #tell=SiSU_Screen::Ansi.new('v',"Setting to system default language")
        #tell.warn #unless cmd =~/[vVM]/
        @language=Info_dir.new.defaults[:language]
      end
    end
    def language
      lang=Hash.new
      case @language
      when /American|^us$/i:                  d,c,l=false,'us','American English'
      when /English|^en$/i:                   d,c,l=false,'en','English'
      when /French|Francais|^fr$/i:           d,c,l=false,'fr','French'
      when /German|^de$/i:                    d,c,l=false,'de','German'
      when /Italian|^it$/i:                   d,c,l=false,'it','Italian'
      when /Spanish|Espanol|^es$/i:           d,c,l=false,'es','Spanish'
      when /Brazilian(?: Portuguese)?|^br$/i: d,c,l=false,'br','Brazilian Portuguese'
      when /Portuguese|^pt$/i:                d,c,l=false,'pt','Portuguese'
      when /Swedish|Svensk|^se$/i:            d,c,l=false,'se','Swedish'
      when /Danish|Dansk|^dk$/i:              d,c,l=false,'dk','Danish'
      when /Finnish|Finsk|Suomi|^fi$/i:       d,c,l=false,'fi','Finnish'
      when /Norwegian|Norsk|^no$/i:           d,c,l=false,'no','Norwegian'
      when /Icelandic|^is$/i:                 d,c,l=false,'is','Icelandic'
      when /Dutch|^nl$/i:                     d,c,l=false,'nl','Dutch'
      when /Estonian|^ee$/i:                  d,c,l=false,'ee','Estonian'
      when /Hungarian|^hu$/i:                 d,c,l=false,'hu','Hungarian'
      when /Polish|^pl$/i:                    d,c,l=false,'pl','Polish'
      when /Romanian|^ro$/i:                  d,c,l=false,'ro','Romanian'
      when /Russian|^ru$/i:                   d,c,l=false,'ru','Russian'
      when /Greek|^gr$/i:                     d,c,l=false,'gr','Greek'
      when /Ukranian|^ua$/i:                  d,c,l=false,'ua','Ukranian'
      when /Turkish|^tr$/i:                   d,c,l=false,'tr','Turkish'
      #when /Serbian/i:                       d,c,l=false,'', 'Serbian'
      #when /Welsh/i:                         d,c,l=false,'', 'Welsh'
      #when /Basque/i:                        d,c,l=false,'', 'Basque'
      #when /Breton/i:                        d,c,l=false,'', 'Breton'
      #when /Catalan/i:                       d,c,l=false,'', 'Catalan'
      #when /Galician/i:                      d,c,l=false,'', 'Galician'
      #when /Saa?mi/i:                        d,c,l=false,'', 'Saami'
      #when /Hebrew/i:                        d,c,l=false,'', 'Hebrew'
      #when /Latin/i:                         d,c,l=false,'', 'Latin'
      #when /Esperanto/i:                     d,c,l=false,'', 'Esperanto'
      when /Slovenian|^si$/i:                 d,c,l=false,'si','Slovenian'
      when /Croatian|^hr$/i:                  d,c,l=false,'hr','Croatian'
      when /Slovak(?:ian)?|^sk$/i:            d,c,l=false,'sk','Slovakian'
      when /Czech|^cz$/i:                     d,c,l=false,'cz','Czech'
      when /Bulgarian|^bg$/i:                 d,c,l=false,'bg','Bulgarian'
      else                                    d,c,l=true,'en','English (default)'
      #else                                   d,c,l=true,'xx','Default'
      end
      lang[:d],lang[:c],lang[:l]=d,c,l
      lang
    end
    def name
      language[:l].downcase
    end
    def title
      language[:l]
    end
    def code
      language[:c]
    end
    def file_to_language(file)
      m=/.+?\~(\w{2,3})\.[_-]?sst$/
      @language=if file =~m : file[m,1]
      else ''
      end
      language
    end
    def codes
      codes=['us','en','fr','de','it','es','br','pt','se','dk','fi','no','is','nl','ee','hu','pl','ro','ru','gr','ua','tr','si','hr','sk','cz','bg']
    end
  end
  class Info_dir < Env_call
    require 'fileutils'
    include FileUtils
    attr_accessor :filename,:sys,:home,:hostname,:user,:env,:rc,:www,:fnb,:fnn,:fnt,:flv,:webserv_path,:output_stub,:webrick_port,:processing,:etc, :yamlrc_dir
    @@image_flag,@@local_image=true,true   #warning on @@image_flag
    @@fb=nil
    def initialize(fns='',md=nil)
      super() #you may not want to re-execute this tatic info so frequently!
      @fns,@md=fns,md
      @env=Env_call.new(fns) if fns
      fnb=if @md and defined? @md.fnb: @md.fnb
      elsif defined? @env.fnb and @env.fnb: @env.fnb
      elsif not @fns.nil? and not @fns.empty?
        m=/(.+)?\.(?:[_-]?sst|ssm)$/im
        @fns[m,1] if not @fns.empty?
      end
      if fnb: @@fb ||=fnb
      end
      @sys=Info_system.instance
      @fnb ||=@@fb #clean up this... used primarily for zap which is not passed normal parameters
      @fixed_websev_root='' # @home
      @pwd=@@pwd||=Dir.pwd
      m=/.+\/(\S+)/im
      @output_stub=@@pwd[m,1]
    end
    def defaults
      @default_dir ||=@sys.default_dir #DEFAULT_DIR
    end
    def home
      @home
    end
    def user
      @sys.user
    end
    def home
      @sys.home
    end
    def hostname
      @sys.hostname
    end
    def pwd
      @sys.pwd
    end
    def etc
      defaults[:sisu_etc]                                                      #live/dynamic
      # @sys.sisu_etc                                                           #broken: live/dynamic
      # defaults[:sisu_etc]                                                     #live/dynamic
    end
    def host
      @sys.host
    end
    def arch
      @sys.arch
    end
    def rbver
      @sys.rbver
    end
    def dir_arch
      @sys.dir_arch
    end
    def dir_sitearch
      @sys.dir_sitearch
    end
    def dir_bin
      @sys.dir_bin
    end
    def locale
      @sys.locale
    end
    def default_dir
      @sys.default_dir
    end
    def rc_path
      @sys.rc_path
    end
    def current_document
      @@current_document||=Dir.pwd
      @@current_document
    end
    def share                                                                 #shared data repository source directory
      defaults[:sisu_share]
    end
    def style
      defaults[:stylesheet_stub]
    end
    def sample_data                                                            #sample data repository source directory
      defaults[:sample_data_path]
    end
    def digest
      if defined? @rc['default']['digest'] and @rc['default']['digest'] != nil
        case @rc['default']['digest']
        when /^sha(?:2|256)?$/: 'sha256'
        when /^md5$/:           'md5'
        else                    'md5'
        end
      else                      'md5'
      end
    end
    def digest_length
      case digest
      when /sha256/: 64
      when /md5/:    32
      else           32
      end
    end
    def digest_pattern
      "[0-9a-f]{#{digest_length}}"
    end
    def yamlrc_info
      rc_path.each do |v|
        if FileTest.exists?("#{v}/sisurc.yaml")
          @yamlrc_dir="#{v}/sisurc.yaml"
          break
        end
      end
      @yamlrc_dir
    end
    def processing_path_tmp_base
      defaults[:processing_path_tmp_base]
    end
    def processing                                                             #processing directory, used/needed for sisu work files, has sub-directories (metaverse,tex etc)
      File.mkpath(defaults[:processing_path_tmp]) unless FileTest.directory?(defaults[:processing_path_tmp])
      path_processing=if defined? @rc['processing']['path']
        File.mkpath("#{home}/#{@rc['processing']['path']}") unless FileTest.directory?("#{home}/#{@rc['processing']['path']}")
        ["#{home}/#{@rc['processing']['path']}",defaults[:processing_path],defaults[:processing_path_tmp]]
      else [defaults[:processing_path],defaults[:processing_path_tmp]]
      end
      @processing=nil
      path_processing.each do |v|                                              #
        if File.writable?("#{v}/.")
          @processing=v
          unless FileTest.directory?(@processing)
            puts "a processing directory (#{@processing}) is being created for use by sisu"
            File.mkpath(@processing) 
          end
          break
        end
      end
      @processing
    end
    def odf_structure
      File.mkpath("#{defaults[:processing_path_tmp]}/odf") unless FileTest.directory?("#{defaults[:processing_path_tmp]}/odf")
      system("
        rm -r #{processing}/odf/*
        cd #{processing}
        tar xzf #{share}/odf/odt.tgz
        cd #@pwd
      ")
      #tar xzf #{etc}/misc/odf/odt.tgz                                       #i find no images in /etc !
      #tar xzf #{share}/odf/odt.tgz                                          #requires new directory structure
    end
    def odf
      "#{processing}/odf"
    end
    def webserv_path                                                          #separation required for webrick which cannot use data_o (different requirements as no file is passed)
      man_path=if defined? @rc['webserv']['path'] and @rc['webserv']['path'] =~/\S\S+/
        man_path=case @rc['webserv']['path']
        when /^\/\S+/:     @rc['webserv']['path']
        when /^\.\/(\S+)/: "#{pwd}/#{$1}"
        when /^~\/(\S+)/:  "#{home}/#{$1}"
        else               "#{home}/#{@rc['webserv']['path']}"
        end
      else nil
      end
      @webserv_path=if defined? man_path and File.writable?("#{man_path}/."): man_path #web server path as configured in rc file
      elsif FileTest.directory?(defaults[:webserv_path]) and File.writable?("#{defaults[:webserv_path]}/.") #web server path default
        defaults[:webserv_path]
      else #create default directory under home and place output there
        File.mkpath(defaults[:output_local]) unless FileTest.directory?(defaults[:output_local])
        defaults[:output_local]
      end
    end
    def webserv_path_dir_stub_ensure
      File.mkpath(webserv_path) unless FileTest.directory?(webserv_path)
      File.mkpath("#{webserv_path}/#@output_stub") unless FileTest.directory?("#{webserv_path}/#@output_stub")
    end
    def data_o                                                                 #web/webserv output directory... subdirectory into which further subdirectories are made based on file names
      sfx_tst=@sfx_fns || @fns #if @fns.empty?
      "#{webserv_path}/#@output_stub" #"#{defaults[:webserv_path]}/#@output_stub"
    end
    def webserv_url                                                           #web url for local webserv (localhost, or hostname)
      sfx_tst=@sfx_fns || @fns #if @fns.empty?
      if webserv_dir and webserv_path =~ /#{webserv_dir}/ #revisit
        "#{webserv_path}/#@output_stub".gsub(/#{webserv_dir}/, "http://#{hostname}/#@output_stub")
      elsif defined? @rc['webserv']['webrick_url'] and @rc['webserv']['webrick_url'] == false
        "file://#{webserv_path}/#@output_stub"
      else "http://#{hostname}:#{webrick_port}/#@output_stub"
      end
    end
    def webserv_url_base                                                           #web url for local webserv (localhost, or hostname)
      sfx_tst=@sfx_fns || @fns #if @fns.empty?
      if webserv_dir and webserv_path =~ /#{webserv_dir}/ #revisit
        "#{webserv_path}/#@output_stub".gsub(/#{webserv_dir}/, "http://#{hostname}")
      elsif defined? @rc['webserv']['webrick_url'] and @rc['webserv']['webrick_url'] == false
        "file://#{webserv_path}"
      else "http://#{hostname}:#{webrick_port}"
      end
    end
    def webrick_url                                                           #web url for local webserv (localhost, or hostname)
      "http://localhost:#{webrick_port}/#@output_stub"
      #"http://#{hostname}:#{webrick_port}/#@output_stub"
    end
    def webrick_base                                                           #web url for local webserv (localhost, or hostname)
      "http://localhost:#{webrick_port}"
      #"http://#{hostname}:#{webrick_port}"
    end
    def webserv_dir                                                           #fixed/hard path to /www web/presentation directory, on Debian /var/www subdirectories are created within it, depending on markup directory stub-name (last segment of markup directory name)
      defaults[:webserv_dir]
    end
    def webserv_image                                                         #web/presentation directory, subdirectories are created within it, depending on markup directory stub-name (last segment of markup directory name)
      images=if defined? @rc['webserv']['images']
        @rc['webserv']['images'] 
      else defaults[:images]
      end
      "#{webserv_path}/#{images}"
    end
    def output_links_path                                                  #introduced for sql db links to file system objects
      pg_lnk=Hash.new
      pg_lnk[:url_root]=if defined? @rc['webserv']['url_root']: "#{@rc['webserv']['url_root']}/#{stub}"
      else "http://localhost/#{stub}"
      end
      pg_lnk[:hostname]="http://#{hostname}"
      pg_lnk[:stub]=stub
      pg_lnk[:dir]="#{webserv_path}/#{stub}"
      pg_lnk[:dir_url]="file://#{webserv_path}/#{stub}"
      pg_lnk[:localhost]="http://localhost/#{stub}"
      pg_lnk[:local]="http://#{hostname}/#@output_stub"
      pg_lnk[:remote]=pg_lnk[:url_root]
      pg_lnk[:webrick]=webrick_url     #see webserv_url (used extensively) and decide on how to tidy logic
      pg_lnk
    end
    def tell_output_location
      output_type=if defined? @rc['show_output_on'] and  @rc['show_output_on'] =~/^(?:filesystem|webserv|(?:local|remote)(?:_webserv)?|webrick)/
        @rc['show_output_on']
      else 'filesystem'
      end
      output=case output_type
      when /^filesystem(?:_url)?/:           output_links_path[:dir_url]
      when /^remote(?:_webserv)?/:           output_links_path[:remote]
      when /^(?:webserv|local_webserv)/:     output_links_path[:local]
      when /^local(:\d+)/:                   output_links_path[:hostname] + $1 + '/' + output_links_path[:stub]
      when /^localhost(:\d+)/:               output_links_path[:localhost] + $1 +  '/' + output_links_path[:stub]
      when /^localhost/:                     output_links_path[:localhost]
      when /^webrick/:                       output_links_path[:webrick]
      when /^path/:                          output_links_path[:dir]
      else                                   output_links_path[:dir]
      end
    end
    def tell_output_path
      output_links_path[:dir]
    end
    def scripts                                                                #used only to include tla version info
      if defined? @rc['project']['path']: "#{home}/#{@rc['project']['path']}"
      end
    end
    def image_source                                                           #image repository source directory
      image_path=if defined? @rc['image']['path'] and defined? @rc['image']['public']
        path="#{@rc['image']['path']}"
        "#{path}/#{@rc['image']['public']}"
      else
        "#{share}/image"
      end
    end
    def image_source_tex                                                       #image repository source directory
      image_path=if defined? @rc['image']['path'] and defined? @rc['image']['public']
        path="#{@rc['image']['path']}"
        "#{path}/#{@rc['image']['public']}"
      else
        image=defaults[:image_stub]
        "#{share}/image"
      end
    end
    def url_images                                                             #ex defaults url.path_png
      '../_sisu/image'
    end
    def url_images_external                                                    #ex defaults url.path_png
      '../_sisu/image_external'
    end
    def url_images_local
      if FileTest.directory?(defaults[:image_local])
        if @@image_flag
          require 'ftools'
          images=Dir.glob("#{defaults[:image_local]}/*.{png,jpg}")
          path="#{webserv_path}/#@output_stub"
          File.mkpath("#{path}/_sisu/image_local") unless FileTest.directory?("#{path}/_sisu/image_local")
          images.each { |i| File.install(i, "#{path}/#{i}") } unless images.length > 0
          @@image_flag=false
        end
        '../_sisu/image_local'
      else
        if @@local_image==true
          cmd=if @cmd: @cmd 
          else ''
          end
          tell=SiSU_Screen::Ansi.new(cmd,"WARNING - no local image directory or images:", defaults[:image_local] )
          tell.warn unless cmd =~/q/
          @@local_image=false
        end
        url_images
      end
    end
    def image_external
      "#{processing}/external_document/image"
    end
    def url_images_external 
      if FileTest.directory?(image_external)
        if @@image_flag
          require 'ftools'
          images=Dir.glob("#{image_external}/*.{png,jpg}")
          path="#{webserv_path}/#@output_stub"
          File.mkpath("#{path}/_sisu/image_external") unless FileTest.directory?("#{path}/_sisu/image_external")
          images.each { |i| File.install(i,"#{path}/#{i}") } unless images.length > 0
          @@image_flag=false
        end
        '../_sisu/image_external'
      else
        if @@local_image==true
          tell=SiSU_Screen::Ansi.new(@cmd,"WARNING - image directory for external images or no such images:", :image_external )
          tell.warn unless @cmd =~/q/
          @@local_image=false
        end
        url_images_external
      end
    end
    def image_source_local_tex
      if FileTest.directory?(defaults[:image_local]):    defaults[:image_local]
      end
    end
    def image_source_remote_tex
      if FileTest.directory?(image_external): image_external
      end
    end
    def texinfo                                                                #texinfo webserv, check
      sfx_tst=@sfx_fns || @fns                                                   #if @fns.empty?
      "#{processing}/#{defaults[:processing_texinfo]}"
    end
    def man
      if defined? @rc['webserv']['man']: "#{webserv}/#{@rc['webserv']['man']}"
      else defaults[:webserv_man]
      end
    end
    def stub                                                                     #200412
      @output_stub
    end
    def sisupod
      #_sisu
      #  sisu_doc
      #    sisu
      #      content.sst              [file content]
      #      filename.sst             [link to content.sst]
      #      _sisu
      #        conf
      #          skin/
      #            doc                [relevant skin if any other than default]
      #        image                  [all images for specific document gathered here]
      system("rm -r #{Dir.pwd}/_sisu/sisu_doc")
      #rm_r("#{Dir.pwd}/_sisu/sisu_doc")
      paths=Array.new
      paths=["#{Dir.pwd}/_sisu/sisu_doc/_sisu/skin/doc","#{Dir.pwd}/_sisu/sisu_doc/_sisu/skin/dir","#{Dir.pwd}/_sisu/sisu_doc/_sisu/skin/site","#{Dir.pwd}/_sisu/sisu_doc/_sisu/image"]
      paths.each {|x| File.mkpath(x) unless FileTest.directory?(x) }
    end
    def metaverse
      path=if defined? @rc['processing']['metaverse']: "#{processing}/#{@rc['processing']['metaverse']}"
      else                                              "#{processing}/#{defaults[:processing_metaverse]}" 
      end
      File.mkpath(path) unless FileTest.directory?(path)
      path
    end
    def tune
      path=if defined? @rc['processing']['tune']:      "#{processing}/#{@rc['processing']['tune']}"
      else                                              "#{processing}/#{defaults[:processing_tune]}" 
      end
      File.mkpath(path) unless FileTest.directory?(path)
      path
    end
    def tex
      path=if defined? @rc['processing']['latex']:     "#{processing}/#{@rc['processing']['latex']}"
      else                                              "#{processing}/#{defaults[:processing_latex]}" 
      end
      File.mkpath(path) unless FileTest.directory?(path)
      path
    end
    def texi
      path=if defined? @rc['processing']['texinfo']:   "#{processing}/#{@rc['processing']['texinfo']}"
      else                                              "#{processing}/#{defaults[:processing_texinfo]}" 
      end
      File.mkpath(path) unless FileTest.directory?(path)
      path
    end
    def concord_max
      concord_max=if defined? @rc['processing']['concord_max'] and @rc['processing']['concord_max']:
        @rc['processing']['concord_max']
      else                                                       defaults[:concord_max]
      end
    end
    def lout
      path=if defined? @rc['processing']['lout']:      "#{processing}/#{@rc['processing']['lout']}"
      else                                              "#{processing}/#{defaults[:processing_lout]}" 
      end
      File.mkpath(path) unless FileTest.directory?(path)
      path
    end
#    def processingpath(path1,path2)
#      path=if defined? path1: "#{processing}/#{path1}"
#      else                    "#{processing}/#{path2}" 
#      end
#      File.mkpath(path) unless FileTest.directory?(path)
#      path
#    end
#    def metaverse
#      processingpath(@rc['processing']['metaverse'],defaults[:processing_metaverse])
#    end
#    def tune
#      processingpath(@rc['processing']['tune'],defaults[:processing_tune])
#    end
#    def tex
#      processingpath(@rc['processing']['latex'],defaults[:processing_latex])
#    end
#    def texi
#      processingpath(@rc['processing']['texinfo'],defaults[:processing_texinfo])
#    end
#    def lout
#      processingpath(@rc['processing']['lout'],defaults[:processing_lout])
#    end
    def encoding
      path="#{processing}/#{defaults[:processing_encoding]}" 
      File.mkpath(path) unless FileTest.directory?(path)
      path
    end
    def sqlite
      "#{home}/sqlite"
    end
    def cgi
      if defined? @rc['webserv']['cgi']:             "#{@rc['webserv']['cgi']}"
      else                                              defaults[:webserv_cgi]
      end
    end
    def php
      if defined? @rc['webserv']['php']:              "#{public_output}/#{@rc['webserv']['php']}"
      else                                              defaults[:webserv_php]
      end
    end
    def feed
      if defined? @rc['webserv']['feed']:             "#{public_output}/#{@rc['webserv']['feed']}"
      else                                              defaults[:webserv_feed]
      end
    end
    def feed_home
      "#{public_output}/#{@rc['webserv']['feed_home']}"
    end
    def bin
      '/usr/local/bin'
    end
                                                                                 # programs
    def text_editor
      if defined? @rc['program_select']['editor'] and @rc['program_select']['editor'] =~/\S\S+/
        @rc['program_select']['editor']
      elsif defined? @rc['program_select']['text_editor'] and @rc['program_select']['text_editor'] =~/\S\S+/
        @rc['program_select']['text_editor']
      else 'editor'                                                            #'gvim -c :R -c :S'
      end
    end
    def pdf_viewer
      if defined? @rc['program_select']['pdf_viewer'] and @rc['program_select']['pdf_viewer'] =~/\S\S+/
        @rc['program_select']['pdf_viewer']
      else 'pdf-viewer'                                                        #'evince'
      end
    end
    def web_browser
      if defined? @rc['program_select']['www_browser'] and @rc['program_select']['www_browser'] =~/\S\S+/
        @rc['program_select']['www_browser']
      elsif defined? @rc['program_select']['web_browser'] and @rc['program_select']['web_browser'] =~/\S\S+/
        @rc['program_select']['web_browser']
      else 'x-www-browser'                                                      #'kazehakase' 'galeon'
      end
    end
    def www_browser
      web_browser
    end
    def console_web_browser
      if defined? @rc['program_select']['console_www_browser'] and @rc['program_select']['console_www_browser'] =~/\S\S+/
        @rc['program_select']['console_www_browser']
      elsif defined? @rc['program_select']['console_web_browser'] and @rc['program_select']['console_web_browser'] =~/\S\S+/
        @rc['program_select']['console_web_browser']
      else 'console-www-browser'                                                #'links2' 'elinks' 'epiphany'
      end
    end
    def console_www_browser
      web_browser
    end
    def xml_viewer
      if defined? @rc['program_select']['xml_viewer'] and @rc['program_select']['xml_viewer'] =~/\S\S+/
        @rc['program_select']['xml_viewer']
      else text_editor
      end
    end
    def xml_editor
      xml_viewer
    end
    def odf_viewer
      if defined? @rc['program_select']['odf_viewer'] and @rc['program_select']['odf_viewer'] =~/\S\S+/
        @rc['program_select']['odf_viewer']
      else 'oowriter'                                                          #'odf-viewer','oowriter'
      end
    end
    def file_encoding #file encoding
      is=''
      if defined? @rc['program_set']['file_encoding']:     is=@rc['program_set']['encoding'] 
      end
      if is.nil? or is==true:                           is='encoding'
      end
      is
    end
    def wc #wordcount
      is=''
      if defined? @rc['program_set']['wc']:                is=@rc['program_set']['wc'] 
      end
      if is.nil? or is==true:                           is='wc'
      end
      is
    end
    def tidy
      if defined? @rc['program_set']['tidy']:              is=@rc['program_set']['tidy'] 
      end
      if is.nil? or is==true:                           is='tidy'
      end
      is
    end
    def rexml                                                                  #should be part of ruby 1.8 but apparently not always
      is=if FileTest.directory?("#{Config::CONFIG['rubylibdir']}/rexml") #Config::CONFIG['sitedir']
        true
      else false
      end
      is=if defined? @rc['program_set']['rexml']:      @rc['program_set']['rexml'] 
      else ''
      end
      if is.nil? or is==true:                           is='rexml'
      end
      is
    end
    def pdflatex
      is=if defined? @rc['program_set']['pdflatex']:   @rc['program_set']['pdflatex'] 
      else ''
      end
      if is.nil? or is==true:                           is='pdflatex'
      end
      is
    end
    def postgresql
      is=if defined? @rc['program_set']['postgresql']:  @rc['program_set']['postgresql'] 
      else ''
      end
      if is.nil? or is==true:                            is='postgresql'
      end
      is
    end
    def sqlite
      is=if defined? @rc['program_set']['sqlite']:      @rc['program_set']['sqlite'] 
      else ''
      end
      if is.nil? or is==true:                           is='sqlite'
      end
      is
    end
    def papersize                                                              # paper settings, default overidden in param if set within document
      if defined? @rc['default']['papersize']:   @rc['default']['papersize'].downcase
      else                                       defaults[:papersize].downcase
      end
    end
    def language                                                              # language settings
##
      m=/.+\/\S+?\~(\S+)/
      pwd=Dir.pwd
      conf=if defined? @rc['default']['language']: @rc['default']['language']
      else nil
      end
      l=if pwd=~m :                                pwd[m1,1]                   #2 directory: by visible directory name
      elsif conf: @rc['default']['language']                                 #3 config: from sisurc.yaml
      else                                        defaults[:language]         #4 sisu: program default
      end                                                                     #1 document: param gets
      SiSU_Env::Standardise_language.new(l)
    end
    def multilingual
      if defined? @rc['default']['multilingual'] and @rc['default']['multilingual'] != nil:  @rc['default']['multilingual']
      else                                         defaults[:multilingual]
      end
    end
    def lang_filename(l)
      @lang=Hash.new
      x=if multilingual
        x=if defined? @rc['default']['language_file'] and @rc['default']['language_file'] != nil: @rc['default']['language_file']
        else 1
        end
      else 0
      end
      if (l != defaults[:language_code]) or (language.code != defaults[:language_code]) #watch
        if x==1:      @lang[:pre],@lang[:mid],@lang[:post]="#{l}.",'',''
        elsif x==2:   @lang[:pre],@lang[:mid],@lang[:post]='',".#{l}",''
        elsif x==3:   @lang[:pre],@lang[:mid],@lang[:post]='','',".#{l}"
        else          @lang[:pre],@lang[:mid],@lang[:post]='','',''
        end
      else            @lang[:pre],@lang[:mid],@lang[:post]='','',''
      end
      @lang
    end
    def webrick_port
      if defined? @rc['webrick']['port']:        @rc['webrick']['port']
      else                                        defaults[:webrick_port]
      end
    end
  end
  class Info_processing_flag
    attr_accessor :color,:cf_1,:cf_2,:cf_3,:cf_4,:cf_5
    def initialize
      @rc=Get_yaml.instance.yamlrc
    end
    def color                                                                  #processing flag shortcuts
      if defined? @rc['flag']['color']: @rc['flag']['color']
      else                               false
      end
    end
    def cf_1                                                                   #processing flag shortcuts
      if defined? @rc['flag']['i']:     @rc['flag']['i']
      else                               '-Nhwpy'
      end
    end
    def cf_2                                                                   #processing flag shortcuts
      if defined? @rc['flag']['ii']:    @rc['flag']['ii']
      else                               '-NhwpAoy'
      end
    end
    def cf_3                                                                   #processing flag shortcuts
      if defined? @rc['flag']['iii']:   @rc['flag']['iii']
      else                               '-NhwpAobxXy'
      end
    end
    def cf_4                                                                   #processing flag shortcuts
      if defined? @rc['flag']['iv']:    @rc['flag']['iv']
      else                               '-NhwpAobxXDy --import'
      end
    end
    def cf_5                                                                   #processing flag shortcuts
      if defined? @rc['flag']['v']:     @rc['flag']['v']
      else                               '-NhwpAobxXDy --update'
      end
    end
  end
  class Info_settings < Info_dir
    def permission?(prog)                                                     #program defaults
      if defined? @rc['permission_set'][prog]: @rc['permission_set'][prog]
      else                                   false
      end
    end
    def program?(prog)                                                                    #program defaults
      if defined? @rc['program_set'][prog]: @rc['program_set'][prog]
      else                                   false
      end
    end
  end
  class File_map < Info_dir
    attr_accessor :local_sisu_source
    def initialize(opt='') #watch
      super()
      @opt=opt #,opt.fns,opt.cmd
      #@file=@opt.fns #not always fns
      @env=if @opt.fns and not @opt.fns.empty?
        SiSU_Env::Info_dir.new(@opt.fns)
      else 
        SiSU_Env::Info_dir.new('dummy.sst')
      end
      if @opt.cmd =~/m/: @md=SiSU_Param::Parameters.new(@opt).get
      end
      ft=Array.new
      if @md and defined? @md.fn and @md.fn                                    # used for multilingual
        if @md.cmd =~ /[hH]/:               ft << @md.fn[:html]
        end
        if @md.cmd =~ /w/ and @md.cmd !~ /[hH]/: ft << @md.fn[:concordance]
        end
        if @md.cmd =~ /y/ and @md.cmd !~ /[hH]/: ft << @md.fn[:manifest]
        end
        if @md.cmd =~ /p/:                  ft << @md.fn[:pdf_l] << @md.fn[:pdf_p]
        end
        if @md.cmd =~ /x/:                  ft << @md.fn[:sax]
        end
        if @md.cmd =~ /X/:                  ft << @md.fn[:dom]
        end
        if @md.cmd =~ /b/:                  ft << @md.fn[:xhtml]
        end
        if @md.cmd =~ /[aAeE]/:             ft << @md.fn[:plain]
        end
        if @md.cmd =~ /N/:                  ft << @md.fn[:digest]
        end
        if @md.cmd =~ /o/:                  ft << @md.fn[:odf]
        end
        if @md.cmd =~ /s/:                  ft << @md.fns
        end
        if @md.cmd =~ /S/:                  ft << 'sisupod.tgz' << '.kdi'
        end
        @fnb=@md.fnb
      else                                                                     # still needed where/when param is not parsed
        if @opt.cmd =~ /[hH]/:               ft << '.html' << '.html.??'
        end
        if @opt.cmd =~ /w/ and @opt.cmd !~ /[hH]/: ft << 'concordance.html' << '??.concordance.html' << 'concordance.??.html'
        end
        if @opt.cmd =~ /y/ and @opt.cmd !~ /[hH]/: ft << 'sisu_manifest.html' << '??.sisu_manifest.html' << 'sisu_manifest.??.html'
        end
        if @opt.cmd =~ /p/:                  ft << 'landscape.pdf' << 'portrait.pdf' << '.pdf'
        end
        if @opt.cmd =~ /x/:                  ft << 'sax.xml' << '??.sax.xml' << 'sax.??.xml'
        end
        if @opt.cmd =~ /X/:                  ft << 'dom.xml' << '??.dom.xml' << 'dom.??.xml'
        end
        if @opt.cmd =~ /b/:                  ft << 'scroll.xhtml' << '??.scroll.xhtml' << 'scroll.??.xhtml'
        end
        if @opt.cmd =~ /[aAeE]/:             ft << 'plain.txt' << '??.plain.txt' << 'plain.??.txt'
        end
        if @opt.cmd =~ /N/:                  ft << 'digest.txt' << '??.digest.txt' << 'digest.??.txt'
        end
        if @opt.cmd =~ /o/:                  ft << 'opendocument.odt' << '??.opendocument.odt' << 'opendocument.??.odt'
        end
        if @opt.cmd =~ /s/:                  ft << '.sst' << '.ssi' << '.ssm'
        end
        if @opt.cmd =~ /S/:                  ft << 'sisupod.tgz' << '.kdi'
        end
      end
      ft=ft.uniq
      filetypes=ft.join(',')
      @filetypes=if filetypes !~/..+/: ''   # -r called alone, copy all
      elsif @opt.cmd =~/u/:                  ''   # -u added, copy all, (used to create remote directory tree see output path), not the usual function of -u
      elsif filetypes =~/\S+?,\S+/:    '*{' + filetypes + '}' # more than one relevant file type
      else                              '*' + filetypes                       # one relevant file type
      end
      @source_path=if @fnb and not @fnb.empty?: "#{@env.data_o}/#@fnb"
      else                                      @env.data_o
      end
      @local_sisu_source=if @filetypes =~/\S/: "#@source_path/#@filetypes"
      else                  @source_path
      end
    end
  end
  class Clean_output
    require 'fileutils'
    include FileUtils::Verbose
    def initialize(opt)
      @opt=opt
      z=File_map.new(@opt)
      @zap=z.local_sisu_source
      if @opt.cmd =~ /[hH]/
        @zap=Dir.glob(@zap).join(' ')
        if @opt.cmd !~ /w/: @zap.gsub!(/#@source_path\/concordance.html/,'')
        end
      end
    end
    def zap
      if @zap !~/\/\//: rm_rf(@zap) if FileTest.directory?(@zap)
      else puts 'suspect zap request, ignored'
      end
      #Dir.unlink(@zap) if FileTest.directory?(@zap)
      #System_call.new(@zap,'',@cmd).rm
    end
  end
  class Info_remote_host
    def initialize
      @rc=Get_yaml.instance.yamlrc
    end
    def remote_host #see Info_remote remote_host_base_general
      r=Hash.new
      r=if (defined? @rc['remote']['user'] and defined? @rc['remote']['host'])
        r[:user]=@rc['remote']['user']
        r[:host]=@rc['remote']['host']
        r[:path]=if defined? @rc['remote']['path']
          @rc['remote']['path'] 
        else ''
        end
        r[:name]="#{r[:user]}@#{r[:host]}:#{r[:path]}"
        r
      else 
        r[:name]='.'
        r[:user]=''
        r[:host]=''
        r[:path]=''
        r
        #puts "no remote host or user"
      end
    end
  end
  class Info_remote < File_map
    @@flag_remote=false
    require 'socket'
    def initialize(opt)
      super(opt) #
      @opt=opt
      @rc=Get_yaml.instance.yamlrc
    end
    def remote_host_base_general
      SiSU_Env::Info_remote_host.new.remote_host
    end
    def remote_host_base
      remote=remote_host_base_general
      #host_ip=IPSocket.getaddress(remote[:host]) unless remote[:host].empty?
      @@flag_remote=true if remote[:name] =~/\S+@\S+/ #and  host_ip =~/\d+\.\d+\.\d+\.\d+/ #very naive check should be enough /[0-255]+\.[0-255]+\.[0-255]+\.[0-255]+/
      #remote[:name]
      remote_host_base_general[:name]
    end
    #def remote_host_base
    #  remote_host_base_general[:name]
    #end
    def scp                                                                    #sort out later using ruby libraries #not ideal, first time each file is sent, -r must be called separately for subdir to be built
      input=@local_sisu_source
      output=case @opt.cmd 
      when /u/:                "#{self.remote_host_base}/#{@env.output_stub}/."             #creates remote directory tree, this is not the usual function of u
      when /[aAbeEhHNopwxXy]/: "#{self.remote_host_base}/#{@env.output_stub}/#{@fnb}/." 
      else                     "#{self.remote_host_base}/#{@env.output_stub}/." 
      end
      if (input =~/\S/ and input !~/\/\//) and (output =~/\S/ and output !~/\/\//) and @@flag_remote==true and @opt.cmd !~/U/
          System_call.new(input,output).scp
      elsif @opt.cmd =~/U/: p "#{input} -> #{output}"
      else puts 'suspect scp request, ignored'
      end
    end
    def rsync
      input=@local_sisu_source
      output=case @opt.cmd 
      when /u/:                "#{self.remote_host_base}/#{@env.output_stub}/."             #creates remote directory tree, this is not the usual function of u
      when /[aAbeEhHNopwxXy]/: "#{self.remote_host_base}/#{@env.output_stub}/#{@fnb}/." 
      else                     "#{self.remote_host_base}/#{@env.output_stub}/." 
      end
      if (input =~/\S/ and input !~/\/\//) and (output =~/\S/ and output !~/\/\//) and @@flag_remote==true and @opt.cmd !~/U/
        if @local_sisu_source == @source_path
          System_call.new(input,output,@opt.cmd).rsync('--delete-after')
        else System_call.new(input,output,@opt.cmd).rsync
        end
      elsif  @opt.cmd =~/U/
        p "#{__FILE__} #{__LINE__}" if @opt.cmd =~/M/
        p "#{input} -> #{output}"
      else 
        puts 'suspect rsync request, ignored'
        puts "#{input} -> #{output} remote flag: #@@flag_remote"
        p "permission not granted #{__FILE__} #{__LINE__}" if @opt.cmd =~/M/
      end
    end
    def scp_base                                                                #base site
      input=@source_path
      output="#{remote_host_base}/#{@env.output_stub}/."
      if defined? @env.rc['permission_set']['remote_base_site'] and @env.rc['permission_set']['remote_base_site'] and @@flag_remote==true and @opt.cmd !~/U/
        puts "begin scp_base: #{input} -> #{output}"
        System_call.new("#{input}/#{@env.style}/",output).scp
      elsif @opt.cmd =~/U/
        p "#{__FILE__} #{__LINE__}" if @opt.cmd =~/M/
        puts "begin scp_base: #{input} -> #{output}"
        p "#{input}/#{@env.style}/ -> #{output}"
      else  p "permission not granted #{__FILE__} #{__LINE__}" if @opt.cmd =~/M/
      end
    end
    def scp_base_all                                                            #base site
      input=@source_path
      output="#{remote_host_base}/#{@env.output_stub}/."
      if defined? @env.rc['permission_set']['remote_base_site'] and @env.rc['permission_set']['remote_base_site'] and @@flag_remote==true and @opt.cmd !~/U/
        puts "begin scp_base_all: #{input} -> #{output}"
        System_call.new("#{input}/_sisu/image/",output).scp
        System_call.new("#{input}/_sisu/image_local/",output).scp
        System_call.new("#{input}/#{@env.style}/",output).scp
      elsif @opt.cmd =~/U/
        p "#{__FILE__} #{__LINE__}" if @opt.cmd =~/M/
        puts "scp_base_all: #{input} -> #{output}"
        p "#{input}/_sisu/image/ -> #{output}"
        p "#{input}/_sisu/image_local/ -> #{output}"
        p "#{input}/#{@env.style}/ -> #{output}"
      else  p "permission not granted #{__FILE__} #{__LINE__}" if @opt.cmd =~/M/
      end
    end
    def rsync_base                                                              #base site
      input=@source_path
      output="#{remote_host_base}/#{@env.output_stub}/."
      if defined? @env.rc['permission_set']['remote_base_site'] and @env.rc['permission_set']['remote_base_site'] and @@flag_remote==true and @opt.cmd !~/U/
        #puts "begin rsync_base: #{input} -> #{output}"
        #System_call.new("#{input}/_sisu/image/",output).rsync
        #System_call.new("#{input}/_sisu/image_local/",output).rsync
        #System_call.new("#{input}/#{@env.style}/",output).rsync
      elsif @opt.cmd =~/U/
        p "#{__FILE__} #{__LINE__}" if @opt.cmd =~/M/
        puts "rsync_base: #{input} -> #{output}"
        p "#{input}/_sisu/image/ -> #{output}"
        p "#{input}/_sisu/image_local/ -> #{output}"
        p "#{input}/#{@env.style}/ -> #{output}"
      else  p "permission not granted #{__FILE__} #{__LINE__}" if @opt.cmd =~/M/
      end
    end
    def rsync_base_sync                                                         #base site
      input=@source_path
      output="#{remote_host_base}/#{@env.output_stub}/."
      if defined? @env.rc['permission_set']['remote_base_site'] and @env.rc['permission_set']['remote_base_site'] and @@flag_remote==true and @opt.cmd !~/U/
        puts "begin rsync_base_sync: #{input} -> #{output}"
        System_call.new("#{input}/_sisu/image/",output).rsync('--delete-after')
        System_call.new("#{input}/_sisu/image_local/",output).rsync('--delete-after')
        System_call.new("#{input}/#{@env.style}/",output).rsync('--delete-after')
      elsif @opt.cmd =~/U/
        p "#{__FILE__} #{__LINE__}" if @opt.cmd =~/M/
        puts "rsync_base_sync: #{input} -> #{output}"
        p "#{input}/_sisu/image/ -> #{output}"
        p "#{input}/_sisu/image_local/ -> #{output}"
        p "#{input}/#{@env.style}/ -> #{output}"
      else  p "permission not granted #{__FILE__} #{__LINE__}" if @opt.cmd =~/M/
      end
    end
  end
  class Info_version <Info_dir
    require 'rbconfig'
    def get_version
      require 'ftools'
      @version=Hash.new
      @pwd=ENV['PWD']
      yst_etc="#{defaults[:sisu_etc]}/version.yaml"
      lib_path=`echo $RUBYLIB`.split(':')
      if File.exist?(yst_etc): @version=YAML::load(File::open(yst_etc)) #unless @@noyaml
      end
      @version
    end
    def rbversion
      #Config::CONFIG['ruby_version']
      %x{ruby -v}.strip
    end
  end
  class Create_system_link #< Info_dir                                          #revisit problems created 2004w41
    def initialize
      @env=SiSU_Env::Info_dir.new
    end
    def images
      File.mkpath("#{@env.data_o}/_sisu") unless FileTest.directory?("#{@env.data_o}/_sisu")
      #File.mkpath("#{@env.data_o}/sisu/image}") unless FileTest.directory?("#{@env.data_o}/#{@env.style}")
      File.symlink("../../_sisu/image", "#{@env.data_o}/_sisu/image") unless File.symlink?("#{@env.data_o}/_sisu/image")
    end
    def man_forms
      #File.symlink("../../man/form", "#{@env.data_o}/man/form") unless File.symlink?("#{@env.data_o}/man/form") == true
    end
    def man_pdf
      #File.symlink("../../man/form", "#{@env.data_o}/man/pdf") unless File.symlink?("#{@env.data_o}/man/pdf") == true
    end
  end
  class Info_file < Info_dir
    def initialize(fns='')
      @env=SiSU_Env::Info_dir.new
      @fns=fns
    end
    def basefilename
      m=/(.+?)\.(?:[_-]?sst|ssm)$/im
      fnb=@fns[m,1]
    end
    def project
      "#{@env.development}/sisu.lnk"
    end
    def project_info
      "#{@env.datapriv_i}/sisu.er3"
    end
    def project_todo
      "#{@env.development}/todo.txt"
    end
    def admin_self
      "#{@env.home}/.corundum/dots.lnk"
    end
    def admin_root
      '/root/.corundum/admin.txt'
    end
  end
  class Info_db < Info_dir
    def initialize
      @@pwd ||=Dir.pwd
      @pwd=Dir.pwd
      @env=SiSU_Env::Info_dir.new
      m=/.+\/(\S+)/im
      @pwd_stub=@pwd[m,1]
    end
    def user
      @env.user #'ralph'
    end
    def db_psql
      "SiSU_#@pwd_stub"
    end
    def db_mysql
      'lm'
    end
    def db_sqlite
      "#{@env.webserv_path}/#@pwd_stub/sisu_sqlite.db"
    end
    def port_psql
      '5432'
    end
    def port_mysql
      '**'
    end
    def dbi_psql
      "DBI:Pg:database=#{db_psql};port=#{port_psql}"
    end
    def dbi_mysql
      "dbi:Mysql:database=#{db_mysql};port=#{port_mysql}"
    end
    def dbi_sqlite
      "DBI:SQLite:#{db_sqlite}" #sqlite3 ?
    end
  end
  class Info_port < Info_dir
    def initialize
      @env=SiSU_Env::Info_dir.new
    end
    def webrick
      @env.webrick_port
    end
  end
  class Info_program < Info_dir                                                  #revisit
    attr_accessor :editor,:wc,:tidy,:rexml,:pdflatex,:postgresql,:sqlite
    def initialize
      env=SiSU_Env::Info_dir.new
      @editor,@wc,@tidy,@rexml,@pdflatex,@postgresql,@sqlite=env.text_editor,env.wc,env.tidy,env.rexml,env.pdflatex,env.postgresql,env.sqlite
    end
  end
  class Info_skin
    def initialize(md=nil)
      @d_sk=if md.class==String : md
      elsif defined? md.doc_skin and md.doc_skin: md.doc_skin
      else nil
      end
      @home,@pwd=ENV['HOME'],ENV['PWD']
      m=/.+\/(\S+)/im
      @pwd_stub=@pwd[m,1]
      @env=SiSU_Env::Info_dir.new
    end
    def select                                                                 # skin loading logic here
      load "#{SiSU_lib}/defaults.rb"
      #skin_path=["#@pwd/_sisu/skin","#@home/.sisu/skin","/etc/sisu/skin","#{@env.processing}/external_document/skin"]
      skin_path=Array.new
      @env.sys.rc_path.each{|x| skin_path << "#{x}/skin"}
      skin_path << "#{@env.processing}/external_document/skin"
      @doc_skin=nil
      unless @d_sk.nil?
        sk_doc="doc/#@d_sk.rb"
        skin_path.each do |v|                                                    #document skin priority 1
          if FileTest.file?("#{v}/#{sk_doc}")
            @doc_skin="#{v}/#{sk_doc}"
            load @doc_skin
            break
          end
        end
      end
      unless @doc_skin
        sk_dir="dir/skin_#@pwd_stub.rb"
        skin_path.each do |v|                                                  #directory skin priority 2
          if FileTest.file?("#{v}/#{sk_dir}")
            @dir_skin="#{v}/#{sk_dir}"
            load @dir_skin
            break
          end
        end
      end
      sk=if @dir_skin: @dir_skin
      elsif @doc_skin: @doc_skin
      else             nil
      end
    end
  end
  class CSS_default
    def html
      'html.css'
    end
    def html_tables
      'html_tables.css'
    end
    def xhtml
      'xhtml.css'
    end
    def xml_sax
      'sax.css'
    end
    def xml_dom
      'dom.css'
    end
    def docbook_xml
      'docbook.css'
    end
    def homepage
      'index.css'
    end
  end
  class CSS_select < Info_dir
    def initialize(md)
      @md=md
      @env=SiSU_Env::Info_dir.new
    end
    def html
      css=if @md.doc_css and FileTest.file?("#{@env.data_o}/#{@env.style}/#{@md.doc_css}_html.css")
        "#{@md.doc_css}_html.css"
      elsif FileTest.file?("#{@env.data_o}/#{@env.style}/#{@env.stub}_html.css")
        "#{@env.stub}_html.css"
      else CSS_default.new.html
      end
    end
    def html_tables
      css=if @md.doc_css and FileTest.file?("#{@env.data_o}/#{@env.style}/#{@md.doc_css}_html_tables.css")
        "#{@md.doc_css}_html_tables.css"
      elsif FileTest.file?("#{@env.data_o}/#{@env.style}/#{@env.stub}_html_tables.css")
        "#{@env.stub}_html_tables.css"
      else CSS_default.new.html_tables
      end
    end
    def xhtml
      css=if @md.doc_css and FileTest.file?("#{@env.data_o}/#{@env.style}/#{@md.doc_css}_xhtml.css")
        "#{@md.doc_css}_xhtml.css"
      elsif FileTest.file?("#{@env.data_o}/#{@env.style}/#{@env.stub}_xhtml.css")
        "#{@env.stub}_xhtml.css"
      else CSS_default.new.xhtml
      end
    end
    def xml_sax
      css=if @md.doc_css and FileTest.file?("#{@env.data_o}/#{@env.style}/#{@md.doc_css}_xml_sax.css")
        "#{@md.doc_css}_xml_sax.css"
      elsif FileTest.file?("#{@env.data_o}/#{@env.style}/#{@env.stub}_xml_sax.css")
        "#{@env.stub}_xml_sax.css"
      else CSS_default.new.xml_sax
      end
    end
    def xml_dom
      css=if @md.doc_css and FileTest.file?("#{@env.data_o}/#{@env.style}/#{@md.doc_css}_xml_dom.css")
        "#{@md.doc_css}_xml_dom.css"
      elsif FileTest.file?("#{@env.data_o}/#{@env.style}/#{@env.stub}_xml_dom.css")
        "#{@env.stub}_xml_dom.css"
      else CSS_default.new.xml_dom
      end
    end
    def docbook_xml
      css=if @md.doc_css and FileTest.file?("#{@env.data_o}/#{@env.style}/#{@md.doc_css}_docbook.css")
        "#{@md.doc_css}_xml_dom.css"
      elsif FileTest.file?("#{@env.data_o}/#{@env.style}/#{@env.stub}_docbook.css")
        "#{@env.stub}_docbook.css"
      else CSS_default.new.docbook_xml
      end
    end
    def homepage
      css=if @md.doc_css and FileTest.file?("#{@env.data_o}/#{@env.style}/#{@md.doc_css}_homepage.css")
        "#{@md.doc_css}_homepage.css"
      elsif FileTest.file?("#{@env.data_o}/#{@env.style}/#{@env.stub}_homepage.css")
        "#{@env.stub}_homepage.css"
      else CSS_default.new.homepage
      end
    end
  end
  class CSS_stylesheet
    def initialize(md)
      @md=md
      @css=CSS_select.new(@md)
      @env=SiSU_Env::Info_dir.new
    end
    def html
      %{  <link rel="stylesheet" href="../#{@env.style}/#{@css.html}" type="text/css" />}
    end
    def html_tables
      %{  <link rel="stylesheet" href="../#{@env.style}/#{@css.html_tables}" type="text/css" />}
    end
  end
  class SiSU_file <Info_dir                                                    #todo unify with Create_file
    def initialize(md,fno='')
      begin
        @fno,@fns,@fnb=fno,md.fns,md.fnb
        @env=SiSU_Env::Info_dir.new(@fns)
        @env_out="#{@env.data_o}/#@fnb"
      rescue: STDERR.puts SiSU_Screen::Ansi.new(@cmd,$!,$@).rescue
      ensure
      end
    end
    def mkdir_initialize                                                       # not used but consider using
      File.mkpath(@env.data_o) unless FileTest.directory?(@env.data_o)
      File.mkpath("#{@env.data_o}/#@fnb") unless FileTest.directory?("#{@env.data_o}/#@fnb")
      File.mkpath("#{@env.data_o}/#{@env.style}") unless FileTest.directory?("#{@env.data_o}/#{@env.style}")
      File.mkpath(@env.metaverse) unless FileTest.directory?(@env.metaverse)
      File.mkpath(@env.tune) unless FileTest.directory?(@env.tune)
    end
    def mkdir
      dir=@env.data_o
      txt_path="#{dir}/#@fnb"
      File.mkpath(dir) unless FileTest.directory?(dir)
      File.mkpath(txt_path) unless FileTest.directory?(txt_path)
    end
    def mkfile #consider using more
      filename="#{@env.data_o}/#@fnb/#@fno"
      File.new(filename,'w+')
    end
  end
  class Create_file <Info_dir                                                  #todo unify with SiSU_file
    def initialize(cmd,fns,operation='')
      @cmd=cmd
      begin
        super(fns)
        @env=SiSU_Env::Info_dir.new(fns)
        ver=Info_version.new
        case operation #watch
        when /pdf/: @env_out=''
        when /sql/
        when /xml|plaintext|ascii/: @env_out="#{@env.data_o}/#@fnb" #check change of name to plaintext from ascii
        else
          if defined? @md.sfx_src and @md.sfx_src =~/ss[ftsumc]/
            @env_suf='lm'
            @env_out_root=@env.data_o
            @env_out="#{@env.data_o}/#@fnb"
            @env_tex=@env.tex
            @env_lout=@env.lout
            @@publisher='SiSU http://www.jus.uio.no/sisu'
          end
          @env_pdf="#@env_out_root/pdf"
        end
      rescue: STDERR.puts SiSU_Screen::Ansi.new(@cmd,$!,$@).rescue
      ensure
      end
    end
    def param_instantiate
      @cX||=SiSU_Screen::Ansi.new(@cmd)
      @@date=Info_date.new
      @@proc=@@filename_txt=@@filename_texinfo=@@filename_lout_portrait=@@filename_lout_landscape=@@filename_html_scroll=@@filename_html_index=@@filename_html_segtoc=@@filename_semantic=@@filename_rss=@@newfile=@@drr=@@yaml=@@yamladdr=nil
      @@publisher='SiSU scribe'
    end
    def html_root
      #@env.data_o
    end
    def mkdir_html
      File.mkpath(@env.data_o) unless FileTest.directory?(@env.data_o)
      File.mkpath("#{@env.data_o}/#@fnb") unless FileTest.directory?("#{@env.data_o}/#@fnb")
      File.mkpath("#{@env.data_o}/#{@env.style}") unless FileTest.directory?("#{@env.data_o}/#{@env.style}")
      File.mkpath(@env.metaverse) unless FileTest.directory?(@env.metaverse)
      File.mkpath(@env.tune) unless FileTest.directory?(@env.tune)
    end
    def mkdir_dirs
      dir="#{@env.data_o}"
      txt_path="#{dir}/#@fnb"
      File.mkpath(dir) unless FileTest.directory?(dir)
      File.mkpath(txt_path) unless FileTest.directory?(txt_path)
    end
    def mkdir_pdf
      Dir.mkdir("#{@env.tex}") unless FileTest.directory?("#{@env.tex}")
    end
    def file_generic(output_file='')
      filename="#{@env.data_o}/#@fnb/#{output_file}"
      File.new(filename,'w+')
    end
    def file_error
      File.new('/tmp/errorlog.sisu','w+')
    end
    def file_meta
      File.new("#{@env.metaverse}/#@fns.meta",'w+')
    end
    def marshal_meta
      "#{@env.metaverse}/#@fns.meta.rbm"
    end
    def meta
      "#{@env.metaverse}/#@fns.meta"
    end
    #def file_yml
    #  File.new("#{@env.metaverse}/#@fnb.yaml",'w+')
    #end
    def file_tune
      File.new("#{@env.tune}/#@fns.tune",'w+')
    end
    def marshal_tune
      "#{@env.tune}/#@fns.marshal_tune"
    end
    def file_digest
      filename_txt="#{@env.data_o}/#@fnb/digest.txt"
      @@filename_txt=File.new(filename_txt,'w+')
    end
    def file_plaintext
      filename_txt="#{@env.data_o}/#@fnb/plain.txt"
      @@filename_txt=File.new(filename_txt,'w+')
    end
    def file_semantic
      filename_semantic="./semantic.yaml"
      @@filename_semantic=File.new(filename_semantic,'w+')
    end
    def file_rss
      filename_rss="./semantic.xml"
      @@filename_rss=File.new(filename_rss,'w+')
    end
    def file_html_scroll(md)
      filename_scroll="#{@env.data_o}/#{md.fnb}/#{md.fnl[:pre]}doc#{md.fnl[:mid]}.html#{md.fnl[:post]}"
      @@filename_html_scroll=File.new(filename_scroll,'w+')
    end
    def file_html_index(md)
      filename_index="#{@env.data_o}/#{md.fnb}/#{md.fnl[:pre]}index#{md.fnl[:mid]}.html#{md.fnl[:post]}"
      @@filename_html_index=File.new(filename_index,'w+')
    end
    def file_html_segtoc(md)
      filename_segtoc="#{@env.data_o}/#{md.fnb}/#{md.fnl[:pre]}toc#{md.fnl[:mid]}.html#{md.fnl[:post]}"
      File.new(filename_segtoc,'w+')
    end
    def file_php_scroll
      filename_scroll="#{@env.data_o}/#@fnb/doc.php"
      @@filename_html_scroll=File.new(filename_scroll,'w+')
    end
    def file_php_index
      filename_index="#{@env.data_o}/#@fnb/index.php"
      @@filename_html_index=File.new(filename_index,'w+')
    end
    def file_php_index_link
    end
    def file_php_segtoc
      filename_segtoc="#{@env.data_o}/#@fnb/toc.php"
      @@filename_html_segtoc=File.new(filename_segtoc,'w+')
    end
    def file_texinfo
      File.new("#{@env.texinfo}/#@fnb.texinfo",'w+')
    end
    def file_loutpdf_portrait
    end
    def file_loutpdf_landscape
    end
    def file_js
    end
  end
  class Create_site < Info_dir
    require "#{SiSU_lib}/css"
    include SiSU_Style
    def initialize(cmd)
      @cmd=cmd
      @env=SiSU_Env::Info_dir.new
      @pwd=Dir.pwd
    end
    def homepage
      homepage_path=nil
      @env.sys.rc_path.each do |v|                                                    #document skin priority 1
        if FileTest.file?("#{v}/home/index.html")
          homepage_path="#{v}/home/index.html"
          break
        end
      end
      @home=SiSU_Viz::Home.new #use sysenv in future
      File.mkpath("#{@env.webserv_path}/#{@env.stub}") unless FileTest.directory?("#{@env.webserv_path}/#{@env.stub}")
      if homepage_path and FileTest.file?(homepage_path)
        cp(homepage_path,"#{@env.webserv_path}/#{@env.stub}/index.html")
        cp(homepage_path,"#{@env.webserv_path}/#{@env.stub}/toc.html")
      else
        @doc_skin=nil
        sk_doc='doc/skin_sisu.rb'
        @env.sys.rc_path.each do |v|                                                    #document skin priority 1
          if FileTest.file?("#{v}/skin/#{sk_doc}")
            @doc_skin="#{v}/skin/#{sk_doc}"
            load @doc_skin
            break
          end
        end
        filename_homepage=File.new("#{@env.webserv_path}/#{@env.stub}/index.html",'w')
			  filename_homepage_toc=File.new("#{@env.webserv_path}/#{@env.stub}/toc.html",'w')
        filename_homepage << @home.index
        filename_homepage_toc << @home.index
      end
    end
    def cp_images(src_path,dest_path)
      if FileTest.directory?(src_path)
        cd(src_path)
        source=Dir.glob("*.{png,jpg,ico}")
        File.mkpath(dest_path) unless FileTest.directory?(dest_path)
        source.each do |i| 
          cp_r(i,"#{dest_path}/#{i}")
        end
        chmod(0755,dest_path)
        cd(@pwd)
      else puts "\tWARN, did not find - #{src_path}"
      end
    end
    def cp_local_images
      src="#@pwd/_sisu/image"
      dest="#{@env.webserv_path}/#{@env.stub}/_sisu/image_local"
      cp_images(src,dest)
    end
    def cp_external_images
      src="#{@env.processing}/external_document/image"
      dest="#{@env.webserv_path}/#{@env.stub}/_sisu/image_external"
      cp_images(src,dest)
    end
    def cp_webserver_images
      src=@env.image_source
      dest="#{@env.webserv_path}/_sisu/image"
      cp_images(src,dest)
    end
    def cp_css
      File.mkpath("#{@env.data_o}/#{@env.style}") unless FileTest.directory?("#{@env.data_o}/#{@env.style}")
      css_path=['/etc/sisu/css',"#@home/.sisu/css","#@pwd/_sisu/css"]
      if defined? @env.rc['permission_set']['css_modify'] and @env.rc['permission_set']['css_modify']
        tell=SiSU_Screen::Ansi.new(@cmd,"modify is css set to: #{@env.rc['permission_set']['css_modify']}") 
        css_path.each do |x|
          if FileTest.directory?(x)
            cd(x)
            source=Dir.glob("*.{css}")
            source.each do |i|
              cp(i,"#{@env.data_o}/#{@env.style}")
            end
            cd(@pwd)
          end
        end
      else tell=SiSU_Screen::Ansi.new(@cmd,"modify css is not set or is set to: false") 
      end
      tell.warn if @cmd=~/[MV]/
      fn_css=SiSU_Env::CSS_default.new
      css=SiSU_Style::CSS.new
      path_style="#{@env.data_o}/#{@env.style}"
      File.mkpath(path_style) unless FileTest.directory?(path_style)
      style=File.new("#{path_style}/#{fn_css.homepage}",'w')
      style << css.homepage
      style.close
      style=File.new("#{path_style}/#{fn_css.html_tables}",'w')
      style << css.html_tables
      style.close
      style=File.new("#{path_style}/#{fn_css.html}",'w') 
      style << css.html
      style.close
      style=File.new("#{path_style}/#{fn_css.xml_sax}",'w') 
      style << css.xml_sax
      style.close
      style=File.new("#{path_style}/#{fn_css.xml_dom}",'w') 
      style << css.xml_dom
      style=File.new("#{path_style}/#{fn_css.docbook_xml}",'w') 
      style << css.docbook_xml
      style.close
      style=File.new("#{path_style}/#{fn_css.xhtml}",'w') 
      style << css.xhtml
      style.close
    end
  end
end
module SiSU_Screen
  class Color
    attr_accessor :off,:marker,:bold,:underline,:invert,:darkgrey_hi,:grey_hi,:pink_hi,:fuschia_hi,:red_hi,:orange_hi,:yellow_hi,:brown_hi,:lightgreen_hi,:green_hi,:cyan_hi,:blue_hi,:navy_hi,:grey,:pink,:fuschia,:ruby,:red,:orange,:yellow,:brown,:green,:darkgreen,:cyan,:blue,:navy,:black
    def initialize(&block)
      instance_eval &block
    end
  end
  class Ansi < Color
    attr_reader :cX
    def initialize(cmd,*txt)
      @cmd,@txt=cmd,txt
      @color_instruct=txt[0]
      flag=SiSU_Env::Info_processing_flag.new
      if @cmd
        if flag.color                                                              #set default colors on or off -c acts as toggle against this default, if default is off -c turns on, if default is on -c turns off
          if @cmd =~/c/: @use_color=false
          else @use_color=true
          end
        else
          if @cmd =~/c/: @use_color=true
          else @use_color=false
          end
        end
        if @cmd =~/k/: @use_color=false
        end
      else @use_color=false
      end
      @cX=@@cX= unless @use_color
        Color.new do
          self.off=self.marker=self.bold=self.underline=self.invert=self.darkgrey_hi=self.grey_hi=self.pink_hi=self.fuschia_hi=self.red_hi=self.orange_hi=self.yellow_hi=self.brown_hi=self.lightgreen_hi=self.green_hi=self.cyan_hi=self.blue_hi=self.navy_hi=self.grey=self.pink=self.fuschia=self.ruby=self.red=self.orange=self.yellow=self.brown=self.green=self.darkgreen=self.cyan=self.blue=self.navy=self.black=''
        end
      else                                                                       #default set to colors on
        Color.new do
          self.off           = "\033[0m"
          self.marker        = "\033[42m"
          self.bold          = "\033[1m"
          self.underline     = "\033[4m"
          self.invert        = "\033[7m"
          self.darkgrey_hi   = "\033[100m"
          self.grey_hi       = "\033[47m"
          self.pink_hi       = "\033[105m"
          self.fuschia_hi    = "\033[45m"
          self.red_hi        = "\033[41m"
          self.orange_hi     = "\033[101m"
          self.yellow_hi     = "\033[103m"
          self.brown_hi      = "\033[43m"
          self.lightgreen_hi = "\033[102m"
          self.green_hi      = "\033[42m"
          self.cyan_hi       = "\033[106m"
          self.blue_hi       = "\033[104m"
          self.navy_hi       = "\033[44m"
          self.grey          = "\033[90m"
          self.pink          = "\033[95m"
          self.fuschia       = "\033[35m"
          self.ruby          = "\033[31m"
          self.red           = "\033[91m" #check
          self.orange        = "\033[91m"
          self.yellow        = "\033[93m"
          self.brown         = "\033[33m"
          self.green         = "\033[92m"
          self.darkgreen     = "\033[32m"
          self.cyan          = "\033[36m"
          self.blue          = "\033[94m"
          self.navy          = "\033[34m"
          self.black         = "\033[30m"
        end
      end
    end
    def colors
      0.upto(109) {|i| print "\033[#{i}m 33[#{i}m \033[m"}
       puts ''
    end
    def color
      case @color_instruct
      when /invert/:        @cX.invert
      when /darkgrey_hi/:   @cX.darkgrey_hi
      when /grey_hi/:       @cX.grey_hi
      when /pink_hi/:       @cX.pink_hi
      when /fuschia_hi/:    @cX.fuschia_hi
      when /red_hi/:        @cX.red_hi
      when /orange_hi/:     @cX.orange_hi
      when /yellow_hi/:     @cX.yellow_hi
      when /brown_hi/:      @cX.brown_hi
      when /lightgreen_hi/: @cX.lightgreen_hi
      when /green_hi/:      @cX.green_hi
      when /cyan_hi/:       @cX.cyan_hi
      when /blue_hi/:       @cX.blue_hi
      when /navy_hi/:       @cX.navy_hi
      when /grey/:          @cX.grey
      when /pink/:          @cX.pink
      when /fuschia/:       @cX.fuschia
      when /ruby/:          @cX.ruby
      when /red/:           @cX.red
      when /orange/:        @cX.orange
      when /yellow/:        @cX.yellow
      when /brown/:         @cX.brown
      when /green/:         @cX.green
      when /darkgreen/:     @cX.darkgreen
      when /cyan/:          @cX.cyan
      when /blue/:          @cX.blue
      when /navy/:          @cX.navy
      when /close/:         @cX.off
      when /off/:           @cX.off
      end
    end
    def colorize
      case @color_instruct
      when /invert/:        puts "#{@cX.invert}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /darkgrey_hi/:   puts "#{@cX.darkgrey_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /grey_hi/:       puts "#{@cX.grey_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /pink_hi/:       puts "#{@cX.pink_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /fuschia_hi/:    puts "#{@cX.fuschia_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /red_hi/:        puts "#{@cX.red_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /orange_hi/:     puts "#{@cX.orange_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /yellow_hi/:     puts "#{@cX.yellow_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /brown_hi/:      puts "#{@cX.brown_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /lightgreen_hi/: puts "#{@cX.lightgreen_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /green_hi/:      puts "#{@cX.green_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /cyan_hi/:       puts "#{@cX.cyan_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /blue_hi/:       puts "#{@cX.blue_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /navy_hi/:       puts "#{@cX.navy_hi}#{@txt[1]}#{@cX.off} #{@cX.blue}#{@txt[2]}#{@cX.off} #{@cX.grey}#{@txt[3]}#{@cX.off}"
      when /bold/:          puts "#{@cX.bold}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /grey/:          puts "#{@cX.grey}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /pink/:          puts "#{@cX.pink}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /fuschia/:       puts "#{@cX.fuschia}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /ruby/:          puts "#{@cX.ruby}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /red/:           puts "#{@cX.red}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /orange/:        puts "#{@cX.orange}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /yellow/:        puts "#{@cX.yellow}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /brown/:         puts "#{@cX.brown}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /green/:         puts "#{@cX.green}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /darkgreen/:     puts "#{@cX.darkgreen}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /cyan/:          puts "#{@cX.cyan}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /blue/:          puts "#{@cX.blue}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      when /navy/:          puts "#{@cX.navy}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}"
      end
    end
    def sourcename(sourcefilename)
      @sourcefilename=sourcefilename
    end
    def basename(sourcefilename)
      @basename=sourcefilename.sub(/\.(?:[_-]?sst|ssm)$/,'')
    end
    def sisu
    end
    def rescue
      %{\t   #{@cX.orange}Rescued#{@cX.off} #{@cX.grey}An#{@cX.off} #{@cX.fuschia}ERROR#{@cX.off} #{@cX.grey}occurred, message:"#{@cX.off} #{@cX.fuschia}#{@txt[0]}#{@cX.off} #{@cX.grey}#{@txt[1]}#{@cX.off} #{@cX.brown}#{@txt[2]}#{@cX.off}}
    end
    def warn
      puts "\t  #{@cX.brown}#{@txt[0]}#{@cX.off} #{@cX.grey}#{@txt[1]}#{@cX.off}" 
    end
    def error
      puts "\t  #{@cX.fuschia}#{@txt[0]}#{@cX.off} #{@cX.brown}#{@txt[1]}#{@cX.off}" 
    end
    def error2
      puts %{\t  #{@cX.grey}#{@txt[0]}#{@cX.off} #{@cX.fuschia}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}}
    end
    def version
      puts "#{@cX.blue_hi}#{@txt[0]} #{@txt[1]}#{@cX.off} #{@cX.grey}of#{@cX.off} #{@cX.ruby}#{@txt[2]}#{@cX.off} #{@cX.grey}(#{@txt[3]})#{@cX.off} & #{@cX.ruby}Ruby#{@cX.off} #{@cX.grey}(#{@txt[4]})#{@cX.off}\n"
    end
    def html3numbers
      puts %{\t#{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.cyan}files processed#{@cX.off}. } +
        %{#{@cX.grey}} +
        %{scroll only: #{@txt[1]}, seg only: #{@txt[2]},} +
        %{#{@cX.off} } +
        %{#{@cX.cyan}joint scroll & seg: #{@txt[3]},#{@cX.off}} +
        %{#{@cX.grey} } +
        %{nav only: #{@txt[4]}} +
        %{#{@cX.off}}
    end
    def html2numbers
      puts %{\t#{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.cyan}files processed#{@cX.off}. } +
        %{#{@cX.grey}} +
        %{scroll only: #{@txt[1]},} +
        %{#{@cX.off} } +
        %{#{@cX.cyan}seg only: #{@txt[2]},#{@cX.off}} +
        %{#{@cX.grey} } +
        %{joint scroll & seg: #{@txt[3]}, nav only: #@n_files_nav} +
        %{#{@cX.off}}
    end
    def html1numbers
      puts %{\t#{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.cyan}files processed#{@cX.off}. } +
        %{#{@cX.cyan}scroll only: #{@txt[1]},#{@cX.off}} +
        %{#{@cX.grey} } +
        %{seg only: #{@txt[2]}, } +
        %{joint scroll & seg: #{@txt[3]}, nav only: #@n_files_nav} +
        %{#{@cX.off}}
    end
    def html0_numbers
      puts %{\t#{@cX.ruby}#{@txt[0]} files processed#{@cX.off}. } +
        %{#{@cX.grey}} +
        %{scroll only: #{@txt[1]}, seg only: #{@txt[2]}, joint scroll & seg: #{@txt[3]},} +
        %{#{@cX.off} } +
        %{#{@cX.cyan}nav only: #{@txt[4]}#{@cX.off}.}
    end
    def php_numbers
      puts %{  #{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.cyan}files processed#{@cX.off}. } +
        %{php_scr only: #{@txt[1]}, } +
        %{#{@cX.cyan}php_seg only: #{@txt[2]}#{@cX.off}, } +
        %{joint php scroll & seg: #{@txt[3]}, } +
        %{php_nav only: #{@txt[4]}}
    end
    def txt_grey
      puts "\t#{@cX.grey}#{@txt[0]}#{@cX.off} #{@cX.cyan}#{@txt[1]}#{@cX.off}" 
    end
    def txt_cyan
      puts "\t#{@cX.cyan}#{@txt[0]}#{@cX.off} #{@cX.grey}#{@txt[1]}#{@cX.off}" 
    end
    def txt_blue
      puts "\t#{@cX.blue}#{@txt[0]}#{@cX.off} #{@cX.grey}#{@txt[1]}#{@cX.off}" 
    end
    def url #clean
      blue
    end
    def result
      puts "\t#{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.blue}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}" 
    end
    def maintenance
      puts "\t#{@cX.grey}#{@txt[0]}#{@cX.off} #{@cX.brown}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off}" 
    end
    def instruct
      puts %{\t  #{@cX.grey}#{@txt[0]}#{@cX.off} #{@cX.cyan}#{@txt[1]}#{@cX.off} #{@cX.grey}#{@txt[2]}#{@cX.off} #{@cX.cyan}#{@txt[3]}#{@cX.off} #{@cX.grey}#{@txt[4]}#{@cX.off} "#{@cX.brown}#@f#{@cX.off}"}
    end
    def grey_open
      print @cX.grey
    end
    def p_off
      print @cX.off
    end
    def p_close
      print @cX.off
    end
    def flow
      puts %{\t#{@cX.grey}#{@txt[0]}#{@cX.off} #{@cX.ruby}->#{@cX.off}\n\t  #{@cX.blue}#{@txt[1]}#{@cX.off}}
    end
    def output
      puts %{\t#{@cX.grey}#{@txt[0]}#{@cX.off} #{@cX.ruby}->#{@cX.off}\n\t  #{@cX.blue}#{@txt[1]}#{@cX.off}}
    end
    def generic
      puts %{\t#{@cX.navy}#{@txt[0]} #{@txt[1]}#{@cX.off}. }
    end
    def files_processed
      puts %{\t#{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.grey}#{@txt[1]}#{@cX.off}}
    end
    def print_blue
      print "#{@cX.blue}#{@txt[0]} #{@txt[1]}#{@cX.off}"
    end
    def print_blue
      print "#{@cX.blue}#{@txt[0]} #{@txt[1]}#{@cX.off}"
    end
    def puts_blue
      puts "#{@cX.blue}#{@txt[0]} #{@txt[1]}#{@cX.off}"
    end
    def print_brown
      print "#{@cX.brown}#{@txt[0]}#{@cX.off}"
    end
    def blue_tab
      print  "\t#{@cX.blue}#{@txt[0]}#{@cX.off}\n"
    end
    def print_grey
      print "#{@cX.grey}#{@txt[0]} #{@txt[1]}#{@cX.off}"
    end
    def puts_grey
      puts "#{@cX.grey}#{@txt[0]} #{@txt[1]}#{@cX.off}"
    end
    def puts_brown
      puts "#{@cX.brown}#{@txt[0]} #{@txt[1]}#{@cX.off}"
    end
    def grey_tab
      print "\t#{@cX.grey}#{@txt[0]}#{@cX.off}\n"
    end
    def green_title
      puts %{#{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.grey}#{@txt[1]}#{@cX.off}}
    end
    def green_title_hi
      puts %{#{@cX.green_hi}#{@cX.black}#{@txt[0]}#{@cX.off*2} #{@cX.grey}#{@txt[1]}#{@cX.off}}
    end
    def green_hi_blue
      puts %{#{@cX.green_hi}#{@cX.black}#{@txt[0]}#{@cX.off*2} #{@cX.blue}#{@txt[1]}#{@cX.off}}
    end
    def grey_title_hi
      puts %{#{@cX.grey_hi}#{@cX.black}#{@txt[0]}#{@cX.off*2} #{@cX.blue}#{@txt[1]}#{@cX.off}}
    end
    def dark_grey_title_hi
      puts %{#{@cX.darkgrey_hi}#{@cX.black}#{@txt[0]}#{@cX.off*2} #{@cX.blue}#{@txt[1]}#{@cX.off}}
    end
    def cyan_title_hi
      puts %{#{@cX.cyan_hi}#{@cX.black}#{@txt[0]}#{@cX.off*2} #{@cX.blue}#{@txt[1]}#{@cX.off}}
    end
    def dbi_title
      puts %{#{@cX.blue_hi}#{@cX.black}#{@txt[0]}#{@cX.off*2} #{@cX.cyan}#{@cX.grey}#{@txt[1]}#{@cX.off*2} #{@cX.green}#{@txt[2]}#{@cX.off}}
    end
    def yellow_title_hi
      puts %{#{@cX.yellow_hi}#{@cX.black}#{@txt[0]}#{@cX.off*2} #{@cX.blue}#{@txt[1]}#{@cX.off}}
    end
    def term_sheet_title
      puts %{\t#{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.red_hi}#{@cX.black}#{@txt[1]}.#{@cX.off*2} }
    end
    def generic_number
      puts "#{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.grey}#{@txt[1]}#{@cX.off}"
    end
    def tex_numbers
      puts %{  #{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.cyan}tex/pdf files processed#{@cX.off}.}
    end
    def tex_info_numbers
      puts %{  #{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.cyan}texinfo files processed#{@cX.off}.}
    end
    def lout_numbers
      puts %{  #{@cX.green}#@n_lout#{@cX.off} #{@cX.cyan}lout/pdf files processed#{@cX.off}.}
    end
    def parameters
    end
    def reserved
      puts %{ #{@cX.grey_hi}#{@cX.black}reserved#{@cX.off*2}}
    end
    def meta_verse_title
      puts %{#{@cX.green_hi}#{@cX.black}MetaVerse#{@cX.off*2} } 
    end
    def meta_verse_title_v
      b=sourcename(@txt[0])
      puts %{#{@cX.green_hi}#{@cX.black}MetaVerse#{@cX.off*2} } +
        %{#{@cX.grey}#{@txt[0]}#{@cX.off} #{@cX.ruby}->#{@cX.off}\n\t  #{@cX.blue}~meta/#{b}.meta#{@cX.off}}
    end
    def meta_verse
      puts "\t#{@cX.grey}MetaVerse#{@cX.off}"
    end
    def meta_verse_skipped
      puts %{\t#{@cX.grey}MetaVerse: MarshalLoad (creation of metaVerse skipped)#{@cX.off}}
    end
    def meta_verse_load
      puts %{\t<<MetaVerse: MarshalLoad>>}
    end
    def html_title
      puts %{#{@cX.green_hi}#{@cX.black}HTML#{@cX.off*2}}
    end
    def html_output
      puts %{\t#{@cX.grey}#{@txt[0]}#{@cX.off} #{@cX.ruby}->#{@cX.off}\n\t  #{@cX.blue}#{@txt[1]}#{@cX.off}}
    end
    def php_title
      puts %{\n#{@cX.green_hi}#{@cX.black}PHP#{@cX.off*2}}
    end
    def segmented
      puts "\t#{@cX.grey}Seg#{@cX.off} #{@cX.green}#{@txt[0]}#{@cX.off} #{@cX.grey}segments#{@cX.off}"
    end
  end
end
module SiSU_Errors
  include SiSU_Env 
  include SiSU_Screen 
  class Info_error <Create_file
    def initialize(error,errorlist,cmd,fns='')
      @fns,@cmd,@error,@errorlist=fns,cmd,error,errorlist
      @cmd=if cmd and cmd =~/c/: 'Vc'
      else                    'V'
      end
    end
    def error                                                                    #creates file errorlog.sisu in pwd of last error reported
      file=if @fns: SiSU_Env::Create_file.new(@cmd,@fns).file_error
      else          File.new('/tmp/errorlog.sisu','w+')
      end
      file << @fns << "\n" << @error << "\n" << @errorlist 
      file.close
      if @cmd=~/[vVM]/: STDERR.puts SiSU_Screen::Ansi.new('',$!,$@).rescue
      else             SiSU_Screen::Ansi.new('',"rescued, exception reaised, silenced").puts_grey
      end
    end
  end
end
__END__
