/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is part of Skrooge and implements classes SKGDocumentBank.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgdocumentbank.h"
#include "skgbankobject.h"
#include "skgaccountobject.h"
#include "skgunitobject.h"
#include "skgunitvalueobject.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgservices.h"

#include <QSqlDatabase>
#include <QFile>

SKGDocumentBank::SKGDocumentBank(DatabaseMode iMode)
                : SKGDocument(iMode)
{
        SKGTRACEIN(10, "SKGDocumentBank::SKGDocumentBank");
        connect(this, SIGNAL(tableModified(QString, int)), this, SLOT(refreshCache(const QString&)));
}

SKGDocumentBank::~SKGDocumentBank()
{
        SKGTRACEIN(10, "SKGDocumentBank::~SKGDocumentBank");
}

QStringList SKGDocumentBank::getImpactedTable(const QString& iTable) const
{
        SKGTRACEIN(10, "SKGDocumentBank::getImpactedTable");
        QStringList output;
        if (iTable=="recurrentoperation") {
                output.push_back("operation");
        } else if (iTable=="operation") {
                output.push_back("account");
                output.push_back("category");
                output.push_back("recurrentoperation");
        } else if (iTable=="unitvalue") {
                output.push_back("unit");
        } else if (iTable=="unit") {
                output.push_back("operation");
        } else if (iTable=="category") {
                output.push_back("suboperation");
        } else if (iTable=="suboperation") {
                output.push_back("operation");
        } else if (iTable=="account") {
                output.push_back("bank");
        }
        return output;
}

QString SKGDocumentBank::getViewsIndexesAndTriggersVersion() const
{
        return "2009.03.16";
}

SKGError SKGDocumentBank::refreshViewsIndexesAndTriggers() const
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocumentBank::refreshViewsIndexesAndTriggers", err);

        QString version = getParameter("SKG_DB_BANK_VIEWS_VERSION");
        if (version==getViewsIndexesAndTriggersVersion()) return err;

        setParameter("SKG_DB_BANK_VIEWS_VERSION", getViewsIndexesAndTriggersVersion());


        //See currencies at http://www.xe.com/iso4217.php
        //WARNING: Don't forget to update getViewVersion when this methode is modified
        /**
         * This constant is used to initialized the data model (trigger creation)
         * IF YOU MODIFY THIS METHOD, DO NOT FORGET TO MODIFY getViewsIndexesAndTriggersVersion TOO
         */
        const QString BankInitialDataModelTrigger
        =
                DELETECASCADEPARAMETER("bank")
                +DELETECASCADEPARAMETER("account")
                +DELETECASCADEPARAMETER("unit")
                +DELETECASCADEPARAMETER("category")
                +DELETECASCADEPARAMETER("operation")
                +DELETECASCADEPARAMETER("suboperation")
                +DELETECASCADEPARAMETER("budget")+

                //Compute fullname
                "DROP TRIGGER IF EXISTS cpt_category_fullname1;;"
                "CREATE TRIGGER cpt_category_fullname1 " //This trigger must be the first
                "AFTER UPDATE OF t_fullname ON category BEGIN "
                "UPDATE category SET t_name=t_name WHERE r_category_id=new.id;"
                "END;;"

                "DROP TRIGGER IF EXISTS cpt_category_fullname2;;"
                "CREATE TRIGGER cpt_category_fullname2 "
                "AFTER INSERT ON category BEGIN "
                "UPDATE category SET t_fullname="
                "CASE WHEN r_category_id IS NULL THEN new.t_name ELSE (SELECT c.t_fullname from category c where c.id=new.r_category_id)||'"OBJECTSEPARATOR"'||new.t_name END "
                "WHERE id=new.id;"
                "END;;"

                "DROP TRIGGER IF EXISTS cpt_category_fullname3;;"
                "CREATE TRIGGER cpt_category_fullname3 "
                "AFTER UPDATE OF t_name ON category BEGIN "
                "UPDATE category SET t_fullname="
                "CASE WHEN r_category_id IS NULL OR r_category_id='' THEN new.t_name ELSE (SELECT c.t_fullname from category c where c.id=new.r_category_id)||'"OBJECTSEPARATOR"'||new.t_name END "
                "WHERE id=new.id;"
                "END;;"

                //-- Cascading delete - WARNING rewriten for skrooge to support recursive mode
                "DROP TRIGGER IF EXISTS fkdc_category_parent_id_category_id;;"
                "CREATE TRIGGER fkdc_category_parent_id_category_id "
                "BEFORE DELETE ON category "
                "FOR EACH ROW BEGIN "
                "    DELETE FROM category WHERE category.t_fullname LIKE OLD.t_fullname||'"OBJECTSEPARATOR"%'; "
                "END;; "

                /*     "CREATE TRIGGER fkdc_category_parent_id_category_id "
                		"BEFORE DELETE ON category "
                		"FOR EACH ROW BEGIN "
                		"    DELETE FROM category WHERE category.r_category_id=OLD.id; "
                		"END;; "*/
                ;

        /**
         * This constant is used to initialized the data model (index creation)
         */
        const QString BankInitialDataModelIndex
        =
                "DROP INDEX IF EXISTS uidx_unit_name;;"
                "CREATE UNIQUE INDEX uidx_unit_name ON unit(t_name);;"
                "DROP INDEX IF EXISTS uidx_unit_symbol;;"
                "CREATE UNIQUE INDEX uidx_unit_symbol ON unit(t_symbol);;"
                "DROP INDEX IF EXISTS uidx_unitvalue;;"
                "CREATE UNIQUE INDEX uidx_unitvalue ON unitvalue(d_date,rd_unit_id);;"
                "DROP INDEX IF EXISTS idx_unit_unit_id;;"
                "CREATE INDEX idx_unit_unit_id ON unitvalue(rd_unit_id);;"
                "DROP INDEX IF EXISTS uidx_bank_name;;"
                "CREATE UNIQUE INDEX uidx_bank_name ON bank(t_name);;"
                "DROP INDEX IF EXISTS uidx_account_name;;"
                "CREATE UNIQUE INDEX uidx_account_name ON account(t_name);;"
                "DROP INDEX IF EXISTS idx_account_bank_id;;"
                "CREATE INDEX idx_account_bank_id ON account(rd_bank_id);;"
                "DROP INDEX IF EXISTS idx_category_category_id;;"
                "CREATE INDEX idx_category_category_id ON category(r_category_id);;"
                "DROP INDEX IF EXISTS idx_category_t_fullname;;"
                "CREATE INDEX idx_category_t_fullname ON category(t_fullname);;"
                "DROP INDEX IF EXISTS uidx_category_parent_id_name;;"
                "CREATE UNIQUE INDEX uidx_category_parent_id_name ON category(t_name,r_category_id);;"
                "DROP INDEX IF EXISTS uidx_category_fullname;;"
                "CREATE UNIQUE INDEX uidx_category_fullname ON  category(t_fullname);;"
                "DROP INDEX IF EXISTS idx_operation_account_id;;"
                "CREATE INDEX  idx_operation_account_id ON operation (rd_account_id);;"
                "DROP INDEX IF EXISTS idx_operation_grouped_operation_id;;"
                "CREATE INDEX  idx_operation_grouped_operation_id ON operation (i_group_id);;"
                "DROP INDEX IF EXISTS idx_operation_t_mode;;"
                "CREATE INDEX  idx_operation_t_mode ON operation (t_mode);;"
                "DROP INDEX IF EXISTS idx_operation_t_payee;;"
                "CREATE INDEX  idx_operation_t_payee ON operation (t_payee);;"
                "DROP INDEX IF EXISTS idx_operation_i_number;;"
                "CREATE INDEX  idx_operation_i_number ON operation (i_number);;"
                "DROP INDEX IF EXISTS idx_operation_rd_account_id;;"
                "CREATE INDEX  idx_operation_rd_account_id ON operation (rd_account_id);;"
                "DROP INDEX IF EXISTS idx_operation_rc_unit_id;;"
                "CREATE INDEX  idx_operation_rc_unit_id ON operation (rc_unit_id);;"
                "DROP INDEX IF EXISTS idx_operation_t_status;;"
                "CREATE INDEX  idx_operation_t_status ON operation (t_status);;"
                "DROP INDEX IF EXISTS idx_operation_t_import_id;;"
                "CREATE INDEX  idx_operation_t_import_id ON operation (t_import_id);;"
                "DROP INDEX IF EXISTS idx_operation_tmp1_found_transfert;;"
                "CREATE INDEX  idx_operation_tmp1_found_transfert ON operation (id, rc_unit_id, d_date, rd_account_id);;"
                "DROP INDEX IF EXISTS idx_suboperation_operation_id;;"
                "CREATE INDEX idx_suboperation_operation_id ON suboperation (rd_operation_id);;"
                "DROP INDEX IF EXISTS idx_suboperation_category_id;;"
                "CREATE INDEX idx_suboperation_category_id ON suboperation (r_category_id);;"
                "DROP INDEX IF EXISTS idx_budget_category_id;;"
                "CREATE INDEX  idx_budget_category_id ON budget (r_category_id);;"
                "DROP INDEX IF EXISTS idx_recurrentoperation_rd_operation_id;;"
                "CREATE INDEX  idx_recurrentoperation_rd_operation_id ON recurrentoperation (rd_operation_id);;"
                ;

        /**
         * This constant is used to initialized the data model (view creation)
         */
        const QString BankInitialDataModelView
        =
                // ==================================================================
                //These following views contains only attributes used by corresponding class (better for performances)
                //unit
                "CREATE VIEW  v_unit_displayname AS "
                "SELECT *, t_name||' ('||t_symbol||')' AS t_displayname from unit;;"

                "CREATE VIEW  v_unit_tmp1 AS "
                "SELECT *,"
                "(SELECT MIN(s.d_date) FROM  unitvalue s WHERE s.rd_unit_id=unit.id) AS d_MINDATE, "
                "(SELECT MAX(s.d_date) FROM  unitvalue s WHERE s.rd_unit_id=unit.id) AS d_MAXDATE "
                "from unit;;"

                "CREATE VIEW  v_unit_tmp2 AS "
                "SELECT *,"
                "(SELECT count(1) FROM unitvalue s WHERE s.rd_unit_id=v_unit_tmp1.id) AS i_NBVALUES, "
                "ifnull((SELECT s.f_quantity FROM unitvalue s WHERE s.rd_unit_id=v_unit_tmp1.id AND s.d_date=v_unit_tmp1.d_MAXDATE),1) AS f_LASTVALUE "
                "from v_unit_tmp1;;"

                "CREATE VIEW  v_unit AS "
                "SELECT *,"
                "v_unit_tmp2.f_LASTVALUE*ifnull((SELECT s2.f_LASTVALUE FROM v_unit_tmp2 s2 WHERE s2.id=v_unit_tmp2.rd_unit_id) , 1) AS f_CURRENTAMOUNT "
                "from v_unit_tmp2;;"

                //unitvalue
                "CREATE VIEW  v_unitvalue_displayname AS "
                "SELECT *, (select t_displayname from v_unit_displayname where unitvalue.rd_unit_id=v_unit_displayname.id)||' '||d_date AS t_displayname from unitvalue;;"

                "CREATE VIEW  v_unitvalue AS "
                "SELECT * "
                "from unitvalue;;"

                //suboperation
                "CREATE VIEW  v_suboperation_displayname AS "
                "SELECT *, 'NOT YET IMPLEMENTED' AS t_displayname from suboperation;;"

                "CREATE VIEW  v_suboperation AS "
                "SELECT * "
                "from suboperation;;"

                //operation
                "CREATE VIEW  v_operation_numbers AS "
                "SELECT DISTINCT i_number, rd_account_id FROM operation;;"

                "CREATE VIEW  v_operation_next_numbers AS "
                "SELECT T1.i_number as i_number FROM v_operation_numbers as T1 LEFT OUTER JOIN v_operation_numbers T2 "
                "ON T2.rd_account_id=T1.rd_account_id AND T2.i_number=T1.i_number+1 "
                "WHERE T1.i_number!=0 AND (T2.i_number IS NULL) ORDER BY T1.i_number;;"

                "CREATE VIEW  v_operation_tmp1 AS "
                "SELECT *,"
                "(SELECT TOTAL(s.f_value) FROM suboperation s WHERE s.rd_operation_id=operation.ID) AS f_QUANTITY,"
                "(SELECT count(1) FROM suboperation s WHERE s.rd_operation_id=operation.ID) AS i_NBSUBCATEGORY "
                "from operation;;"

                "CREATE VIEW  v_operation AS "
                "SELECT *,"
                "((SELECT s.f_CURRENTAMOUNT FROM v_unit s WHERE s.id=v_operation_tmp1.rc_unit_id)*v_operation_tmp1.f_QUANTITY) AS f_CURRENTAMOUNT "
                "from v_operation_tmp1;;"

                "CREATE VIEW  v_operation_displayname AS "
                "SELECT *, d_date||' '||t_payee||' '||v_operation.f_CURRENTAMOUNT||' '||(SELECT (CASE WHEN s.t_symbol!='' THEN s.t_symbol ELSE s.t_name END) FROM unit s WHERE s.id=v_operation.rc_unit_id) AS t_displayname from v_operation;;"

                //account
                "CREATE VIEW  v_account AS "
                "SELECT "
                "*,"
                "(SELECT TOTAL(s.f_CURRENTAMOUNT) FROM v_operation s WHERE s.rd_account_id=account.id) AS f_CURRENTAMOUNT "
                "from account;;"

                //bank
                "CREATE VIEW  v_bank_displayname AS "
                "SELECT *, t_name AS t_displayname from bank;;"

                "CREATE VIEW  v_account_displayname AS "
                "SELECT *, (select t_displayname from v_bank_displayname where account.rd_bank_id=v_bank_displayname.id)||'-'||t_name AS t_displayname from account;;"

                "CREATE VIEW  v_bank AS "
                "SELECT *,"
                "(SELECT TOTAL(s.f_CURRENTAMOUNT) FROM v_account s WHERE s.rd_bank_id=bank.id) AS f_CURRENTAMOUNT "
                "from bank;;"

                //category
                "CREATE VIEW  v_category_displayname AS "
                "SELECT *, t_fullname AS t_displayname from category;;"

                "CREATE VIEW  v_category AS SELECT * "
                "from category;;"

                //recurrentoperation
                "CREATE VIEW  v_recurrentoperation AS "
                "SELECT *,"
                "i_period_increment||' '||(CASE WHEN t_period_unit='Y' THEN '"+SKGServices::stringToSqlString(tr("year(s)"))+"' ELSE (CASE WHEN t_period_unit='M' THEN '"+SKGServices::stringToSqlString(tr("month(s)"))+"' ELSE '"+SKGServices::stringToSqlString(tr("day(s)"))+"' END) END) AS t_PERIODNLS,"
                "date(d_date,i_period_increment||(CASE WHEN t_period_unit='Y' THEN ' year' ELSE (CASE WHEN t_period_unit='M' THEN ' month' ELSE ' day' END) END)) AS d_DUEDATE "
                "from recurrentoperation;;"

                "CREATE VIEW  v_recurrentoperation_displayname AS "
                "SELECT *, d_date||' '||t_PERIODNLS||' '||(select t_displayname from v_operation_displayname where v_operation_displayname.id=v_recurrentoperation.rd_operation_id) AS t_displayname from v_recurrentoperation;;"


                // ==================================================================
                //These following views contains all attributes needed for display
                //unitvalue
                "CREATE VIEW  v_unitvalue_display AS "
                "SELECT *,"
                "(SELECT (CASE WHEN s.t_symbol!='' THEN s.t_symbol ELSE s.t_name END) FROM unit s WHERE s.id=(SELECT s2.rd_unit_id FROM unit s2 WHERE s2.id=unitvalue.rd_unit_id)) AS t_UNIT,"
                "STRFTIME('%Y-%m',unitvalue.d_date) AS d_DATEMONTH,"
                "STRFTIME('%Y',unitvalue.d_date) AS d_DATEYEAR "
                "from unitvalue;;"

                //suboperation
                "CREATE VIEW  v_suboperation_display AS "
                "SELECT *,"
                "ifnull((SELECT s.t_fullname FROM category s WHERE s.id=v_suboperation.r_category_id),'') AS t_CATEGORY, "
                "(CASE WHEN v_suboperation.f_value>=0 THEN v_suboperation.f_value ELSE 0 END) AS f_VALUE_INCOME, "
                "(CASE WHEN v_suboperation.f_value<=0 THEN v_suboperation.f_value ELSE 0 END) AS f_VALUE_EXPENSE "

                "from v_suboperation;;"

                //operation
                "DROP VIEW IF EXISTS v_operation_display;;"
                "CREATE VIEW  v_operation_display AS "
                "SELECT *,"
                "(SELECT s.t_name FROM account s WHERE s.id=v_operation.rd_account_id) AS t_ACCOUNT,"
                "(SELECT (CASE WHEN s.t_symbol!='' THEN s.t_symbol ELSE s.t_name END) FROM unit s WHERE s.id=v_operation.rc_unit_id) AS t_UNIT,"
                "(CASE WHEN (SELECT count(1) FROM v_suboperation_display s WHERE s.rd_operation_id=v_operation.ID)>1 THEN '' ELSE (SELECT MAX(s.t_CATEGORY) FROM v_suboperation_display s WHERE s.rd_operation_id=v_operation.ID) END) AS t_CATEGORY,"
                "(CASE WHEN v_operation.f_QUANTITY<=0 THEN '-' ELSE  '+' END) AS t_TYPEEXPENSE,"
                "(CASE WHEN v_operation.f_QUANTITY<=0 THEN '"+SKGServices::stringToSqlString(tr("Expense"))+"' ELSE '"+SKGServices::stringToSqlString(tr("Income"))+"' END) AS t_TYPEEXPENSENLS, "
                "STRFTIME('%Y-%m',v_operation.d_date) AS d_DATEMONTH,"
                "STRFTIME('%Y',v_operation.d_date) AS d_DATEYEAR, "
                "(SELECT count(1) FROM v_recurrentoperation s WHERE s.rd_operation_id=v_operation.id) AS i_NBRECURRENT,  "
                "(CASE WHEN v_operation.f_QUANTITY>=0 THEN v_operation.f_CURRENTAMOUNT ELSE 0 END) AS f_CURRENTAMOUNT_INCOME, "
                "(CASE WHEN v_operation.f_QUANTITY<=0 THEN v_operation.f_CURRENTAMOUNT ELSE 0 END) AS f_CURRENTAMOUNT_EXPENSE "
                "from v_operation;;"

                //unit
                "CREATE VIEW  v_unit_display AS "
                "SELECT *,"
                "(SELECT (CASE WHEN s.t_symbol!='' THEN s.t_symbol ELSE s.t_name END) FROM unit s WHERE s.id=v_unit.rd_unit_id) AS t_UNIT,"
                "(CASE WHEN v_unit.t_type IN ('1', '2', 'C') THEN '"+SKGServices::stringToSqlString(tr("Currency"))+"' ELSE (CASE WHEN v_unit.t_type='S' THEN '"+SKGServices::stringToSqlString(tr("Share"))+"' ELSE (CASE WHEN v_unit.t_type='I' THEN '"+SKGServices::stringToSqlString(tr("Index"))+"' ELSE '"+SKGServices::stringToSqlString(tr("Object"))+"' END) END) END) AS t_TYPENLS "
                "from v_unit;;"

                "CREATE VIEW  v_unit_op AS "
                "SELECT *,"
                "(SELECT TOTAL(o.f_QUANTITY) FROM v_operation_display o WHERE o.rc_unit_id=v_unit_display.id) AS f_OPQUANTITY "
                "from v_unit_display;;"

                //account
                "CREATE VIEW  v_account_display AS "
                "SELECT "
                "(CASE WHEN t_type='C' THEN '"+SKGServices::stringToSqlString(tr("Current"))+"' ELSE (CASE WHEN t_type='D' THEN '"+SKGServices::stringToSqlString(tr("Credit card"))+"' ELSE (CASE WHEN t_type='A' THEN '"+SKGServices::stringToSqlString(tr("Actif"))+"' ELSE (CASE WHEN t_type='I' THEN '"+SKGServices::stringToSqlString(tr("Investment"))+"' ELSE (CASE WHEN t_type='O' THEN '"+SKGServices::stringToSqlString(tr("Other"))+"' END) END) END) END) END) AS t_TYPENLS,"
                "(SELECT s.t_name FROM bank s WHERE s.id=v_account.rd_bank_id) AS t_BANK,"
                "(SELECT s.t_bank_number FROM bank s WHERE s.id=v_account.rd_bank_id) AS t_BANK_NUMBER,"
                "(SELECT s.t_icon FROM bank s WHERE s.id=v_account.rd_bank_id) AS t_ICON,"
                "*,"
                "(SELECT TOTAL(s.f_CURRENTAMOUNT) FROM v_operation s WHERE s.rd_account_id=v_account.id AND s.t_status!='N') AS f_CHECKED, "
                "(SELECT TOTAL(s.f_CURRENTAMOUNT) FROM v_operation s WHERE s.rd_account_id=v_account.id AND s.t_status='N') AS f_COMING_SOON "
                "from v_account;;"

                //operations
                "CREATE VIEW  v_operation_consolidated AS "
                "SELECT op.*, "
                "(SELECT s.t_TYPENLS||'"OBJECTSEPARATOR"'||op.t_ACCOUNT FROM v_account_display s WHERE s.id=op.rd_account_id) AS t_ACCOUNTTYPE,"
                "(SELECT s.t_TYPENLS||'"OBJECTSEPARATOR"'||op.t_UNIT FROM v_unit_display s WHERE s.id=op.rc_unit_id) AS t_UNITTYPE,"
                "sop.id as i_SUBOPID, "
                "sop.t_comment as t_REALCOMMENT, "
                "sop.t_CATEGORY as t_REALCATEGORY, "
                "sop.r_category_id as i_IDCATEGORY, "
                "(CASE WHEN op.f_QUANTITY<0 THEN '-' ELSE (CASE WHEN op.f_QUANTITY=0 THEN '' ELSE '+' END) END) AS t_TYPEEXPENSE, "
                "(CASE WHEN op.f_QUANTITY<0 THEN '"+SKGServices::stringToSqlString(tr("Expense"))+"' ELSE (CASE WHEN op.f_QUANTITY=0 THEN '' ELSE '"+SKGServices::stringToSqlString(tr("Income"))+"' END) END) AS t_TYPEEXPENSENLS, "
                "sop.f_value AS f_REALQUANTITY, "
                "((SELECT s.f_CURRENTAMOUNT FROM v_unit s WHERE s.id=op.rc_unit_id)*sop.f_value) AS f_REALCURRENTAMOUNT, "
                "((SELECT s.f_CURRENTAMOUNT FROM v_unit s WHERE s.id=op.rc_unit_id)*sop.f_VALUE_INCOME) AS f_REALCURRENTAMOUNT_INCOME, "
                "((SELECT s.f_CURRENTAMOUNT FROM v_unit s WHERE s.id=op.rc_unit_id)*sop.f_VALUE_EXPENSE) AS f_REALCURRENTAMOUNT_EXPENSE "
                "from v_operation_display as op, v_suboperation_display as sop WHERE sop.rd_operation_id=op.ID;;"

                //category
                "CREATE VIEW  v_category_display_tmp AS SELECT *,"
                "(SELECT count(1) FROM v_operation_consolidated o WHERE o.i_IDCATEGORY=v_category.ID) AS i_NBOPERATIONS, "
                "(SELECT TOTAL(o.f_REALCURRENTAMOUNT) FROM v_operation_consolidated o WHERE o.i_IDCATEGORY=v_category.ID) AS f_REALCURRENTAMOUNT "
                "from v_category;;"

                "CREATE VIEW  v_category_display AS SELECT *,"
                "(SELECT TOTAL(c.f_REALCURRENTAMOUNT) FROM v_category_display_tmp c WHERE c.id=v_category_display_tmp.id OR c.t_fullname like v_category_display_tmp.t_fullname||'"OBJECTSEPARATOR"%') AS f_SUMCURRENTAMOUNT, "
                "(SELECT CAST(TOTAL(c.i_NBOPERATIONS) as INTEGER) FROM v_category_display_tmp c WHERE c.id=v_category_display_tmp.id OR c.t_fullname like v_category_display_tmp.t_fullname||'"OBJECTSEPARATOR"%') AS i_SUMNBOPERATIONS, "
                "(CASE WHEN v_category_display_tmp.f_REALCURRENTAMOUNT<0 THEN '"+SKGServices::stringToSqlString(tr("Expense"))+"' ELSE (CASE WHEN v_category_display_tmp.f_REALCURRENTAMOUNT=0 THEN '' ELSE '"+SKGServices::stringToSqlString(tr("Income"))+"' END) END) AS t_TYPEEXPENSE "
                "from v_category_display_tmp;;"

                //recurrentoperation
                "CREATE VIEW  v_recurrentoperation_display AS SELECT rop.*, op.t_displayname as  t_operationname "
                "from v_recurrentoperation rop, v_operation_displayname as op WHERE rop.rd_operation_id=op.ID;;"
                ;

        err=SKGDocument::refreshViewsIndexesAndTriggers();
        if (err.isSucceeded()) {
                QStringList views;
                SKGServices::getDistinctValues(this, "sqlite_master", "name", "type='view'", views);
                int nb=views.count();
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        QString view=views.at(i);
                        QString table=SKGServices::getRealTable(view);
                        if (table=="account" ||
                                        table=="unit" ||
                                        table=="unitvalue" ||
                                        table=="bank" ||
                                        table=="recurrentoperation" ||
                                        table=="operation" ||
                                        table=="suboperation" || table=="category") {
                                err = SKGServices::executeSqliteOrder(this, "DROP VIEW IF EXISTS "+view+";;");
                        }
                }
        }
        if (err.isSucceeded()) err = SKGServices::executeSqliteOrder(this, BankInitialDataModelIndex);
        if (err.isSucceeded()) err = SKGServices::executeSqliteOrder(this, BankInitialDataModelView);
        if (err.isSucceeded()) err = SKGServices::executeSqliteOrder(this, BankInitialDataModelTrigger);

        return err;
}

SKGError SKGDocumentBank::migrate(bool& oMigrationDone)
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocumentBank::migrate", err);
        oMigrationDone=false;

        err = beginTransaction("#INTERNAL#");

        QString version = getParameter("SKG_DB_BANK_VERSION");
        QString initialversion = version;
        if (version.isEmpty()) {
                SKGTRACEL(10) << "Initial creation" << endl;
                /**
                 * This constant is used to initialized the data model.
                 * Rules for attribute name:
                 *    t_xxx for TEXT and VARCHAR
                 *    d_xxx for DATE
                 *    f_xxx for FLOAT
                 *    i_xxx for INTEGER
                 *    r_xxx for a link without constraint
                 *    rc_pointed_table_pointed_attribute_xxx for link on other an objet in named "pointed_table" with "pointed_attribute"=id of pointing object
                 *                                       a constraint will be created without DELETE CASCADE
                 *    rd_pointed_table_pointed_attribute_xxx for link on other an objet in named "pointed_table" with "pointed_attribute"=id of pointing object
                 *                                       a constraint will be created with DELETE CASCADE
                 *    xxx must be in lower case for R/W attributes and in upper case for R/O attributes
                 * Rules for table name:
                 *    v_yyy for views
                 */
                const QString BankInitialDataModel
                =
                        // ==================================================================
                        //Table unit
                        "DROP TABLE IF EXISTS unit;;"
                        "CREATE TABLE unit("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "t_name TEXT NOT NULL,"
                        "t_symbol TEXT NOT NULL DEFAULT '',"
                        "t_country TEXT NOT NULL DEFAULT '',"
                        "t_type VARCHAR(1) NOT NULL DEFAULT 'C' CHECK (t_type IN ('1', '2', 'C', 'S', 'I', 'O')),"
                        //1=main currency, 2=secondary currency, C=currencies S=share, I=index, O=object
                        "t_internet_code TEXT NOT NULL DEFAULT '',"
                        "rd_unit_id INTEGER NOT NULL DEFAULT 0);;"

                        // ==================================================================
                        //Table unitvalue
                        "DROP TABLE IF EXISTS unitvalue;;"
                        "CREATE TABLE unitvalue("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "rd_unit_id INTEGER NOT NULL,"
                        "d_date DATE NOT NULL,"
                        "f_quantity FLOAT NOT NULL CHECK (f_quantity>=0));;"

                        // ==================================================================
                        //Table bank
                        "DROP TABLE IF EXISTS bank;;"
                        "CREATE TABLE bank ("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "t_name TEXT NOT NULL DEFAULT '',"
                        "t_bank_number TEXT NOT NULL DEFAULT '',"
                        "t_icon TEXT NOT NULL DEFAULT '');;"

                        // ==================================================================
                        //Table account
                        "DROP TABLE IF EXISTS account;;"
                        "CREATE TABLE account("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "t_name TEXT NOT NULL,"
                        "t_number TEXT NOT NULL DEFAULT '',"
                        "t_agency_number TEXT NOT NULL DEFAULT '',"
                        "t_agency_address TEXT NOT NULL DEFAULT '',"
                        "t_comment TEXT NOT NULL DEFAULT '',"
                        "t_close VARCHAR(1) DEFAULT 'N' CHECK (t_close IN ('Y', 'N')),"
                        "t_type VARCHAR(1) NOT NULL DEFAULT 'C' CHECK (t_type IN ('C', 'D', 'A', 'I', 'O')),"
                        //C=current D=credit card A=actif (for objects) I=Investment O=other
                        "rd_bank_id INTEGER NOT NULL);;"

                        // ==================================================================
                        //Table category
                        "DROP TABLE IF EXISTS category;;"
                        "CREATE TABLE category ("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "t_name TEXT NOT NULL DEFAULT '' CHECK (t_name NOT LIKE '%"OBJECTSEPARATOR"%'),"
                        "t_fullname TEXT,"
                        "r_category_id INT);;"

                        // ==================================================================
                        //Table operation
                        "DROP TABLE IF EXISTS operation;;"
                        "CREATE TABLE operation("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "i_group_id INTEGER NOT NULL DEFAULT 0,"
                        "i_number INTEGER DEFAULT 0 CHECK (i_number>=0),"
                        "d_date DATE NOT NULL DEFAULT '0000-00-00',"
                        "d_effective_date DATE,"
                        "rd_account_id INTEGER NOT NULL,"
                        "t_mode TEXT NOT NULL DEFAULT '',"
                        "t_payee TEXT NOT NULL DEFAULT '',"
                        "t_comment TEXT NOT NULL DEFAULT '',"
                        "rc_unit_id INTEGER NOT NULL,"
                        "t_status VARCHAR(1) DEFAULT 'N' CHECK (t_status IN ('N', 'P', 'C')),"
                        "t_bookmarked VARCHAR(1) DEFAULT 'N' CHECK (t_bookmarked IN ('Y', 'N')),"
                        "t_imported VARCHAR(1) DEFAULT 'N' CHECK (t_imported IN ('Y', 'N', 'P')),"
                        "t_import_id TEXT DEFAULT '',"
                        "r_recurrentoperation_id INTEGER NOT NULL DEFAULT 0);;"

                        // ==================================================================
                        //Table suboperation
                        "DROP TABLE IF EXISTS suboperation;;"
                        "CREATE TABLE suboperation("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "t_comment TEXT NOT NULL DEFAULT '',"
                        "rd_operation_id INTEGER NOT NULL,"
                        "r_category_id INTEGER NOT NULL DEFAULT 0,"
                        "f_value FLOAT NOT NULL DEFAULT 0.0);;"

                        // ==================================================================
                        //Table budget
                        "DROP TABLE IF EXISTS budget;;"
                        "CREATE TABLE budget ("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "r_category_id INTEGER NOT NULL ,"
                        "f_amount FLOAT NOT NULL"
                        ");;"

                        // ==================================================================
                        //Table recurrentoperation
                        "DROP TABLE IF EXISTS recurrentoperation;;"
                        "CREATE TABLE recurrentoperation ("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "d_date DATE NOT NULL DEFAULT '0000-00-00',"
                        "rd_operation_id INTEGER NOT NULL,"
                        "i_period_increment INTEGER NOT NULL DEFAULT 1 CHECK (i_period_increment>=0),"
                        "t_period_unit TEXT NOT NULL DEFAULT 'M' CHECK (t_period_unit IN ('D', 'M', 'Y')),"
                        "t_auto_write VARCHAR(1) DEFAULT 'Y' CHECK (t_auto_write IN ('Y', 'N')),"
                        "i_auto_write_days INTEGER NOT NULL DEFAULT 5 CHECK (i_auto_write_days>=0),"
                        "t_warn VARCHAR(1) DEFAULT 'Y' CHECK (t_auto_write IN ('Y', 'N')),"
                        "i_warn_days INTEGER NOT NULL DEFAULT 5 CHECK (i_warn_days>=0)"
                        ");;"
                        ;

                if (err.isSucceeded())  err = SKGServices::executeSqliteOrder(this, BankInitialDataModel);

                //Set new version
                version = "1.3";
                if (err.isSucceeded()) err = SKGDocument::setParameter("SKG_DB_BANK_VERSION", version);
        }

        if (err.isSucceeded() && version == "0.1") {
                SKGTRACEL(10) << "Migration from 0.1 to 0.2" <<endl;

                err = SKGServices::executeSqliteOrder(this, "ALTER TABLE unit ADD COLUMN rc_unit_id INTEGER NOT NULL DEFAULT 0");

                //Set new version
                version="0.2";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.2") {
                SKGTRACEL(10) << "Migration from 0.2 to 0.3" <<endl;

                QString sql=
                        "DROP TABLE IF EXISTS unitvalue2;;"
                        "CREATE TABLE unitvalue2("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "rd_unit_id INTEGER NOT NULL,"
                        "d_date DATE NOT NULL,"
                        "f_quantity FLOAT NOT NULL CHECK (f_quantity>=0));;"

                        "INSERT INTO unitvalue2 (id,rd_unit_id,d_date,f_quantity) SELECT id,rd_unit_id,d_date,f_value from unitvalue;;"

                        "DROP TABLE IF EXISTS unitvalue;;"
                        "ALTER TABLE unitvalue2 RENAME TO unitvalue;;"
                        ;

                err = SKGServices::executeSqliteOrder(this, sql);

                //Set new version
                version="0.3";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.3") {
                SKGTRACEL(10) << "Migration from 0.3 to 0.4" <<endl;

                err = SKGServices::executeSqliteOrder(this, "ALTER TABLE operation ADD COLUMN t_import_id TEXT DEFAULT ''");

                //Set new version
                version="0.4";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.4") {
                SKGTRACEL(10) << "Migration from 0.4 to 0.5" <<endl;

                QString sql=
                        "DROP TABLE IF EXISTS recurrentoperation;;"
                        "CREATE TABLE recurrentoperation ("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "d_date DATE NOT NULL DEFAULT '0000-00-00',"
                        "rd_operation_id INTEGER NOT NULL,"
                        "i_period_increment INTEGER NOT NULL DEFAULT 1 CHECK (i_period_increment>=0),"
                        "t_period_unit TEXT NOT NULL DEFAULT 'M' CHECK (t_period_unit IN ('D', 'M', 'Y')),"
                        "t_auto_write VARCHAR(1) DEFAULT 'Y' CHECK (t_auto_write IN ('Y', 'N')),"
                        "i_auto_write_days INTEGER NOT NULL DEFAULT 5 CHECK (i_auto_write_days>=0),"
                        "t_warn VARCHAR(1) DEFAULT 'Y' CHECK (t_auto_write IN ('Y', 'N')),"
                        "i_warn_days INTEGER NOT NULL DEFAULT 5 CHECK (i_warn_days>=0)"
                        ");;"
                        "ALTER TABLE operation ADD COLUMN r_recurrentoperation_id INTEGER NOT NULL DEFAULT 0;;"
                        ;
                err = SKGServices::executeSqliteOrder(this, sql);

                //Set new version
                version="0.5";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.5") {
                SKGTRACEL(10) << "Migration from 0.5 to 0.6" <<endl;

                err = SKGServices::executeSqliteOrder(this, "ALTER TABLE account ADD COLUMN t_comment TEXT NOT NULL DEFAULT ''");

                //Set new version
                version="0.6";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.6") {
                SKGTRACEL(10) << "Migration from 0.6 to 0.7" <<endl;

                QString sql=
                        "DROP TABLE IF EXISTS unit2;;"
                        "CREATE TABLE unit2("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "t_name TEXT NOT NULL,"
                        "t_symbol TEXT NOT NULL DEFAULT '',"
                        "t_country TEXT NOT NULL DEFAULT '',"
                        "t_type VARCHAR(1) NOT NULL DEFAULT 'C' CHECK (t_type IN ('1', '2', 'C', 'S', 'O')),"
                        //1=main currency, 2=secondary currency, C=currencies S=share O=object
                        "t_internet_code TEXT NOT NULL DEFAULT '',"
                        "rd_unit_id INTEGER NOT NULL DEFAULT 0);;"

                        "INSERT INTO unit2 (id,t_name,t_symbol,t_country,t_type,t_internet_code,rd_unit_id) SELECT id,t_name,t_symbol,t_country,t_type,t_internet_code,rc_unit_id from unit;;"

                        "DROP TABLE IF EXISTS unit;;"
                        "ALTER TABLE unit2 RENAME TO unit;;"
                        ;

                err = SKGServices::executeSqliteOrder(this, sql);

                //Set new version
                version="0.7";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.7") {
                SKGTRACEL(10) << "Migration from 0.7 to 0.8" <<endl;

                QString sql="DELETE FROM operation where id in (select id from operation op where (select count(1) from suboperation sop where sop.rd_operation_id=op.id)=0)"
                            ;
                err = SKGServices::executeSqliteOrder(this, sql);

                //Set new version
                version="0.8";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.8") {
                SKGTRACEL(10) << "Migration from 0.8 to 0.9" <<endl;

                QString sql="UPDATE operation set i_group_id=0 where i_group_id=''"
                            ;
                err = SKGServices::executeSqliteOrder(this, sql);

                //Set new version
                version="0.9";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.9") {
                SKGTRACEL(10) << "Migration from 0.9 to 1.0" <<endl;

                QString sql=
                        "DROP TABLE IF EXISTS unit2;;"
                        "CREATE TABLE unit2("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                        "t_name TEXT NOT NULL,"
                        "t_symbol TEXT NOT NULL DEFAULT '',"
                        "t_country TEXT NOT NULL DEFAULT '',"
                        "t_type VARCHAR(1) NOT NULL DEFAULT 'C' CHECK (t_type IN ('1', '2', 'C', 'S', 'I', 'O')),"
                        //1=main currency, 2=secondary currency, C=currencies S=share, I=index, O=object
                        "t_internet_code TEXT NOT NULL DEFAULT '',"
                        "rd_unit_id INTEGER NOT NULL DEFAULT 0);;"

                        "INSERT INTO unit2 (id,t_name,t_symbol,t_country,t_type,t_internet_code,rd_unit_id) SELECT id,t_name,t_symbol,t_country,t_type,t_internet_code,rd_unit_id from unit;;"

                        "DROP TABLE IF EXISTS unit;;"
                        "ALTER TABLE unit2 RENAME TO unit;;"
                        ;

                err = SKGServices::executeSqliteOrder(this, sql);

                //Set new version
                version="1.0";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "1.0") {
                SKGTRACEL(10) << "Migration from 1.0 to 1.1" <<endl;

                QString sql= "DELETE from parameters where t_name like 'SKG_MONTHLY_REPORT_%';;" ;
                err = SKGServices::executeSqliteOrder(this, sql);

                //Set new version
                version="1.1";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "1.1") {
                SKGTRACEL(10) << "Migration from 1.1 to 1.2" <<endl;

                QString sql=
                        "ALTER TABLE suboperation ADD COLUMN t_comment TEXT NOT NULL DEFAULT '';;"
                        ;

                err = SKGServices::executeSqliteOrder(this, sql);

                //Set new version
                version="1.2";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "1.2") {
                SKGTRACEL(10) << "Migration from 1.2 to 1.3" <<endl;

                QString sql=
                        "UPDATE node set f_sortorder=id where f_sortorder is NULL or f_sortorder='';;"
                        ;

                err = SKGServices::executeSqliteOrder(this, sql);

                //Set new version
                version="1.3";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_BANK_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded()) {
                bool mig=false;
                err = SKGDocument::migrate(mig);
                oMigrationDone=oMigrationDone|mig;
        }

        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocument::migrate() failed. Migration database from version %1 to version %2 failed").arg(initialversion).arg(version));

        //close temporary transaction
        if (err.isSucceeded()) err=endTransaction(true);
        else  endTransaction(false);

        return err;
}

SKGError SKGDocumentBank::dump(int iMode) const
{
        SKGError err;
        if (getDatabase()) {
                //dump parameters
                SKGTRACE << "=== START DUMP BANK DOCUMENT ===" << endl;
                err = SKGDocument::dump(iMode);

                if (iMode&DUMPUNIT) {
                        SKGTRACE << "=== DUMPUNIT (UNITS))===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "select * from v_unit_display order by id"));

                        SKGTRACE << "=== DUMPUNIT (VALUES) ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "select * from v_unitvalue_display order by rd_unit_id, d_date"));
                }

                if (iMode&DUMPACCOUNT) {
                        SKGTRACE << "=== DUMPACCOUNT (BANKS) ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "select * from v_bank order by id"));

                        SKGTRACE << "=== DUMPACCOUNT (ACCOUNTS) ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "select * from v_account_display order by rd_bank_id, id"));
                }

                if (iMode&DUMPOPERATION) {
                        SKGTRACE << "=== DUMPOPERATION (OPERATIONS) ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "select * from v_operation_display order by id"));

                        SKGTRACE << "=== DUMPOPERATION (SUBOPERATIONS) ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "select * from v_suboperation_display order by rd_operation_id, id"));

                        SKGTRACE << "=== DUMPOPERATION (RECURRENT) ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "select * from v_recurrentoperation order by rd_operation_id, id"));
                }

                if (iMode&DUMPCATEGORY) {
                        SKGTRACE << "=== DUMPCATEGORY ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "select * from v_category_display order by r_category_id, id"));
                }

                SKGTRACE << "=== END DUMP BANK DOCUMENT ===" << endl;
        }

        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocument::dump() failed"));
        return err;
}

SKGError SKGDocumentBank::addOrModifyUnitValue(const QString& iUnitName, const QDate& iDate, double iValue, SKGUnitValueObject* oValue) const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocumentBank::addOrModifyUnitValue", err);

        //Creation or update of the unit
        SKGUnitObject unit(this);
        err = unit.setName(iUnitName);
        if (!unit.exist()) {
                if (err.isSucceeded()) err = unit.setSymbol(iUnitName);
                if (err.isSucceeded()) err = unit.save();
        } else err=unit.load();

        //Creation or update of the value
        SKGUnitValueObject value;
        if (err.isSucceeded()) err = unit.addUnitValue(value);
        if (err.isSucceeded()) err = value.setDate(iDate);
        if (err.isSucceeded()) err = value.setQuantity(iValue);
        if (err.isSucceeded()) err = value.save();

        if (oValue) *oValue=value;

        //Add error if needed
        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocumentBank::addOrModifyUnitValue on [%1] failed").arg(iUnitName));
        return err;
}

SKGError SKGDocumentBank::ModifyUnit(const QString& iUnitName, const QString& iInternetCode) const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocumentBank::ModifyUnit", err);

        SKGNamedObject temporaryResult;
        err = SKGNamedObject::getObjectByName(this, "unit", iUnitName, temporaryResult);
        if (err.isSucceeded()) {
                err = temporaryResult.setAttribute("t_internet_code", iInternetCode);
                if (err.isSucceeded())
                        err = temporaryResult.save();
        }

        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocumentBank::ModifyUnit on [%1] failed").arg(iUnitName));
        return err;
}

SKGError SKGDocumentBank::RenameUnit(const QString& iUnitName, const QString& inewName) const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocumentBank::RenameUnit", err);

        //Modify unit
        SKGNamedObject temporaryResult;
        err = SKGNamedObject::getObjectByName(this, "unit", iUnitName, temporaryResult);
        if (err.isSucceeded()) {
                err = temporaryResult.setName(inewName);
                if (err.isSucceeded())
                        err = temporaryResult.save();
        }

        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocumentBank::RenameUnit on [%1] failed").arg(iUnitName));
        return err;
}

SKGError SKGDocumentBank::deleteUnit(const QString& iUnitName) const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocumentBank::deleteUnit", err);

        //delete unit
        SKGNamedObject temporaryResult;
        err = SKGNamedObject::getObjectByName(this, "unit", iUnitName, temporaryResult);
        if (err.isSucceeded()) err = temporaryResult.remove();
        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocumentBank::deleteUnit on [%1] failed").arg(iUnitName));
        return err;
}

QString SKGDocumentBank::getPrimaryUnit()
{
        QString primaryUnitCache=getCachedValue("primaryUnitCache");
        if (primaryUnitCache.isEmpty()) {
                this->refreshCache("unit");
                primaryUnitCache=getCachedValue("primaryUnitCache");
        }
        return primaryUnitCache;
}

QString SKGDocumentBank::getSecondaryUnit()
{
        QString secondaryUnitCache=getCachedValue("secondaryUnitCache");
        if (secondaryUnitCache.isEmpty()) {
                this->refreshCache("unit");
                secondaryUnitCache=getCachedValue("secondaryUnitCache");
        }
        return secondaryUnitCache;
}

double SKGDocumentBank::getSecondaryUnitValue()
{
        QString secondaryUnitValueCache=getCachedValue("secondaryUnitValueCache");
        if (secondaryUnitValueCache.isEmpty()) {
                this->refreshCache("unit");
                secondaryUnitValueCache=getCachedValue("secondaryUnitValueCache");
        }
        return SKGServices::stringToDouble(secondaryUnitValueCache);
}

void SKGDocumentBank::refreshCache(const QString& iTable)
{
        if (iTable=="unit") {
                SKGStringListList result;
                SKGServices::executeSelectSqliteOrder(this, "SELECT t_symbol from unit where t_type='1'", result);
                if (result.size()==2) addValueInCache("primaryUnitCache", result.at(1).at(0));
                else addValueInCache("primaryUnitCache", "");

                SKGServices::executeSelectSqliteOrder(this, "SELECT t_symbol, f_CURRENTAMOUNT from v_unit where t_type='2'", result);
                if (result.size()==2) {
                        addValueInCache("secondaryUnitCache", result.at(1).at(0));
                        addValueInCache("secondaryUnitValueCache", result.at(1).at(1));
                } else {
                        addValueInCache("secondaryUnitCache", "");
                        addValueInCache("secondaryUnitValueCache", "1");
                }
        }
}

SKGError SKGDocumentBank::addOrModifyAccount(const QString& iName, const QString& iNumber, const QString& iBankName) const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocumentBank::addOrModifyAccount", err);

        //Creation or update of the bank
        SKGBankObject bank(this);
        err = bank.setName(iBankName);
        if (err.isSucceeded())  err = bank.save();

        //Creation or update of the account
        SKGAccountObject account;
        if (err.isSucceeded()) err = bank.addAccount(account);
        if (err.isSucceeded()) err = account.setAttribute("rd_bank_id", SKGServices::intToString(bank.getID()));
        if (err.isSucceeded()) err = account.setName(iName);
        if (err.isSucceeded()) err = account.setAttribute("t_number", iNumber);
        if (err.isSucceeded()) err = account.save();

        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocumentBank::addOrModifyAccount on [%1] failed").arg(iName));
        return err;
}

SKGError SKGDocumentBank::RenameAccount(const QString& iName, const QString& inewName) const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocumentBank::RenameAccount", err);

        //Rename account
        SKGNamedObject temporaryResult;
        err = SKGNamedObject::getObjectByName(this, "account", iName, temporaryResult);
        if (err.isSucceeded()) {
                err = temporaryResult.setName(inewName);
                if (err.isSucceeded())
                        err = temporaryResult.save();
        }

        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocumentBank::RenameAccount on [%1] failed").arg(iName));
        return err;
}

SKGError SKGDocumentBank::deleteAccount(const QString& iName) const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocumentBank::deleteAccount", err);

        //delete account
        SKGNamedObject temporaryResult;
        err = SKGNamedObject::getObjectByName(this, "account", iName, temporaryResult);
        if (err.isSucceeded()) {
                err = temporaryResult.remove();
        }

        if (err.isFailed()) err.addError(ERR_FAIL, tr("Operation [%1] on [%2] failed").arg("SKGDocumentBank::deleteAccount").arg(iName));
        return err;
}

QString SKGDocumentBank::getDisplay(const QString& iString) const
{
        QString output=iString.toLower();

        //Internationallization
        if (output.endsWith("account.t_name")) return tr("Account");
        else if (output.endsWith("t_operationname")) return tr("Operation");
        else if (output.endsWith("t_name")) return tr("Name");
        else if (output.endsWith("account.f_value")) return tr("Balance");
        else if (output.endsWith("f_value")) return tr("Amount");
        else if (output.endsWith("f_currentamount") || output.endsWith("f_realcurrentamount")) return tr("Current amount");
        else if (output.endsWith("f_currentamount_income") || output.endsWith("f_realcurrentamount_income")) return tr("Income");
        else if (output.endsWith("f_currentamount_expense") || output.endsWith("f_realcurrentamount_expense")) return tr("Expense");
        else if (output.endsWith("quantity")) return tr("Quantity");
        else if (output.endsWith("account.t_number")) return tr("Account number");
        else if (output.endsWith("t_number") || output.endsWith("i_number")) return tr("Number");
        else if (output.endsWith("t_bank_number")) return tr("Bank number");
        else if (output.endsWith("t_agency_number")) return tr("Agency number");
        else if (output.endsWith("t_agency_address")) return tr("Agency address");
        else if (output.endsWith("t_payee")) return tr("Payee");
        else if (output.endsWith("t_comment")|| output.endsWith("t_realcomment")) return tr("Comment");
        else if (output.endsWith("t_mode")) return tr("Mode");
        else if (output.endsWith("d_date")) return tr("Date");
        else if (output.endsWith("t_account")) return tr("Account");
        else if (output.endsWith("t_category")|| output.endsWith("t_realcategory")) return tr("Category");
        else if (output.endsWith("t_bank")) return tr("Bank");
        else if (output.endsWith("t_unit")) return tr("Unit");
        else if (output.endsWith("f_checked")) return tr("Checked");
        else if (output.endsWith("f_coming_soon")) return tr("Foreseen");
        else if (output.endsWith("t_symbol")) return tr("Symbol");
        else if (output.endsWith("t_country")) return tr("Country");
        else if (output.endsWith("t_type")) return tr("Type");
        else if (output.endsWith("t_internet_code")) return tr("Internet code");
        else if (output.endsWith("i_nboperations")) return tr("Number of operations");
        else if (output.endsWith("t_periodnls")) return tr("Periodicity");
        else if (output.endsWith("i_auto_write_days")) return tr("Auto write");
        else if (output.endsWith("i_warn_days")) return tr("Warn");
        else if (output.endsWith("d_duedate")) return tr("Due date");
        else if (output.endsWith("t_close")) return tr("Closed");
        else if (output.endsWith("t_bookmarked")) return tr("Bookmarked");
        else if (output.endsWith("t_status")) return tr("Status");
        else if (output.endsWith("i_nbrecurrent")) return tr("Scheduled");
        else if (output.endsWith("i_sumnboperations")) return tr("Number of operations (cumulative)");
        else if (output.endsWith("f_sumcurrentamount")) return tr("Current amount (cumulative)");
        else output=SKGDocument::getDisplay(iString);
        return output;
}
#include "skgdocumentbank.moc"
