/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for undo and redo operation.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgundoredoplugin.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgmainpanel.h"
#include "skgundoredoplugindockwidget.h"
#include "skgservices.h"
#include "skrooge_undoredo.h"

#include <QtGui/QWidget>

#include <kactioncollection.h>
#include <kstandardaction.h>
#include <ktoolbarpopupaction.h>

K_PLUGIN_FACTORY(SKGUndoRedoPluginFactory, registerPlugin<SKGUndoRedoPlugin>();)
K_EXPORT_PLUGIN(SKGUndoRedoPluginFactory("skrooge_undoredo", "skrooge_undoredo"))

SKGUndoRedoPlugin::SKGUndoRedoPlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent), dockWidget(NULL)
{
        SKGTRACEIN(10, "SKGUndoRedoPlugin::SKGUndoRedoPlugin");
}

SKGUndoRedoPlugin::~SKGUndoRedoPlugin()
{
        SKGTRACEIN(10, "SKGUndoRedoPlugin::~SKGUndoRedoPlugin");
        parent=NULL;
        currentBankDocument=NULL;
}

void SKGUndoRedoPlugin::setupActions(SKGMainPanel* iParent, SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGUndoRedoPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;
        parent=iParent;

        setComponentData( SKGUndoRedoPluginFactory::componentData() );
        setXMLFile("skrooge_undoredo.rc");

        dockWidget = new QDockWidget(parent);
        dockWidget->setObjectName(QString::fromUtf8("skrooge_undoredo_docwidget"));
        dockWidget->setAllowedAreas(Qt::LeftDockWidgetArea|Qt::RightDockWidgetArea);
        dockWidget->setWindowTitle(title());
        dockWidget->setWidget(new SKGUndoRedoPluginDockWidget(parent, currentBankDocument));

        //Menu
        undoSaveAction = new KAction(KIcon("document-revert"), i18n("&Undo document"), this);
        connect(undoSaveAction, SIGNAL(triggered(bool)), this, SLOT(actionUndoSave()));
        actionCollection()->addAction( QLatin1String("edit_undolastsave"), undoSaveAction );
        undoSaveAction->setShortcut(Qt::CTRL+Qt::ALT+Qt::Key_Z);
        iParent->registedGlobalAction("edit_undolastsave", undoSaveAction);

        undoAction = new KToolBarPopupAction(KIcon("edit-undo"), i18n("&Undo"), this);
        connect(undoAction, SIGNAL(triggered(bool)), this, SLOT(actionUndo()));
        actionCollection()->addAction( QLatin1String("edit_undo"), undoAction );
        undoAction->setShortcut(Qt::CTRL+Qt::Key_Z);
        undoMenu=(KMenu*) undoAction->menu();
        connect(undoMenu , SIGNAL( aboutToShow() ), this, SLOT( onShowUndoMenu() ) );

        undoAction->setStickyMenu(false);
        undoAction->setData(1);

        //undoAction=KStandardAction::undo(this, SLOT(actionUndo()), actionCollection());
        iParent->registedGlobalAction("edit-undo", undoAction);

        redoAction = new KToolBarPopupAction(KIcon("edit-redo"), i18n("&Redo"), this);
        connect(redoAction, SIGNAL(triggered(bool)), this, SLOT(actionRedo()));
        actionCollection()->addAction( QLatin1String("edit_redo"), redoAction );
        redoAction->setShortcut(Qt::CTRL+Qt::SHIFT+Qt::Key_Z);
        redoMenu=(KMenu*) redoAction->menu();
        connect(redoMenu , SIGNAL( aboutToShow() ), this, SLOT( onShowRedoMenu() ) );

        redoAction->setStickyMenu(false);
        redoAction->setData(1);

        //redoAction=KStandardAction::redo(this, SLOT(actionRedo()), actionCollection());
        iParent->registedGlobalAction("edit-redo", redoAction);

        // add action to control hide / display of history
        dockWidget->toggleViewAction()->setShortcut(Qt::SHIFT+Qt::Key_F11);
        actionCollection()->addAction("view_transactions", dockWidget->toggleViewAction());
}

void SKGUndoRedoPlugin::refresh()
{
        SKGTRACEIN(10, "SKGUndoRedoPlugin::refresh");

        bool undoPossible=(currentBankDocument->getNbTransaction(SKGDocument::UNDO)>0);
        undoSaveAction->setEnabled(undoPossible);
        undoAction->setEnabled(undoPossible);
        redoAction->setEnabled(currentBankDocument->getNbTransaction(SKGDocument::REDO)>0);

        //Refresh undo redo
        QString name;
        currentBankDocument->getTransactionToTreat(SKGDocument::UNDO, &name);
        QString message = i18n("Undo operation [%1].", name);
        if (!name.length()) message="";
        undoAction->setStatusTip(message);

        currentBankDocument->getTransactionToTreat(SKGDocument::REDO, &name);
        message = i18n("Redo operation [%1].", name);
        if (!name.length()) message="";
        redoAction->setStatusTip(message);

}


QWidget* SKGUndoRedoPlugin::getPreferenceWidget()
{
        SKGTRACEIN(10, "SKGUndoRedoPlugin::getPreferenceWidget");
        //Read Setting
        KSharedConfigPtr config=KSharedConfig::openConfig("skrooge_undoredorc");
        KConfigGroup pref=config->group("skrooge_undoredo");
        pref.writeEntry("maxNumberOfUndo",SKGServices::stringToInt(currentBankDocument->getParameter("SKG_UNDO_MAX_DEPTH")));

        //Create widget
        QWidget* widget=new QWidget();
        ui.setupUi(widget);
        return widget;
}

KConfigSkeleton* SKGUndoRedoPlugin::getPreferenceSkeleton()
{
        return skrooge_undoredo::self();
}

SKGError SKGUndoRedoPlugin::savePreferences() const
{
        SKGError err;
        //Read Setting
        KSharedConfigPtr config=KSharedConfig::openConfig("skrooge_undoredorc");
        KConfigGroup pref=config->group("skrooge_undoredo");
        int max = pref.readEntry("maxNumberOfUndo", 50);

        //Save setting in document
        QString newValue=SKGServices::intToString(max);
        if (newValue!=currentBankDocument->getParameter("SKG_UNDO_MAX_DEPTH")) err=currentBankDocument->setParameter("SKG_UNDO_MAX_DEPTH",newValue);
        return err;
}

QString SKGUndoRedoPlugin::title() const
{
        return i18n("History");
}

QString SKGUndoRedoPlugin::icon() const
{
        return "edit-undo";
}

QString SKGUndoRedoPlugin::statusTip () const
{
        return i18n("History");
}

QString SKGUndoRedoPlugin::toolTip () const
{
        return i18n("History");
}

QStringList SKGUndoRedoPlugin::tips() const
{
        QStringList output;
        output.push_back(i18n("<p>... you can undo and redo yours modifications.</p>"));
        output.push_back(i18n("<p>... you can modify the maximum size of undo/redo stack.</p>"));
        return output;
}

int SKGUndoRedoPlugin::getOrder() const
{
        //Must be one of the first
        return 1;
}

void SKGUndoRedoPlugin::close()
{
        SKGTRACEIN(10, "SKGUndoRedoPlugin::close");
}

void SKGUndoRedoPlugin::actionUndoSave()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGUndoRedoPlugin::actionUndoSave",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        err=currentBankDocument->undoRedoTransaction(SKGDocument::UNDOLASTSAVE);
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Undo successfully done."));
        else err.addError(ERR_FAIL, i18n("Undo failed"));

        //Display error
        parent->displayErrorMessage(err);
}

void SKGUndoRedoPlugin::actionUndo()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGUndoRedoPlugin::actionUndo",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        int pos=((QAction*) sender())->data().toInt();
        for ( int i = 1 ; err.isSucceeded() && i <=pos; ++i )
                err=currentBankDocument->undoRedoTransaction(SKGDocument::UNDO);
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Undo successfully done."));
        else err.addError(ERR_FAIL, i18n("Undo failed"));

        //Display error
        parent->displayErrorMessage(err);
}

void SKGUndoRedoPlugin::onShowUndoMenu()
{
        undoMenu->clear();
        SKGStringListList listTmp;
        SKGServices::executeSelectSqliteOrder(currentBankDocument,
                                              "SELECT t_name FROM doctransaction where t_mode='U' order by  d_date DESC LIMIT 7",
                                              listTmp);
        int nb=listTmp.count();
        for (int i=1; i<nb; ++i) {
                QAction* act=undoMenu->addAction(listTmp.at(i).at(0));
                act->setData(i);
                connect(act, SIGNAL(triggered() ), this, SLOT( actionUndo() ) );
        }
}

void SKGUndoRedoPlugin::actionRedo()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGUndoRedoPlugin::actionRedo",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        int pos=((QAction*) sender())->data().toInt();
        for ( int i = 1 ; err.isSucceeded() && i <=pos; ++i )

                err=currentBankDocument->undoRedoTransaction(SKGDocument::REDO);
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Redo successfully done."));
        else err.addError(ERR_FAIL, i18n("Redo failed"));

        //Display error
        parent->displayErrorMessage(err);
}

void SKGUndoRedoPlugin::onShowRedoMenu()
{
        redoMenu->clear();
        SKGStringListList listTmp;
        SKGServices::executeSelectSqliteOrder(currentBankDocument,
                                              "SELECT t_name FROM doctransaction where t_mode='R' order by  d_date ASC LIMIT 7",
                                              listTmp);
        int nb=listTmp.count();
        for (int i=1; i<nb; ++i) {
                QAction* act=redoMenu->addAction(listTmp.at(i).at(0));
                act->setData(i);
                connect(act, SIGNAL(triggered() ), this, SLOT( actionRedo() ) );
        }
}

QDockWidget* SKGUndoRedoPlugin::getDockWidget()
{
        return dockWidget;
}
#include "skgundoredoplugin.moc"
