/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A class to automotically hide some widget on mousse hoverEvent.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgautohide.h"
#include "skgtraces.h"

#include <kmenu.h>
#include <kicon.h>

#include <QWidget>
#include <QHoverEvent>
#include <QTimer>

SKGAutoHide::SKGAutoHide ( QWidget* iAutoHiddenWidget, QWidget* iManager, int iMSec )
                :QObject(), autoHiddenWidget ( iAutoHiddenWidget ), manager ( iManager ), mSec ( iMSec ), menu(NULL)
{
        _SKGTRACEIN ( 10, "SKGAutoHide::SKGAutoHide" );

        timer=new QTimer();
        timer->setSingleShot ( true );
        timer->setInterval ( mSec );
        connect ( timer, SIGNAL ( timeout() ), this, SLOT ( hideWidget() ) );

        if ( manager ) {
                manager->setAttribute ( Qt::WA_Hover, true );
                manager->installEventFilter ( this );

                //Menu
                menu = new KMenu ( manager );

                actAutoHide = menu->addAction ( KIcon ( "show-menu" ), tr ( "Auto hide" ) );
                if (actAutoHide) {
                        actAutoHide->setCheckable ( true );
                        actAutoHide->setChecked ( false );
                }
                manager->setContextMenuPolicy ( Qt::CustomContextMenu );
                connect ( manager,SIGNAL ( customContextMenuRequested ( const QPoint & ) ),this,SLOT ( showContextMenu ( const QPoint& ) ) );
        }

        if ( autoHiddenWidget ) {
                autoHiddenWidget->setAttribute ( Qt::WA_Hover, true );
                autoHiddenWidget->installEventFilter ( this );

                if (menu) {
                        autoHiddenWidget->setContextMenuPolicy ( Qt::CustomContextMenu );
                        connect ( autoHiddenWidget,SIGNAL ( customContextMenuRequested ( const QPoint & ) ),this,SLOT ( showContextMenu ( const QPoint& ) ) );
                }
        }
        resetTimer();
}

SKGAutoHide::~SKGAutoHide()
{
        _SKGTRACEIN ( 10, "SKGAutoHide::~SKGAutoHide" );
        autoHiddenWidget=NULL;
        actAutoHide=NULL;
        manager=NULL;
        menu=NULL;

        if ( timer ) {
                delete timer;
                timer=NULL;
        }
}

void SKGAutoHide::setAutoHideMode ( bool iMode )
{
        if (actAutoHide) actAutoHide->setChecked ( iMode );
}

bool SKGAutoHide::autoHideMode()
{
        return (actAutoHide && actAutoHide->isChecked());
}

void SKGAutoHide::showContextMenu ( const QPoint& pos )
{
        QWidget* sender=static_cast<QWidget*> ( this->sender() );
        if ( menu && sender ) menu->popup ( sender->mapToGlobal ( pos ) );
}

void SKGAutoHide::resetTimer()
{
        _SKGTRACEIN ( 10, "SKGAutoHide::resetTimer" );
        if ( timer && autoHiddenWidget ) {
                if ( timer->isActive() ) timer->stop();

                autoHiddenWidget->show();
                timer->start();
        }
}

bool SKGAutoHide::eventFilter ( QObject *object, QEvent *event )
{
        if ( event && object && timer && autoHiddenWidget ) {
                QHoverEvent *hoverEvent = dynamic_cast<QHoverEvent*> ( event );
                if ( hoverEvent ) {
                        if ( hoverEvent->type() ==QEvent::HoverEnter ) {
                                autoHiddenWidget->show();
                                if ( timer->isActive() ) timer->stop();
                        } else if ( hoverEvent->type() ==QEvent::HoverLeave ) {
                                resetTimer();
                        }
                }
        }
        return false;
}

void SKGAutoHide::hideWidget()
{
        if (autoHideMode() && autoHiddenWidget) autoHiddenWidget->hide();
}

#include "skgautohide.moc"

