/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin for monthlu report.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgmonthlypluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgdocumentbank.h"
#include "skgaccountobject.h"
#include "skgunitobject.h"
#include "skgtransactionmng.h"

#include <kstandarddirs.h>

#include <QDomDocument>
#include <QPrinter>
#include <QPrintDialog>
#include <QPainter>
#include <QTextStream>
#include <QFile>
#include <QDir>

SKGMonthlyPluginWidget::SKGMonthlyPluginWidget(SKGMainPanel* iParent, SKGDocument* iDocument)
                : SKGTabWidget(iParent, iDocument)
{
        SKGTRACEIN(1, "SKGMonthlyPluginWidget::SKGMonthlyPluginWidget");

        ui.setupUi(this);

        ui.kRefresh->setIcon(KIcon("view-refresh"));
        ui.kPrint->setIcon(KIcon("printer"));

        connect(ui.kMonth, SIGNAL(currentIndexChanged(int)), this, SLOT(onMonthChanged()));

        //Refresh
        connect((const QObject*) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)), Qt::QueuedConnection );
        dataModified("", 0);
}

SKGMonthlyPluginWidget::~SKGMonthlyPluginWidget()
{
        SKGTRACEIN(1, "SKGMonthlyPluginWidget::~SKGMonthlyPluginWidget");
}

QString SKGMonthlyPluginWidget::getState()
{
        SKGTRACEIN(10, "SKGMonthlyPluginWidget::getState");
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);

        //Get state
        root.setAttribute("month", ui.kMonth->text());

        return doc.toString();
}

void SKGMonthlyPluginWidget::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGMonthlyPluginWidget::setState");
        QDomDocument doc("SKGML");
        doc.setContent(iState);
        QDomElement root = doc.documentElement();

        //Set state
        QString month=root.attribute ( "month");
        if (!month.isEmpty()) ui.kMonth->setText(month);
}

QWidget* SKGMonthlyPluginWidget::getWidgetForPrint()
{
        return ui.kWebView;
}

void SKGMonthlyPluginWidget::onPrint()
{
        _SKGTRACEIN(10, "SKGMonthlyPluginWidget::onPrint");
#ifndef QT_NO_PRINTER
        QPrinter printer;
        QPrintDialog dialog(&printer, this);
        if (dialog.exec() == QDialog::Accepted) {
                QPainter painter(&printer);
                ui.kWebView->render(&painter);
                painter.end();
        }
#endif
}

void SKGMonthlyPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
        SKGTRACEIN(1, "SKGMonthlyPluginWidget::dataModified");

        QSqlDatabase* db = getDocument()->getDatabase();
        setEnabled(db!=NULL);
        if (db!=NULL &&  (iTableName=="parameters" || iTableName=="operation" || iTableName.isEmpty())) {
                //Fill combo
                disconnect(ui.kMonth, SIGNAL(currentIndexChanged(int)), this, SLOT(onMonthChanged()));

                QStringList list;
                SKGServices::getDistinctValues(getDocument(), "v_operation_display", "d_DATEMONTH", "d_date<=CURRENT_DATE", list);
                list.removeOne(QDate::currentDate().toString ("yyyy-MM"));
                qSort(list.begin(), list.end(), qGreater<QString>());

                QString month=ui.kMonth->text();
                ui.kMonth->clear();
                ui.kMonth->addItems(list);
                if (!month.isEmpty()) ui.kMonth->setText(month);
                else if (list.count()) month=list.at(0);
                connect(ui.kMonth, SIGNAL(currentIndexChanged(int)), this, SLOT(onMonthChanged()));

                onMonthChanged();
        }
}

void SKGMonthlyPluginWidget::onMonthChanged()
{
        SKGTRACEIN(1, "SKGMonthlyPluginWidget::onMonthChanged");
        QString month=ui.kMonth->text();

        //Display report
        QString htmlReport=getDocument()->getParameter("SKG_MONTHLY_REPORT_"+month);
        if (htmlReport.isEmpty()) {
                //Report not yet generated
                ui.kWebView->setHtml(i18n("The monthly report has not yet been computed: click on refresh to do so."));
        } else {
                //Report generated
                //Set style sheet
                QDir dirCss(KStandardDirs::locate("data", QString::fromLatin1("skrooge/css/")));
                QFile file(dirCss.absoluteFilePath("skrooge_monthly-default.css"));
                if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
                        return;

                QString style;
                QTextStream in(&file);
                while (!in.atEnd()) {
                        style+=in.readLine();
                }

                htmlReport.replace("##STYLE##", style);

                //Display html report
                ui.kWebView->setHtml(htmlReport);
        }
}

void SKGMonthlyPluginWidget::onReportRefresh()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGMonthlyPluginWidget::onReportRefresh",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        //Get month
        QString month=ui.kMonth->text();
        QDate date=QDate::fromString(month, "yyyy-MM");
        QDate date1=date.addDays(-1);
        QDate date2=date.addMonths(1).addDays(-1);
        QDate date3=date2.addYears(-1);
        QString previousmonth=date1.toString("yyyy-MM");
        QString primary=((SKGDocumentBank*) getDocument())->getPrimaryUnit();
        KLocale* locale=KGlobal::locale();

        {
                int step=0;
                SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18n("Compute monthly report for [%1]", month), err, 4);

                //Generate HTML report
                QString html="<?xml version=\"1.0\" encoding=\"utf-8\"?>"
                             "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">"
                             "<html xmlns=\"http://www.w3.org/1999/xhtml\">"
                             "<head>"
                             "<meta http-equiv=\"content-type\" content=\"text/html;charset=utf-8\" />"
                             "<meta http-equiv=\"Content-Style-Type\" content=\"text/css\" />"
                             "<style type=\"text/css\">##STYLE##</style>"
                             "</head>"
                             "<body>\n";

                //Main title
                html+="<h1>"+i18n("Report for %1", month)+"</h1>";

                //Income & Expenditure
                if (err.isSucceeded()) {
                        SKGStringListList listTmp;
                        err=SKGServices::executeSelectSqliteOrder(getDocument(),
                                        "SELECT TOTAL(f_CURRENTAMOUNT), d_DATEMONTH  from v_operation_display WHERE (d_DATEMONTH='"+month+"' OR d_DATEMONTH='"+previousmonth+"') AND i_group_id=0 group by d_DATEMONTH, t_TYPEEXPENSE",
                                        listTmp);
                        if (err.isSucceeded()) {
                                html+="<h2>"+i18n("Income & Expenditure")+"</h2><table class=\"table\">";
                                html+="<tr class=\"tabletitle\"><td align=\"center\"></td>"
                                      "<td align=\"center\">"+previousmonth+"</td>"
                                      "<td align=\"center\">"+month+"</td>"
                                      "</tr>";
                                double income_previous_month=0;
                                double expense_previous_month=0;
                                double income_month=0;
                                double expense_month=0;

                                int nbval=listTmp.count();
                                for (int i=1; i<nbval; ++i) { //Ignore header
                                        QString m=listTmp.at(i).at(1);
                                        double v=SKGServices::stringToDouble(listTmp.at(i).at(0));
                                        if (v>0 && m==month) income_month=v;
                                        else if (v<0 && m==month) expense_month=v;
                                        else if (v>0 && m==previousmonth) income_previous_month=v;
                                        else if (v<0 && m==previousmonth) expense_previous_month=v;
                                }
                                double saving_previous_month=income_previous_month+expense_previous_month;
                                double saving_month=income_month+expense_month;

                                html+=QString("<tr><td>")+i18n("Income")+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(income_previous_month, primary,2)+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(income_month, primary,2)+"</td>"
                                      "</tr>";
                                html+=QString("<tr><td>")+i18n("Expenditure")+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(abs(expense_previous_month), primary,2)+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(abs(expense_month), primary,2)+"</td>"
                                      "</tr>";
                                html+="<tr class=\"tabletotal\"><td>"+i18n("Savings possible")+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(saving_previous_month, primary,2)+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(saving_month, primary,2)+"</td>"
                                      "</tr>";
                                html+="</table>";
                                QString range="0,"+SKGServices::doubleToString(qMax(abs(income_month), abs(expense_month)));
                                QString img="http://chart.apis.google.com/chart?cht=bvs&chbh=100&chxt=x,y&chxr=1,"+range+"&chco=FF0000|00FF00&chd=t:"+
                                            SKGServices::doubleToString(abs(expense_month))+','+
                                            SKGServices::doubleToString(abs(income_month))+
                                            "&chds="+range+
                                            "&chs=300x200&chl="+i18n("Expenditure")+'|'+i18n("Income")+"&chtt="+i18n("Expenditure vs Income")+'|'+month;
                                // SKGTRACE << img << endl;
                                html+="<img src=\""+img+"\" />";
                        }
                }
                ++step;
                if (err.isSucceeded()) err=getDocument()->stepForward(step);

                //Account
                if (err.isSucceeded()) {
                        SKGObjectBase::SKGListSKGObjectBase accounts;
                        SKGError err=SKGObjectBase::getObjects(getDocument(), "v_account_display", "1=1 ORDER BY t_TYPENLS", accounts);
                        if (err.isSucceeded()) {
                                html+="<h2>"+i18n("Amounts in accounts")+"</h2><table class=\"table\">";
                                html+="<tr class=\"tabletitle\"><td align=\"center\">"+i18n("Account")+"</td>"
                                      "<td align=\"center\">"+locale->formatDate (date1, KLocale::ShortDate)+"</td>"
                                      "<td align=\"center\">"+locale->formatDate (date2, KLocale::ShortDate)+"</td>"
                                      "<td align=\"center\">%</td>"
                                      "<td align=\"center\">"+locale->formatDate (date3, KLocale::ShortDate)+"</td>"
                                      "<td align=\"center\">"+locale->formatDate (date2, KLocale::ShortDate)+"</td>"
                                      "<td align=\"center\">%</td>"
                                      "</tr>";
                                double sumTypeV1=0;
                                double sumTypeV2=0;
                                double sumTypeV3=0;
                                double sumV1=0;
                                double sumV2=0;
                                double sumV3=0;
                                QString currentType;
                                int nb=accounts.count();
                                err=getDocument()->beginTransaction("#INTERNAL#", nb);
                                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                                        SKGAccountObject account=accounts.at(i);
                                        double v1=account.getAmount(date1);
                                        double v2=account.getAmount(date2);
                                        double v3=account.getAmount(date3);
                                        QString type=account.getAttribute("t_TYPENLS");
                                        bool closed=account.isClosed();
                                        if (type!=currentType) {
                                                if (!currentType.isEmpty()) {
                                                        html+="<tr class=\"tabletotal\"><td>"+i18n("Total of %1", currentType)+"</td>"
                                                              "<td align=\"right\">"+locale->formatMoney(sumTypeV1, primary,2)+"</td>"
                                                              "<td align=\"right\">"+locale->formatMoney(sumTypeV2, primary,2)+"</td>"
                                                              "<td"+(sumTypeV1>sumTypeV2 ? " class=\"negativevalue\"" : "")+
                                                              " align=\"right\">"+(sumTypeV1==0 ? "" : locale->formatMoney(100.0*(sumTypeV2-sumTypeV1)/sumTypeV1,"%",2))+"</td>"
                                                              "<td align=\"right\">"+locale->formatMoney(sumTypeV3, primary,2)+"</td>"
                                                              "<td align=\"right\">"+locale->formatMoney(sumTypeV2, primary,2)+"</td>"
                                                              "<td"+(sumTypeV3>sumTypeV2 ? " class=\"negativevalue\"" : "")+
                                                              " align=\"right\">"+(sumTypeV3==0 ? "" : locale->formatMoney(100.0*(sumTypeV2-sumTypeV3)/sumTypeV3,"%",2))+"</td>"
                                                              "</tr>";
                                                        sumTypeV1=0;
                                                        sumTypeV2=0;
                                                        sumTypeV3=0;
                                                }
                                                currentType=type;
                                        }
                                        if (!closed || abs(v1)>0.01 || abs(v2)>0.01 || abs(v3)>0.01) {
                                                html+=QString("<tr><td>")+(closed ? "<i>" : "")+account.getName()+(closed ? "</i>" : "")+"</td>"
                                                      "<td align=\"right\">"+locale->formatMoney(v1, primary,2)+"</td>"
                                                      "<td align=\"right\">"+locale->formatMoney(v2, primary,2)+"</td>"
                                                      "<td"+(v1>v2 ? " class=\"negativevalue\"" : "")+
                                                      " align=\"right\">"+(v1==0 ? "" :locale->formatMoney(100.0*(v2-v1)/v1,"%",2))+"</td>"
                                                      "<td align=\"right\">"+locale->formatMoney(v3, primary,2)+"</td>"
                                                      "<td align=\"right\">"+locale->formatMoney(v2, primary,2)+"</td>"
                                                      "<td"+(v3>v2 ? " class=\"negativevalue\"" : "")+
                                                      " align=\"right\">"+(v3==0 ? "" : locale->formatMoney(100.0*(v2-v3)/v3,"%",2))+"</td>"
                                                      "</tr>";
                                        }
                                        sumTypeV1+=v1;
                                        sumTypeV2+=v2;
                                        sumTypeV3+=v3;
                                        sumV1+=v1;
                                        sumV2+=v2;
                                        sumV3+=v3;

                                        if (err.isSucceeded()) err=getDocument()->stepForward(i+1);
                                }
                                getDocument()->endTransaction(err.isSucceeded());

                                html+="<tr class=\"tabletotal\"><td>"+i18n("Total of %1", currentType)+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(sumTypeV1, primary,2)+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(sumTypeV2, primary,2)+"</td>"
                                      "<td"+(sumTypeV1>sumTypeV2 ? " class=\"negativevalue\"" : "")+
                                      " align=\"right\">"+(sumTypeV1==0 ? "" : locale->formatMoney(100.0*(sumTypeV2-sumTypeV1)/sumTypeV1,"%",2))+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(sumTypeV3, primary,2)+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(sumTypeV2, primary,2)+"</td>"
                                      "<td"+(sumTypeV3>sumTypeV2 ? " class=\"negativevalue\"" : "")+
                                      " align=\"right\">"+(sumTypeV3==0 ? "" : locale->formatMoney(100.0*(sumTypeV2-sumTypeV3)/sumTypeV3,"%",2))+"</td>"
                                      "</tr>";
                                html+="<tr class=\"tabletotal\"><td>"+i18n("Total")+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(sumV1, primary,2)+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(sumV2, primary,2)+"</td>"
                                      "<td"+(sumV1>sumV2 ? " class=\"negativevalue\"" : "")+
                                      " align=\"right\">"+(sumV1==0 ? "" : locale->formatMoney(100.0*(sumV2-sumV1)/sumV1,"%",2))+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(sumV3, primary,2)+"</td>"
                                      "<td align=\"right\">"+locale->formatMoney(sumV2, primary,2)+"</td>"
                                      "<td"+(sumV3>sumV2 ? " class=\"negativevalue\"" : "")+
                                      " align=\"right\">"+(sumV3==0 ? "" : locale->formatMoney(100.0*(sumV2-sumV3)/sumV3,"%",2))+"</td>"
                                      "</tr>";
                                html+="</table>";
                        }
                }
                ++step;
                if (err.isSucceeded()) err=getDocument()->stepForward(step);

                //Unit
                if (err.isSucceeded()) {
                        SKGObjectBase::SKGListSKGObjectBase units;
                        SKGError err=SKGObjectBase::getObjects(getDocument(), "v_unit_display", "1=1 ORDER BY t_TYPENLS", units);
                        if (err.isSucceeded()) {
                                html+="<h2>"+i18n("Amounts of units")+"</h2><table class=\"table\">";
                                html+="<tr class=\"tabletitle\"><td align=\"center\">"+i18n("Unit")+"</td>"
                                      "<td align=\"center\">"+locale->formatDate (date1, KLocale::ShortDate)+"</td>"
                                      "<td align=\"center\">"+locale->formatDate (date2, KLocale::ShortDate)+"</td>"
                                      "<td align=\"center\">%</td></tr>";
                                foreach(const SKGUnitObject& unit, units) {
                                        double v1=unit.getAmount(date1);
                                        double v2=unit.getAmount(date2);
                                        html+="<tr><td>"+unit.getName()+"</td>"
                                              "<td align=\"right\">"+locale->formatMoney(v1, primary,2)+"</td>"
                                              "<td align=\"right\">"+locale->formatMoney(v2, primary,2)+"</td>"
                                              "<td"+(v1>v2 ? " class=\"negativevalue\"" : "")+
                                              " align=\"right\">"+(v1==0 ? "" : locale->formatMoney(100.0*(v2-v1)/v1,"%",2))+"</td></tr>";
                                }
                                html+="</table>";
                        }
                }
                ++step;
                if (err.isSucceeded()) err=getDocument()->stepForward(step);

                html+="</body></html>";
                if (err.isSucceeded()) err=getDocument()->setParameter("SKG_MONTHLY_REPORT_"+month, html);
                IFSKGTRACEL(1) {
                        QFile file("/tmp/skrooge_monthy_report.html");
                        if (!file.open(QIODevice::WriteOnly | QIODevice::Text))
                                return;

                        QTextStream out(&file);
                        out << html << "\n";
                }
                ++step;
                if (err.isSucceeded()) err=getDocument()->stepForward(step);
        }
        QApplication::restoreOverrideCursor();


        //status bar
        if (err.isSucceeded())  err=SKGError(0, i18n("Report computed."));
        else err.addError(ERR_FAIL, i18n("Report computation failed."));

        //Display error
        getMainPanel()->displayErrorMessage(err);
}
#include "skgmonthlypluginwidget.moc"


