/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for operation management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgoperationpluginwidget.h"
#include "skgsortfilterproxymodel.h"
#include "skgmainpanel.h"
#include "skgobjectmodel.h"
#include "skgobjectbase.h"
#include "skgbankincludes.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgsplittabledelegate.h"

#include <kstandarddirs.h>
#include <kstandardaction.h>
#include <kmessagebox.h>
#include <kinputdialog.h>
#include <skgcalculatoredit.h>

#include <QtGui/QHeaderView>
#include <QDomDocument>
#include <QDir>

/**
 * @def NOUPDATE
 * For a not modified field
 */
#define NOUPDATE "-------"

SKGOperationPluginWidget::SKGOperationPluginWidget ( SKGMainPanel* iParent, SKGDocumentBank* iDocument, KAction* iFastEditionAction )
                : SKGTabWidget ( iParent, iDocument ), fastEditionAction ( iFastEditionAction ), lastFastEditionOperationFound ( 0 ), showClosedAccounts ( false ),
                numberFieldIsNotUptodate ( true ),modeInfoZone ( 0 )
{
        SKGTRACEIN ( 1, "SKGOperationPluginWidget::SKGOperationPluginWidget" );

        ui.setupUi ( this );

        ui.kTitle->hide();
        ui.kresetInternalFiler->hide();
        ui.kReconciliatorFrame2->hide();

        {
                SKGTRACEIN ( 10, "SKGOperationPluginWidget::SKGOperationPluginWidget-model creation" );
                //Bind operation view
                objectModel = new SKGObjectModel ( ( SKGDocumentBank* ) getDocument(), "v_operation_display", "1=0", this );
                SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel ( this );
                modelproxy->setSourceModel ( objectModel );
                modelproxy->setSortRole ( Qt::UserRole );
                modelproxy->setDynamicSortFilter ( true );

                ui.kOperationView->setModel ( modelproxy );
                ui.kOperationView->setWindowTitle ( i18n ( "Operations" ) );

                //Add registered global action in contextual menu
                if ( iParent ) {
                        ui.kOperationView->insertAction ( 0, iParent->getGlobalAction ( "edit_delete" ) );
                        QAction* sep=new QAction ( this );
                        sep->setSeparator ( true );
                        ui.kOperationView->insertAction ( 0, sep );
                        ui.kOperationView->insertAction ( 0, iParent->getGlobalAction ( "edit_duplicate_operation" ) );
                        ui.kOperationView->insertAction ( 0, iParent->getGlobalAction ( "edit_switch_bookmark" ) );
                        ui.kOperationView->insertAction ( 0, iParent->getGlobalAction ( "edit_point_selected_operation" ) );
                        ui.kOperationView->insertAction ( 0, iParent->getGlobalAction ( "schedule_operation" ) );
                        ui.kOperationView->insertAction ( 0, sep );
                        ui.kOperationView->insertAction ( 0, iParent->getGlobalAction ( "open_report" ) );
                }

                connect ( ui.kOperationView->selectionModel(), SIGNAL ( selectionChanged ( QItemSelection,QItemSelection ) ), this, SLOT ( onSelectionChanged() ) );
                connect ( objectModel, SIGNAL ( beforeReset() ), ui.kOperationView, SLOT ( saveSelection() ) );
                connect ( objectModel, SIGNAL ( afterReset() ), ui.kOperationView, SLOT ( resetSelection() ) );
        }

        //Add completions
        KCompletion *comp = ui.kTypeEdit->completionObject();
        connect ( ui.kTypeEdit,SIGNAL ( returnPressed ( const QString& ) ),comp,SLOT ( addItem ( const QString& ) ) );

        //Add Standard KDE Icons to buttons to Operations
        ui.kModifyOperationBtn->setIcon ( KIcon ( "dialog-ok-apply" ) );
        ui.kAddOperationBtn->setIcon ( KIcon ( "list-add" ) );
        ui.kresetInternalFiler->setIcon ( KIcon ( "dialog-cancel" ) );
        ui.kRemoveSubOperation->setIcon ( KIcon ( "edit-delete" ) );
        ui.kReconciliatorButton->setIcon ( KIcon ( "object-rotate-left" ) );
        ui.kValidate->setIcon( KIcon( "dialog-ok-apply" ) );

        ui.kStandardBtn->setIcon (KIcon ( "dialog-ok-apply" ) );
        ui.kTransferBtn->setIcon (KIcon ( "skrooge_transfer" ) );
        ui.kSplitBtn->setIcon (KIcon ( "skrooge_split" ) );
        ui.kShareBtn->setIcon (KIcon ( "skrooge_unit" ) );

        //Refresh
        connect ( ( const QObject* ) getDocument(), SIGNAL ( transactionSuccessfullyEnded ( int ) ), this, SLOT ( refresh() ), Qt::QueuedConnection );
        connect ( ( const QObject* ) getDocument(), SIGNAL ( transactionSuccessfullyEnded ( int ) ), this, SLOT ( onFilterChanged() ), Qt::QueuedConnection );
        connect ( ( const QObject* ) getDocument(), SIGNAL ( transactionSuccessfullyEnded ( int ) ), this, SLOT ( onRefreshInformationZone() ), Qt::QueuedConnection );
        connect ( ( const QObject* ) getDocument(), SIGNAL ( transactionSuccessfullyEnded ( int ) ), this, SLOT ( onSelectionChanged() ), Qt::QueuedConnection );
        connect ( ui.kHideUselessOperation, SIGNAL ( stateChanged ( int ) ), this, SLOT ( onFilterChanged() ), Qt::QueuedConnection );

        //Fast edition
        connect ( qApp, SIGNAL ( focusChanged ( QWidget*, QWidget* ) ), this, SLOT ( onFocusChanged() ) );
        connect ( fastEditionAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( onFastEdition() ) );

        // SubOperations
        connect ( ui.kAmountEdit,SIGNAL ( textChanged ( const QString& ) ),this,SLOT ( onQuantityChanged() ) );
        connect ( ui.kSubOperationsTable,SIGNAL ( cellChanged ( int,int ) ),this,SLOT ( onSubopCellChanged ( int,int ) ) );
        connect ( ui.kRemoveSubOperation,SIGNAL ( clicked ( bool ) ),this,SLOT ( onRemoveSubOperation() ) );
        ui.kSubOperationsTable->verticalHeader()->setDefaultSectionSize ( ui.kSubOperationsTable->verticalHeader()->minimumSectionSize() );
        ui.kSubOperationsTable->horizontalHeader()->setResizeMode ( QHeaderView::Interactive );
        ui.kSubOperationsTable->setWordWrap ( false );
        ui.kSubOperationsTable->setItemDelegate ( new SKGSplitTableDelegate ( ui.kSubOperationsTable, getDocument() ) );

        ui.kTransferFrm->hide();
        ui.kSplitFrm->hide();
        ui.kShareFrm->hide();

        setCurrentMode(0);

        refresh();
        onOperationCreatorModified();
}

SKGOperationPluginWidget::~SKGOperationPluginWidget()
{
        SKGTRACEIN ( 1, "SKGOperationPluginWidget::~SKGOperationPluginWidget" );
        objectModel=NULL;
        fastEditionAction=NULL;
}

void SKGOperationPluginWidget::onSelectionChanged()
{
        SKGTRACEIN ( 10, "SKGOperationPluginWidget::onSelectionChanged" );

        int mode=getCurrentMode();

        //Mapping
        QItemSelectionModel *selModel=ui.kOperationView->selectionModel();
        if ( selModel ) {
                QModelIndexList indexes=selModel->selectedRows();
                int nbSelect=indexes.count();
                bool onConsolidatedTable=false;
                if ( nbSelect && objectModel ) {
                        QModelIndex idx=indexes[indexes.count()-1];

                        QSortFilterProxyModel* proxyModel= ( QSortFilterProxyModel* ) ui.kOperationView->model();
                        QModelIndex idxs=proxyModel->mapToSource ( idx );

                        SKGOperationObject obj ( objectModel->getObject ( idxs ) );
                        onConsolidatedTable= ( obj.getTable() =="v_operation_consolidated" );

                        ui.kDateEdit->setDate ( SKGServices::stringToTime ( obj.getAttribute ( "d_date" ) ).date() );
                        ui.kCommentEdit->setText ( obj.getAttribute ( obj.getTable() =="v_operation_consolidated" ? "t_REALCOMMENT" : "t_comment" ) );
                        QString number=obj.getAttribute ( "i_number" );
                        if ( number=="0" ) number="";
                        ui.kNumberEdit->setText ( number );
                        ui.kAccountEdit->setText ( obj.getAttribute ( "t_ACCOUNT" ) );
                        ui.kPayeeEdit->setText ( obj.getAttribute ( "t_payee" ) );
                        ui.kTypeEdit->setText ( obj.getAttribute ( "t_mode" ) );
                        ui.kUnitEdit->setText ( obj.getAttribute ( "t_UNIT" ) );
                        QString cat=obj.getAttribute ( "t_REALCATEGORY" );
                        if ( cat.isEmpty() ) cat=obj.getAttribute ( "t_CATEGORY" );
                        ui.kCategoryEdit->setText ( cat );
                        ui.kTrackerEdit->setText ( obj.getAttribute ( "t_REFUND" ) );
                        QString quantity=obj.getAttribute ( "f_REALQUANTITY" );
                        if ( quantity.isEmpty() ) quantity=obj.getAttribute ( "f_QUANTITY" );
                        if ( SKGServices::stringToDouble ( quantity ) >0 ) quantity='+'+quantity;
                        ui.kAmountEdit->setText ( quantity );

                        if ( nbSelect>1 ) {
                                if (mode>=0) setCurrentMode(0);
                                ui.kAccountEdit->setText ( NOUPDATE );
                                ui.kTypeEdit->setText ( NOUPDATE );
                                ui.kUnitEdit->setText ( NOUPDATE );
                                ui.kCategoryEdit->setText ( NOUPDATE );
                                ui.kTrackerEdit->setText ( NOUPDATE );
                                ui.kCommentEdit->setText ( NOUPDATE );
                                ui.kPayeeEdit->setText ( NOUPDATE );
                        } else {
                                int nbSubOperations = obj.getNbSubOperations();
                                if ( nbSubOperations > 1 && !onConsolidatedTable ) {
                                        if (mode>=0) setCurrentMode(1);;

                                        displaySubOperations();
                                } else {
                                        if (mode>=0) setCurrentMode(0);
                                }
                        }
                } else {
                        ui.kDateEdit->setDate ( QDate::currentDate() );
                        ui.kPayeeEdit->setText ( "" );
                        ui.kCategoryEdit->setText ( "" );
                        ui.kTrackerEdit->setText ( "" );
                        ui.kAmountEdit->setText ( "" );
                        ui.kTypeEdit->setText ( "" );
                }

                ui.kDateEdit->setEnabled ( nbSelect<=1 );
                ui.kAmountEdit->setEnabled ( nbSelect<=1 );
                ui.kNumberEdit->setEnabled ( nbSelect<=1 );

                bool splitTest=nbSelect<=1 && !onConsolidatedTable;
                ui.kSplitBtn->setEnabled ( splitTest );
                if (!splitTest && mode==1) setCurrentMode(0);

                onOperationCreatorModified();

                emit selectionChanged();
        }

        if (modeInfoZone==2) onRefreshInformationZone();
}

void SKGOperationPluginWidget::onOperationCreatorModified()
{
        SKGTRACEIN ( 10, "SKGOperationPluginWidget::onOperationCreatorModified" );

        int mode=getCurrentMode();

        //Is it an existing unit ?
        QString unitName=ui.kUnitEdit->currentText();
        SKGUnitObject unit ( getDocument() );
        unit.setName ( unitName );
        unit.setSymbol ( unitName );
        if ( unit.load().isSucceeded() ) {
                ui.kShareBtn->setEnabled(true);
                if ( mode==3 && unit.getType() ==SKGUnitObject::SHARE ) {
                        //Update units
                        QString unitOfUnitName= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
                        SKGUnitObject unitOfUnit;
                        unit.getUnit ( unitOfUnit );
                        if ( unitOfUnit.exist() ) unitOfUnitName=unitOfUnit.getSymbol();

                        ui.kUnitShare->setText ( unitOfUnitName );
                        ui.kUnitCommission->setText ( unitOfUnitName );
                        ui.kUnitTax->setText ( unitOfUnitName );

                        //Update total in "purchase / sale share" page
                        double total=ui.kAmountSharesEdit->value() + ( ui.kCommissionEdit->value() +ui.kTaxEdit->value() ) * ( ui.kAmountEdit->value() >0 ? 1 : -1 );
                        ui.KTotal->setText ( KGlobal::locale()->formatMoney ( total, unitOfUnitName,2 ) );
                } else {
                        //BUG 2692665
                        ui.kUnitShare->setText ( unitName );
                        ui.kUnitCommission->setText ( unitName );
                        ui.kUnitTax->setText ( unitName );
                }

        } else {
                ui.kShareBtn->setEnabled(false);
                if (mode==3) setCurrentMode(0);
        }

        bool activated=ui.kAccountEdit->currentText().length() >0 &&
                       ui.kAmountEdit->text().length() >0 &&
                       unitName.length() >0 &&
                       ( mode!=3 || ui.kAmountSharesEdit->text().length() >0 );

        int nbSelect=getNbSelectedObjects();

        ui.kAddOperationBtn->setEnabled ( activated );
        ui.kModifyOperationBtn->setEnabled ( activated && nbSelect>0 && (getCurrentMode()==0 ||  getCurrentMode()==1) );

        QList<QTableWidgetItem*> selectedItems = ui.kSubOperationsTable->selectedItems();
        ui.kRemoveSubOperation->setEnabled ( selectedItems.count() >0 );
}

void SKGOperationPluginWidget::onUpdateOperationClicked()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPluginWidget::onUpdateOperationClicked",err );
        bool confirmed=true;
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

        int nb=selection.count();
        {
                SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18n ( "Operation update" ), err, nb );

                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                err=updateSelection ( selection );
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if ( err.isSucceeded() ) {
                if ( confirmed ) err=SKGError ( 0, i18n ( "Operation updated." ) );
                else err=SKGError ( 0, i18n ( "Operation canceled" ) );
        } else err.addError ( ERR_FAIL, i18n ( "Operation update failed" ) );

        //Display error
        getMainPanel()->displayErrorMessage ( err );
}

SKGError SKGOperationPluginWidget::updateSelection ( const SKGObjectBase::SKGListSKGObjectBase& iSelection )
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPluginWidget::updateSelection",err );

        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        int nb=iSelection.count();

        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                SKGObjectBase obj=iSelection.at ( i );
                SKGOperationObject operationObj=SKGOperationObject ( obj.getDocument(), obj.getID() );

                //Update operation if single selection
                if ( getCurrentMode()==0 ) {
                        //Get subop
                        SKGSubOperationObject subOp;
                        int nbSubop=0;
                        if ( obj.getTable() =="v_operation_consolidated" ) {
                                //It's a sub operation
                                subOp=SKGSubOperationObject ( obj.getDocument(), SKGServices::stringToInt ( obj.getAttribute ( "i_SUBOPID" ) ) );
                                nbSubop=1;
                        } else {
                                //It's a real operation, we take the first one
                                SKGObjectBase::SKGListSKGObjectBase subOps;
                                if ( err.isSucceeded() ) err=operationObj.getSubOperations ( subOps );
                                nbSubop=subOps.count();
                                if ( nbSubop ) subOp=subOps[0];
                        }

                        QString trackerName=ui.kTrackerEdit->text();
                        if ( err.isSucceeded() && trackerName !=NOUPDATE ) {
                                if (trackerName.isEmpty()) err=subOp.removeRefundTracker();
                                else {
                                        SKGRefundTrackerObject refund;
                                        err=SKGRefundTrackerObject::createRefund((SKGDocumentBank*) getDocument(), trackerName, refund, true);
                                        if ( err.isSucceeded() ) err=subOp.setRefundTracker( refund );
                                }
                        }

                        SKGCategoryObject cat;
                        if ( err.isSucceeded() && ui.kCategoryEdit->text() !=NOUPDATE ) {
                                if ( err.isSucceeded() && nbSubop>1 ) {
                                        err=SKGError ( 25, i18n ( "Cannot update a split operation" ) );
                                } else {
                                        err=SKGCategoryObject::createPathCategory ( ( SKGDocumentBank* ) getDocument(), ui.kCategoryEdit->text(), cat, true );
                                        if ( err.isSucceeded() ) err=subOp.setCategory ( cat );
                                }
                        } else {
                                //Get current category to be able to
                                subOp.getCategory ( cat );
                        }

                        if ( err.isSucceeded() && nb==1 ) {
                                if ( getCurrentMode()==0 && nbSubop>1 ) {
                                        err=SKGError ( 25, i18n ( "Cannot update a split operation" ) );
                                } else {
                                        double val=ui.kAmountEdit->value();

                                        //Is the sign forced ?
                                        if ( ui.kAmountEdit->sign() ==0 ) {
                                                SKGObjectBase cat2 ( cat.getDocument(), "v_category_display", cat.getID() );

                                                //Are we able to find to sign with the category ?
                                                if ( cat2.getAttribute ( "t_TYPEEXPENSE" ) ==tr ( "Expenditure" ) ) val=-val;
                                        }
                                        err=subOp.setQuantity ( val );
                                }
                        }

                        if ( err.isSucceeded() && ui.kCommentEdit->text() !=NOUPDATE ) {
                                if ( obj.getTable() =="v_operation_consolidated" ) err=subOp.setComment ( ui.kCommentEdit->text() );
                        }
                        if ( err.isSucceeded() ) err=subOp.save();
                } else {
                        int nbsubop=ui.kSubOperationsTable->rowCount();
                        QList<int> listIdSubOp;
                        for ( int j=0; err.isSucceeded() && j<nbsubop; ++j ) {
                                //Get values
                                QTableWidgetItem* item=ui.kSubOperationsTable->item ( j,0 );
                                int id=item->data ( Qt::UserRole ).toInt();
                                QString catName=item->text();
                                item=ui.kSubOperationsTable->item ( j,1 );
                                QString comment=item->text();
                                item=ui.kSubOperationsTable->item ( j,2 );
                                double val=SKGServices::stringToDouble ( item->text() );
                                item=ui.kSubOperationsTable->item ( j,3 );
                                QString trackerName=item->text();

                                SKGSubOperationObject subOperation;
                                if ( id ) {
                                        //Update existing sub op
                                        subOperation=SKGSubOperationObject ( ( SKGDocumentBank* ) getDocument(), id );
                                } else {
                                        //Create new sub op
                                        err=operationObj.addSubOperation ( subOperation );
                                }

                                //Create sub operation object
                                SKGCategoryObject cat;
                                if ( err.isSucceeded() && !catName.isEmpty() ) {
                                        err=SKGCategoryObject::createPathCategory ( ( SKGDocumentBank* ) getDocument(), catName, cat, true );
                                        if ( err.isSucceeded() ) err=subOperation.setCategory ( cat );
                                }
                                if (err.isSucceeded()) {
                                        if (trackerName.isEmpty()) err=subOperation.removeRefundTracker();
                                        else {
                                                SKGRefundTrackerObject refund;
                                                err=SKGRefundTrackerObject::createRefund(( SKGDocumentBank* ) getDocument(), trackerName, refund, true );
                                                if ( err.isSucceeded() ) err=subOperation.setRefundTracker(refund );
                                        }
                                }
                                if ( err.isSucceeded() ) err=subOperation.setComment ( comment );
                                if ( err.isSucceeded() ) err=subOperation.setQuantity ( val );
                                if ( err.isSucceeded() ) err=subOperation.save();

                                //The sub operation created or updated mustn't be removed
                                listIdSubOp.push_back ( subOperation.getID() );
                        }

                        //Remove useless subop
                        if ( err.isSucceeded() ) {
                                SKGObjectBase::SKGListSKGObjectBase subOps;
                                err=operationObj.getSubOperations ( subOps );
                                int nbsubop=subOps.count();
                                for ( int j=0; err.isSucceeded() && j<nbsubop; ++j ) {
                                        SKGObjectBase sop=subOps.at ( j );
                                        if ( !listIdSubOp.contains ( sop.getID() ) ) {
                                                err=sop.remove();
                                        }
                                }
                        }
                }
                if (nb==1) {
                        if ( err.isSucceeded() ) err=operationObj.setNumber ( SKGServices::stringToInt ( ui.kNumberEdit->text() ) );
                        if ( err.isSucceeded() ) err=operationObj.setDate ( ui.kDateEdit->date() );
                }
                if ( err.isSucceeded() && ui.kCommentEdit->text() !=NOUPDATE ) {
                        if ( obj.getTable() !="v_operation_consolidated" )  err=operationObj.setComment ( ui.kCommentEdit->text() );
                }

                if ( err.isSucceeded() && ui.kAccountEdit->text() !=NOUPDATE ) {
                        SKGAccountObject account ( getDocument() );
                        err=account.setName ( ui.kAccountEdit->text() );
                        if ( err.isSucceeded() ) err=account.load();
                        if ( err.isSucceeded() ) err=operationObj.setParentAccount ( account );
                }
                if ( err.isSucceeded() && ui.kTypeEdit->text() !=NOUPDATE ) err=operationObj.setMode ( ui.kTypeEdit->text() );
                if ( err.isSucceeded() && ui.kPayeeEdit->text() !=NOUPDATE ) err=operationObj.setPayee ( ui.kPayeeEdit->text() );
                if ( err.isSucceeded() && ui.kUnitEdit->text() !=NOUPDATE ) {
                        SKGUnitObject unit ( getDocument() );
                        err=unit.setSymbol ( ui.kUnitEdit->text() );
                        if ( !unit.exist() ) {
                                if ( err.isSucceeded() ) err= unit.setName ( ui.kUnitEdit->text() );
                                if ( err.isSucceeded() ) err = unit.save();

                                SKGUnitValueObject unitVal;
                                if ( err.isSucceeded() ) err=unit.addUnitValue ( unitVal );
                                if ( err.isSucceeded() ) err=unitVal.setDate ( QDate::currentDate() );
                                if ( err.isSucceeded() ) err=unitVal.setQuantity ( 1 );
                                if ( err.isSucceeded() ) err=unitVal.save();

                                if ( err.isSucceeded() ) getDocument()->sendMessage ( tr ( "Unit [%1] has been created" ).arg ( ui.kUnitEdit->text() ), true );
                        } else err=unit.load();

                        if ( err.isSucceeded() ) err=operationObj.setUnit ( unit );
                }

                //Save
                if ( err.isSucceeded() ) err=operationObj.save();

                if ( err.isSucceeded() ) err=getDocument()->stepForward ( i+1 );
        }

        return err;
}

void SKGOperationPluginWidget::onAddOperationClicked()
{
        SKGError err;
        SKGTRACEIN ( 10, "SKGOperationPluginWidget::onAddOperationClicked" );

        //Get parameters
        QString accountName=ui.kAccountEdit->currentText();
        QString catName=ui.kCategoryEdit->currentText();
        QString trackerName=ui.kTrackerEdit->currentText();

        bool confirmed=true;
        SKGOperationObject operation;
        {
                SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Operation creation" ), err );

                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

                //Get account
                SKGAccountObject accountObj ( getDocument() );
                if ( err.isSucceeded() ) err=accountObj.setName ( accountName );
                if ( err.isSucceeded() ) err=accountObj.load();

                //Create operation object
                if ( err.isSucceeded() ) err=accountObj.addOperation ( operation );
                if ( err.isSucceeded() ) err=operation.setMode ( ui.kTypeEdit->currentText() );
                if ( err.isSucceeded() ) err=operation.setPayee ( ui.kPayeeEdit->currentText() );
                if ( err.isSucceeded() ) err=operation.setNumber ( SKGServices::stringToInt ( ui.kNumberEdit->text() ) );
                if ( err.isSucceeded() ) err=operation.setComment ( ui.kCommentEdit->text() );
                if ( err.isSucceeded() ) err=operation.setDate ( ui.kDateEdit->date() );

                SKGUnitObject unit ( getDocument() );
                err=unit.setSymbol ( ui.kUnitEdit->text() );
                if ( !unit.exist() ) {
                        if ( err.isSucceeded() ) err= unit.setName ( ui.kUnitEdit->text() );
                        if ( err.isSucceeded() ) err = unit.save();

                        SKGUnitValueObject unitVal;
                        if ( err.isSucceeded() ) err=unit.addUnitValue ( unitVal );
                        if ( err.isSucceeded() ) err=unitVal.setDate ( QDate::currentDate() );
                        if ( err.isSucceeded() ) err=unitVal.setQuantity ( 1 );
                        if ( err.isSucceeded() ) err=unitVal.save();

                        if ( err.isSucceeded() ) getDocument()->sendMessage ( tr ( "Unit [%1] has been created" ).arg ( ui.kUnitEdit->text() ), true );
                } else err=unit.load();

                if ( err.isSucceeded() ) err=operation.setUnit ( unit );
                if ( err.isSucceeded() ) err=operation.save();

                if ( getCurrentMode()<=1 ) {
                        //STD OPERATION (SPLITTED OR NOT)
                        if ( getCurrentMode()==0 ) {
                                //Create sub operation object
                                SKGSubOperationObject subOperation;
                                if ( err.isSucceeded() ) err=operation.addSubOperation ( subOperation );
                                SKGCategoryObject cat;
                                if ( err.isSucceeded() && !catName.isEmpty() && getCurrentMode()==0 ) {
                                        err=SKGCategoryObject::createPathCategory ( ( SKGDocumentBank* ) getDocument(), catName, cat, true );
                                        if ( err.isSucceeded() ) err=subOperation.setCategory ( cat );
                                }
                                if ( err.isSucceeded() && getCurrentMode()==0 ) {
                                        if (trackerName.isEmpty()) err=subOperation.removeRefundTracker();
                                        else {
                                                SKGRefundTrackerObject refund;
                                                err=SKGRefundTrackerObject::createRefund((SKGDocumentBank*) getDocument(), trackerName, refund, true);
                                                if ( err.isSucceeded() ) err=subOperation.setRefundTracker( refund );
                                        }
                                }

                                double val=ui.kAmountEdit->value();
                                if ( err.isSucceeded() ) {
                                        if ( cat.exist() ) {
                                                //Is the sign forced ?
                                                if ( ui.kAmountEdit->sign() ==0 ) {
                                                        SKGObjectBase cat2 ( cat.getDocument(), "v_category_display", cat.getID() );

                                                        //Are we able to find to sign with the category ?
                                                        if ( cat2.getAttribute ( "t_TYPEEXPENSE" ) ==tr ( "Expenditure" ) ) val=-val;
                                                }
                                        }
                                        err=subOperation.setQuantity ( val );
                                }
                                if ( err.isSucceeded() ) err=subOperation.save();
                        } else {
                                int nbsubop=ui.kSubOperationsTable->rowCount();
                                for ( int i=0; err.isSucceeded() && i<nbsubop; ++i ) {
                                        //Get values
                                        QTableWidgetItem* item=ui.kSubOperationsTable->item ( i,0 );
                                        QString catName=item->text();
                                        item=ui.kSubOperationsTable->item ( i,1 );
                                        QString comment=item->text();
                                        item=ui.kSubOperationsTable->item ( i,2 );
                                        double val=SKGServices::stringToDouble ( item->text() );

                                        //Create sub operation object
                                        SKGSubOperationObject subOperation;
                                        if ( err.isSucceeded() ) err=operation.addSubOperation ( subOperation );
                                        SKGCategoryObject cat;
                                        if ( err.isSucceeded() && !catName.isEmpty() && getCurrentMode()==0 ) {
                                                err=SKGCategoryObject::createPathCategory ( ( SKGDocumentBank* ) getDocument(), catName, cat, true );
                                                if ( err.isSucceeded() ) err=subOperation.setCategory ( cat );
                                        }
                                        if (err.isSucceeded()) {
                                                if (trackerName.isEmpty()) err=subOperation.removeRefundTracker();
                                                else {
                                                        SKGRefundTrackerObject refund;
                                                        err=SKGRefundTrackerObject::createRefund(( SKGDocumentBank* ) getDocument(), trackerName, refund, true );
                                                        if ( err.isSucceeded() ) err=subOperation.setRefundTracker(refund );
                                                }
                                        }
                                        if ( err.isSucceeded() ) err=subOperation.setComment ( comment );
                                        if ( err.isSucceeded() ) err=subOperation.setQuantity ( val );
                                        if ( err.isSucceeded() ) err=subOperation.save();
                                }
                        }
                } else if ( getCurrentMode()==2 ) {
                        //TRANSFER
                        //Create sub operation object
                        double operationQuantity=ui.kAmountEdit->value();

                        //Bug 192832 – Do not take amount sign into account vvv
                        if (abs(operationQuantity)!=operationQuantity) {
                                operationQuantity=-operationQuantity;
                                err=getDocument()->sendMessage(i18n("Absolute value has been used for transfer creation."));
                        }
                        //Bug 192832 – Do not take amount sign into account ^^^

                        SKGSubOperationObject subOperation;
                        if ( err.isSucceeded() ) err=operation.addSubOperation ( subOperation );
                        if ( err.isSucceeded() ) err=subOperation.setQuantity ( -operationQuantity );
                        if ( err.isSucceeded() ) err=subOperation.save();

                        //Get account
                        SKGAccountObject accountObj2 ( getDocument() );
                        if ( err.isSucceeded() ) err=accountObj2.setName ( ui.kTargetAccountEdit->currentText() );
                        if ( err.isSucceeded() ) err=accountObj2.load();

                        //Check unit of target account
                        //Correction bug 2299303 vvv
                        SKGUnitObject unitTargetAccount;
                        if ( err.isSucceeded() ) err=accountObj2.getUnit ( unitTargetAccount );
                        if ( err.isSucceeded() && unitTargetAccount.exist() && unit!=unitTargetAccount ) {
                                //The unit of the operation is not compliant with the unit of the target account
                                //We ask to the user if he wants to continue or convert into the target account
                                bool ok=false;
                                QApplication::setOverrideCursor ( QCursor ( Qt::ArrowCursor ) );
                                double newval=KInputDialog::getDouble ( i18n ( "Confirmation" ),
                                                                        i18n ( "The operation's unit is not compatible with the target account.\n"
                                                                               "Click Cancel if you want to continue anyway; "
                                                                               "otherwise, enter the value in the target account's unit (%1):", unitTargetAccount.getSymbol() ),
                                                                        SKGUnitObject::convert ( operationQuantity, unit, unitTargetAccount ),
                                                                        -DBL_MAX, DBL_MAX, 2, &ok, this );
                                QApplication::restoreOverrideCursor();
                                if ( ok ) {
                                        operationQuantity=newval;
                                        unit=unitTargetAccount;
                                }

                        }
                        //Correction bug 2299303 ^^^

                        //create transferred operation
                        SKGOperationObject operation2;
                        if ( err.isSucceeded() ) err=accountObj2.addOperation ( operation2 );
                        if ( err.isSucceeded() ) err=operation2.setMode ( ui.kTypeEdit->currentText() );
                        if ( err.isSucceeded() ) err=operation2.setPayee ( ui.kPayeeEdit->currentText() );
                        if ( err.isSucceeded() ) err=operation2.setNumber ( SKGServices::stringToInt ( ui.kNumberEdit->text() ) );
                        if ( err.isSucceeded() ) err=operation2.setComment ( ui.kCommentEdit->text() );
                        if ( err.isSucceeded() ) err=operation2.setDate ( ui.kDateEdit->date() );
                        if ( err.isSucceeded() ) err=operation2.setUnit ( unit );
                        if ( err.isSucceeded() ) err=operation2.setGroupOperation ( operation );
                        if ( err.isSucceeded() ) err=operation2.save();

                        //Create sub operation object
                        SKGSubOperationObject subOperation2;
                        if ( err.isSucceeded() ) err=operation2.addSubOperation ( subOperation2 );
                        if ( err.isSucceeded() ) err=subOperation2.setQuantity ( operationQuantity );
                        if ( err.isSucceeded() ) err=subOperation2.save();
                } else if ( getCurrentMode()==3 ) {
                        //PURCHASE OR SALE SHARE
                        //Create sub operation object
                        SKGSubOperationObject subOperation;
                        double val=ui.kAmountEdit->value();
                        if ( err.isSucceeded() ) err=operation.addSubOperation ( subOperation );
                        if ( err.isSucceeded() ) err=subOperation.setQuantity ( val );
                        if ( err.isSucceeded() ) err=subOperation.save();

                        if ( err.isSucceeded() && val>0 ) err=operation.setProperty ( "SKG_OP_ORIGINAL_AMOUNT", SKGServices::doubleToString ( ui.kAmountSharesEdit->value() ) );
                        if ( err.isSucceeded() ) err=operation.save();

                        //Get account
                        SKGAccountObject accountObj2 ( getDocument() );
                        if ( err.isSucceeded() ) err=accountObj2.setName ( ui.kPaymentAccountEdit->currentText() );
                        if ( err.isSucceeded() ) err=accountObj2.load();

                        //create paiement operation for shares
                        SKGOperationObject operation2;
                        if ( err.isSucceeded() ) err=accountObj2.addOperation ( operation2 );
                        if ( err.isSucceeded() ) err=operation2.setMode ( ui.kTypeEdit->currentText() );
                        if ( err.isSucceeded() ) err=operation2.setPayee ( ui.kPayeeEdit->currentText() );
                        if ( err.isSucceeded() ) err=operation2.setNumber ( SKGServices::stringToInt ( ui.kNumberEdit->text() ) );
                        if ( err.isSucceeded() ) err=operation2.setComment ( ui.kCommentEdit->text() );
                        if ( err.isSucceeded() ) err=operation2.setDate ( ui.kDateEdit->date() );
                        if ( err.isSucceeded() ) {
                                SKGUnitObject unitOfUnit;
                                err=unit.getUnit ( unitOfUnit );
                                if ( err.isSucceeded() ) err=operation2.setUnit ( unitOfUnit );
                        }
                        if ( err.isSucceeded() ) err=operation2.setGroupOperation ( operation );
                        if ( err.isSucceeded() ) err=operation2.save();

                        //Create main sub operation
                        SKGSubOperationObject subOperation2;
                        if ( err.isSucceeded() ) err=operation2.addSubOperation ( subOperation2 );
                        if ( err.isSucceeded() ) err=subOperation2.setComment ( i18n ( "Shares" ) );
                        if ( err.isSucceeded() ) err=subOperation2.setQuantity ( ui.kAmountSharesEdit->value() * ( val>0 ? -1 : 1 ) );
                        if ( err.isSucceeded() ) err=subOperation2.save();

                        //Create commission sub operation
                        if ( ui.kAmountSharesEdit->value() ) {
                                SKGSubOperationObject subOperation2;
                                if ( err.isSucceeded() ) err=operation2.addSubOperation ( subOperation2 );
                                if ( err.isSucceeded() ) err=subOperation2.setComment ( i18n ( "Commission" ) );
                                if ( err.isSucceeded() ) err=subOperation2.setQuantity ( -ui.kCommissionEdit->value() );
                                if ( err.isSucceeded() ) err=subOperation2.save();
                        }

                        //Create tax sub operation
                        if ( ui.kTaxEdit->value() ) {
                                SKGSubOperationObject subOperation2;
                                if ( err.isSucceeded() ) err=operation2.addSubOperation ( subOperation2 );
                                if ( err.isSucceeded() ) err=subOperation2.setComment ( i18n ( "Tax" ) );
                                if ( err.isSucceeded() ) err=subOperation2.setQuantity ( -ui.kTaxEdit->value() );
                                if ( err.isSucceeded() ) err=subOperation2.save();
                        }
                }

                QApplication::restoreOverrideCursor();
        }

        //status bar
        if ( err.isSucceeded() ) {
                if ( confirmed ) {
                        err=SKGError ( 0, i18n ( "Operation created" ) );
                        ui.kOperationView->selectObject ( operation.getUniqueID() );
                } else err=SKGError ( 0, i18n ( "Operation canceled" ) );
        } else err.addError ( ERR_FAIL, i18n ( "Operation creation failed" ) );

        //Display error
        getMainPanel()->displayErrorMessage ( err );
}

SKGObjectBase::SKGListSKGObjectBase SKGOperationPluginWidget::getSelectedObjects()
{
        return ui.kOperationView->getSelectedObjects();
}

int SKGOperationPluginWidget::getNbSelectedObjects()
{
        return ui.kOperationView->getNbSelectedObjects();
}

QString SKGOperationPluginWidget::getState()
{
        SKGTRACEIN ( 10, "SKGOperationPluginWidget::getState" );
        QDomDocument doc ( "SKGML" );
        QDomElement root;
        if ( lastState.hasChildNodes() ) {
                doc=lastState;
                root = doc.documentElement();
        } else {
                root = doc.createElement ( "parameters" );
                doc.appendChild ( root );
        }
        QString account=root.attribute ( "account" );
        QString currentPage=root.attribute ( "currentPage" );
        QString hideUselessOperation=root.attribute ( "hideUselessOperation" );
        QString filter=root.attribute ( "filter" );
        QString modeInfoZoneS=root.attribute ( "modeInfoZone" );
        QString reconcilitorAmountS=root.attribute ( "reconcilitorAmount" );

        if ( account.isEmpty() ) root.setAttribute ( "account", ui.kDisplayAccountCombo->currentText() );
        if ( currentPage.isEmpty() ) root.setAttribute ( "currentPage", getCurrentMode() );
        if ( hideUselessOperation.isEmpty() ) root.setAttribute ( "hideUselessOperation", ui.kHideUselessOperation->checkState() ==Qt::Checked ? "Y" : "N" );
        if ( filter.isEmpty() ) root.setAttribute ( "filter", ui.kFilterEdit->text() );
        if ( modeInfoZoneS.isEmpty() ) root.setAttribute ( "modeInfoZone", SKGServices::intToString ( modeInfoZone ) );
        if ( reconcilitorAmountS.isEmpty() ) root.setAttribute ( "reconcilitorAmount", ui.kReconcilitorAmountEdit->text() );

        //Memorize table settings
        root.setAttribute ( "view", ui.kOperationView->getState() );

        return doc.toString();
}

void SKGOperationPluginWidget::setState ( const QString& iState )
{
        SKGTRACEIN ( 10, "SKGOperationPluginWidget::setState" );
        QDomDocument doc ( "SKGML" );
        doc.setContent ( iState );
        QDomElement root = doc.documentElement();

        QString account=root.attribute ( "account" );
        QString currentPage=root.attribute ( "currentPage" );
        QString hideUselessOperation=root.attribute ( "hideUselessOperation" );
        QString filter=root.attribute ( "filter" );
        QString title=root.attribute ( "title" );
        QString title_icon=root.attribute ( "title_icon" );
        QString modeInfoZoneS=root.attribute ( "modeInfoZone" );
        QString reconcilitorAmountS=root.attribute ( "reconcilitorAmount" );

        if ( !account.isEmpty() ) {
                SKGAccountObject acc;
                SKGNamedObject::getObjectByName ( getDocument(), "v_account", account, acc );
                if ( acc.isClosed() && showClosedAccounts==false ) {
                        showClosedAccounts=true;
                        refresh();
                }
                ui.kDisplayAccountCombo->setText ( account );
        }
        if ( !modeInfoZoneS.isEmpty() ) {
                modeInfoZone=SKGServices::stringToInt ( modeInfoZoneS )-1;
                onRotateAccountTools();
        }
        if ( !reconcilitorAmountS.isEmpty() ) ui.kReconcilitorAmountEdit->setText ( reconcilitorAmountS );
        if ( !currentPage.isEmpty() ) setCurrentMode ( SKGServices::stringToInt ( currentPage ) );
        if ( !hideUselessOperation.isEmpty() ) ui.kHideUselessOperation->setCheckState ( hideUselessOperation=="Y" ? Qt::Checked : Qt::Unchecked );
        if ( !filter.isEmpty() ) ui.kFilterEdit->setText ( filter );
        if ( !title.isEmpty() ) {
                ui.kTitle->setComment("<html><body><b>"+title+"</b></body></html>");
                ui.kTitle->show();
        } else {
                ui.kTitle->hide();
        }
        if ( !title_icon.isEmpty() ) ui.kTitle->setPixmap ( KIcon ( title_icon ).pixmap ( 22, 22 ), KTitleWidget::ImageLeft );

        QSortFilterProxyModel* modelproxy = ( QSortFilterProxyModel* ) ui.kOperationView->model();
        SKGObjectModel* objectModel= ( SKGObjectModel* ) modelproxy->sourceModel ();
        QString operationTable=root.attribute ( "operationTable" );
        operationWhereClause=root.attribute ( "operationWhereClause" );
        if ( objectModel && !operationTable.isEmpty() )  objectModel->setTable ( operationTable );
        if ( objectModel && !operationWhereClause.isEmpty() )  objectModel->setFilter ( operationWhereClause );
        if ( !operationTable.isEmpty() || !operationWhereClause.isEmpty() ) {
                //We keep a copy of given state in case of bookmark
                lastState=doc;
        }

        onFilterChanged();
        onRefreshInformationZone();

        //!!! Must be done here after onFilterChanged
        ui.kOperationView->setState ( root.attribute ( "view" ) );
}

QString SKGOperationPluginWidget::getDefaultStateAttibute()
{
        return "SKGOPERATION_DEFAULT_PARAMETERS";
}

void SKGOperationPluginWidget::refresh()

{
        SKGTRACEIN ( 1, "SKGOperationPluginWidget::refresh" );

        //Refresh account summary
        QSqlDatabase* db = getDocument()->getDatabase();
        setEnabled ( db!=NULL );
        if ( db!=NULL ) {

                //Correction bug 2299394 vvv
                if ( ui.kOperationView->isAutoResized() ) ui.kOperationView->resizeColumnsToContents();
                //Correction bug 2299394 ^^^

                //Disconnect combo filter account
                disconnect ( ui.kDisplayAccountCombo, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onFilterChanged() ) );
                disconnect ( ui.kDisplayAccountCombo, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onRefreshInformationZone() ) );
                disconnect ( ui.kDisplayAccountCombo, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );
                disconnect ( ui.kUnitEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );
                disconnect ( ui.kAmountEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );
                disconnect ( ui.kAmountSharesEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );
                disconnect ( ui.kCommissionEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );
                disconnect ( ui.kTaxEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );

                //Set account combos
                QString current=ui.kDisplayAccountCombo->currentText();

                //Clear
                ui.kDisplayAccountCombo->clear();
                ui.kAccountEdit->clear();
                ui.kPaymentAccountEdit->clear();
                ui.kTargetAccountEdit->clear();

                ui.kDisplayAccountCombo->addItem ( i18n ( "All" ) );

                SKGStringListList listAccount;
                SKGServices::executeSelectSqliteOrder(getDocument(), QString("SELECT t_ICON, t_name from v_account_display ")+(showClosedAccounts ? "" :"where t_close='N' order by t_name"), listAccount);

                int nbAccounts=listAccount.count();
                if ( nbAccounts ==0 ) {
                        ui.kTitle->setText ( i18n ( "First you have to create an account." ) );
                        ui.kTitle->setPixmap ( KIcon ( "dialog-information" ).pixmap ( 22, 22 ) , KTitleWidget::ImageLeft);
                        ui.kTitle->show();
                } else {
                        if ( !lastState.hasChildNodes() ) ui.kTitle->hide();

                        QDir dirLogo ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/images/logo/" ) ) );
                        for (int i=1; i<nbAccounts; ++i) { //Ignore header
                                QIcon icon( dirLogo.absoluteFilePath ( listAccount.at(i).at(0) ) );
                                QString text=listAccount.at(i).at(1);
                                ui.kDisplayAccountCombo->addItem (icon, text);
                                ui.kAccountEdit->addItem (icon, text);
                                ui.kPaymentAccountEdit->addItem (icon, text);
                                ui.kTargetAccountEdit->addItem (icon, text);
                        }
                }

                int pos=ui.kDisplayAccountCombo->findText ( current );
                if ( pos==-1 ) pos=0;
                ui.kDisplayAccountCombo->setCurrentIndex ( pos );

                //Set type combo
                SKGMainPanel::fillComboWithDistinctValue ( ui.kTypeEdit, getDocument(), "operation", "t_mode", "" );

                //Set type unit
                SKGMainPanel::fillComboWithDistinctValue ( ui.kUnitEdit, getDocument(), "unit", "ifnull(t_symbol,t_name)", "t_type!='I'" );
                QString primary= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
                if ( !primary.isEmpty() ) ui.kUnitEdit->setText ( primary );

                //Set type category
                SKGMainPanel::fillComboWithDistinctValue ( ui.kCategoryEdit, getDocument(), "category", "t_fullname", "" );

                //Set type refund
                SKGMainPanel::fillComboWithDistinctValue ( ui.kTrackerEdit, getDocument(), "refund", "t_name", "t_close='N'" );

                //Set type payee
                SKGMainPanel::fillComboWithDistinctValue ( ui.kPayeeEdit, getDocument(), "operation", "t_payee", "" );

                //Set type number
                numberFieldIsNotUptodate=true;

                //Connect combo filter account
                connect ( ui.kDisplayAccountCombo, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onFilterChanged() ), Qt::QueuedConnection );
                connect ( ui.kDisplayAccountCombo, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onRefreshInformationZone() ), Qt::QueuedConnection );
                connect ( ui.kDisplayAccountCombo, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
                connect ( ui.kUnitEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
                connect ( ui.kAmountEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
                connect ( ui.kAmountSharesEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
                connect ( ui.kCommissionEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
                connect ( ui.kTaxEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
        }
}

void SKGOperationPluginWidget::openOperation ( const SKGOperationObject& iOperation, SKGMainPanel* iParent )
{
        _SKGTRACEIN ( 10, "SKGOperationPluginWidget::openOperation" );

        //Build where clause and title
        int opid=iOperation.getID();
        QString wc="id="+SKGServices::intToString ( opid );

        opid=SKGServices::stringToInt ( iOperation.getAttribute ( "i_group_id" ) );
        if ( opid!=0 ) wc+=" or i_group_id="+SKGServices::intToString ( opid );
        wc='('+wc+')';
        QString title=i18n ( "Operations grouped or split" );

        //Open
        if ( QApplication::keyboardModifiers() &Qt::ControlModifier && QApplication::keyboardModifiers() &Qt::ShiftModifier ) {
                //Call debug plugin
                QDomDocument doc ( "SKGML" );
                QDomElement root = doc.createElement ( "parameters" );
                doc.appendChild ( root );
                root.setAttribute ( "sqlOrder", "SELECT * from v_operation_consolidated WHERE "+wc );

                if ( iParent ) iParent->setNewTabContent ( iParent->getPluginByName ( "Skrooge debug plugin" ), -1, doc.toString() );
        } else {
                //Call operation plugin
                QDomDocument doc("SKGML");
                doc.setContent ( iOperation.getDocument()->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
                QDomElement root = doc.documentElement();
                if (root.isNull()) {
                        root=doc.createElement("parameters");
                        doc.appendChild(root);
                }

                root.setAttribute ( "operationTable", "v_operation_consolidated" );
                root.setAttribute ( "operationWhereClause", wc );
                root.setAttribute ( "title", title );
                root.setAttribute ( "title_icon", "view-statistics" );
                root.setAttribute ( "currentPage", "-1" );

                if ( iParent ) iParent->setNewTabContent ( iParent->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
        }
}

void SKGOperationPluginWidget::onDoubleClick()
{
        _SKGTRACEIN ( 10, "SKGOperationPluginWidget::onDoubleClick" );

        //Get selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        if ( selection.count() ==1 ) {
                SKGOperationObject op ( selection.at ( 0 ) );
                openOperation ( op, getMainPanel() );
        }
}

void SKGOperationPluginWidget::onResetInternalFilter()
{
        SKGTRACEIN ( 10, "SKGOperationPluginWidget::onResetInternalFilter" );
        QSortFilterProxyModel* modelproxy = ( QSortFilterProxyModel* ) ui.kOperationView->model();
        SKGObjectModel* objectModel= ( SKGObjectModel* ) modelproxy->sourceModel ();

        operationWhereClause="";
        if ( objectModel ) objectModel->setTable ( "v_operation_display" );

        ui.kTitle->hide();
        lastState.clear();

        onFilterChanged();
}

void SKGOperationPluginWidget::onFilterRegExpChanged()
{
        SKGTRACEIN ( 10, "SKGOperationPluginWidget::onFilterRegExpChanged" );
        QRegExp regExp ( ui.kFilterEdit->text(), Qt::CaseInsensitive );
        ( ( SKGSortFilterProxyModel* ) ui.kOperationView->model() )->setFilterRegExp ( regExp );
}

void SKGOperationPluginWidget::onRefreshInformationZone()
{
        SKGTRACEIN ( 1, "SKGOperationPluginWidget::onRefreshInformationZone" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        KLocale* locale=KGlobal::locale();
        QString unit1= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
        double unit1Value=1;
        QString unit2= ( ( SKGDocumentBank* ) getDocument() )->getSecondaryUnit();
        double unit2Value= ( ( SKGDocumentBank* ) getDocument() )->getSecondaryUnitValue();

        if ( modeInfoZone==0 ) {
                //Refresh info area
                //Compute where clause
                QString filter="1=1";
                if ( ui.kDisplayAccountCombo->currentIndex() >0 ) {
                        filter="t_name='"+SKGServices::stringToSqlString ( ui.kDisplayAccountCombo->currentText() ) +'\'';
                }
                SKGStringListList listTmp;
                SKGServices::executeSelectSqliteOrder ( getDocument(),
                                                        "SELECT SUM(f_CURRENTAMOUNT), SUM(f_CHECKED), SUM(f_COMING_SOON) from v_account_display WHERE "+filter,
                                                        listTmp );
                if ( listTmp.count() ==2 ) {
                        if ( ui.kDisplayAccountCombo->currentIndex() >0 ) {
                                SKGUnitObject unitAccount;
                                SKGAccountObject account ( getDocument() );
                                if ( account.setName ( ui.kDisplayAccountCombo->currentText() ).isSucceeded() ) {
                                        if ( account.load().isSucceeded() ) {
                                                if ( account.getUnit ( unitAccount ).isSucceeded() ) {
                                                        if ( !unitAccount.getSymbol().isEmpty() ) {
                                                                unit1=unitAccount.getSymbol();
                                                                unit1Value=SKGServices::stringToDouble ( unitAccount.getAttribute ( "f_CURRENTAMOUNT" ) );

                                                                if ( unit1!= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit() ) {
                                                                        unit2= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
                                                                        unit2Value=1;
                                                                }
                                                        }
                                                }
                                        }
                                }
                        }

                        double v1=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 0 ) );
                        double v2=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 1 ) );
                        double v3=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 2 ) );
                        QString s1=QString ( "<font color=\"" ) + ( v1<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v1/unit1Value, unit1,2 ) +"</font>";
                        QString s2=QString ( "<font color=\"" ) + ( v2<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v2/unit1Value, unit1,2 ) +"</font>";
                        QString s3=QString ( "<font color=\"" ) + ( v3<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v3/unit1Value, unit1,2 ) +"</font>";
                        ui.kInfo->setText ( i18n ( "Balance: %1     Cleared: %2     In Transit: %3", s1, s2, s3 ) );
                        if ( !unit2.isEmpty() && unit2Value ) {
                                s1=QString ( "<font color=\"" ) + ( v1<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v1/unit2Value, unit2,2 ) +"</font>";
                                s2=QString ( "<font color=\"" ) + ( v2<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v2/unit2Value, unit2,2 ) +"</font>";
                                s3=QString ( "<font color=\"" ) + ( v3<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v3/unit2Value, unit2,2 ) +"</font>";
                        }
                        ui.kInfo->setToolTip ( i18n ( "<p>Balance: %1</p><p>Cleared: %2</p><p>In Transit: %3</p>", s1, s2, s3 ) );
                }
        } else if (modeInfoZone==1) {
                //Refresh reconciliation area
                //Compute where clause
                QString filter='\''+SKGServices::stringToSqlString ( ui.kDisplayAccountCombo->currentText() ) +'\'';
                SKGStringListList listTmp;
                double diff=0;
                SKGServices::executeSelectSqliteOrder ( getDocument(),
                                                        "SELECT SUM(f_CHECKED) from v_account_display WHERE t_name="+filter,
                                                        listTmp );
                if ( listTmp.count() ==2 ) {
                        diff=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 0 ) )-ui.kReconcilitorAmountEdit->value();
                }
                SKGServices::executeSelectSqliteOrder ( getDocument(),
                                                        "SELECT ABS(TOTAL(f_CURRENTAMOUNT_EXPENSE)),TOTAL(f_CURRENTAMOUNT_INCOME) FROM v_operation_display WHERE t_status='P' AND t_ACCOUNT="+filter,
                                                        listTmp );
                if ( listTmp.count() ==2 ) {
                        if ( ui.kDisplayAccountCombo->currentIndex() >0 ) {
                                SKGUnitObject unitAccount;
                                SKGAccountObject account ( getDocument() );
                                if ( account.setName ( ui.kDisplayAccountCombo->currentText() ).isSucceeded() ) {
                                        if ( account.load().isSucceeded() ) {
                                                if ( account.getUnit ( unitAccount ).isSucceeded() ) {
                                                        if ( !unitAccount.getSymbol().isEmpty() ) {
                                                                unit1=unitAccount.getSymbol();
                                                                unit1Value=SKGServices::stringToDouble ( unitAccount.getAttribute ( "f_CURRENTAMOUNT" ) );

                                                                if ( unit1!= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit() ) {
                                                                        unit2= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
                                                                        unit2Value=1;
                                                                }
                                                        }
                                                }
                                        }
                                }
                        }

                        double v1=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 0 ) );
                        double v2=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 1 ) );
                        QString d=locale->formatMoney ( diff/unit1Value, unit1,2 );
                        QString sdiff=QString ( "<font color=\"" ) + ( diff<0 ? "red" : "black" ) +"\">"+d+"</font>";
                        QString s1=QString ( "<font color=\"" ) + ( v1<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v1/unit1Value, unit1,2 ) +"</font>";
                        QString s2=QString ( "<font color=\"" ) + ( v2<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v2/unit1Value, unit1,2 ) +"</font>";
                        ui.kReconciliatorInfo->setText ( i18n ( "%1 - Delta: %2     Expenditure: %3     Income: %4", unit1, sdiff, s1, s2 ) );
                        ui.kValidate->setEnabled(d==locale->formatMoney ( 0, unit1,2 ));

                        if ( !unit2.isEmpty() && unit2Value ) {
                                sdiff=QString ( "<font color=\"" ) + ( diff<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( diff/unit2Value, unit2,2 ) +"</font>";
                                s1=QString ( "<font color=\"" ) + ( v1<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v1/unit2Value, unit2,2 ) +"</font>";
                                s2=QString ( "<font color=\"" ) + ( v2<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v2/unit2Value, unit2,2 ) +"</font>";
                        }
                        ui.kReconciliatorInfo->setToolTip ( i18n ( "<p>Delta: %1</p><p>Expenditure: %2</p><p>Income: %3</p>", sdiff, s1, s2 ) );

                }

        } else {
                //Refresh info area with selection
                SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
                double amount=0;
                int nb=selection.count();
                for (int i=0; i<nb; ++i) {
                        SKGOperationObject obj=selection.at(i);
                        amount+=obj.getAmount(QDate::currentDate());
                }

                QString v2=QString ( "<font color=\"" ) + ( amount<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( amount/unit1Value, unit1,2 ) +"</font>";
                ui.kInfo->setText ( i18np ( "Selection: 1 operation for %2", "Selection: %1 operations for %2", nb, v2 ) );
                if ( !unit2.isEmpty() && unit2Value ) {
                        v2=QString ( "<font color=\"" ) + ( amount<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( amount/unit2Value, unit2,2 ) +"</font>";
                }
                ui.kInfo->setToolTip ( i18np ( "Selection: 1 operation for %2", "Selection: %1 operations for %2", nb, v2 ) );
        }
        QApplication::restoreOverrideCursor();
}

void SKGOperationPluginWidget::onFilterChanged()
{
        SKGTRACEIN ( 1, "SKGOperationPluginWidget::onFilterChanged" );
        if ( !isEnabled() ) return;
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

        //Enable/ disable widgets
        bool onOneAccount= ( ui.kDisplayAccountCombo->currentIndex() >0 );
        ui.kAccountEdit->setEnabled ( !onOneAccount );
        ui.kReconciliatorFrame2->setEnabled ( onOneAccount );
        if ( !onOneAccount && modeInfoZone==1 ) {
                ui.kReconciliatorFrame2->hide();
                ui.kInfo->show();
                modeInfoZone=0;
        }

        ui.kAccountLabel->setEnabled ( operationWhereClause.isEmpty() );
        ui.kDisplayAccountCombo->setEnabled ( operationWhereClause.isEmpty() );
        ui.kHideUselessOperation->setEnabled ( operationWhereClause.isEmpty() );
        if ( operationWhereClause.isEmpty() ) ui.kresetInternalFiler->hide();
        else ui.kresetInternalFiler->show();

        //Compute where clause
        QString filter2=operationWhereClause;
        if ( onOneAccount ) {
                QString account=ui.kDisplayAccountCombo->currentText();
                if ( operationWhereClause.isEmpty() ) filter2="t_ACCOUNT='"+SKGServices::stringToSqlString ( account ) +'\'';

                ui.kAccountEdit->setText ( account );
        }

        //Add status filter
        if ( operationWhereClause.isEmpty() && ui.kHideUselessOperation->checkState() ==Qt::Checked ) {
                if ( !filter2.isEmpty() ) filter2+=" AND ";
                filter2+="t_status!='C'";
        }

        //Update model
        QSortFilterProxyModel* modelproxy = ( QSortFilterProxyModel* ) ui.kOperationView->model();
        SKGObjectModel* objectModel= ( SKGObjectModel* ) modelproxy->sourceModel ();
        if ( objectModel ) {
                objectModel->setFilter ( filter2 );
                objectModel->refresh();
        }

        //Correction bug 2299394 vvv
        if ( ui.kOperationView->isAutoResized() ) ui.kOperationView->resizeColumnsToContents();
        //Correction bug 2299394 ^^^

        QApplication::restoreOverrideCursor();
}

void SKGOperationPluginWidget::onSplitOperationClicked()
{
        setCurrentMode(1);;
}

void SKGOperationPluginWidget::fillNumber()
{
        SKGTRACEIN ( 10, "SKGOperationPluginWidget::fillNumber" );
        QStringList list;
        SKGServices::getDistinctValues ( getDocument(), "v_operation_next_numbers", "i_number", "", list );

        //Fill completion
        KCompletion *comp = ui.kNumberEdit->completionObject();
        comp->clear ();
        comp->insertItems ( list );

        numberFieldIsNotUptodate=false;
}

void SKGOperationPluginWidget::onFocusChanged()
{
        _SKGTRACEIN ( 10, "SKGOperationPluginWidget::onFocusChanged" );

        if ( numberFieldIsNotUptodate && ui.kNumberEdit->hasFocus() ) {
                fillNumber();
        }

        bool test=ui.kTypeEdit->hasFocus() ||
//                  ui.kAmountEdit->hasFocus() ||
//  		    ui.kNumberEdit->hasFocus() ||
                  ui.kUnitEdit->hasFocus() ||
                  ui.kCategoryEdit->hasFocus() ||
                  ui.kTrackerEdit->hasFocus() ||
                  ui.kCommentEdit->hasFocus() ||
                  ui.kPayeeEdit->hasFocus()
                  ;
        if ( fastEditionAction ) fastEditionAction->setEnabled ( test );
}

void SKGOperationPluginWidget::onFastEdition()
{
        SKGTRACEIN ( 10, "SKGOperationPluginWidget::onFocusChanged" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        SKGError err;

        //Get widget item
        QWidget* w=QApplication::focusWidget();

        //Build the where clause
        QString wc;
        if ( ui.kTypeEdit->hasFocus() ) wc="t_mode LIKE '"+SKGServices::stringToSqlString ( ui.kTypeEdit->text() ) +"%'";
        else if ( ui.kUnitEdit->hasFocus() ) wc="t_UNIT LIKE '"+SKGServices::stringToSqlString ( ui.kUnitEdit->text() ) +"%'";
        else if ( ui.kCategoryEdit->hasFocus() ) wc="t_CATEGORY LIKE '"+SKGServices::stringToSqlString ( ui.kCategoryEdit->text() ) +"%'";
        else if ( ui.kTrackerEdit->hasFocus() ) wc="t_REFUND LIKE '"+SKGServices::stringToSqlString ( ui.kTrackerEdit->text() ) +"%'";
        else if ( ui.kCommentEdit->hasFocus() ) wc="t_comment LIKE '"+SKGServices::stringToSqlString ( ui.kCommentEdit->text() ) +"%'";
        else if ( ui.kPayeeEdit->hasFocus() ) wc="t_payee LIKE '"+SKGServices::stringToSqlString ( ui.kPayeeEdit->text() ) +"%'";

        if ( !wc.isEmpty() ) {
                if ( wc!=lastFastEditionWhereClause ) {
                        lastFastEditionWhereClause=wc;
                        lastFastEditionOperationFound=0;
                }

                //Look for last operation
                if ( lastFastEditionOperationFound!=0 ) {
                        wc+=" AND id<"+SKGServices::intToString ( lastFastEditionOperationFound );
                }

                wc+=" ORDER BY d_date DESC, id DESC LIMIT 1";

                SKGObjectBase::SKGListSKGObjectBase operations;
                err=SKGObjectBase::getObjects ( getDocument(), "v_operation_display", wc, operations );
                if ( err.isSucceeded() && operations.count() ) {
                        SKGOperationObject op=operations.at ( 0 );

                        lastFastEditionOperationFound=op.getID();
                        ui.kTypeEdit->setText ( op.getMode() );
                        ui.kUnitEdit->setText ( op.getAttribute ( "t_UNIT" ) );
                        ui.kCategoryEdit->setText ( op.getAttribute ( "t_CATEGORY" ) );
                        ui.kCommentEdit->setText ( op.getComment() );
                        ui.kPayeeEdit->setText ( op.getPayee() );
                        ui.kTrackerEdit->setText(op.getAttribute ( "t_REFUND" ));
                        if ( ui.kAccountEdit->isEnabled() ) ui.kAccountEdit->setText ( op.getAttribute ( "t_ACCOUNT" ) );
                        ui.kAmountEdit->setValue ( SKGServices::stringToDouble ( op.getAttribute ( "f_QUANTITY" ) ) );

                        //set next number
                        int number=op.getNumber();
                        if (number==0) {
                                ui.kNumberEdit->setText ("");
                        } else {
                                if (numberFieldIsNotUptodate) {
                                        fillNumber();
                                }

                                KCompletion *comp = ui.kNumberEdit->completionObject();
                                if (comp) {
                                        QStringList list=comp->items();
                                        int nb=list.count();
                                        int cpt=0;
                                        while ( nb>=0 && cpt>=0 && cpt<1000 ) {
                                                ++number;

                                                if (list.contains(SKGServices::intToString(number))) {
                                                        cpt=-2;
                                                }
                                                ++cpt;
                                        }

                                        if (cpt<0) ui.kNumberEdit->setText(SKGServices::intToString(number));
                                }
                        }

                        //Get nb operation linked
                        SKGObjectBase::SKGListSKGObjectBase groupedOperations;
                        op.getGroupedOperations ( groupedOperations );
                        int nbGroupedOp=groupedOperations.count();

                        //Get nb sub op
                        SKGObjectBase::SKGListSKGObjectBase subOperations;
                        op.getSubOperations ( subOperations );
                        int nbSupOp=subOperations.count();

                        if ( nbSupOp>1 ) {
                                //It is a SPLITTED operation
                                setCurrentMode(1);;
                                displaySubOperations ( op );
                        } else {
                                if ( nbGroupedOp>1 ) {
                                        //It is a TRANSFER
                                        SKGOperationObject op2=groupedOperations.at ( 0 );
                                        if ( op2==op ) op2=groupedOperations.at ( 1 );

                                        SKGAccountObject targetAccount;
                                        op2.getParentAccount ( targetAccount );

                                        ui.kTargetAccountEdit->setText ( targetAccount.getName() );
                                } else setCurrentMode(0);
                        }

                } else {
                        lastFastEditionWhereClause="";
                        lastFastEditionOperationFound=0;
                }
        }

        w->setFocus(Qt::OtherFocusReason);
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage ( err );

}

int SKGOperationPluginWidget::getCurrentMode()
{
        if (ui.kStandardFrm->isVisible()) return 0;
        else if (ui.kSplitFrm->isVisible()) return 1;
        else if (ui.kTransferFrm->isVisible()) return 2;
        else if (ui.kShareFrm->isVisible()) return 3;
        return -1;
}

void SKGOperationPluginWidget::onBtnModeClicked()
{
        QWidget* sender=static_cast<QWidget*> ( this->sender() );

        int currentMode=getCurrentMode();
        int newMode=0;

        if (sender==ui.kStandardBtn) newMode=0;
        else if (sender==ui.kSplitBtn) newMode=1;
        else if (sender==ui.kTransferBtn) newMode=2;
        else if (sender==ui.kShareBtn) newMode=3;

        if (currentMode==newMode) newMode=-1;
        setCurrentMode(newMode);
}

void SKGOperationPluginWidget::setCurrentMode(int iMode)
{
        ui.kStandardFrm->hide();
        ui.kSplitFrm->hide();
        ui.kTransferFrm->hide();
        ui.kShareFrm->hide();
        ui.kCommunFrm->setVisible(iMode>=0);
        ui.kBtnFrm->setVisible(iMode>=0);

        ui.kStandardBtn->setChecked(false);
        ui.kSplitBtn->setChecked(false);
        ui.kTransferBtn->setChecked(false);
        ui.kShareBtn->setChecked(false);

        if (iMode!=1 && iMode!=-1) {
                ui.kSubOperationsTable->setRowCount(0);
                ui.kSubOperationsTable->clearContents();
        }

        if (iMode==0) {
                ui.kStandardFrm->show();
                ui.kStandardBtn->setChecked(true);
        } else if (iMode==1) {
                if (ui.kSubOperationsTable->rowCount()==0) {
                        addSubOperationLine ( 0, ui.kCategoryEdit->text(), ui.kTrackerEdit->text(), ui.kCommentEdit->text(), ui.kAmountEdit->value(), 0 );
                }
                ui.kSplitFrm->show();
                ui.kSplitBtn->setChecked(true);
                //ui.kStandardBtn->setChecked(true);
        } else if (iMode==2) {
                ui.kTransferFrm->show();
                ui.kTransferBtn->setChecked(true);
        } else if (iMode==3) {
                ui.kShareFrm->show();
                ui.kShareBtn->setChecked(true);
        }


        onOperationCreatorModified();
}

void SKGOperationPluginWidget::displaySubOperations ( const SKGOperationObject& iOperation )
{
        ui.kSubOperationsTable->setRowCount ( 0 );
        ui.kSubOperationsTable->clearContents();

        int nbSubOperations=0;

        QList<SKGObjectBase> subOperations;
        SKGError err =  iOperation.getSubOperations ( subOperations );
        nbSubOperations = subOperations.count();
        for ( int i = 0; i < nbSubOperations; ++i ) {
                SKGSubOperationObject subOperation = subOperations.at ( i );

                SKGCategoryObject category;
                subOperation.getCategory ( category );

                SKGRefundTrackerObject refund;
                subOperation.getRefundTracker( refund );

                addSubOperationLine ( i, category.getFullName(), refund.getName(),
                                      subOperation.getComment(), subOperation.getQuantity(),
                                      subOperation.getID() );
        }

        onQuantityChanged();
}

void SKGOperationPluginWidget::displaySubOperations()
{
        SKGOperationObject operation;
        if ( getSelectedOperation ( operation ).isSucceeded() ) displaySubOperations ( operation );
}

double SKGOperationPluginWidget::getRemainingQuantity()
{
        double sumQuantities = 0;
        int nbSubOperations = ui.kSubOperationsTable->rowCount();

        for ( int i = 0; i < nbSubOperations ; ++i ) {
                QTableWidgetItem* quantityItem = ui.kSubOperationsTable->item ( i,2 );
                sumQuantities = sumQuantities + SKGServices::stringToDouble ( quantityItem->text() );
        }

        return ui.kAmountEdit->value() - sumQuantities;
}

void SKGOperationPluginWidget::onQuantityChanged()
{
        int nbSubOperations = ui.kSubOperationsTable->rowCount();

        QTableWidgetItem* remainingQuantityItem = ui.kSubOperationsTable->item ( nbSubOperations-1,2 );
        if ( remainingQuantityItem ) {
                bool previous=ui.kSubOperationsTable->blockSignals ( true ); // Disable signals so that filling cell doesn't create new lines
                remainingQuantityItem->setText ( SKGServices::doubleToString ( SKGServices::stringToDouble ( remainingQuantityItem->text() ) +getRemainingQuantity() ) );
                ui.kSubOperationsTable->blockSignals ( previous ); // Reenable signals
        }
}

void SKGOperationPluginWidget::onSubopCellChanged ( int row, int column )
{
        int nbSubOperations = ui.kSubOperationsTable->rowCount();
        if ( row == nbSubOperations-1 && column == 2 ) {
                // If the quantity in the last line is edited, we add a new
                // line with the new remaining quantity
                addSubOperationLine ( nbSubOperations,"","","",0 );
                onQuantityChanged();
        } else if ( column == 2 ) {
                onQuantityChanged();
        }
}

void SKGOperationPluginWidget::onRemoveSubOperation()
{
        QList<int> rowsToRemove;
        QList<QTableWidgetItem*> selectedItems = ui.kSubOperationsTable->selectedItems();
        int nb=selectedItems.count();
        for ( int i = 0; i < nb; ++i ) {
                QTableWidgetItem* item = selectedItems.at ( i );
                int row = item->row();
                if ( !rowsToRemove.contains ( row ) ) {
                        rowsToRemove.append ( row );
                }
        }
        for ( int j = rowsToRemove.count()-1; j >=0 ; --j ) {
                ui.kSubOperationsTable->removeRow ( rowsToRemove.at ( j ) );
        }

        // If all rows removed, add an empty line
        if ( ui.kSubOperationsTable->rowCount() == 0 ) {
                addSubOperationLine ( 0,"","","",0 );
        }

        onQuantityChanged();
}

void SKGOperationPluginWidget::onRotateAccountTools()
{
        if ( modeInfoZone==0 ) {
                //Switch to reconciliation mode
                ui.kReconciliatorFrame2->show();
                ui.kInfo->hide();
                modeInfoZone=1;
        } else if ( modeInfoZone==1 ) {
                //Switch to information on selection mode
                ui.kReconciliatorFrame2->hide();
                ui.kInfo->show();
                modeInfoZone=2;
        } else {
                //Switch to information mode
                ui.kReconciliatorFrame2->hide();
                ui.kInfo->show();
                modeInfoZone=0;
        }
        onRefreshInformationZone();
}

void SKGOperationPluginWidget::onValidatePointedOperations()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPluginWidget::onValidatePointedOperations",err );

        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        QString account=ui.kDisplayAccountCombo->currentText();
        SKGObjectBase::SKGListSKGObjectBase list;
        err=SKGObjectBase::getObjects ( getDocument(), "v_operation_display", "t_status='P' AND t_ACCOUNT='"+SKGServices::stringToSqlString ( account ) +'\'', list );
        int nb=list.count();
        if ( err.isSucceeded() ) {
                SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18n ( "Switch to checked" ), err, nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                        SKGOperationObject op=list[i];
                        err=op.setStatus ( SKGOperationObject::CHECKED );
                        if ( err.isSucceeded() ) err=op.save();
                        if ( err.isSucceeded() ) err=getDocument()->stepForward ( i+1 );
                }
        }
        //status bar
        if ( err.isSucceeded() ) err=SKGError ( 0, i18n ( "Operation checked." ) );
        else err.addError ( ERR_FAIL, i18n ( "Switch failed" ) );
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage ( err );
}

void SKGOperationPluginWidget::addSubOperationLine ( int row, const QString& category, const QString& refund, const QString& comment, double quantity, int id )
{
        bool previous=ui.kSubOperationsTable->blockSignals ( true );

        ui.kSubOperationsTable->insertRow ( row );

        // Category
        QTableWidgetItem* categoryItem = new QTableWidgetItem ( category );
        categoryItem->setData ( Qt::UserRole, id );
        ui.kSubOperationsTable->setItem ( row,0,categoryItem );

        // Comment
        ui.kSubOperationsTable->setItem ( row,1,new QTableWidgetItem ( comment ) );

        // Quantity
        QTableWidgetItem* quantityItem = new QTableWidgetItem ( SKGServices::doubleToString ( quantity ) );
        ui.kSubOperationsTable->setItem ( row,2,quantityItem );

        // Refund
        QTableWidgetItem* refundItem = new QTableWidgetItem ( refund );
        categoryItem->setData ( Qt::UserRole, id );
        ui.kSubOperationsTable->setItem ( row,3,refundItem );

        ui.kSubOperationsTable->blockSignals ( previous );

        ui.kSubOperationsTable->resizeColumnsToContents();
}

QWidget* SKGOperationPluginWidget::getWidgetForPrint()
{
        return ui.kOperationView;
}

SKGError SKGOperationPluginWidget::getSelectedOperation ( SKGOperationObject& operation )
{
        SKGError err;
        SKGObjectBase::SKGListSKGObjectBase selectedOperations = getSelectedObjects();
        if ( selectedOperations.count() > 0 ) {
                operation = selectedOperations.at ( 0 );
                err.setReturnCode ( 0 );
        } else {
                err.setReturnCode ( 1 );
                err.setMessage ( i18n ( "No Operation Selected" ) );
        }
        return err;
}

#include "skgoperationpluginwidget.moc"
