/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to follow refund.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgrefundpluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgobjectmodel.h"
#include "skgsortfilterproxymodel.h"
#include "skgdocumentbank.h"
#include "skgrefundtrackerobject.h"
#include "skgtransactionmng.h"

#include <QDomDocument>

SKGRefundPluginWidget::SKGRefundPluginWidget(SKGMainPanel* iParent, SKGDocument* iDocument)
                : SKGTabWidget(iParent, iDocument)
{
        SKGTRACEIN(1, "SKGRefundPluginWidget::SKGRefundPluginWidget");

        ui.setupUi(this);

        ui.kAddButton->setIcon(KIcon("list-add"));
        ui.kModifyButton->setIcon(KIcon("dialog-ok-apply"));

        objectModel = new SKGObjectModel((SKGDocumentBank*) getDocument(), "v_refund_display", "1=0", this);
        SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel(this);
        modelproxy->setSourceModel(objectModel);
        modelproxy->setSortRole(Qt::UserRole);
        modelproxy->setDynamicSortFilter(true);
        ui.kView->setModel(modelproxy);
        ui.kView->setWindowTitle(i18n("Refund"));

        //Add registered global action in contextual menu
        if (iParent) {
                ui.kView->insertAction(0, iParent->getGlobalAction("edit_delete"));
                QAction* sep=new QAction(this);
                sep->setSeparator(true);
                ui.kView->insertAction(0, sep);
                ui.kView->insertAction(0, iParent->getGlobalAction("open_report"));

                ui.kView->resizeColumnToContents(0);
        }

        ui.kTitle->setPixmap(KIcon("dialog-information").pixmap(22, 22), KTitleWidget::ImageLeft);

        connect((const QObject*) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)), Qt::QueuedConnection );
        dataModified("", 0);

        connect(ui.kView->selectionModel(), SIGNAL(selectionChanged(QItemSelection,QItemSelection)), this, SLOT(onSelectionChanged()));

        connect(objectModel, SIGNAL(beforeReset()), ui.kView, SLOT(saveSelection()));
        connect(objectModel, SIGNAL(afterReset()), ui.kView, SLOT(resetSelection()));
}

SKGRefundPluginWidget::~SKGRefundPluginWidget()
{
        SKGTRACEIN(1, "SKGRefundPluginWidget::~SKGRefundPluginWidget");
        objectModel=NULL;
}

void SKGRefundPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
        SKGTRACEIN(1, "SKGRefundPluginWidget::dataModified");
        if (iTableName=="refund" || iTableName.isEmpty()) {
                //Correction bug 2299394 vvv
                if (ui.kView->isAutoResized()) ui.kView->resizeColumnsToContents();
                //Correction bug 2299394 ^^^
                onEditorModified();
        }
}

void SKGRefundPluginWidget::onSelectionChanged()
{
        SKGTRACEIN(10, "SKGRefundPluginWidget::onSelectionChanged");

        SKGObjectBase::SKGListSKGObjectBase objs=getSelectedObjects();
        if (objs.count()) {
                SKGRefundTrackerObject obj=objs.at(0);
                ui.kNameInput->setText(obj.getName());
                ui.kCommentEdit->setText(obj.getComment());
        } else {
                ui.kNameInput->setText("");
                ui.kCommentEdit->setText("");
        }
        onEditorModified();
        emit selectionChanged();
}

SKGObjectBase::SKGListSKGObjectBase SKGRefundPluginWidget::getSelectedObjects()
{
        return ui.kView->getSelectedObjects();
}

int SKGRefundPluginWidget::getNbSelectedObjects()
{
        return ui.kView->getNbSelectedObjects();
}

QString SKGRefundPluginWidget::getState()
{
        SKGTRACEIN(10, "SKGRefundPluginWidget::getState");
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);

        root.setAttribute ( "hideUseless", ui.kHideUseless->checkState() ==Qt::Checked ? "Y" : "N" );
        root.setAttribute ( "filter", ui.kFilterEdit->text() );

        //Memorize table settings
        root.setAttribute("view", ui.kView->getState());
        return doc.toString();
}

void SKGRefundPluginWidget::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGRefundPluginWidget::setState");
        QDomDocument doc("SKGML");
        doc.setContent(iState);
        QDomElement root = doc.documentElement();

        QString hideUseless=root.attribute ( "hideUseless" );
        QString filter=root.attribute ( "filter" );

        if ( !hideUseless.isEmpty() ) ui.kHideUseless->setCheckState ( hideUseless=="Y" ? Qt::Checked : Qt::Unchecked );
        if ( !filter.isEmpty() ) ui.kFilterEdit->setText ( filter );

        onFilterChanged();

        //!!! Must be done here after onFilterChanged
        ui.kView->setState(root.attribute ( "view"));
}

QString SKGRefundPluginWidget::getDefaultStateAttibute()
{
        return "SKGREFUND_DEFAULT_PARAMETERS";
}

QWidget* SKGRefundPluginWidget::getWidgetForPrint()
{
        return ui.kView;
}

void SKGRefundPluginWidget::onEditorModified()
{
        _SKGTRACEIN(10, "SKGRefundPluginWidget::onEditorModified");
        int nb=getNbSelectedObjects();
        ui.kModifyButton->setEnabled(!ui.kNameInput->text().isEmpty() && nb==1);
        ui.kAddButton->setEnabled(!ui.kNameInput->text().isEmpty() && nb<=1);
}

void SKGRefundPluginWidget::onFilterRegExpChanged()
{
        SKGTRACEIN ( 10, "SKGRefundPluginWidget::onFilterRegExpChanged" );
        QRegExp regExp ( ui.kFilterEdit->text(), Qt::CaseInsensitive );
        ( ( SKGSortFilterProxyModel* ) ui.kView->model() )->setFilterRegExp ( regExp );
}

void SKGRefundPluginWidget::onFilterChanged()
{
        SKGTRACEIN ( 10, "SKGRefundPluginWidget::onFilterChanged" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

        //Compute where clause
        QString filter;

        if ( ui.kHideUseless->checkState() ==Qt::Checked ) {
                filter=" t_close='N'";
        }

        //Update model
        if ( objectModel ) {
                objectModel->setFilter ( filter );
                objectModel->refresh();

                //Correction bug 2299394 vvv
                if ( ui.kView->isAutoResized() ) ui.kView->resizeColumnsToContents();
                //Correction bug 2299394 ^^^
        }

        QApplication::restoreOverrideCursor();
}

void SKGRefundPluginWidget::onAddTracker()
{
        SKGError err;
        _SKGTRACEINRC(10, "SKGRefundPluginWidget::onAddTracker", err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        QString name=ui.kNameInput->text();
        SKGRefundTrackerObject tracker;
        {
                SKGBEGINTRANSACTION(*getDocument(), i18n("Tracker creation [%1]", name)  , err);

                err=SKGRefundTrackerObject::createRefund((const SKGDocumentBank*) getDocument(), name, tracker);
                if (err.isSucceeded()) err=tracker.setComment(ui.kCommentEdit->text());
                if (err.isSucceeded()) err=tracker.save();
        }
        //status bar
        if (err.isSucceeded()) {
                err=SKGError(0, i18n("Tracker [%1] created", name));
                ui.kView->selectObject ( tracker.getUniqueID() );
        } else err.addError(ERR_FAIL, i18n("Tracker creation failed"));
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage(err);
}

void SKGRefundPluginWidget::onModifyTracker()
{
        SKGError err;
        _SKGTRACEINRC(10, "SKGRefundPluginWidget::onModifyTracker", err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

        int nb=selection.count();
        if (nb==1) {
                SKGBEGINTRANSACTION(*getDocument(), i18n("Tracker [%1] update", ui.kNameInput->text())  , err);

                //Modification of category object
                SKGRefundTrackerObject tracker=selection[0];
                err=tracker.setName(ui.kNameInput->text());
                if (err.isSucceeded()) err=tracker.setComment(ui.kCommentEdit->text());
                if (err.isSucceeded()) err=tracker.save();

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Tracker [%1] updated", ui.kNameInput->text()));
                else err.addError(ERR_FAIL, i18n("Tracker update failed"));
        }
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage(err);
}

void SKGRefundPluginWidget::onDoubleClick()
{
        _SKGTRACEIN(10, "SKGRefundPluginWidget::onDoubleClick");

        //Get selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        if (selection.count()==1) {
                SKGRefundTrackerObject refund(selection.at(0));
                QString name=refund.getName();

                //Build where clause and title
                QString wc="t_REFUND = '"+SKGServices::stringToSqlString(name)+'\'';
                QString title=i18n("Operations followed by tracker '%1'",  name);

                //Open
                if (QApplication::keyboardModifiers()&Qt::ControlModifier && QApplication::keyboardModifiers()&Qt::ShiftModifier) {
                        //Call debug plugin
                        QDomDocument doc("SKGML");
                        QDomElement root = doc.createElement("parameters");
                        doc.appendChild(root);
                        root.setAttribute("sqlOrder", "SELECT * from v_operation_consolidated WHERE "+wc);

                        getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName("Skrooge debug plugin"), -1, doc.toString());
                } else {
                        //Call operation plugin
                        QDomDocument doc("SKGML");
                        doc.setContent ( getDocument()->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
                        QDomElement root = doc.documentElement();
                        if (root.isNull()) {
                                root=doc.createElement("parameters");
                                doc.appendChild(root);
                        }

                        root.setAttribute("operationTable", "v_operation_consolidated");
                        root.setAttribute("operationWhereClause", wc);
                        root.setAttribute("title", title);
                        root.setAttribute("title_icon", "view-statistics");
                        root.setAttribute ( "currentPage", "-1" );

                        getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
                }
        }
}
#include "skgrefundpluginwidget.moc"


