/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to manage scheduled operations
 *
 * @author Stephane MANKOWSKI
 */
#include "skgscheduledplugin.h"
#include "skgscheduledpluginwidget.h"
#include "skgscheduledboardwidget.h"
#include "skrooge_scheduled.h"
#include "skgrecurrentoperationobject.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"
#include "skgoperationobject.h"
#include "skgdocumentbank.h"

#include <kactioncollection.h>
#include <kstandardaction.h>

K_PLUGIN_FACTORY ( SKGScheduledPluginFactory, registerPlugin<SKGScheduledPlugin>(); )
K_EXPORT_PLUGIN ( SKGScheduledPluginFactory ( "skrooge_scheduled", "skrooge_scheduled" ) )

SKGScheduledPlugin::SKGScheduledPlugin ( QObject* iParent, const QVariantList& /*iArg*/ ) : SKGInterfacePlugin ( iParent )
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::SKGScheduledPlugin" );
}

SKGScheduledPlugin::~SKGScheduledPlugin()
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::~SKGScheduledPlugin" );
        parent=NULL;
        currentBankDocument=NULL;
        scheduleOperationAction=NULL;
}

void SKGScheduledPlugin::setupActions ( SKGMainPanel* iParent, SKGDocument* iDocument, const QStringList& iArgument )
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::setupActions" );
        Q_UNUSED ( iArgument );

        currentBankDocument=iDocument;
        parent=iParent;

        setComponentData ( SKGScheduledPluginFactory::componentData() );
        setXMLFile ( "skrooge_scheduled.rc" );

        //Create yours actions here
        scheduleOperationAction = new KAction ( KIcon ( "skrooge_schedule" ), i18n ( "&Schedule" ), this );
        connect ( scheduleOperationAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionScheduleOperation() ) );
        actionCollection()->addAction ( QLatin1String ( "schedule_operation" ), scheduleOperationAction );
        scheduleOperationAction->setShortcut ( Qt::CTRL+Qt::Key_I );

        if ( parent ) parent->registedGlobalAction ( "schedule_operation", scheduleOperationAction );
}

void SKGScheduledPlugin::refresh()
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::refresh" );
        if ( parent ) {
                SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();

                if ( selection.count() >0 ) {
                        bool onOperation= ( selection.at ( 0 ).getRealTable() =="operation" );
                        if ( scheduleOperationAction ) scheduleOperationAction->setEnabled ( onOperation );
                } else {
                        if ( scheduleOperationAction ) scheduleOperationAction->setEnabled ( false );
                }

                //Automatic insert
                if ( currentBankDocument && currentBankDocument->getDatabase() !=NULL ) {
                        QString doc_id=currentBankDocument->getUniqueIdentifier();
                        if ( docUniqueIdentifier!=doc_id && currentBankDocument->getParameter ( "SKG_DB_BANK_VERSION" ) >="0.5" ) {
                                docUniqueIdentifier=doc_id;

                                SKGError err;
                                //Read Setting
                                KSharedConfigPtr config=KSharedConfig::openConfig ( "skrooge_scheduledrc" );
                                KConfigGroup pref=config->group ( "skrooge_scheduled" );
                                bool check_on_open = pref.readEntry ( "check_on_open",true );

                                if ( check_on_open ) {
                                        SKGBEGINTRANSACTION ( *currentBankDocument, i18n ( "Insert recurrent operations" ), err );

                                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                                        int nbi=0;
                                        err=SKGRecurrentOperationObject::process ( currentBankDocument, nbi );
                                        QApplication::restoreOverrideCursor();
                                }
                                //Display error
                                if ( parent ) parent->displayErrorMessage ( err );
                        }
                }
        }
}

void SKGScheduledPlugin::close()
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::close" );
}

SKGWidget* SKGScheduledPlugin::getDashboardWidget()
{
        return new SKGScheduledBoardWidget(currentBankDocument);
}

SKGTabWidget* SKGScheduledPlugin::getWidget()
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::getWidget" );
        return new SKGScheduledPluginWidget ( parent, ( SKGDocumentBank* ) currentBankDocument );
}

QWidget* SKGScheduledPlugin::getPreferenceWidget()
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::getPreferenceWidget" );
        QWidget* widget=new QWidget();
        ui.setupUi ( widget );
        return widget;
}

KConfigSkeleton* SKGScheduledPlugin::getPreferenceSkeleton()
{
        return skrooge_scheduled::self();
}

SKGError SKGScheduledPlugin::savePreferences() const
{
        return SKGError();
}

QString SKGScheduledPlugin::title() const
{
        return i18n ( "Scheduled" ); //MUST BE CHANGED
}

QString SKGScheduledPlugin::icon() const
{
        return "chronometer";
}

QString SKGScheduledPlugin::statusTip () const
{
        return i18n ( "Manage scheduled operations" );//MUST BE CHANGED
}

QString SKGScheduledPlugin::toolTip () const
{
        return i18n ( "Manage scheduled operations" );//MUST BE CHANGED
}


int SKGScheduledPlugin::getOrder() const
{
        return 20;
}

QStringList SKGScheduledPlugin::tips() const
{
        QStringList output;
        output.push_back ( i18n ( "<p>... you can schedule operations.</p>" ) );
        return output;
}

bool SKGScheduledPlugin::isInContext() const
{
        return true;
}

bool SKGScheduledPlugin::isEnabled() const
{
        return true;
}

void SKGScheduledPlugin::actionScheduleOperation()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGScheduledPlugin::actionScheduleOperation",err );

        //Get Selection
        if ( parent ) {
                SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
                int nb=selection.count();
                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                if ( nb && currentBankDocument ) {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Operation schedule" ), err, nb );

                        KSharedConfigPtr config=KSharedConfig::openConfig("skrooge_scheduledrc");
                        KConfigGroup pref=config->group("skrooge_scheduled");

                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                SKGRecurrentOperationObject recOp;
                                err=operationObj.addRecurrentOperation ( recOp );
                                if ( err.isSucceeded() ) err=recOp.warnEnabled(pref.readEntry("remind_me",true));
                                if ( err.isSucceeded() ) err=recOp.setWarnDays(pref.readEntry("remind_me_days",5));
                                if ( err.isSucceeded() ) err=recOp.autoWriteEnabled(pref.readEntry("auto_write",true));
                                if ( err.isSucceeded() ) err=recOp.setAutoWriteDays(pref.readEntry("auto_write_days",0));
                                if ( err.isSucceeded() ) err=recOp.setPeriodIncrement(pref.readEntry("once_every",1));
                                if ( err.isSucceeded() ) err=recOp.setPeriodUnit((SKGRecurrentOperationObject::PeriodUnit) pref.readEntry("once_every_unit",1));
                                if ( err.isSucceeded() ) err=recOp.save();
                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
                        }
                }
                QApplication::restoreOverrideCursor();

                //status bar
                if ( err.isSucceeded() )  err=SKGError ( 0, i18n ( "Operation scheduled." ) );
                else err.addError ( ERR_FAIL, i18n ( "Operation schedule failed" ) );

                //Display error
                parent->displayErrorMessage ( err );
        }
}
#include "skgscheduledplugin.moc"
