/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGDATEEDIT_H
#define SKGDATEEDIT_H
/** @file
 * A date edit with more features.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgbasegui_export.h"
#include "skgcombobox.h"

#include <QtDesigner/QDesignerExportWidget>
#include <QDateTime>
#include <QEvent>
#include <QMap>
#include <QMouseEvent>

class QEvent;
class SKGDatePickerPopup;

/**
 * This file is a tab widget used by plugins
 * based on KDateEdit of PIM
 */
class SKGBASEGUI_EXPORT SKGDateEdit : public SKGComboBox
{
        Q_OBJECT
        /**
         * Date of the editor
         */
        Q_PROPERTY( QDate date READ date WRITE setDate USER true)
        /**
         * Mode of the editor
         */
        Q_PROPERTY( SKGDateEdit::Mode mode READ mode WRITE setMode)
        /**
         * Mode of the editor
         */
        Q_ENUMS(Mode);
public:
        /**
         * Mode of the editor
         */
        enum Mode {PREVIOUS,   /**< if date is incompleted, the previous one is selected */
                   CURRENT,    /**< if date is incompleted, the current month is selected */
                   NEXT        /**< if date is incompleted, the next one is selected */
                  };

        /**
         * Constructor
         * @param parent the parent
         * @param name name
         */
        explicit SKGDateEdit( QWidget *parent = 0, const char *name = 0 );

        /**
         * Destructor
         */
        virtual ~SKGDateEdit();

        /**
         * Get date
         * @return The date entered
         */
        QDate date();

        /**
         * Get the mode
         * @return the mode
         */
        virtual Mode mode() const;

        /**
         * Set the mode
         * @param iMode the mode
         */
        virtual void setMode (Mode iMode);

        /**
         * Sets whether the widget is read-only for the user. If read-only, the
         * date pop-up is inactive, and the displayed date cannot be edited.
         *
         * @param readOnly True to set the widget read-only, false to set it read-write.
         */
        virtual void setReadOnly( bool readOnly );

        /**
         * @return True if the widget is read-only, false if read-write.
         */
        virtual bool isReadOnly() const;
Q_SIGNALS:
        /**
         * This signal is emitted whenever the user has entered a new date.
         * When the user changes the date by editing the line edit field,
         * the signal is not emitted until focus leaves the line edit field.
         * The passed date can be invalid.
         */
        void dateEntered( const QDate &date );

        /**
         * This signal is emitted whenever the user modifies the date.
         * The passed date can be invalid.
         */
        void dateChanged( const QDate &date );

public Q_SLOTS:
        /**
         * Sets the date.
         * @param date The new date to display. This date must be valid or
         *             it will not be set
         */
        virtual void setDate( const QDate &date );

        /**
         * Display the calendar
         */
        virtual void showPopup();

protected:
        /**
         * Event filter
         * @param  object the object
         * @param  event the event
         * @return true or false
         */
        virtual bool eventFilter( QObject *object, QEvent *event );

        /**
         * Event when focus is out
         * @param  event the event
         */
        virtual void focusOutEvent( QFocusEvent *event );

        /**
         * Event when a key is pressed
         * @param  event the event
         */
        virtual void keyPressEvent( QKeyEvent* event );

private Q_SLOTS:
        void lineEnterPressed();
        void dateSelected( const QDate & );

private:
        virtual bool assignDate( const QDate &date );
        QDate parseDate( bool *replaced = 0 );
        void updateView();

        SKGDatePickerPopup *mPopup;

        QDate mDate;
        bool mReadOnly;
        bool mTextChanged;
        bool mDiscardNextMousePress;

        QMap<QString, int> mKeywordMap;
        Mode mMode;

        QColor font_color;
};

#endif
