/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for import and export operation.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportexportplugin.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgbankincludes.h"
#include "skgmainpanel.h"
#include "skgoperationobject.h"
#include "skgimportexport_settings.h"

#include <kactioncollection.h>
#include <kaction.h>
#include <kactionmenu.h>
#include <kfiledialog.h>
#include <kencodingfiledialog.h>

#include <QTextCodec>
#include <QDomDocument>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGImportExportPluginFactory, registerPlugin<SKGImportExportPlugin>();)
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN(SKGImportExportPluginFactory("skrooge_importexport", "skrooge_importexport"))

SKGImportExportPlugin::SKGImportExportPlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
        SKGTRACEIN(10, "SKGImportExportPlugin::SKGImportExportPlugin");
}

SKGImportExportPlugin::~SKGImportExportPlugin()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::~SKGImportExportPlugin");
        currentBankDocument=NULL;

        importAction=NULL;
        importCsvUnitAction=NULL;
        exportCsvAction=NULL;
        exportQifAction=NULL;
        processingFoundTransfert=NULL;
        processingBankPerfect=NULL;
        processingBank=NULL;
        validateImportedOperationsAction=NULL;
        openNotValidatedAction=NULL;
        mergeImportedOperationAction=NULL;
}

void SKGImportExportPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGImportExportPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;

        // Tell the host application to load my GUI component
        setComponentData( SKGImportExportPluginFactory::componentData() );
        setXMLFile("skrooge_importexport.rc");

        //Imports
        KActionMenu* imports=new  KActionMenu(KIcon("document-import"),i18nc("Verb, action to import items from another format","Import"), this);
        actionCollection()->addAction( QLatin1String("import"), imports );

        //Import
        QStringList overlay;
        overlay.push_back("skrooge");

        importAction = new KAction(KIcon("document-import", NULL, overlay), i18nc("Verb, action to import items from another format","Import..."), this);
        importAction->setShortcut ( Qt::CTRL+Qt::ALT+Qt::Key_I );
        connect(importAction, SIGNAL(triggered(bool)), SLOT(import()));
        actionCollection()->addAction( QLatin1String("import_operation"), importAction );
        imports->addAction(importAction);
        importAction->setShortcut ( Qt::CTRL+Qt::META+Qt::Key_I );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("import_operation", importAction);

        //Import CSV Unit
        QStringList overlaycsv;
        overlaycsv.push_back("text-csv");
        importCsvUnitAction = new KAction(KIcon("document-import", NULL, overlaycsv), i18nc("Verb, action to import","Import CSV Unit..."), this);
        connect(importCsvUnitAction, SIGNAL(triggered(bool)), SLOT(import()));
        actionCollection()->addAction( QLatin1String("import_csv_unit"), importCsvUnitAction );
        imports->addAction(importCsvUnitAction);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("import_csv_unit", importCsvUnitAction);

        //Exports
        KActionMenu* exports=new  KActionMenu(KIcon("document-export"),i18nc("Verb, action to export items in another format","Export"), this);
        actionCollection()->addAction( QLatin1String("export"), exports );

        //Export QIF
        exportQifAction = new KAction(KIcon("document-export"), i18nc("Verb, action to export items to another format","Export QIF..."), this);
        connect(exportQifAction, SIGNAL(triggered(bool)), SLOT(exportQif()));
        actionCollection()->addAction( QLatin1String("export_qif"), exportQifAction );
        exports->addAction(exportQifAction);
        exportQifAction->setShortcut ( Qt::CTRL+Qt::META+Qt::Key_E );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("export_qif", exportQifAction);

        //Export CSV
        exportCsvAction = new KAction(KIcon("document-export", NULL, overlaycsv), i18nc("Verb, action to export items to another format","Export CSV..."), this);
        connect(exportCsvAction, SIGNAL(triggered(bool)), SLOT(exportCsv()));
        actionCollection()->addAction( QLatin1String("export_csv"), exportCsvAction );
        exports->addAction(exportCsvAction);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("export_csv", exportCsvAction);

        //Processing
        KActionMenu* processing=new  KActionMenu(KIcon("tools-wizard"),i18nc("Noun, apply some kind of transformation on an item","Processing"), this);
        actionCollection()->addAction( QLatin1String("processing"), processing );

        //Processing found and group
        QStringList overlaytransfers;
        overlaytransfers.push_back("skrooge_transfer");

        processingFoundTransfert = new KAction(KIcon("tools-wizard", NULL, overlaytransfers), i18nc("Verb, action to find and group transfers","Find and group transfers"), this);
        connect(processingFoundTransfert, SIGNAL(triggered(bool)), SLOT(foundTransfer()));
        actionCollection()->addAction( QLatin1String("process_foundtransfer"), processingFoundTransfert );
        processing->addAction(processingFoundTransfert);
        processingFoundTransfert->setShortcut ( Qt::CTRL+Qt::META+Qt::Key_G );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("process_foundtransfer", processingFoundTransfert);

        //Processing bankperfect
        processingBankPerfect = new KAction(KIcon("tools-wizard"), i18nc("Verb, action to clean an import","Clean BankPerfect's imports"), this);
        connect(processingBankPerfect, SIGNAL(triggered(bool)), SLOT(cleanBankPerfect()));
        actionCollection()->addAction( QLatin1String("process_bankperfect"), processingBankPerfect );
        processing->addAction(processingBankPerfect);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("process_bankperfect", processingBankPerfect);

        //Processing banks
        processingBank = new KAction(KIcon("tools-wizard"), i18nc("Verb, action to clean an import","Clean bank's imports"), this);
        connect(processingBank, SIGNAL(triggered(bool)), SLOT(cleanBanks()));
        actionCollection()->addAction( QLatin1String("process_banks"), processingBank );
        processing->addAction(processingBank);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("process_banks", processingBank);

        //Processing banks
        QStringList overlayValidate;
        overlayValidate.push_back("dialog-ok-apply");
        validateImportedOperationsAction = new KAction(KIcon("document-import", NULL, overlayValidate), i18nc("Verb, action to validate imported operations","Validate imported operations"), this);
        connect(validateImportedOperationsAction, SIGNAL(triggered(bool)), SLOT(validateImportedOperations()));
        actionCollection()->addAction( QLatin1String("validate_imported_operation"), validateImportedOperationsAction );
        validateImportedOperationsAction->setShortcut ( Qt::CTRL+Qt::SHIFT+Qt::Key_V );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("validate_imported_operation", validateImportedOperationsAction);

        QStringList overlayopen;
        overlayopen.push_back("skg_open");
        openNotValidatedAction = new KAction ( KIcon("document-import", NULL, overlayopen), i18nc("Verb, action to open","Open imported operations not yet validated..." ), this);
        connect ( openNotValidatedAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( openNotValidated() ) );
        actionCollection()->addAction ( QLatin1String ( "view_open_not_validated" ), openNotValidatedAction );
        openNotValidatedAction->setShortcut ( Qt::META+Qt::Key_V );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "view_open_not_validated", openNotValidatedAction );

        mergeImportedOperationAction = new KAction(KIcon("skrooge_merge_import"), i18nc("Verb, action to merge","Merge imported operations"), this);
        connect(mergeImportedOperationAction, SIGNAL(triggered(bool)), SLOT(mergeImportedOperation()));
        actionCollection()->addAction( QLatin1String("merge_imported_operation"), mergeImportedOperationAction );
        mergeImportedOperationAction->setShortcut ( Qt::CTRL+Qt::Key_M );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("merge_imported_operation", mergeImportedOperationAction);
}

void SKGImportExportPlugin::processArguments(const QStringList& iArgument)
{
        int nbArg=iArgument.count();
        if (nbArg) {
                QString filename=iArgument.at(nbArg-1);
                QString extension=QFileInfo(filename).suffix().toUpper();
                if (QFile(filename).exists() && (extension=="QIF" || extension=="CSV" || extension=="OFX" || extension=="QFX")) {
                        import(filename);
                }
        }
}

QWidget* SKGImportExportPlugin::getPreferenceWidget()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::getPreferenceWidget");
        QWidget* widget=new QWidget();
        ui.setupUi(widget);

        ui.kHeaderPositionFrm->hide();
        ui.kColumnsPositionsFrm->hide();

        return widget;
}

KConfigSkeleton* SKGImportExportPlugin::getPreferenceSkeleton()
{
        return skgimportexport_settings::self();
}

QString SKGImportExportPlugin::title() const
{
        return i18nc("Noun","Import / Export");
}

QString SKGImportExportPlugin::icon() const
{
        return "utilities-file-archiver";
}

QString SKGImportExportPlugin::toolTip () const
{
        return i18nc("Noun","Import / Export management");
}

QStringList SKGImportExportPlugin::tips() const
{
        QStringList output;
        output.push_back(i18nc("Description of a tips", "<p>... skrooge is able to detect automatically transfers after an import.</p>"));
        output.push_back(i18nc("Description of a tips", "<p>... you can import many files in one shot.</p>"));
        output.push_back(i18nc("Description of a tips", "<p>... unit amounts can be imported through a CSV file.</p>"));
        return output;
}

int SKGImportExportPlugin::getOrder() const
{
        return 70;
}

bool SKGImportExportPlugin::isInContext() const
{
        return false;
}

void SKGImportExportPlugin::refresh()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::refresh");

        if (currentBankDocument && SKGMainPanel::getMainPanel()) {
                bool test=(currentBankDocument->getDatabase()!=NULL);
                if (openNotValidatedAction) openNotValidatedAction->setEnabled (test);
                if (importAction) importAction->setEnabled(test);
                if (exportQifAction) exportQifAction->setEnabled(test);
                if (exportCsvAction) exportCsvAction->setEnabled(test);
                if (importCsvUnitAction) importCsvUnitAction->setEnabled(test);
                if (processingFoundTransfert) processingFoundTransfert->setEnabled(test);
                if (processingBankPerfect) processingBankPerfect->setEnabled(test);
                if (processingBank) processingBank->setEnabled(test);

                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                if ( selection.count() >0 && test) {
                        bool onOperation= ( selection.at ( 0 ).getRealTable() =="operation" &&  selection.at ( 0 ).getTable()!="v_operation_consolidated");
                        if ( validateImportedOperationsAction ) validateImportedOperationsAction->setEnabled ( onOperation );
                        if ( mergeImportedOperationAction ) mergeImportedOperationAction->setEnabled ( onOperation );
                } else {
                        if ( validateImportedOperationsAction ) validateImportedOperationsAction->setEnabled ( false );
                        if ( mergeImportedOperationAction ) mergeImportedOperationAction->setEnabled ( false );
                }
        }
}

void SKGImportExportPlugin::close()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::close");
}

void SKGImportExportPlugin::import(const QString& iFile)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::import",err);
        if (currentBankDocument) {
                QSet<QString> fileNames;
                QString lastCodecUsed=currentBankDocument->getParameter("SKG_LAST_CODEC_USED_FOR_IMPORT");
                if (lastCodecUsed.length()==0) lastCodecUsed=QTextCodec::codecForLocale()->name();
                QString codec;

                if (iFile.isEmpty()) {
                        //Panel to ask files
                        KEncodingFileDialog::Result result=KEncodingFileDialog::getOpenFileNamesAndEncoding(lastCodecUsed, "kfiledialog:///IMPEXP",
                                                           sender()==importCsvUnitAction ? "*.csv|"+i18nc("A file format", "CSV Files") :
                                                           QString(SKGImportExportManager::isOFXSupported() ? "*.ofx *.qfx ": "")+"*.qif *.csv|"+i18nc("A file format", "All supported formats")+'\n'+
                                                           (SKGImportExportManager::isOFXSupported() ? "*.ofx *.qfx|"+i18nc("A file format", "OFX Files")+'\n': "")+
                                                           "*.qif|"+i18nc("A file format", "QIF Files")+'\n'+
                                                           "*.csv|"+i18nc("A file format", "CSV Files") ,
                                                           SKGMainPanel::getMainPanel());
                        fileNames=result.fileNames.toSet();
                        codec=result.encoding;
                } else {
                        fileNames.insert(iFile);
                        QString codec=lastCodecUsed;
                }

                int nbFiles=fileNames.count();
                if (nbFiles) {

                        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                        {
                                SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Import with codec %1", codec), err, nbFiles);

                                //Read Setting
                                bool automatic_validation = skgimportexport_settings::automatic_validation();
                                bool automatic_rule = skgimportexport_settings::apply_rules();

                                if (err.isSucceeded()) err=currentBankDocument->setParameter("SKG_LAST_CODEC_USED_FOR_IMPORT", codec);

                                QSetIterator<QString> f(fileNames);
                                int i=1;
                                while (err.isSucceeded() && f.hasNext()) {
                                        //Get Filename
                                        QString fileName=f.next();

                                        //Compute mode
                                        SKGImportExportManager::ImportExportMode mode=SKGImportExportManager::CSV;
                                        if (sender()==importCsvUnitAction) mode=SKGImportExportManager::CSVUNIT;
                                        QString extension=QFileInfo(fileName).suffix().toUpper();
                                        if (extension=="QIF") mode=SKGImportExportManager::QIF;
                                        else if (extension=="OFX" || extension=="QFX") mode=SKGImportExportManager::OFX;

                                        //Import
                                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, mode, fileName);
                                        imp1.setAutomaticValidation(automatic_validation);
					imp1.setAutomaticApplyRules(automatic_rule);
                                        imp1.setCodec(codec);

                                        if (mode==SKGImportExportManager::CSV || mode==SKGImportExportManager::CSVUNIT) {
                                                //Set header line
                                                bool automatic_search_header = skgimportexport_settings::automatic_search_header();
                                                if (!automatic_search_header) {
                                                        int header_position = skgimportexport_settings::header_position();
                                                        if (err.isSucceeded()) err=imp1.setCSVHeaderIndex(header_position-1);

                                                }

                                                bool automatic_search_columns = skgimportexport_settings::automatic_search_columns();
                                                if (!automatic_search_columns) {
                                                        QStringList columns_positions = skgimportexport_settings::columns_positions().split('|');
                                                        if (err.isSucceeded()) err=imp1.setCSVMapping(&columns_positions);
                                                }
                                        }

                                        if (err.isSucceeded()) err=imp1.importFile();
                                        if (err.isFailed()) {
                                                err.addError(ERR_FAIL, i18nc("Error message", "Import of %1 file named [%2] failed", extension, fileName));
                                        }

                                        if (err.isSucceeded()) err=currentBankDocument->stepForward(i);
                                        i++;
                                }
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if (err.isSucceeded()) {
                        err=SKGError(0, i18np("%1 file successfully imported.", "%1 files successfully imported.", nbFiles));
                }

                //Display error
                if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);

                //Open last modified operations if setting activated
                if (err.isSucceeded()) openLastModifiedIfSetting();
        }
}

void SKGImportExportPlugin::exportCsv()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::exportCsv",err);
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "*.csv|"+i18nc("A file format", "CSV Files") , SKGMainPanel::getMainPanel());
        if (fileName.isEmpty() || !currentBankDocument) return;

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                SKGBEGINTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Export CSV"), err);
                if (err.isSucceeded()) {
                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::CSV, fileName);
                        err=imp1.exportFile();
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18nc("Successful message after an user action", "File [%1] successfully exported.", fileName));
        else err.addError(ERR_FAIL, i18nc("Error message", "Export of [%1] failed", fileName));

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGImportExportPlugin::exportQif()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::exportQif",err);
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "*.qif|"+i18nc("A file format", "QIF Files") , SKGMainPanel::getMainPanel());
        if (fileName.isEmpty() || !currentBankDocument) return;

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                SKGBEGINTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Export QIF"), err);
                if (err.isSucceeded()) {
                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, fileName);
                        err=imp1.exportFile();
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18nc("Successful message after an user action", "File [%1] successfully exported.", fileName));
        else err.addError(ERR_FAIL, i18nc("Error message", "Export of [%1] failed", fileName));

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGImportExportPlugin::foundTransfer()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::foundTransfer",err);
        if (currentBankDocument) {
                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                int NbOperationsMerged=0;
                {
                        SKGBEGINTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Find and group transfers"), err);
                        if (err.isSucceeded()) {
                                SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, "");
                                err=imp1.findAndGroupTransfers(NbOperationsMerged);
                        }
                }
                QApplication::restoreOverrideCursor();

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18np("Document successfully processed. %1 operation merged.",
                                                            "Document successfully processed. %1 operations merged.", NbOperationsMerged));
                else err.addError(ERR_FAIL, i18nc("Error message", "Processing failed."));

                //Display error
                if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);

                //Open last modified operations if setting activated
                if (err.isSucceeded()) openLastModifiedIfSetting();
        }
}

void SKGImportExportPlugin::cleanBankPerfect()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::cleanBankPerfect",err);
        if (currentBankDocument) {
                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                {
                        SKGBEGINTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Clean BankPerfect's imports"), err);
                        if (err.isSucceeded()) {
                                SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, "");
                                err=imp1.cleanBankPerfectImport();
                        }
                }
                QApplication::restoreOverrideCursor();

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18nc("Successful message after an user action", "Document successfully cleaned."));
                else err.addError(ERR_FAIL, i18nc("Error message", "Clean failed."));

                //Display error
                if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);

                //Open last modified operations if setting activated
                if (err.isSucceeded()) openLastModifiedIfSetting();
        }
}

void SKGImportExportPlugin::cleanBanks()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::cleanBanks",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                SKGBEGINTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Clean bank's imports"), err);
                if (err.isSucceeded()) {
                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, "");
                        err=imp1.cleanBankImport();
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18nc("Successful message after an user action", "Document successfully cleaned."));
        else err.addError(ERR_FAIL, i18nc("Error message", "Clean failed."));

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);

        //Open last modified operations if setting activated
        if (err.isSucceeded()) openLastModifiedIfSetting();
}

void SKGImportExportPlugin::validateImportedOperations()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::validateImportedOperations",err);

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        if (SKGMainPanel::getMainPanel() && currentBankDocument) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Validate imported operations"), err, nb);
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGOperationObject op=selection[i];
                        if (op.getAttribute("t_imported")=="P") {
                                err=op.setImported(true);
                                if (err.isSucceeded()) err=op.save();
                        }
                        if (err.isSucceeded()) err=currentBankDocument->stepForward(i+1);
                }

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18nc("Successful message after an user action", "Imported operations validated."));
                else err.addError(ERR_FAIL, i18nc("Error message", "Validation failed"));
        }
        QApplication::restoreOverrideCursor();

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGImportExportPlugin::mergeImportedOperation()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::mergeImportedOperation",err);

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        if (SKGMainPanel::getMainPanel() && currentBankDocument) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                err=SKGError(ERR_INVALIDARG, i18nc("Error message", "Invalid selection, you must select one imported operation and one manual operation with same amounts"));
                if (nb==2) {
                        SKGOperationObject opImported=selection.at(0);
                        SKGOperationObject opManual=selection.at(1);
                        if (opImported.isImported()!=opManual.isImported() && opImported.getCurrentAmount()==opManual.getCurrentAmount()) {
                                SKGBEGINTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Merge imported operations"), err);
                                if (!opImported.isImported())  qSwap(opImported, opManual);
                                err=opManual.merge(opImported);
                                if (err.isFailed()) err.addError(ERR_FAIL, i18nc("Error message", "Merge failed"));
                        }
                }

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18nc("Successful message after an user action", "Imported operations merged."));
        }
        QApplication::restoreOverrideCursor();

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGImportExportPlugin::openLastModifiedIfSetting()
{
        //Read Setting
        bool open_after_import_or_processing = skgimportexport_settings::open_after_import_or_processing();
        if (open_after_import_or_processing) {
                //Open last operations
                KAction* action=SKGMainPanel::getMainPanel()->getGlobalAction ("view_open_last_modified");
                if (action) {
                        action->trigger();
                }
        }
}

void SKGImportExportPlugin::openNotValidated()
{
        SKGTRACEIN ( 10, "SKGImportExportPlugin::openNotValidated");
        QString wc="t_imported='P'";
        QString title=i18nc("Noun, a list of items", "Operations imported and not yet validated" );

        //Call operation plugin
        QDomDocument doc("SKGML");
        if(currentBankDocument) doc.setContent ( currentBankDocument->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
        QDomElement root = doc.documentElement();
        if (root.isNull()) {
                root=doc.createElement("parameters");
                doc.appendChild(root);
        }

        root.setAttribute ( "account", i18nc("For all accounts", "All") );
        root.setAttribute ( "operationTable", "v_operation_display" );
        root.setAttribute ( "operationWhereClause", wc );
        root.setAttribute ( "title", title );
        root.setAttribute ( "title_icon", icon() );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
}

#include "skgimportexportplugin.moc"
