/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for unit management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgunitboardwidget.h"
#include "skgdocumentbank.h"
#include "skgtraces.h"
#include "skgunitobject.h"
#include "skgmainpanel.h"

#include <QDomDocument>

SKGUnitBoardWidget::SKGUnitBoardWidget ( SKGDocument* iDocument )
        : SKGWidget ( iDocument )
{
    SKGTRACEIN ( 10, "SKGUnitBoardWidget::SKGUnitBoardWidget" );

    ui.setupUi ( this );

    //Create menu
    setContextMenuPolicy ( Qt::ActionsContextMenu );

    menuIndexes=new KAction ( i18nc ("Noun, example: CAC40", "Indexes" ), this );
    menuIndexes->setCheckable ( true );
    menuIndexes->setChecked ( true );
    connect ( menuIndexes, SIGNAL ( triggered ( bool ) ), this, SLOT ( dataModified() ) );
    addAction ( menuIndexes );

    menuShares=new KAction ( i18nc ("Noun", "Shares" ), this );
    menuShares->setCheckable ( true );
    menuShares->setChecked ( true );
    connect ( menuShares, SIGNAL ( triggered ( bool ) ), this, SLOT ( dataModified() ) );
    addAction ( menuShares );

    menuSharesOwnedOnly=new KAction ( i18nc ("Noun","Shares owned only" ), this );
    menuSharesOwnedOnly->setCheckable ( true );
    menuSharesOwnedOnly->setChecked ( false );
    connect ( menuSharesOwnedOnly, SIGNAL ( triggered ( bool ) ), this, SLOT ( dataModified() ) );
    addAction ( menuSharesOwnedOnly );

    //Refresh
    connect ( ( const QObject* ) getDocument(), SIGNAL ( tableModified ( QString, int ) ), this, SLOT ( dataModified ( QString, int ) ), Qt::QueuedConnection );
}

SKGUnitBoardWidget::~SKGUnitBoardWidget()
{
    SKGTRACEIN ( 10, "SKGUnitBoardWidget::~SKGUnitBoardWidget" );
    menuIndexes=NULL;
    menuShares=NULL;
    menuSharesOwnedOnly=NULL;
}

QString SKGUnitBoardWidget::getState()
{
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    root.setAttribute ( "menuIndexes", menuIndexes && menuIndexes->isChecked() ? "Y" : "N" );
    root.setAttribute ( "menuShares", menuShares && menuShares->isChecked() ? "Y" : "N" );
    root.setAttribute ( "menuSharesOwnedOnly", menuSharesOwnedOnly && menuSharesOwnedOnly->isChecked() ? "Y" : "N" );
    return doc.toString();
}

void SKGUnitBoardWidget::setState ( const QString& iState )
{
    QDomDocument doc ( "SKGML" );
    if ( doc.setContent ( iState ) )
    {
        QDomElement root = doc.documentElement();

        if(menuIndexes) menuIndexes->setChecked ( root.attribute ( "menuIndexes" ) =="Y" );
        if(menuShares) menuShares->setChecked ( root.attribute ( "menuShares" ) =="Y" );
        if(menuSharesOwnedOnly) menuSharesOwnedOnly->setChecked ( root.attribute ( "menuSharesOwnedOnly" ) =="Y" );
    }

    dataModified ( "", 0 );
}

void SKGUnitBoardWidget::dataModified ( const QString& iTableName, int iIdTransaction )
{
    SKGTRACEIN ( 10, "SKGUnitBoardWidget::dataModified" );
    Q_UNUSED ( iIdTransaction );

    if ( iTableName=="unit" || iTableName.isEmpty() )
    {
        KLocale* locale=KGlobal::locale();
        SKGDocumentBank* doc= dynamic_cast<SKGDocumentBank*> ( getDocument() );
        if ( doc && locale )
        {
            SKGServices::SKGUnitInfo primary= doc->getPrimaryUnit();

	    //Build where clause
	    QString wc;
	    if(menuSharesOwnedOnly && menuShares && menuIndexes)
	    {
	      menuSharesOwnedOnly->setEnabled(menuShares->isChecked());
	      menuShares->setEnabled(!menuSharesOwnedOnly->isChecked());
	      if(!menuShares->isChecked()) menuSharesOwnedOnly->setChecked(false);

	      if ( menuIndexes->isChecked() )
	      {
		  wc="t_type='I'";
	      }
	      if ( menuShares->isChecked() )
	      {
		  if ( !wc.isEmpty() ) wc+=" OR ";
		  if ( menuSharesOwnedOnly->isChecked() ) wc+="(t_type='S' AND f_QUANTITYOWNED>0)";
		  else wc+="t_type='S'";
	      }
	    }

	    if ( wc.isEmpty() ) wc="1=0";

            QString html;
            SKGObjectBase::SKGListSKGObjectBase objs;
            SKGError err=SKGObjectBase::getObjects ( getDocument(), "v_unit_display", wc+" ORDER BY t_type DESC, t_name ASC", objs );
            if ( err.isSucceeded() )
            {
                html+="<html><body>";
                int nb=objs.count();
                if ( nb )
                {
                    html+="<table class=\"table\">";
                    QString lastTitle;
                    for ( int i=0; i<nb; ++i )
                    {
                        SKGUnitObject obj=objs.at ( i );
                        QString type=obj.getAttribute ( "t_TYPENLS" );
                        if ( lastTitle!=type )
                        {
                            lastTitle=type;
                            html+="<tr class=\"tabletotal\"><td><b>"+SKGServices::stringToHtml ( lastTitle ) +"</b></td></tr>";
                        }
                        html+=QString ( "<tr><td><a href=\""+SKGServices::stringToHtml ( obj.getSymbol() ) +"\">" ) +SKGServices::stringToHtml ( obj.getDisplayName() ) +"</a></td><td align=\"right\">";
                        QString unitSymbol=primary.Name;
                        if ( obj.getType() ==SKGUnitObject::INDEX ) unitSymbol=' ';
                        html+=SKGServices::stringToHtml ( locale->formatMoney ( SKGServices::stringToDouble ( obj.getAttribute ( "f_CURRENTAMOUNT" ) ),unitSymbol , primary.NbDecimal ) );
                        html+="</td><td>(";
                        double val=obj.getDailyChange();
                        html+="<font color=\"";
                        html+= ( val<0 ? "red" : "black" );
                        html+="\">";
                        html+=SKGServices::stringToHtml ( locale->formatNumber ( val ) );
                        html+="</font> %)</td></tr>";
                    }
                    html+="</table>";
                }
                else
                {
                    html+=i18nc ("Message", "No share or index defined<br>on the <a href=\"___\">\"Units\"</a> page." );
                }
                html+="</body></html>";
                ui.kLabel->setText ( html );
            }

	    //No widget if no account
	    int nbAccount=0;
	    SKGObjectBase::getNbObjects(doc, "account", "", nbAccount);
	    if(parentWidget()) setVisible(nbAccount>0);
        }
    }
}

void SKGUnitBoardWidget::onOpen ( const QString& iLink )
{
    QDomDocument doc ( "SKGML" );
    if ( iLink!="___" )
    {
        SKGUnitObject unit ( ( SKGDocumentBank* ) getDocument() );
        unit.setSymbol ( iLink );
        unit.load();
        int unitid=unit.getID();

        //Build where clause and title
        QString wc="rc_unit_id="+SKGServices::intToString ( unitid );
        QString title=i18nc("Noun, a list of items", "Operations with unit equal to '%1'",  unit.getName() );

        //Open
        //Call operation plugin
        doc.setContent ( getDocument()->getParameter ( "SKGOPERATION_DEFAULT_PARAMETERS" ) );
        QDomElement root = doc.documentElement();
        if ( root.isNull() )
        {
            root=doc.createElement ( "parameters" );
            doc.appendChild ( root );
        }

        root.setAttribute ( "operationTable", "v_operation_display" );
        root.setAttribute ( "operationWhereClause", wc );
        root.setAttribute ( "title", title );
        root.setAttribute ( "title_icon", "skrooge_unit" );
        root.setAttribute ( "currentPage", "-1" );

        SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
    }
    else
    {
        doc.setContent ( getDocument()->getParameter ( "SKGUNIT_DEFAULT_PARAMETERS" ) );
        SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge unit plugin" ), -1, doc.toString() );
    }
}
#include "skgunitboardwidget.moc"
