/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A dashboard.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgdashboardpluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgdocument.h"
#include "skginterfaceplugin.h"
#include "skgservices.h"
#include <kcmdlineargs.h>
#include <kaboutdata.h>

#include <kmenu.h>

#include <QDomDocument>
#include <QSpacerItem>
#include <QMouseEvent>

#include <math.h>

SKGDashboardPluginWidget::SKGDashboardPluginWidget ( SKGDocument* iDocument )
        : SKGTabPage ( iDocument ), flowLayout ( NULL ), menu ( NULL ), addMenu ( NULL ), removeMenu ( NULL )
{
    SKGTRACEIN ( 1, "SKGDashboardPluginWidget::SKGDashboardPluginWidget" );

    ui.setupUi ( this );

    //Drag and drop
    clickedPoint=QPoint(-1, -1);

    const KAboutData* about=KCmdLineArgs::aboutData();
    if (about)
    {
        ui.kTitle->setPixmap ( KIcon ( about->programIconName() ).pixmap ( 22, 22 ), KTitleWidget::ImageLeft );
        ui.kTitle->setComment ( "<html><body><b>"+i18nc ( "Message", "Welcome to %1", about->programName()  ) +"</b></body></html>" );
    }

    //Build menu
    setContextMenuPolicy ( Qt::CustomContextMenu );
    menu = new KMenu ( this );
    connect ( this ,SIGNAL ( customContextMenuRequested ( const QPoint & ) ),this,SLOT ( showHeaderMenu ( const QPoint& ) ) );

    addMenu=menu->addMenu ( KIcon ( "list-add" ),  i18nc ( "Verb", "Add" ) );
    //Build menu
    if ( addMenu )
    {
        addMenu->clear();

        int index=0;
        while ( index>=0 )
        {
            SKGInterfacePlugin* plugin=SKGMainPanel::getMainPanel()->getPluginByIndex ( index );
            if ( plugin )
            {
                int nbdbw=plugin->getNbDashboardWidgets();
                for ( int j=0; j<nbdbw; ++j )
                {
                    //Create menu
                    QAction* act = addMenu->addAction ( plugin->getDashboardWidgetTitle ( j ) );
                    if (act)
                    {
                        act->setIcon ( KIcon ( plugin->icon() ) );
                        act->setData ( plugin->objectName() +'-'+SKGServices::intToString ( j ) );

                        connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onAddWidget() ) );
                    }
                }
            }
            else index=-2;
            ++index;
        }
    }

    removeMenu=menu->addMenu ( KIcon ( "list-remove" ),  i18nc ( "Verb", "Remove" ) );
    previousMenu=menu->addMenu ( KIcon ( "arrow-left" ),  i18nc ( "Verb", "Move before" ) );
    nextMenu=menu->addMenu ( KIcon ( "arrow-right" ),  i18nc ( "Verb", "Move after" ) );
    zoomInMenu=menu->addMenu ( KIcon ( "zoom-in" ),  i18nc ( "Verb", "Zoom in" ) );
    zoomOutMenu=menu->addMenu ( KIcon ( "zoom-out" ),  i18nc ( "Verb", "Zoom out" ) );

    //Plug buttons with menus
    ui.kAddWidget->setIcon ( addMenu->icon() );
    ui.kAddWidget->setMenu ( addMenu );
    ui.kAddWidget->setPopupMode ( QToolButton::InstantPopup );

    ui.kRemoveWidget->setIcon ( removeMenu->icon() );
    ui.kRemoveWidget->setMenu ( removeMenu );
    ui.kRemoveWidget->setPopupMode ( QToolButton::InstantPopup );

    ui.kPreviousWidget->setIcon ( previousMenu->icon() );
    ui.kPreviousWidget->setMenu ( previousMenu );
    ui.kPreviousWidget->setPopupMode ( QToolButton::InstantPopup );

    ui.kNextWidget->setIcon ( nextMenu->icon() );
    ui.kNextWidget->setMenu ( nextMenu );
    ui.kNextWidget->setPopupMode ( QToolButton::InstantPopup );

    ui.kZoomInWidget->setIcon ( zoomInMenu->icon() );
    ui.kZoomInWidget->setMenu ( zoomInMenu );
    ui.kZoomInWidget->setPopupMode ( QToolButton::InstantPopup );

    ui.kZoomOutWidget->setIcon ( zoomOutMenu->icon() );
    ui.kZoomOutWidget->setMenu ( zoomOutMenu );
    ui.kZoomOutWidget->setPopupMode ( QToolButton::InstantPopup );

    //Build layout
    flowLayout = new SKGFlowLayout ( ui.kContent );
    flowLayout->setSpacing(0);
}

SKGDashboardPluginWidget::~SKGDashboardPluginWidget()
{
    SKGTRACEIN ( 1, "SKGDashboardPluginWidget::~SKGDashboardPluginWidget" );
    menu=NULL;
    addMenu=NULL;
    removeMenu=NULL;
    nextMenu=NULL;
    previousMenu=NULL;
    flowLayout=NULL;
    zoomInMenu=NULL;
    zoomOutMenu=NULL;
}

QString SKGDashboardPluginWidget::getState()
{
    SKGTRACEIN ( 10, "SKGDashboardPluginWidget::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    root.setAttribute ( "zoomPosition", SKGServices::intToString(zoomPosition()) );

    int nb=items.count();
    for ( int i=0; i<nb; ++i )
    {
        QDomElement element = doc.createElement ( "ITEM-"+SKGServices::intToString ( i+1 ) );
        root.appendChild ( element );

        QStringList param=SKGServices::splitCSVLine ( items.at ( i ),'-' );
        SKGWidget* item=itemsPointers.at ( i );
        if ( item )
        {
            element.setAttribute ( "name", param.at ( 0 ) );
            element.setAttribute ( "index", param.at ( 1 ) );
            element.setAttribute ( "state", item->getState() );
            element.setAttribute ( "zoom", SKGServices::intToString(itemsSizes.at(i)) );
        }
    }

    return doc.toString();
}

void SKGDashboardPluginWidget::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGDashboardPluginWidget::setState" );

    //Initialisation
    int nb=items.count();
    for ( int i=0; flowLayout && i<nb; ++i )
    {
        SKGWidget* item=itemsPointers.at ( i );
        if ( item )
        {
            flowLayout->removeWidget ( item );
            item->hide();
            delete item;
        }
    }

    removeMenu->clear ();
    nextMenu->clear();
    previousMenu->clear();
    zoomInMenu->clear();
    zoomOutMenu->clear();

    items.clear();
    itemsPointers.clear();
    itemsSizes.clear();

    if ( iState.isEmpty() )
    {
        int index=0;
        while ( index>=0 )
        {
            SKGInterfacePlugin* plugin=SKGMainPanel::getMainPanel()->getPluginByIndex ( index );
            if ( plugin )
            {
                int nb=plugin->getNbDashboardWidgets();
                for ( int j=0; j<nb; ++j )
                {
                    addItem ( plugin, j );
                }
            }
            else index=-2;
            ++index;
        }
    }
    else
    {
        QDomDocument doc ( "SKGML" );
        doc.setContent ( iState );
        QDomElement root = doc.documentElement();

        QString zoomPositionS=root.attribute ( "zoomPosition" );
        if ( !zoomPositionS.isEmpty() ) setZoomPosition(SKGServices::stringToInt ( zoomPositionS ));

        int index=1;
        while ( index>0 )
        {
            QDomElement element=root.firstChildElement ( "ITEM-"+SKGServices::intToString ( index ) );
            if ( !element.isNull() )
            {
                SKGInterfacePlugin* plugin=SKGMainPanel::getMainPanel()->getPluginByName ( element.attribute ( "name" ) );
                QString index=element.attribute ( "index" );
                if ( index.isEmpty() ) index='0';
                QString zoom=element.attribute ( "zoom" );
                if ( zoom.isEmpty() ) zoom='0';
                if ( plugin ) addItem ( plugin, SKGServices::stringToInt ( index ), SKGServices::stringToInt ( zoom ), element.attribute ( "state" ) );
            }
            else index=-1;
            ++index;
        }
    }
}

QString SKGDashboardPluginWidget::getDefaultStateAttribute()
{
    return "SKGDASHBOARD_DEFAULT_PARAMETERS";
}

void SKGDashboardPluginWidget::refresh()
{
    SKGTRACEIN ( 1, "SKGDashboardPluginWidget::refresh" );

}

bool SKGDashboardPluginWidget::eventFilter ( QObject* object, QEvent* event )
{
    if ( event && object &&
            (event->type() == QEvent::MouseButtonPress ||
             event->type() == QEvent::DragEnter ||
             event->type() == QEvent::DragMove ||
             event->type() == QEvent::Drop ||
             event->type() == QEvent::MouseMove ))
    {
        //Search SKGWidget corresponding to this widget
        SKGWidget* toMove=NULL;
        int toMoveIndex=-1;
        int nb=itemsPointers.count();
        for ( int i=0; toMove==NULL && i<nb; ++i )
        {
            SKGWidget* w=itemsPointers.at(i);
            if (w && w->getDragWidget()==object)
            {
                toMove=w;
                toMoveIndex=i;
            }
        }

        if ( event->type() == QEvent::MouseButtonPress )
        {
            //Drag
            QMouseEvent *mevent = ( QMouseEvent * ) event;
            if ( mevent->button() == Qt::LeftButton )
            {
                clickedPoint=mevent->pos();

                return true;
            }
        }
        else if ( event->type() == QEvent::MouseMove )
        {
            //Drag
            if (clickedPoint!=QPoint(-1, -1) && toMoveIndex!=-1)
            {
                QMouseEvent *mevent = ( QMouseEvent * ) event;
                int distance = ( mevent->pos( ) - clickedPoint ).manhattanLength( );
                if ( distance >= QApplication::startDragDistance( ))
                {
                    QMimeData *mimeData = new QMimeData;
                    mimeData->setData("application/x-skgdashboardpluginwidget", SKGServices::intToString(toMoveIndex).toLatin1());

                    QDrag *drag = new QDrag(this);
                    drag->setMimeData(mimeData);
                    drag->exec(); // krazy:exclude=crashy

                    return true;
                }
            }
        }
        else if ( event->type() == QEvent::DragEnter )
        {
            //Drop move
            QDragEnterEvent *devent = ( QDragEnterEvent * ) event;
            if (devent->mimeData()->hasFormat("application/x-skgdashboardpluginwidget")) {
                devent->accept();

                return true;
            }
        }

        else if ( event->type() == QEvent::DragMove )
        {
            //Drop move
            QDragMoveEvent *devent = ( QDragMoveEvent * ) event;
            if (devent->mimeData()->hasFormat("application/x-skgdashboardpluginwidget")) {
                int oldPos=SKGServices::stringToInt(devent->mimeData()->data("application/x-skgdashboardpluginwidget"));
                if (oldPos!=toMoveIndex) devent->accept();
                else devent->ignore();

                return true;
            }
        }

        else if ( event->type() == QEvent::Drop )
        {
            //Drop
            QDropEvent *devent = ( QDropEvent * ) event;
            if (devent->mimeData()->hasFormat("application/x-skgdashboardpluginwidget")) {
                int oldPos=SKGServices::stringToInt(devent->mimeData()->data("application/x-skgdashboardpluginwidget"));

                if (oldPos+1==toMoveIndex) ++toMoveIndex;

                //Get item
                QString id=items.at ( oldPos );
                SKGWidget* wgt=itemsPointers.at ( oldPos );
                int s=itemsSizes.at(oldPos);

                //Remove item
                items.removeAt ( oldPos );
                itemsPointers.removeAt ( oldPos );
                itemsSizes.removeAt ( oldPos );
                if (toMoveIndex>oldPos) --toMoveIndex;

                //Add it after
                items.insert ( toMoveIndex, id );
                itemsPointers.insert ( toMoveIndex, wgt );
                itemsSizes.insert ( toMoveIndex, s );

                //Get new state
                QString state=getState();

                //Reset state
                setState ( state );

                clickedPoint=QPoint(-1, -1);

                return true;
            }
        }
    }

    return false;
}

void SKGDashboardPluginWidget::showHeaderMenu ( const QPoint& pos )
{
    //Display menu
    if ( menu ) menu->popup ( mapToGlobal ( pos ) );
}

void SKGDashboardPluginWidget::onAddWidget()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        QString id=sender->data().toString();
        QStringList param=SKGServices::splitCSVLine ( id,'-' );

        SKGInterfacePlugin* db=SKGMainPanel::getMainPanel()->getPluginByName ( param.at ( 0 ) );
        if ( db )
        {
            SKGWidget* dbw=db->getDashboardWidget ( SKGServices::stringToInt ( param.at ( 1 ) ) );
            if ( dbw )
            {
                //Add
                items.push_back ( id );
                itemsPointers.push_back ( dbw );
                itemsSizes.push_back(0);

                //Get new state
                QString state=getState();

                //Remove added item
                items.pop_back();
                itemsPointers.pop_back();
                itemsSizes.pop_back();

                //Reset state
                setState ( state );
            }
        }
    }
}

void SKGDashboardPluginWidget::onMoveAfterWidget()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        int pos=sender->data().toInt();
        if ( pos>=0 && pos<items.count()-1 ) //last one cannot be moved after
        {
            //Get item
            QString id=items.at ( pos );
            SKGWidget* wgt=itemsPointers.at ( pos );
            int s=itemsSizes.at(pos);

            //Remove item
            items.removeAt ( pos );
            itemsPointers.removeAt ( pos );
            itemsSizes.removeAt ( pos );

            //Add it after
            items.insert ( pos+1, id );
            itemsPointers.insert ( pos+1, wgt );
            itemsSizes.insert ( pos+1, s );

            //Get new state
            QString state=getState();

            //Reset state
            setState ( state );
        }
    }
}

void SKGDashboardPluginWidget::onMoveBeforeWidget()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        int pos=sender->data().toInt();
        if ( pos>0 && pos<=items.count()-1 ) //first one cannot be moved before
        {
            //Get item
            QString id=items.at ( pos );
            SKGWidget* wgt=itemsPointers.at ( pos );
            int s=itemsSizes.at(pos);

            //Remove item
            items.removeAt ( pos );
            itemsPointers.removeAt ( pos );
            itemsSizes.removeAt ( pos );

            //Add it after
            items.insert ( pos-1, id );
            itemsPointers.insert ( pos-1, wgt );
            itemsSizes.insert ( pos-1, s );

            //Get new state
            QString state=getState();

            //Reset state
            setState ( state );
        }
    }
}

void SKGDashboardPluginWidget::onRemoveWidget()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        int pos=sender->data().toInt();
        if ( pos>=0 )
        {
            //Get item
            QString id=items.at ( pos );
            SKGWidget* wgt=itemsPointers.at ( pos );
            int s=itemsSizes.at(pos);

            //Remove item
            items.removeAt ( pos );
            itemsPointers.removeAt ( pos );
            itemsSizes.removeAt ( pos );

            //Get new state
            QString state=getState();

            //We add removed object to be sure that it will be removed
            items.push_back ( id );
            itemsPointers.push_back ( wgt );
            itemsSizes.push_back ( s );

            //Reset state
            setState ( state );
        }
    }
}

void SKGDashboardPluginWidget::onZoomInWidget()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        int pos=sender->data().toInt();
        if ( pos>=0 )
        {
            //Get item
            int s=itemsSizes.at(pos);

            //Remove item
            itemsSizes.removeAt ( pos );

            //Add it after
            itemsSizes.insert ( pos, s+1 );

            //Get new state
            QString state=getState();

            //Reset state
            setState ( state );
        }
    }
}

void SKGDashboardPluginWidget::onZoomOutWidget()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        int pos=sender->data().toInt();
        if ( pos>=0 )
        {
            //Get item
            int s=itemsSizes.at(pos);

            //Remove item
            itemsSizes.removeAt ( pos );

            //Add it after
            itemsSizes.insert ( pos, qMax(s-1, 0) );

            //Get new state
            QString state=getState();

            //Reset state
            setState ( state );
        }
    }
}

void SKGDashboardPluginWidget::addItem ( SKGInterfacePlugin* iDashboard, int iIndex, int iZoom, const QString& iState )
{
    if ( iDashboard && flowLayout)
    {
        SKGWidget* dbw=iDashboard->getDashboardWidget ( iIndex );
        if ( dbw )
        {
            //Add widget
            dbw->setParent ( ui.kContent );
            dbw->setState ( iState );
            flowLayout->addWidget ( dbw);

            //Install filter
            QWidget* drag=dbw->getDragWidget();
            if (drag)
            {
                drag->installEventFilter(this);
                drag->setAcceptDrops(true);
            }

            //Set size
            QSize size=dbw->minimumSize();
            QSize newSize(size.width()*pow(10,((qreal) iZoom+1)/10.0), size.height()*pow(10,((qreal) iZoom+1)/10.0));
            dbw->setMinimumSize(newSize);

            //Add remove item in menu
            QString id=iDashboard->objectName() +'-'+SKGServices::intToString ( iIndex );
            {
                QAction* act=removeMenu->addAction ( iDashboard->getDashboardWidgetTitle ( iIndex ) );
                if ( act )
                {
                    act->setIcon ( KIcon ( iDashboard->icon() ) );
                    act->setData ( items.count() );
                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onRemoveWidget() ) );
                }
            }

            //Add move next item in menu
            {
                QAction* act=nextMenu->addAction ( iDashboard->getDashboardWidgetTitle ( iIndex ) );
                if ( act )
                {
                    act->setIcon ( KIcon ( iDashboard->icon() ) );
                    act->setData ( items.count() );
                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onMoveAfterWidget() ) );
                }
            }

            //Add remove item in menu
            {
                QAction* act=previousMenu->addAction ( iDashboard->getDashboardWidgetTitle ( iIndex ) );
                if ( act )
                {
                    act->setIcon ( KIcon ( iDashboard->icon() ) );
                    act->setData ( items.count() );
                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onMoveBeforeWidget() ) );
                }
            }

            //Add remove item in menu
            if (size.height() && size.width())
            {
                QAction* act=zoomInMenu->addAction ( iDashboard->getDashboardWidgetTitle ( iIndex ) );
                if ( act )
                {
                    act->setIcon ( KIcon ( iDashboard->icon() ) );
                    act->setData ( items.count() );
                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onZoomInWidget() ) );
                }
            }

            //Add remove item in menu
            if (size.height() && size.width())
            {
                QAction* act=zoomOutMenu->addAction ( iDashboard->getDashboardWidgetTitle ( iIndex ) );
                if ( act )
                {
                    act->setIcon ( KIcon ( iDashboard->icon() ) );
                    act->setData ( items.count() );
                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onZoomOutWidget() ) );
                }
            }

            items.push_back ( id );
            itemsPointers.push_back ( dbw );
            itemsSizes.push_back(iZoom);
        }
    }
}

#include "skgdashboardpluginwidget.moc"
