/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file defines classes SKGObjectBase.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgobjectbase.h"
#include "skgtraces.h"
#include "skgdocument.h"
#include "skgnamedobject.h"

#include <klocale.h>

#include <QSqlDatabase>

SKGError SKGObjectBase::getObjects(SKGDocument* iDocument, const QString& iTable,
                                   const QString& iWhereClause, SKGListSKGObjectBase& oListObject)
{
    SKGError err;
    SKGTRACEINRC(20, "SKGObjectBase::getObjects", err);
    SKGTRACEL(20) << "Input parameter [iTable]=[" << iTable << ']' << endl;
    SKGTRACEL(20) << "Input parameter [iWhereClause]=[" << iWhereClause << ']' << endl;

    //Initialisation
    oListObject.clear();

    //Execute sqlorder
    SKGStringListList result;
    err = SKGServices::executeSelectSqliteOrder(iDocument,
            "SELECT * FROM " + iTable +
            (!iWhereClause.isEmpty() ? " WHERE " + iWhereClause : ""),
            result);

    //Create output
    if (err.isSucceeded()) {
        SKGStringListListIterator itrow = result.begin();
        QStringList columns = *(itrow);
        ++itrow;
        for (; err.isSucceeded() && itrow != result.end(); ++itrow) {
            QStringList values = *(itrow);
            SKGObjectBase tmp(iDocument, iTable);
            err=tmp.setAttributes(columns, values);
            oListObject.push_back(tmp);
        }
    }
    return err;
}

SKGError SKGObjectBase::getNbObjects(SKGDocument* iDocument, const QString& iTable,
                                     const QString& iWhereClause, int& oNbObjects)
{
    SKGError err;
    SKGTRACEINRC(20, "SKGObjectBase::getNbObjects", err);
    SKGTRACEL(20) << "Input parameter [iTable]=[" << iTable << ']' << endl;
    SKGTRACEL(20) << "Input parameter [iWhereClause]=[" << iWhereClause << ']' << endl;

    //Initialisation
    oNbObjects=0;

    //Execute sqlorder
    SKGStringListList result;
    err = SKGServices::executeSelectSqliteOrder(iDocument,
            "SELECT count(1) FROM " + iTable +
            (!iWhereClause.isEmpty() ? " WHERE " + iWhereClause : ""),
            result);

    //Create output
    if (err.isSucceeded()) oNbObjects=SKGServices::stringToInt(result.at(1).at(0));
    return err;
}

SKGError SKGObjectBase::getObject(SKGDocument* iDocument, const QString& iTable,
                                  const QString& iWhereClause, SKGObjectBase& oObject)
{
    SKGListSKGObjectBase temporaryResult;
    oObject.resetID();
    SKGError err = SKGObjectBase::getObjects(iDocument, iTable, iWhereClause, temporaryResult);
    if (err.isSucceeded()) {
        int size=temporaryResult.size();
        if (size> 1)  err=SKGError(ERR_INVALIDARG, i18nc("Error message", "More than one object returned in '%1' for '%2'",iTable,iWhereClause));
        else {
            if (size==0)  err=SKGError(ERR_INVALIDARG, i18nc("Error message", "No object returned in '%1' for '%2'",iTable,iWhereClause));
            else  oObject = *(temporaryResult.begin());
        }
    }
    return err;
}

SKGError SKGObjectBase::getObject(SKGDocument* iDocument, const QString& iTable,
                                  int iId, SKGObjectBase& oObject)
{
    return getObject(iDocument, iTable, "id=" + SKGServices::intToString(iId), oObject);
}

SKGObjectBase::SKGObjectBase(SKGDocument* iDocument, const QString& iTable, int iID)
        :QObject(), id(iID), table(iTable), document(iDocument)
{
    if (id != 0) load();
}

SKGObjectBase::~SKGObjectBase()
{
    document=NULL;
}

SKGObjectBase::SKGObjectBase(const SKGObjectBase& iObject)
        :QObject()
{
    copyFrom(iObject);
}

const SKGObjectBase& SKGObjectBase::operator= (const SKGObjectBase& iObject)
{
    copyFrom(iObject);
    return *this;
}

bool SKGObjectBase::operator==(const SKGObjectBase& iObject) const
{
    return (getUniqueID()==iObject.getUniqueID());
}

bool SKGObjectBase::operator!=(const SKGObjectBase& iObject) const
{
    return !(*this==iObject);
}

bool SKGObjectBase::operator<(const SKGObjectBase& iObject) const
{
    double d1=SKGServices::stringToDouble(getAttribute("f_sortorder"));
    double d2=SKGServices::stringToDouble(iObject.getAttribute("f_sortorder"));
    return (d1<d2);
}

bool SKGObjectBase::operator>(const SKGObjectBase& iObject) const
{
    double d1=SKGServices::stringToDouble(getAttribute("f_sortorder"));
    double d2=SKGServices::stringToDouble(iObject.getAttribute("f_sortorder"));
    return (d1>d2);
}

QString SKGObjectBase::getUniqueID() const
{
    return SKGServices::intToString(id) + '-' + getRealTable();
}

int SKGObjectBase::getID() const
{
    return id;
}

QString SKGObjectBase::getDisplayName() const
{
    QString output;

    SKGStringListList result;
    QString wc = getWhereclauseId();
    if (wc.isEmpty()) wc = "id=" + SKGServices::intToString(id);
    QString sql = "SELECT t_displayname FROM v_" + getRealTable() + "_displayname WHERE " + wc;
    SKGServices::executeSelectSqliteOrder(document, sql, result);
    if (result.count()==2) output=result.at(1).at(0);

    return output;
}

void SKGObjectBase::copyFrom(const SKGObjectBase& iObject)
{
    id = iObject.id;
    table = iObject.table;
    document = iObject.document;
    attributes = iObject.attributes;
}

SKGObjectBase SKGObjectBase::cloneInto(SKGDocument* iDocument)
{
    SKGDocument* targetDocument=iDocument;
    if (targetDocument==NULL) targetDocument=document;

    SKGObjectBase output;
    output.copyFrom(*this);
    output.id=0;
    output.document=targetDocument;

    return output;
}

SKGError SKGObjectBase::resetID()
{
    SKGError err;
    id = 0;
    return err;
}

QString SKGObjectBase::getTable() const
{
    return table;
}

QString SKGObjectBase::getRealTable() const
{
    return SKGServices::getRealTable(table);
}

SKGDocument* SKGObjectBase::getDocument() const
{
    return document;
}

SKGQStringQStringMap SKGObjectBase::getAttributes() const
{
    return attributes;
}

int SKGObjectBase::getNbAttributes() const
{
    return attributes.count();
}

SKGError SKGObjectBase::setAttributes(const QStringList& iNames, const QStringList& iValues)
{
    SKGError err;
    int nb=iNames.size();
    for (int i = 0; err.isSucceeded() && i < nb; ++i) {
        QString att = iNames.at(i);
        QString val = iValues.at(i);

        if (att != "id") err = setAttribute(att, val);
        else  id = SKGServices::stringToInt(val);
    }
    return err;
}

SKGError SKGObjectBase::setAttribute(const QString& iName, const QString& iValue)
{
    SKGError err;
    attributes[iName] = iValue;
    return err;
}

QString SKGObjectBase::getAttribute(const QString& iName) const
{
    QString output;
    if (attributes.contains(iName)) {
        output=attributes[iName];
    } else {
        //Is the iName a number ?
        bool ok;
        int pos=iName.toInt(&ok);
        if (ok) {
            //What is the key corresponding to this name ?
            QStringList keys=attributes.keys();
            if (pos>=0 && pos<keys.count()) output=attributes[keys[pos]];
        }
    }

    return output;
}

bool SKGObjectBase::exist() const
{
    SKGTRACEIN(20, "SKGObjectBase::exist");

    SKGStringListList result;
    QString wc = getWhereclauseId();
    if (wc.isEmpty() && id) wc = "id=" + SKGServices::intToString(id);
    if (wc.isEmpty()) return false;

    QString sql = "SELECT count(1) FROM " + table + " WHERE " + wc;
    SKGServices::executeSelectSqliteOrder(document, sql, result);
    return (result.size()>=2 && result.at(1).at(0)!="0");
}

SKGError SKGObjectBase::load()
{
    SKGError err;
    SKGTRACEINRC(20, "SKGObjectBase::load", err);

    //Prepare where clause
    QString wc = getWhereclauseId();
    if (wc.isEmpty()) wc = "id=" + SKGServices::intToString(id);

    //Execute sql order
    SKGStringListList result;
    err = SKGServices::executeSelectSqliteOrder(document, "SELECT * FROM " + table + " WHERE " + wc, result);
    if (err.isSucceeded()) {
        int size=result.size();
        if ( size== 1)  err = SKGError(ERR_INVALIDARG, i18nc("Error message", "Load of '%1' with '%2' failed because it was not found in the database",table,wc));
        else if (size != 2)  err = SKGError(ERR_INVALIDARG, i18np("Load of '%2' with '%3' failed because of bad size of result (found one object)",
                                                "Load of '%2' with '%3' failed because of bad size of result (found %1 objects)",
                                                size-1,table,wc));
        else {
            SKGStringListListIterator itrow = result.begin();
            QStringList columns = *(itrow);
            ++itrow;
            QStringList values = *(itrow);
            err=setAttributes(columns, values);
        }
    }

    return err;
}

QString SKGObjectBase::getWhereclauseId() const
{
    int id = getID();
    if (id != 0) return "id=" + SKGServices::intToString(id);
    return "";
}

SKGError SKGObjectBase::save(bool iInsertOrUpdate, bool iReloadAfterSave)
{
    SKGError err;
    SKGTRACEINRC(20, "SKGObjectBase::save", err);

    if (!document) err=SKGError(ERR_POINTER, i18nc("Error message", "Operation impossible because the document is missing"));
    else {
        //Check if we are in a transaction
        err = document->checkExistingTransaction();
        if (err.isSucceeded()) {
            //Table to use
            QString tablename = getRealTable();

            //Build order
            QString part1Insert;
            QString part2Insert;
            QString partUpdate;

            SKGQStringQStringMapIterator it;
            for (it = attributes.begin() ; it != attributes.end(); ++it) {
                QString att = SKGServices::stringToSqlString(it.key());
                QString attlower = att.toLower();
                if (att.length() > 2 && att==attlower) { //We must ignore attributes coming from views
                    QString value = SKGServices::stringToSqlString(it.value());

                    //Attribute
                    if (!part1Insert.isEmpty()) part1Insert.append(',');
                    part1Insert.append('\''+att+'\'');

                    //Value
                    if (!part2Insert.isEmpty()) part2Insert.append(',');
                    part2Insert.append('\''+value+'\'');

                    //Attribute=Value for update
                    if (!partUpdate.isEmpty()) partUpdate.append(',');
                    partUpdate.append(att+"='"+value+'\'');
                }
            }

            //We try an Insert
            if (id == 0) {
                //We have to try un insert
                QString sql=QString("INSERT INTO %1 (%2) VALUES (%3);") .arg(tablename) .arg(part1Insert) .arg(part2Insert);
                err = SKGServices::executeSqliteOrder( document, sql, &id);
            } else {
                //We must try an update
                err = SKGError(ERR_ABORT, ""); //Just to go in UPDATE code
            }

            if (err.isFailed() && iInsertOrUpdate) {
                //INSERT failed, could we try an update ?
                QString wc = this->getWhereclauseId();
                if (!wc.isEmpty()) {
                    //Yes ==> Update
                    err = SKGServices::executeSqliteOrder(document, "UPDATE " + tablename + " SET " + partUpdate + " WHERE " + wc);
                }
            }
        }
    }

    //Reload object is updated
    if (err.isSucceeded() && iReloadAfterSave) {
        //The object has been updated ==>load
        err = load();
    }

    return err;
}

SKGError SKGObjectBase::remove(bool iSendMessage) const
{
    SKGError err;
    SKGTRACEINRC(20, "SKGObjectBase::remove", err);
    if (!document) err=SKGError(ERR_POINTER, i18nc("Error message", "Operation impossible because the document is missing"));
    else {
        //Check if we are in a transaction
        err = document->checkExistingTransaction();

        //delete order
        QString viewForDelete=QString("v_")+getRealTable()+"_delete";

        //Check if the delete view exist
        SKGStringListList temporaryResult;
        SKGServices::executeSelectSqliteOrder(document, "PRAGMA table_info( " + viewForDelete + " );",temporaryResult);
        if (temporaryResult.count()>1) { //At least one attribute
            //Delete view exists, check if the delete is authorized
            int nb=0;
            SKGObjectBase::getNbObjects(document, viewForDelete,"id="+SKGServices::intToString(id), nb);
            if (nb!=1)  err=SKGError(ERR_FAIL, i18nc("Error message", "You are not authorized to delete this object '%1'", getDisplayName()));
        }

        QString displayname=getDisplayName(); //Must be done before the delete order
        if (err.isSucceeded()) err=SKGServices::executeSqliteOrder(document, "DELETE FROM "+getRealTable()+" WHERE id="+SKGServices::intToString(id));
        if (iSendMessage && err.isSucceeded() && !displayname.isEmpty())
            err=document->sendMessage(i18nc("An information message", "'%1' has been deleted",displayname), false);
    }

    return err;
}

SKGError SKGObjectBase::dump()
{
    SKGError err;

    //dump
    SKGTRACE << "=== START DUMP [" << getUniqueID() << "]===" << endl;
    SKGQStringQStringMapIterator it;
    for (it = attributes.begin() ; it != attributes.end(); ++it)
        SKGTRACE << it.key() << "=[" << it.value() << ']' << endl;
    SKGTRACE << "=== END DUMP [" << getUniqueID() << "]===" << endl;
    return err;
}

QStringList SKGObjectBase::getProperties() const
{
    return getDocument()->getParameters(getUniqueID());
}

QString SKGObjectBase::getProperty(const QString& iName) const
{
    return getDocument()->getParameter(iName, getUniqueID());
}

QVariant SKGObjectBase::getPropertyBlob(const QString& iName) const
{
    return getDocument()->getParameterBlob(iName, getUniqueID());
}

SKGError SKGObjectBase::setProperty(const QString& iName, const QString& iValue, const QString& iFileName, SKGPropertyObject* oObjectCreated) const
{
    return getDocument()->setParameter(iName, iValue, iFileName, getUniqueID(), oObjectCreated);
}

SKGError SKGObjectBase::setProperty(const QString& iName, const QString& iValue, const QVariant& iBlob, SKGPropertyObject* oObjectCreated) const
{
    return getDocument()->setParameter(iName, iValue, iBlob, getUniqueID(), oObjectCreated);

}

#include "skgobjectbase.moc"

