/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for for CSV import / export.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportplugincsv.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgbankincludes.h"
#include "skgimportexportmanager.h"

#include <klocale.h>
#include <kfilterdev.h>
#include <ksavefile.h>
#include <QCryptographicHash>
#include <qfileinfo.h>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGImportPluginCsvFactory, registerPlugin<SKGImportPluginCsv>();)
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN(SKGImportPluginCsvFactory("skrooge_import_csv", "skrooge_import_csv"))

SKGImportPluginCsv::SKGImportPluginCsv(QObject* iImporter, const QVariantList& iArg)
    : SKGImportPlugin(iImporter)
{
    SKGTRACEIN(10, "SKGImportPluginCsv::SKGImportPluginCsv");
    Q_UNUSED(iArg);
}

SKGImportPluginCsv::~SKGImportPluginCsv()
{
}

bool SKGImportPluginCsv::isImportPossible()
{
    SKGTRACEIN(10, "SKGImportPluginCsv::isImportPossible");
    return isExportPossible();
}

SKGError SKGImportPluginCsv::importFile()
{
    if(!m_importer) return SKGError(ERR_ABORT, i18nc("Error message", "Invalid parameters"));

    SKGError err;
    SKGTRACEINRC(2, "SKGImportPluginCsv::importFile", err);
    QString iFileName = m_importer->getFileName();
    SKGTRACEL(10) << "Input filename=" << iFileName << endl;

    //Begin transaction
    err = m_importer->getDocument()->beginTransaction("#INTERNAL#", 3);
    if(!err) {
        //Initialize some variables
        QDateTime now = QDateTime::currentDateTime();
        QString postFix = SKGServices::dateToSqlString(now);

        //Default mapping
        if(m_importer->getCSVMapping().count() == 0) {
            err = m_importer->setCSVMapping(NULL);
            if(!err) err = m_importer->getDocument()->sendMessage(i18nc("An information message",  "Use automatic search of the columns"));
        }
        if(!err) err = m_importer->getDocument()->sendMessage(i18nc("An information message",  "Mapping used: %1", m_importer->getCSVMapping().join("|")));

        //Step 1 done
        if(!err) err = m_importer->getDocument()->stepForward(1);

        //Open file
        if(!err) {
            QFile file(iFileName);
            if(!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
                err.setReturnCode(ERR_INVALIDARG);
                err.setMessage(i18nc("Error message",  "Open file '%1' failed", iFileName));
            } else {
                QTextStream stream(&file);
                if(!m_importer->getCodec().isEmpty()) stream.setCodec(m_importer->getCodec().toAscii().constData());

                //Ignore useless lines
                int headerIndex = m_importer->getCSVHeaderIndex();
                if(headerIndex == -1) {
                    err.setReturnCode(ERR_FAIL);
                    err.setMessage(i18nc("Error message",  "Header not found in CSV file"));
                }

                for(int i = 1; i <= headerIndex; ++i)
                    stream.readLine();

                //Get data column
                QStringList dates;
                QStringList lines;
                if(!err) {
                    int posdate = m_importer->getCSVMapping().indexOf("date");
                    if(posdate != -1) {
                        while(!stream.atEnd()) {
                            //Read line
                            QString line = stream.readLine().trimmed();
                            if(!line.isEmpty()) {
                                lines.push_back(line);

                                //Get date
                                QStringList field = SKGServices::splitCSVLine(line, m_importer->getCSVSeparator());
                                if(posdate < field.count()) dates.push_back(field.at(posdate).trimmed());
                            }
                        }
                    }
                }

                //close file
                file.close();

                //Select dateformat
                QString dateFormat = SKGServices::getDateFormat(dates);
                if(!err && dateFormat.isEmpty()) {
                    err.setReturnCode(ERR_FAIL);
                    err.setMessage(i18nc("Error message",  "Date format not supported"));
                }
                if(!err)
                    err = m_importer->getDocument()->sendMessage(i18nc("An information message",  "Import of '%1' with code '%2' and date format '%3'", iFileName, m_importer->getCodec(), dateFormat));

                //Step 2 done
                if(!err) err = m_importer->getDocument()->stepForward(2);

                //Treat all lines
                if(!err) {
                    int nb = lines.size();
                    err = m_importer->getDocument()->beginTransaction("#INTERNAL#", nb);

                    //Save last mapping used in a settings
                    QString mappingDesc;
                    int nbMap = m_importer->getCSVMapping().count();
                    for(int i = 0; i < nbMap; ++i) {
                        if(i) mappingDesc += '|';
                        mappingDesc += m_importer->getCSVMapping().at(i);
                    }
                    if(!err) err = m_importer->getDocument()->setParameter("SKG_LAST_CSV_MAPPING_USED", mappingDesc);

                    SKGUnitObject defUnit;
                    SKGAccountObject defAccount;
                    QMap<int, SKGOperationObject> mapGroup;
                    QMap<int, SKGOperationObject> mapOperation;
                    for(int i = 0; !err && i < nb; ++i) {
                        QString currentCategory;
                        SKGOperationObject currentOperation(m_importer->getDocument());
                        SKGSubOperationObject currentSubOperation(m_importer->getDocument());

                        //Valuate mandatory attribute with default value
                        if(m_importer->getCSVMapping().indexOf("unit") == -1) {
                            err = m_importer->getDefaultUnit(defUnit);
                            if(!err) err = currentOperation.setUnit(defUnit);
                        }
                        if(!err && m_importer->getCSVMapping().indexOf("account") == -1) {
                            err = m_importer->getDefaultAccount(defAccount);
                            if(!err) err = currentOperation.setParentAccount(defAccount);
                        }

                        QString line = lines.at(i);
                        QByteArray hash = QCryptographicHash::hash(line.toUtf8(), QCryptographicHash::Md5);

                        QString skg_op_original_amount;
                        QStringList atts = SKGServices::splitCSVLine(line, m_importer->getCSVSeparator());
                        int nbcol = m_importer->getCSVMapping().count();
                        if(atts.count() < nbcol) {
                            err = SKGError(ERR_INVALIDARG, i18nc("Error message", "Invalid number of columns in line %1. Expected %2. Found %3.",
                                                                 headerIndex + i + 1, nbcol, atts.count()));
                        }
                        int initialBalance = false;
                        int idgroup = 0;
                        int idtransaction = 0;
                        QStringList propertiesAtt;
                        QStringList propertiesVal;
                        double sign = 1.0;
                        bool amountSet = false;
                        for(int c = 0; !err && c < nbcol; ++c) {
                            QString col = m_importer->getCSVMapping()[c];
                            if(!col.isEmpty()) {
                                QString val;
                                if(c >= 0 && c < atts.count()) val = atts.at(c).trimmed();
                                if(col == "date") {
                                    err = currentOperation.setDate(SKGServices::stringToTime(SKGServices::dateToSqlString(val, dateFormat)).date());
                                    if(val == "0000-00-00") initialBalance = true;
                                } else if(col == "number") {
                                    if(!val.isEmpty()) err = currentOperation.setNumber(SKGServices::stringToInt(val));
                                } else if(col == "mode") {
                                    err = currentOperation.setMode(val);
                                } else if(col == "payee") {
                                    SKGPayeeObject payeeObj;
                                    err = SKGPayeeObject::createPayee(m_importer->getDocument(), val, payeeObj);
                                    if(!err) err = currentOperation.setPayee(payeeObj);
                                } else if(col == "comment") {
                                    QString comment = currentOperation.getComment();
                                    if(!comment.isEmpty()) comment += ' ';
                                    comment += val;
                                    err = currentOperation.setComment(comment);
                                    if(!err) err = currentSubOperation.setComment(comment);
                                } else if(col == "status") {
                                    err = currentOperation.setStatus(val == "C" || val == "Y" ?  SKGOperationObject::CHECKED : val == "P" ?  SKGOperationObject::POINTED : SKGOperationObject::NONE);
                                } else if(col == "bookmarked") {
                                    err = currentOperation.bookmark(val == "Y");
                                } else if(col == "idgroup") {
                                    idgroup = SKGServices::stringToInt(val);
                                } else if(col == "idtransaction") {
                                    idtransaction = SKGServices::stringToInt(val);
                                } else if(col == "amount") {
                                    if(!val.isEmpty() && !amountSet) {
                                        amountSet = true;
                                        if(m_importer->getCSVMapping().contains("quantity")) {
                                            //209705 vvvv
                                            skg_op_original_amount = val;
                                            //209705 ^^^^
                                        } else {
                                            err = currentSubOperation.setQuantity(sign * SKGServices::stringToDouble(val));
                                        }
                                    }
                                } else if(col == "quantity") {
                                    err = currentSubOperation.setQuantity(SKGServices::stringToDouble(val));
                                } else if(col == "sign") {
                                    if(QRegExp(m_importer->getCSVMappingRules()["debit"], Qt::CaseInsensitive).indexIn(val) != -1) {
                                        sign = -1;
                                        double cval = currentSubOperation.getQuantity();
                                        if(cval > 0) err = currentSubOperation.setQuantity(-cval);
                                    }
                                } else if(col == "unit") {
                                    //Looking for unit
                                    SKGUnitObject unit(m_importer->getDocument());
                                    if(val != defUnit.getName()) {  //For performance
                                        err = unit.setName(val);
                                        if(!err) err = unit.setSymbol(val);
                                        if(!err && unit.load().isFailed())  err = unit.save(false);     //Save only

                                        //This unit is now the default one, it's better for performance
                                        defUnit = unit;
                                    } else {
                                        unit = defUnit;
                                    }

                                    SKGUnitValueObject unitval;
                                    if(!err) err = unit.addUnitValue(unitval);
                                    if(!err) {
                                        int posAmount = m_importer->getCSVMapping().indexOf("amount");
                                        int posQuantity = m_importer->getCSVMapping().indexOf("quantity");
                                        if(posAmount != -1 && posQuantity != -1) {
                                            err = unitval.setQuantity(SKGServices::stringToDouble(atts.at(posAmount)) / SKGServices::stringToDouble(atts.at(posQuantity)));
                                        } else {
                                            err = unitval.setQuantity(1);
                                        }
                                    }
                                    if(!err) err = unitval.setDate(now.date());
                                    if(!err) err = unitval.save();
                                    if(!err) err = currentOperation.setUnit(unit);
                                } else if(col == "account") {
                                    //Looking for account
                                    if(val != defAccount.getName()) {  //For performance
                                        SKGAccountObject account(m_importer->getDocument());
                                        account.setName(val);
                                        err = account.load();
                                        if(!!err) {
                                            //Not found, we have to create one
                                            SKGBankObject bank(m_importer->getDocument());
                                            QString name = i18nc("Noun",  "Bank for import %1", postFix);
                                            err = bank.setName(name);
                                            if(!err && bank.load().isFailed()) {
                                                err = bank.save(false);   //Save only
                                                if(!err) err = m_importer->getDocument()->sendMessage(i18nc("An information message",  "Default bank '%1' created for import", name));
                                            }
                                            if(!err) err = bank.addAccount(account);
                                            if(!err) err = account.setName(val);
                                            if(!err && account.load().isFailed())  err = account.save(false);      //Save only
                                        }

                                        //This account is now the default one, it's better for performance
                                        defAccount = account;
                                    }
                                    if(!err) err = currentOperation.setParentAccount(defAccount);
                                } else if(col == "category") {
                                    //Set Category
                                    if(!val.isEmpty()) {
                                        //Prepare val
                                        val.replace('/', OBJECTSEPARATOR);
                                        val.replace(':', OBJECTSEPARATOR);
                                        val.replace(',', OBJECTSEPARATOR);
                                        val.replace(';', OBJECTSEPARATOR);
                                        //Get previous category
                                        if(!currentCategory.isEmpty()) val = currentCategory % OBJECTSEPARATOR % val;
                                        currentCategory = val;

                                        //Create and set category
                                        SKGCategoryObject Category;
                                        err = SKGCategoryObject::createPathCategory(m_importer->getDocument(), val, Category);
                                        if(!err)  err = currentSubOperation.setCategory(Category);
                                    }
                                } else {
                                    //A property
                                    propertiesAtt.push_back(col);
                                    propertiesVal.push_back(val);
                                }
                            }
                        }

                        if(!err && initialBalance) {
                            //Specific values for initial balance
                            err = currentOperation.setStatus(SKGOperationObject::CHECKED);
                            if(!err) err = currentOperation.setAttribute("d_date", "0000-00-00");
                        }
                        if(!err) err = currentOperation.setImportID(hash.toHex());
                        if(!err) {
                            if(idtransaction != 0) {
                                if(mapOperation.contains(idtransaction)) {
                                    currentOperation = mapOperation[idtransaction];
                                    skg_op_original_amount = "";
                                } else {
                                    err = currentOperation.save();
                                    mapOperation[idtransaction] = currentOperation;
                                }
                            } else {
                                err = currentOperation.save(false);   //Save only
                            }
                        }
                        if(!err && idgroup != 0) {
                            if(mapGroup.contains(idgroup)) {
                                err = currentOperation.setGroupOperation(mapGroup[idgroup]);
                                if(!err) err = currentOperation.save();
                            }
                            mapGroup[idgroup] = currentOperation;
                        }

                        if(!err) err = currentSubOperation.setParentOperation(currentOperation);
                        if(!err) err = currentSubOperation.save(false, false);      //Save only without reload

                        //209705 vvvv
                        if(!err && !skg_op_original_amount.isEmpty()) {
                            err = currentOperation.setProperty("SKG_OP_ORIGINAL_AMOUNT", skg_op_original_amount);
                        }
                        //209705 ^^^^

                        //Add properties
                        int nbp = propertiesAtt.count();
                        for(int p = 0; !err && p < nbp; ++p) {
                            err = currentOperation.setProperty(propertiesAtt.at(p) , propertiesVal.at(p));
                        }

                        if(!err && i % 20 == 0) err = m_importer->getDocument()->executeSqliteOrder("ANALYZE");
                        if(!err) err = m_importer->getDocument()->stepForward(i + 1);
                    }

                    if(!err) err = m_importer->getDocument()->endTransaction(true);
                    else  m_importer->getDocument()->endTransaction(false);

                    //Lines treated
                    if(!err) err = m_importer->getDocument()->stepForward(3);
                }
            }
        }
    }
    if(!err) err = m_importer->getDocument()->endTransaction(true);
    else  m_importer->getDocument()->endTransaction(false);

    return err;
}


bool SKGImportPluginCsv::isExportPossible()
{
    SKGTRACEIN(10, "SKGImportPluginCsv::isExportPossible");
    return (!m_importer ? true : QFileInfo(m_importer->getFileName()).suffix().toUpper() == "CSV");
}

SKGError SKGImportPluginCsv::exportFile()
{
    if(!m_importer) return SKGError(ERR_ABORT, i18nc("Error message", "Invalid parameters"));
    SKGError err;
    SKGTRACEINRC(2, "SKGImportCsv::exportFile", err);
    QString iFileName = m_importer->getFileName();
    SKGTRACEL(10) << "Input filename=" << iFileName << endl;

    //Open file
    KSaveFile file(iFileName);
    if(!file.open()) {
        err.setReturnCode(ERR_INVALIDARG);
        err.setMessage(i18nc("Error message",  "Save file '%1' failed", iFileName));
    } else {
        QTextStream out(&file);
        if(!m_importer->getCodec().isEmpty()) out.setCodec(m_importer->getCodec().toAscii().constData());
        err = m_importer->getDocument()->dumpSelectSqliteOrder(
                  "SELECT d_date as date, t_ACCOUNT as account, i_number as number, t_mode as mode, "
                  "t_PAYEE as payee, t_REALCOMMENT as comment, f_REALQUANTITY as quantity, "
                  "t_UNIT as unit, f_REALCURRENTAMOUNT as amount, t_TYPEEXPENSE as sign, t_REALCATEGORY as category, t_status as status, "
                  "t_bookmarked as bookmarked, i_SUBOPID id, id idtransaction, i_group_id idgroup "
                  "FROM v_operation_consolidated ORDER BY d_date, id, i_SUBOPID", &out, SKGServices::DUMP_CSV);
    }

    //Close file
    file.finalize();
    file.close();

    return err;
}

QString SKGImportPluginCsv::getMimeTypeFilter() const
{
    return "*.csv|" % i18nc("A file format", "CSV file");
}

#include "skgimportplugincsv.moc"
