/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGINTERFACEPLUGIN_H
#define SKGINTERFACEPLUGIN_H
/** @file
* This file is a plugin interface definition.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include <QStringList>
#include <kparts/plugin.h>

#include "skgdocument.h"
#include "skgtabpage.h"
#include "skgerror.h"
#include "skgbasegui_export.h"
class SKGMainPanel;
class KConfigSkeleton;
class QDockWidget;

/**
 * This file is a plugin interface definition.
 */
class SKGBASEGUI_EXPORT SKGInterfacePlugin : public KParts::Plugin
{
    Q_OBJECT ;
public:
    /**
     * Describe a modification of an object
     */
    struct SKGAdvice {
        QString uuid;               /**< The uuid of the advice */
        int priority;               /**< The priority of the advice (0=low, 10=high) */
        QString shortMessage;           /**< The short message of the advice */
        QString longMessage;            /**< The long message of the advice */
        QStringList autoCorrection;     /**< The message in case of this advice could be auto cleaned. One line per possible solutions */
    } ;

    /**
     * Default constructor
     * @param parent the parent of the plugin
     */
    SKGInterfacePlugin(QObject * parent = 0);

    /**
     * Default destructor
     */
    virtual ~SKGInterfacePlugin();

    /**
     * Called to initialise the plugin
     * @param iDocument the main document
     * @param iArgument the arguments
     * @return true if the plugin is compatible with the document
     */
    virtual bool setupActions(SKGDocument* iDocument, const QStringList& iArgument) = 0;

    /**
     * This function is called when the application is launched again with new arguments
     * @param iArgument the arguments
     */
    virtual void processArguments(const QStringList& iArgument);

    /**
     * Must be modified to close properly the plugin.
     */
    virtual void close();

    /**
     * Must be modified to refresh widgets after a modification.
     */
    virtual void refresh();

    /**
     * The page widget of the plugin.
     * @return The page widget of the plugin
     */
    virtual SKGTabPage* getWidget();

    /**
     * The number of dashboard widgets of the plugin.
     * @return The number of dashboard widgets of the plugin
     */
    virtual int getNbDashboardWidgets();

    /**
     * Get a dashboard widget title of the plugin.
     * @param iIndex the index of the widget
     * @return The title
     */
    virtual QString getDashboardWidgetTitle(int iIndex);

    /**
     * Get a dashboard widget of the plugin.
     * @param iIndex the index of the widget
     * @return The dashboard widget of the plugin
     */
    virtual SKGWidget* getDashboardWidget(int iIndex);

    /**
     * The dock widget of the plugin.
     * @return The dock widget of the plugin
     */
    virtual QDockWidget* getDockWidget();

    /**
     * The preference widget of the plugin.
     * @return The preference widget of the plugin
     */
    virtual QWidget* getPreferenceWidget();

    /**
     * The preference skeleton of the plugin.
     * @return The preference skeleton of the plugin
     */
    virtual KConfigSkeleton* getPreferenceSkeleton();

    /**
     * This function is called when preferences have been modified. Must be used to save some parameters into the document.
     * A transaction is already opened
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError savePreferences() const;

    /**
     * The title of the plugin.
     * @return The title of the plugin
     */
    virtual QString title() const = 0;

    /**
     * The icon of the plugin.
     * @return The icon of the plugin
     */
    virtual QString icon() const;

    /**
     * The statusTip of the plugin.
     * @return The toolTip of the plugin
     */
    virtual QString statusTip() const;

    /**
     * The toolTip of the plugin.
     * @return The toolTip of the plugin
     */
    virtual QString toolTip() const;

    /**
     * The tips list of the plugin.
     * @return The tips list of the plugin
     */
    virtual QStringList tips() const;

    /**
     * Must be implemented to set the position of the plugin.
     * @return integer value between 0 and 999 (default = 999)
     */
    virtual int getOrder() const;

    /**
     * Must be implemented to know if a plugin must be display in pages chooser.
     * @return true of false (default = false)
     */
    virtual bool isInPagesChooser() const;

    /**
     * Must be implemented to know if this plugin is enabled
     * @return true of false (default = true)
     */
    virtual bool isEnabled() const;

    /**
     * The advice list of the plugin.
     * @return The advice list of the plugin
     */
    virtual QList<SKGInterfacePlugin::SKGAdvice> advices() const;

    /**
     * Must be implemented to execute the automatic correction for the advice.
     * @param iAdviceIdentifier the identifier of the advice
     * @param iSolution the identifier of the possible solution
     * @return an object managing the error. MUST return ERR_NOTIMPL if iAdviceIdentifier is not known
     *   @see SKGError
     */
    virtual SKGError executeAdviceCorrection(const QString& iAdviceIdentifier, int iSolution) const;

private:
    Q_DISABLE_COPY(SKGInterfacePlugin);

};
/**
 * This plugin interface definition.
 */
Q_DECLARE_INTERFACE(SKGInterfacePlugin, "skrooge.com.SKGInterfacePlugin/1.0");
/**
 * Declare the class
 */
Q_DECLARE_TYPEINFO(SKGInterfacePlugin::SKGAdvice, Q_PRIMITIVE_TYPE);

#endif // SKGINTERFACEPLUGIN_H
