/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin for monthly report.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgmonthlypluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgdocumentbank.h"
#include "skgaccountobject.h"
#include "skgunitobject.h"
#include "skgtransactionmng.h"

#include <kstandarddirs.h>
#include <grantlee/engine.h>
#include <grantlee/templateloader.h>
#ifdef SKGGRANTLEE17ENABLED
#include <grantlee/metatype.h>
#endif
#include <knewstuff3/downloaddialog.h>
#include <knewstuff3/uploaddialog.h>
#include <kio/netaccess.h>
#include <kmenu.h>
#include <kzip.h>

#include <QDomDocument>
#include <QPrinter>
#include <QPrintDialog>
#include <QTextStream>
#include <QFile>
#include <QDir>
#include <kcolorscheme.h>
#include <QDesktopServices>
#include <QWebFrame>

#ifdef SKGGRANTLEE17ENABLED
GRANTLEE_BEGIN_LOOKUP(SKGAdvice)
if (property == "uuid") return object.getUUID();
else if (property == "shortMessage") return object.getShortMessage();
else if (property == "longMessage") return object.getLongMessage();
else if (property == "priority") return object.getPriority();
else if (property == "autoCorrections") return object.getAutoCorrections();
GRANTLEE_END_LOOKUP

GRANTLEE_BEGIN_LOOKUP(SKGObjectBase)
Q_UNUSED(object);
Q_UNUSED(property);
GRANTLEE_END_LOOKUP
#endif

SKGMonthlyPluginWidget::SKGMonthlyPluginWidget(SKGDocument* iDocument)
    : SKGTabPage(iDocument)
{
    SKGTRACEIN(1, "SKGMonthlyPluginWidget::SKGMonthlyPluginWidget");
    if (!iDocument) return;

    ui.setupUi(this);
    ui.kDeleteTemplate->hide();

    //Make needed paths
    QString path = KStandardDirs::locateLocal("data", QString::fromLatin1("skrooge/html/default/"));

    //Copy "default" directory in locale
    foreach(const QString & file, KStandardDirs().findAllResources("data", QString::fromLatin1("skrooge/html/default/*.html"))) {
        QFile(file).copy(path % QFileInfo(file).fileName());
    }

    //Grantlee initialization
#ifdef SKGGRANTLEE17ENABLED
    Grantlee::MetaType::init();
    Grantlee::registerMetaType<SKGAdvice>();
    Grantlee::registerMetaType<SKGObjectBase>();
#endif
    ui.kRefresh->setIcon(KIcon("view-refresh"));
    ui.kGetNewHotStuff->setIcon(KIcon("get-hot-new-stuff"));
    ui.kDeleteTemplate->setIcon(KIcon("edit-delete"));

    connect((const QObject*) getDocument(), SIGNAL(tableModified(QString,int)), this, SLOT(dataModified(QString,int)), Qt::QueuedConnection);

    QStringList overlays;
    overlays.push_back("list-add");
    m_upload = new KAction(KIcon("get-hot-new-stuff", NULL, overlays), i18n("Upload"), this);
    connect(m_upload, SIGNAL(triggered(Qt::MouseButtons,Qt::KeyboardModifiers)), SLOT(onPutNewHotStuff()));

    KMenu* menu = new KMenu();
    menu->addAction(m_upload);
    ui.kGetNewHotStuff->setMenu(menu);

    //Refresh
    fillTemplateList();
    dataModified("", 0);
}

SKGMonthlyPluginWidget::~SKGMonthlyPluginWidget()
{
    SKGTRACEIN(1, "SKGMonthlyPluginWidget::~SKGMonthlyPluginWidget");
}

QString SKGMonthlyPluginWidget::getState()
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::getState");
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);

    //Get state
    root.setAttribute("month", ui.kMonth->text());
    root.setAttribute("template", ui.kTemplate->text());
    root.setAttribute("web", ui.kWebView->getState());

    return doc.toString();
}

void SKGMonthlyPluginWidget::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::setState");
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    //Set state
    QString month = root.attribute("month");
    if (!month.isEmpty()) ui.kMonth->setText(month);

    QString webS = root.attribute("web");
    if (!webS.isEmpty()) ui.kWebView->setState(webS);

    QString templat = root.attribute("template");
    if (!templat.isEmpty()) {
        bool p = ui.kTemplate->blockSignals(true);
        ui.kTemplate->setText(templat);
        ui.kTemplate->blockSignals(p);
        onTemplateChanged();
    }
}

QString SKGMonthlyPluginWidget::getDefaultStateAttribute()
{
    return "SKGMONTHLY_DEFAULT_PARAMETERS";
}

QWidget* SKGMonthlyPluginWidget::mainWidget() const
{
    return ui.kWebView;
}

void SKGMonthlyPluginWidget::fillTemplateList()
{
    disconnect(ui.kTemplate, SIGNAL(currentIndexChanged(int)), this, SLOT(onMonthChanged()));
    //Get previous selected item
    QString current = ui.kTemplate->text();

    //Fill
    ui.kTemplate->clear();
    foreach(const QString & file, KStandardDirs().findAllResources("data", QString::fromLatin1("skrooge/html/*.txt"))) {
        QFileInfo f(file);
        QString file2 = f.completeBaseName();
        if (!ui.kTemplate->contains(file2)) ui.kTemplate->addItem(file2, file);
    }

    //Set previous selected itemData
    if (!current.isEmpty() && ui.kTemplate->contains(current)) ui.kTemplate->setCurrentItem(current);
    connect(ui.kTemplate, SIGNAL(currentIndexChanged(int)), this, SLOT(onMonthChanged()), Qt::QueuedConnection);
}

void SKGMonthlyPluginWidget::onAddTemplate()
{
    QString templat = ui.kTemplate->text().trimmed();
    QString templatFileName = KStandardDirs::locateLocal("data", QString::fromLatin1("skrooge/html/")) % templat % ".txt";
    QStringList templates = KStandardDirs().findAllResources("data", QString::fromLatin1("skrooge/html/*.txt"));

    if (!templat.isEmpty() && (!templates.contains(templatFileName) || QFileInfo(templatFileName).isWritable())) {
        SKGError err;
        if (!templates.contains(templatFileName)) {
            //Create the new template
            QString source = KStandardDirs().findResource("data", QString::fromLatin1("skrooge/html/default.txt"));
            if (!KIO::NetAccess::file_copy(source, templatFileName)) {
                //Set error message
                err.setReturnCode(ERR_FAIL);
                err.setMessage(i18nc("An error message", "Impossible to copy file from '%1' to '%2'", source, templatFileName));
            } else {
                fillTemplateList();
            }
        }

        //Open the created or already existing file
        QDesktopServices::openUrl(KUrl(templatFileName));

        onTemplateChanged();

        //Display error
        SKGMainPanel::displayErrorMessage(err);
    }
}

void SKGMonthlyPluginWidget::onDeleteTemplate()
{
    QString templat = ui.kTemplate->text().trimmed();
    QString templatFileName = KStandardDirs::locateLocal("data", QString::fromLatin1("skrooge/html/")) % templat % ".txt";
    if (!templat.isEmpty()) {
        //This is a new source
        SKGError err;

        //Delete the file
        QFile file(templatFileName);
        if (!file.remove()) {
            err.setReturnCode(ERR_INVALIDARG);
            err.setMessage(i18nc("Error message",  "Deletion of '%1' failed", templatFileName));
        }

        if (!err) ui.kTemplate->removeItem(ui.kTemplate->findText(templat));

        //Display error
        SKGMainPanel::displayErrorMessage(err);
    }
}

void SKGMonthlyPluginWidget::onTemplateChanged()
{
    QString templat = ui.kTemplate->text().trimmed();
    QString templatFileName = KStandardDirs::locateLocal("data", QString::fromLatin1("skrooge/html/")) % templat % ".txt";
    bool local = !templat.isEmpty() && QFileInfo(templatFileName).isWritable();
    ui.kDeleteTemplate->setVisible(local);
    m_upload->setEnabled(local);
}

void SKGMonthlyPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
    SKGTRACEIN(1, "SKGMonthlyPluginWidget::dataModified");
    Q_UNUSED(iIdTransaction);

    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled(db != NULL);
    if (db != NULL && (iTableName == "v_operation_display" || iTableName.isEmpty())) {
        //Fill combo
        disconnect(ui.kMonth, SIGNAL(currentIndexChanged(int)), this, SLOT(onMonthChanged()));

        QStringList list;
        getDocument()->getDistinctValues("v_operation_display", "d_DATEMONTH", "d_date<=CURRENT_DATE", list);
        list.removeOne(QDate::currentDate().toString("yyyy-MM"));
        qSort(list.begin(), list.end(), qGreater<QString>());

        QString month = ui.kMonth->text();
        ui.kMonth->clear();
        ui.kMonth->addItem(i18nc("The month before the current month", "Last month"));
        ui.kMonth->addItems(list);
        if (!month.isEmpty()) ui.kMonth->setText(month);
        ui.kRefresh->setEnabled(list.count() > 0);
        connect(ui.kMonth, SIGNAL(currentIndexChanged(int)), this, SLOT(onMonthChanged()), Qt::QueuedConnection);

        onMonthChanged();
    }
}

QString SKGMonthlyPluginWidget::getMonth()
{
    QString month = m_cache["getMonth"].toString();
    if (month.isEmpty()) {
        if (ui.kMonth->currentIndex() == 0) {
            //Last month
            if (ui.kMonth->count() > 1) month = ui.kMonth->itemText(1);
        } else {
            //Specific month
            month = ui.kMonth->text();
        }
        m_cache["getMonth"] = month;
    }
    return month;
}

QString SKGMonthlyPluginWidget::getPreviousMonth()
{
    QString previousmonth = m_cache["getPreviousMonth"].toString();
    if (previousmonth.isEmpty()) {
        QString month = getMonth();
        if (!month.isEmpty()) {
            QDate date = QDate::fromString(month, "yyyy-MM");
            QDate date1 = date.addDays(-1);
            previousmonth = date1.toString("yyyy-MM");
        }
        m_cache["getPreviousMonth"] = previousmonth;
    }
    return previousmonth;
}

QVariantList SKGMonthlyPluginWidget::getBudgetTable()
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::getBudgetTable");
    QVariantList table = m_cache["getBudgetTable"].toList();
    if (table.count() == 0) {
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
        table = doc ? doc->getBudget(getMonth()) : QVariantList();
        m_cache["getBudgetTable"] = table;
    }
    return table;
}

QVariantList SKGMonthlyPluginWidget::getUnitTable()
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::getUnitTable");
    QVariantList table = m_cache["getUnitTable"].toList();
    if (table.count() == 0) {
        QString month = getMonth();
        if (!month.isEmpty()) {
            QDate date = QDate::fromString(month, "yyyy-MM");
            QDate date1 = date.addDays(-1);
            QDate date2 = date.addMonths(1).addDays(-1);
            QString previousmonth = date1.toString("yyyy-MM");
            SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
            KLocale* locale = KGlobal::locale();
            if (doc && locale) {
                SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();

                SKGObjectBase::SKGListSKGObjectBase units;
                SKGError err = doc->getObjects("v_unit_display", "1=1 ORDER BY t_TYPENLS", units);
                if (units.count()) {
                    QVariantList line;
                    line << "sum" << doc->getDisplay("t_UNIT") << locale->formatDate(date1, KLocale::ShortDate) << locale->formatDate(date2, KLocale::ShortDate) << "%";
                    table << QVariant(line);
                    foreach(const SKGUnitObject & unit, units) {
                        double v1 = unit.getAmount(date1);
                        double v2 = unit.getAmount(date2);
                        QVariantList line;
                        line << false << unit.getName() << v1 << v2 << (100.0 * (v2 - v1) / qAbs(v1));
                        table << QVariant(line);
                    }
                }
            }
        }
        m_cache["getUnitTable"] = table;
    }

    return table;
}

QVariantList SKGMonthlyPluginWidget::getAccountTable()
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::getAccountTable");
    QVariantList table = m_cache["getAccountTable"].toList();
    if (table.count() == 0) {
        QString month = getMonth();
        if (!month.isEmpty()) {
            QDate date = QDate::fromString(month, "yyyy-MM");
            QDate date1 = date.addDays(-1);
            QDate date2 = date.addMonths(1).addDays(-1);
            QDate date3 = date2.addYears(-1);
            QString previousmonth = date1.toString("yyyy-MM");
            SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
            KLocale* locale = KGlobal::locale();
            if (doc && locale) {
                SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();

                SKGObjectBase::SKGListSKGObjectBase accounts;
                SKGError err = doc->getObjects("v_account_display", "1=1 ORDER BY t_TYPENLS", accounts);
                if (!err) {
                    QVariantList line;
                    line << "sum" << doc->getDisplay("t_ACCOUNT") << locale->formatDate(date1, KLocale::ShortDate) << locale->formatDate(date2, KLocale::ShortDate) <<
                         "%" << locale->formatDate(date3, KLocale::ShortDate) << locale->formatDate(date2, KLocale::ShortDate) << "%";
                    table.push_back(line);
                    double sumTypeV1 = 0;
                    double sumTypeV2 = 0;
                    double sumTypeV3 = 0;
                    double sumV1 = 0;
                    double sumV2 = 0;
                    double sumV3 = 0;
                    QString currentType;
                    int nb = accounts.count();
                    for (int i = 0; !err && i < nb; ++i) {
                        SKGAccountObject account = accounts.at(i);
                        double v1 = account.getAmount(date1);
                        double v2 = account.getAmount(date2);
                        double v3 = account.getAmount(date3);
                        QString type = account.getAttribute("t_TYPENLS");
                        bool closed = account.isClosed();
                        if (type != currentType) {
                            if (!currentType.isEmpty()) {
                                QVariantList line;
                                line << true << i18nc("Noun",  "Total of %1", currentType) << sumTypeV1 << sumTypeV2 <<
                                     (100.0 * (sumTypeV2 - sumTypeV1) / qAbs(sumTypeV1)) << sumTypeV3 << sumTypeV2 << (100.0 * (sumTypeV2 - sumTypeV3) / qAbs(sumTypeV3));
                                table.push_back(line);
                                sumTypeV1 = 0;
                                sumTypeV2 = 0;
                                sumTypeV3 = 0;
                            }
                            currentType = type;
                        }
                        if (!closed || qAbs(v1) > 0.01 || qAbs(v2) > 0.01 || qAbs(v3) > 0.01) {
                            QVariantList line;
                            line << false << account.getName() << v1 << v2 << (100.0 * (v2 - v1) / qAbs(v1)) << v3 << v2 << (100.0 * (v2 - v3) / qAbs(v3));
                            table.push_back(line);
                            ;
                        }
                        sumTypeV1 += v1;
                        sumTypeV2 += v2;
                        sumTypeV3 += v3;
                        sumV1 += v1;
                        sumV2 += v2;
                        sumV3 += v3;
                    }

                    {
                        QVariantList line;
                        line << true << i18nc("Noun",  "Total of %1", currentType) << sumTypeV1 << sumTypeV2 << (100.0 * (sumTypeV2 - sumTypeV1) / qAbs(sumTypeV1)) << sumTypeV3 << sumTypeV2 << (100.0 * (sumTypeV2 - sumTypeV3) / qAbs(sumTypeV3));
                        table.push_back(line);
                    }

                    {
                        QVariantList line;
                        line << true << i18nc("Noun, the numerical total of a sum of values", "Total") << sumV1 << sumV2 << (100.0 * (sumV2 - sumV1) / qAbs(sumV1)) << sumV3 << sumV2 << (100.0 * (sumV2 - sumV3) / qAbs(sumV3));
                        table.push_back(line);
                    }
                }
            }
        }
        m_cache["getAccountTable"] = table;
    }
    return table;
}

QVariantList SKGMonthlyPluginWidget::get5MainCategoriesMonth()
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::get5MainCategoriesMonth");
    QVariantList table = m_cache["get5MainCategoriesMonth"].toList();
    if (table.count() == 0) {
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
        table = doc ? doc->get5MainCategories(getMonth()) : QVariantList();
        m_cache["get5MainCategoriesMonth"] = table;
    }
    return table;
}

QVariantList SKGMonthlyPluginWidget::get5MainCategoriesPreviousMonth()
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::get5MainCategoriesPreviousMonth");
    QVariantList table = m_cache["get5MainCategoriesPreviousMonth"].toList();
    if (table.count() == 0) {
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
        table = doc ? doc->get5MainCategories(getPreviousMonth()) : QVariantList();
        m_cache["get5MainCategoriesPreviousMonth"] = table;
    }
    return table;
}

QStringList SKGMonthlyPluginWidget::get5MainCategoriesVariation()
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::get5MainCategoriesVariation");
    QStringList table = m_cache["get5MainCategoriesVariation"].toStringList();
    if (table.count() == 0) {
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
        table = doc ? doc->get5MainCategoriesVariationList(getMonth(), getPreviousMonth()) : QStringList();
        m_cache["get5MainCategoriesVariation"] = table;
    }
    return table;
}

QStringList SKGMonthlyPluginWidget::getTitles()
{
    QStringList output;
    output.push_back(i18nc("A monthly report title", "Report for %1", getMonth()));
    output.push_back(i18nc("A monthly report title", "Budget"));
    output.push_back(i18nc("A monthly report title", "5 main categories of expenditure"));
    output.push_back(i18nc("A monthly report title", "5 main variations"));
    output.push_back(i18nc("A monthly report title", "Amounts in accounts"));
    output.push_back(i18nc("A monthly report title", "Amounts of units"));
    output.push_back(i18nc("A monthly report title", "Advices"));

    return output;
}

QVariant SKGMonthlyPluginWidget::getAdvices()
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::getAdvices");
    QVariant output = m_cache["getAdvices"];
    if (!output.isValid()) {
        output = qVariantFromValue(SKGMainPanel::getMainPanel()->getAdvices());
        m_cache["getAdvices"] = output;
    }
    return output;
}

QVariantList SKGMonthlyPluginWidget::getIncomeVsExpenditure()
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::getIncomeVsExpenditure");
    QVariantList table = m_cache["getIncomeVsExpenditure"].toList();
    if (table.count() == 0) {
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
        if (doc) {
            QString month = getMonth();
            QString previousmonth = getPreviousMonth();
            SKGStringListList listTmp;
            SKGError err = doc->executeSelectSqliteOrder(
                               "SELECT TOTAL(f_CURRENTAMOUNT), d_DATEMONTH  from v_operation_display WHERE d_DATEMONTH IN ('" % month % "', '" % previousmonth % "') AND t_TRANSFER='N' group by d_DATEMONTH, t_TYPEEXPENSE",
                               listTmp);
            int nbval = listTmp.count();
            if (!err && nbval > 1) {
                QVariantList line;
                line << "sum" << "" << previousmonth << month << "max";
                table.push_back(line);

                double income_previous_month = 0;
                double expense_previous_month = 0;
                double income_month = 0;
                double expense_month = 0;

                for (int i = 1; i < nbval; ++i) { //Ignore header
                    QString m = listTmp.at(i).at(1);
                    double v = SKGServices::stringToDouble(listTmp.at(i).at(0));
                    if (v > 0 && m == month) income_month = v;
                    else if (v < 0 && m == month) expense_month = v;
                    else if (v > 0 && m == previousmonth) income_previous_month = v;
                    else if (v < 0 && m == previousmonth) expense_previous_month = v;
                }
                double saving_previous_month = income_previous_month + expense_previous_month;
                double saving_month = income_month + expense_month;

                {
                    QVariantList line;
                    line << false << doc->getDisplay("f_CURRENTAMOUNT_INCOME") << qAbs(income_previous_month) << qAbs(income_month);
                    table.push_back(line);
                }
                {
                    QVariantList line;
                    line << false << doc->getDisplay("f_CURRENTAMOUNT_EXPENSE") << qAbs(expense_previous_month) << qAbs(expense_month);
                    table.push_back(line);
                }
                {
                    QVariantList line;
                    line << true << i18nc("Noun",  "Savings possible") << saving_previous_month << saving_month;
                    table.push_back(line);
                }
                {
                    QVariantList line;
                    line << true << i18nc("Noun",  "Max") << qMax(qAbs(income_previous_month), qAbs(expense_previous_month)) << qMax(qAbs(income_month), qAbs(expense_month));
                    table.push_back(line);
                }
            }
        }
    }
    return table;
}

void SKGMonthlyPluginWidget::onMonthChanged()
{
    SKGTRACEIN(1, "SKGMonthlyPluginWidget::onMonthChanged");
    m_cache.clear();
    QString month = getMonth();
    if (!month.isEmpty()) {
        //Display report
        QString htmlReport = getDocument()->getParameter("SKG_MONTHLY_REPORT_" % month);
        if (htmlReport.isEmpty() || sender() == ui.kRefresh || sender() == ui.kTemplate) {
            SKGError err;
            SKGBEGINLIGHTTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Compute monthly report for '%1'", month), err);
            htmlReport = getReport();
            err = getDocument()->setParameter("SKG_MONTHLY_REPORT_" % month, htmlReport);
        }

        //Display html report
        ui.kWebView->setHtml(htmlReport);
    }
}

void SKGMonthlyPluginWidget::onGetNewHotStuff()
{
    QPointer<KNS3::DownloadDialog> dialog = new KNS3::DownloadDialog("skrooge_monthly.knsrc", this);
    dialog->exec();
    delete dialog;

    fillTemplateList();
}

void SKGMonthlyPluginWidget::onPutNewHotStuff()
{
    QString templat = ui.kTemplate->text().trimmed();

    //Create zip file
    QString templatFileName = KStandardDirs::locateLocal("data", QString::fromLatin1("skrooge/html/")) % templat % ".txt";
    QString templatHFileName = KStandardDirs::locateLocal("data", QString::fromLatin1("skrooge/html/")) % templat % ".html";
    QString zipFileName = QDir::tempPath() % "/" % templat % ".zip";
    KZip zip(zipFileName);
    if (zip.open(QIODevice::WriteOnly)) {
        zip.addLocalFile(templatFileName, templat % ".txt");
        if (QFile(templatHFileName).exists()) zip.addLocalFile(templatHFileName, templat % ".html"); //To support Skrooge 1.0.0 version
        zip.close();

        //Create screen shots
        QWebFrame* frame = ui.kWebView->page()->mainFrame();
        QString preview2 = QDir::tempPath() % "/" % templat % "_preview2.png";
        frame->setScrollBarValue(Qt::Vertical, frame->scrollBarMaximum(Qt::Vertical));
        ui.kWebView->exportInFile(preview2);

        QString preview3 = QDir::tempPath() % "/" % templat % "_preview3.png";
        frame->setScrollBarValue(Qt::Vertical, frame->scrollBarMaximum(Qt::Vertical) / 2);
        ui.kWebView->exportInFile(preview3);

        QString preview1 = QDir::tempPath() % "/" % templat % "_preview1.png";
        frame->setScrollBarValue(Qt::Vertical, 0);
        ui.kWebView->exportInFile(preview1);

        //Open dialog
        QPointer<KNS3::UploadDialog> dialog = new KNS3::UploadDialog("skrooge_monthly.knsrc", this);

        dialog->setUploadFile(zipFileName);
        dialog->setUploadName(templat);
#ifdef SKGKDE46ENABLED
        dialog->setPreviewImageFile(0, KUrl(preview1));
        dialog->setPreviewImageFile(1, KUrl(preview2));
        dialog->setPreviewImageFile(2, KUrl(preview3));
#endif
        dialog->setDescription(i18nc("Default description for a monthly report template", "My favorite template"));
        dialog->exec();
        delete dialog;

        //Delete temporary files
        QFile(zipFileName).remove();
        QFile(preview1).remove();
        QFile(preview2).remove();
        QFile(preview3).remove();
    }
}

QString SKGMonthlyPluginWidget::getReport()
{
    QString html;
    SKGError err;
    SKGTRACEINRC(10, "SKGMonthlyPluginWidget::getReport", err);
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    //Get month
    QString month = getMonth();
    if (!month.isEmpty()) {
        QString previousmonth = getPreviousMonth();
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
        KLocale* locale = KGlobal::locale();
        if (doc && locale) {
            SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();

            //Prepare grantlee engine
            Grantlee::Engine gEngine(this);
            gEngine.addDefaultLibrary("grantlee_skroogefilters");

            Grantlee::FileSystemTemplateLoader::Ptr gLoader = Grantlee::FileSystemTemplateLoader::Ptr(new Grantlee::FileSystemTemplateLoader());
            gLoader->setTemplateDirs(QStringList(QFileInfo(ui.kTemplate->itemData(ui.kTemplate->currentIndex()).toString()).dir().absolutePath()));
            gEngine.addTemplateLoader(gLoader);

            Grantlee::Template gTemplate = gEngine.loadByName(ui.kTemplate->text() % ".txt");

            QVariantHash mapping;
            mapping.insert("current_date", QDate::currentDate());
            mapping.insert("report", QVariant::fromValue<QObject*>(this));
            mapping.insert("document", QVariant::fromValue<QObject*>(doc));

            KColorScheme scheme(QPalette::Normal);
            mapping.insert("color_negativetext", scheme.foreground(KColorScheme::NegativeText).color().name().right(6));
            mapping.insert("color_positivetext", scheme.foreground(KColorScheme::PositiveText).color().name().right(6));
            mapping.insert("color_neutraltext", scheme.foreground(KColorScheme::NeutralText).color().name().right(6));
            mapping.insert("color_normaltext", scheme.foreground(KColorScheme::NormalText).color().name().right(6));
            mapping.insert("color_inactivetext", scheme.foreground(KColorScheme::InactiveText).color().name().right(6));
            mapping.insert("color_activetext", scheme.foreground(KColorScheme::ActiveText).color().name().right(6));
            mapping.insert("color_linktext", scheme.foreground(KColorScheme::LinkText).color().name().right(6));
            mapping.insert("color_visitedtext", scheme.foreground(KColorScheme::VisitedText).color().name().right(6));
            mapping.insert("color_normalbackground", scheme.background(KColorScheme::NormalBackground).color().name().right(6));
            mapping.insert("color_activebackground", scheme.background(KColorScheme::ActiveBackground).color().name().right(6));

            Grantlee::Context gContext(mapping);

            //Generation
            {
                SKGTRACEINRC(10, "SKGMonthlyPluginWidget::getReport-grantlee render", err);
                m_cache.clear();
                html += gTemplate->render(&gContext);
                if (gTemplate->error()) err = SKGError(gTemplate->error(), gTemplate->errorString());
            }
        }
    }
    QApplication::restoreOverrideCursor();

    //status bar
    if (err) html += err.getFullMessageWithHistorical();
    SKGTRACEL(1) << html << endl;

    return html;
}

#include "skgmonthlypluginwidget.moc"
