/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A plugin to select all
 *
 * @author Stephane MANKOWSKI
 */
#include "skgselectallplugin.h"

#include <kactioncollection.h>
#include <kstandardaction.h>
#include <kaboutdata.h>
#include <kpluginfactory.h>
#include <klocalizedstring.h>

#include <qabstractitemview.h>

#include "skgmainpanel.h"
#include "skgtreeview.h"
#include "skgtraces.h"
/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGSelectAllPluginFactory, registerPlugin<SKGSelectAllPlugin>();)

SKGSelectAllPlugin::SKGSelectAllPlugin(QWidget* iWidget, QObject* iParent, const QVariantList& /*iArg*/) :
    SKGInterfacePlugin(iParent), m_currentDocument(nullptr), m_selectionMessage(nullptr)
{
    Q_UNUSED(iWidget);
    SKGTRACEINFUNC(10);

    connect(SKGMainPanel::getMainPanel(), &SKGMainPanel::selectionChanged, this, &SKGSelectAllPlugin::onSelectionChanged);
}

SKGSelectAllPlugin::~SKGSelectAllPlugin()
{
    SKGTRACEINFUNC(10);
    m_currentDocument = nullptr;
    if (m_selectionMessage) {
        delete m_selectionMessage;
        m_selectionMessage = nullptr;
    }
}

bool SKGSelectAllPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
    SKGTRACEINFUNC(10);
    Q_UNUSED(iArgument);

    m_currentDocument = iDocument;

    setComponentName(QStringLiteral("skg_selectall"), title());
    setXMLFile(QStringLiteral("skg_selectall.rc"));

    // Menu
    registerGlobalAction(QStringLiteral("edit_select_all"), actionCollection()->addAction(KStandardAction::SelectAll, QStringLiteral("edit_select_all"), this, SLOT(onSelectAll())), QStringList(), -1);
    return true;
}

QString SKGSelectAllPlugin::title() const
{
    return i18nc("Select all objects in a list", "Select all");
}

QString SKGSelectAllPlugin::icon() const
{
    return QStringLiteral("edit-select-all");
}

QString SKGSelectAllPlugin::toolTip() const
{
    return i18nc("Select all objects in a list", "Select all");
}

int SKGSelectAllPlugin::getOrder() const
{
    return 6;
}

QStringList SKGSelectAllPlugin::tips() const
{
    QStringList output;
    return output;
}

void SKGSelectAllPlugin::onSelectionChanged()
{
    SKGTRACEINFUNC(10);
    if (SKGMainPanel::getMainPanel()) {
        if (!m_selectionMessage) {
            // Add statusbar
            m_selectionMessage = new QLabel(SKGMainPanel::getMainPanel());
            SKGMainPanel::getMainPanel()->statusBar()->insertPermanentWidget(1, m_selectionMessage);
        }
        SKGObjectBase::SKGListSKGObjectBase selection = SKGMainPanel::getMainPanel()->getSelectedObjects();
        int nb = selection.count();
        if (nb) {
            double sum = 0.0;
            bool computed = false;
            for (int i = 0; i < nb; ++i) {
                SKGObjectBase obj(selection.at(i));
                QString val = obj.getAttribute(QStringLiteral("f_REALCURRENTAMOUNT"));
                if (val.isEmpty()) {
                    val = obj.getAttribute(QStringLiteral("f_CURRENTAMOUNT"));
                }
                if (!val.isEmpty()) {
                    sum += SKGServices::stringToDouble(val);
                    computed = true;
                }
            }
            m_selectionMessage->setText(computed ? i18np("Selection: %1 line for %2", "Selection: %1 lines for %2", nb, m_currentDocument->formatPrimaryMoney(sum)) : i18np("Selection: %1 line", "Selection: %1 lines", nb));
        } else {
            m_selectionMessage->clear();
        }
    }
}

void SKGSelectAllPlugin::onSelectAll()
{
    SKGTRACEINFUNC(10);
    if (SKGMainPanel::getMainPanel()) {
        SKGTabPage* page = SKGMainPanel::getMainPanel()->currentPage();
        if (page) {
            QAbstractItemView* view = qobject_cast<QAbstractItemView*>(page->mainWidget());
            if (view) {
                view->selectAll();
            }
        }
    }
}

#include <skgselectallplugin.moc>
