/*
 	sks.h
 
	Copyright (C) 2004-2006  Manuel Pancorbo Castro

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Manuel Pancorbo Castro <mpancorbo@gmail.com>
 
*/


#ifndef __SKS_H
#define __SKS_H

#define PROGRAM_NAME	"SKS"
#define COPYRIGHT \
 PROGRAM_NAME " v" VERSION " Copyright (C) 2004-2006  Manuel Pancorbo Castro\n"
 
#ifdef LINUX
#include <pwd.h>

#define HOME_VAR		"HOME"
#define KEYRING		"/.skspkr"
#define PRNGFILE	"/.sksprng"

#elif defined WIN32

#define HOME_VAR		"APPDATA"
#define KEY_DIR			"\\SKS"
#define KEYRING		"\\pubkeys.txt"
#define PRNGFILE	"\\prng.bin"

#else
#error "WIN32 or LINUX must be defined"
#endif /*? LINUX*/

#include "eccrypt.h"

#define HASH_SIZE 		GF_SIZE /* 32 */
#define BLOCK_SIZE		16
#define HMAC_SIZE		20 /* Less or equal than HASH_SIZE */
#define PASSWD_SIZE		0x100
#define KEY_SIZE		GF_SIZE

#define SMALL_BUF		0x100
#define MEDIUM_BUF		0x4000
#define LARGE_BUF		0x40000

#define MAX_ALLOC_SIZE  0x100000L /** 1 Mbyte **/

#define DEFAULT_COMP_LEVEL 9 /** From 1-9 **/

/** 
 **  Internal identifiers. No RCF, ISO, NIST recommendation at all
 **/
#define SHA256_CTR	(0x7F)		/** Actually not used **/
#define AES128_CTR  (0x71)
#define AES128_CTR2 (0x73)		/** As AES128_CTR but uses different keys
									for encrytion and HMAC **/
#define COMPRESS_FLAG (0x80)
#define BINARY_FLAG	(0x200)

#define PK_MODE			0
#define SYM_MODE		0x100


typedef struct {
	char lab[SMALL_BUF];
	PubKey val;
	unsigned long int id;
} keychain;

typedef struct{
	keychain *vector, *actual;
	int N;
} keyring;

typedef unsigned char hashout[HASH_SIZE];
typedef unsigned char byte;


/** Input functions **/
ERROR get_text(FILE *inp, char *frase, const char *msg);
ERROR get_privkey(FILE *inp, PrivKey p, const char *msg);

/** Hash functions **/
ERROR hash_binary(byte *buf, byte *bin, size_t len);
ERROR hash_binary_file( FILE *inp);
ERROR hash_password(char *, PrivKey);
ERROR hash_ascii_text( FILE * f_inp, FILE * f_out, int write);
ERROR hash_time (vlPoint c, time_t t);

/** Entropy function **/
int get_entropy(unsigned char *seed, size_t nbytes);

/** AES cipher functions **/
int aes_close(byte *, byte *, int);
ERROR aes_init(byte *, byte *, unsigned int, byte *);
ERROR aes_process(byte *, int, int);
ERROR sym_encrypt(FILE *, FILE *, PubKey, int);
ERROR sym_decrypt(FILE *, FILE *, PubKey, int);

/** Keyring functions **/
FILE * chkopen( char * s, char * mode );
void get_full_path(char * name, char * buf);
FILE * open_keyring(char * mode );
FILE * open_prngfile(void);
ERROR write_pubkey(FILE *, keychain *);
ERROR get_pubkey(FILE *, keychain *);
ERROR get_keyid(PubKey key, unsigned long *id);
ERROR new_key_chain(keyring *kr);
void close_keyring(keyring *kr);
ERROR add_to_keyring(keyring *kr, keychain *);
ERROR load_keyring(FILE *inp, keyring *kr);
ERROR write_keyring();
int search_pubkey(keyring *, int *, char **, FILE *, int);
ERROR import_keyring(FILE *inp);
ERROR delete_element(char *idstring);

/** Most utility functions **/
void write_longint(FILE *ptr, unsigned long int x, int binmode);
int read_longint(FILE *ptr, unsigned long int *x, int binmode);
#define write_timestamp(A, B) write_longint(A, (unsigned long int )B, 0)
#define read_timestamp(A, B) read_longint(A, (unsigned long int *)B, 0)
ERROR write_signature(FILE *ptr, sgPair *sig);
ERROR read_signature(FILE *ptr, sgPair *sig, int binmode);
int position(FILE  * ,/* FILE *,*/ const char *);
int str_match(char *s1, char *s2);
ERROR fputs_base36(PubKey a, FILE *out);
void burn_stack(unsigned long len);
void test_charset();
char * utf8_to_latin1( char *);
#define CHCONV(A) utf8_to_latin1(A)
#define zerokey(A)	memset(A, 0, KEY_SIZE)

enum {
	ES = 0,
	EN
};

#endif	/* ? _SKS_H */
