/*
 * The Sleuth Kit
 *
 * $Date: 2005/09/02 23:34:04 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2003-2005 Brian Carrier.  All rights reserved
 *
 * SUN - Sun VTOC code
 *
 *
 * This software is distributed under the Common Public License 1.0
 *
 */

#include "mm_tools.h"
#include "sun.h"


/*
 * Return a buffer with a description of the partition type
 */
static char *
sun_get_desc(uint8_t fstype)
{
    char *str = mymalloc(64);
    switch (fstype) {

    case 0:
	strncpy(str, "Unassigned (0x00)", 64);
	break;
    case 1:
	strncpy(str, "boot (0x01)", 64);
	break;
    case 2:
	strncpy(str, "/ (0x02)", 64);
	break;
    case 3:
	strncpy(str, "swap (0x03)", 64);
	break;
    case 4:
	strncpy(str, "/usr/ (0x04)", 64);
	break;
    case 5:
	strncpy(str, "backup (0x05)", 64);
	break;
    case 6:
	strncpy(str, "stand (0x06)", 64);
	break;
    case 7:
	strncpy(str, "/var/ (0x07)", 64);
	break;
    case 8:
	strncpy(str, "/home/ (0x08)", 64);
	break;
    case 9:
	strncpy(str, "alt sector (0x09)", 64);
	break;
    case 10:
	strncpy(str, "cachefs (0x0A)", 64);
	break;
    default:
	snprintf(str, 64, "Unknown Type (0x%.2x)", fstype);
	break;
    }

    return str;
}


/* 
 * Load an Intel disk label, this is called by sun_load_table 
 */

static uint8_t
sun_load_table_i386(MM_INFO * mm, sun_dlabel_i386 * dlabel_x86,
		    uint8_t test)
{
    uint32_t idx = 0;

    if (verbose)
	fprintf(stderr, "load_table_i386: Number of partitions: %d\n",
		getu16(mm, dlabel_x86->num_parts));

    /* Cycle through the partitions, there are either 8 or 16 */
    for (idx = 0; idx < getu16(mm, dlabel_x86->num_parts); idx++) {

	if (verbose)
	    fprintf(stderr,
		    "load_table_i386: %d  Starting Sector: %lu  Size: %lu  Type: %d\n",
		    idx,
		    (ULONG) getu32(mm, dlabel_x86->part[idx].start_sec),
		    (ULONG) getu32(mm, dlabel_x86->part[idx].size_sec),
		    getu16(mm, dlabel_x86->part[idx].type));

	if (getu32(mm, dlabel_x86->part[idx].size_sec) == 0)
	    continue;

	/* Add the partition to the internal sorted list */
	mm_part_add(mm,
		    (DADDR_T) getu32(mm, dlabel_x86->part[idx].start_sec),
		    (DADDR_T) getu32(mm, dlabel_x86->part[idx].size_sec),
		    MM_TYPE_VOL,
		    sun_get_desc(getu16(mm, dlabel_x86->part[idx].type)),
		    -1, idx);
    }

    return 0;
}


/* load a sparc disk label, this is called from the general
 * sun_load_table
 */
static uint8_t
sun_load_table_sparc(MM_INFO * mm, sun_dlabel_sparc * dlabel_sp,
		     uint8_t test)
{
    uint32_t idx = 0;
    uint32_t cyl_conv;

    /* The value to convert cylinders to sectors */
    cyl_conv = getu16(mm, dlabel_sp->sec_per_tr) *
	getu16(mm, dlabel_sp->num_head);

    if (verbose)
	fprintf(stderr, "load_table_sparc: Number of partitions: %d\n",
		getu16(mm, dlabel_sp->num_parts));

    /* Cycle through the partitions, there are either 8 or 16 */
    for (idx = 0; idx < getu16(mm, dlabel_sp->num_parts); idx++) {
	uint32_t part_start = cyl_conv *
	    getu32(mm, dlabel_sp->part_layout[idx].start_cyl);

	uint32_t part_size = getu32(mm,
				    dlabel_sp->part_layout[idx].size_blk);

	if (verbose)
	    fprintf(stderr,
		    "load_table_sparc: %d  Starting Sector: %lu  Size: %lu  Type: %d\n",
		    idx,
		    (ULONG) part_start,
		    (ULONG) part_size,
		    getu16(mm, dlabel_sp->part_meta[idx].type));

	if (part_size == 0)
	    continue;

	/* Add the partition to the internal sorted list */
	mm_part_add(mm, (DADDR_T) part_start, (DADDR_T) part_size,
		    MM_TYPE_VOL,
		    sun_get_desc(getu16
				 (mm, dlabel_sp->part_meta[idx].type)), -1,
		    idx);
    }

    return 0;
}


/* 
 * Process the partition table at the sector address 
 *
 * This method just finds out if it is sparc or Intel and then
 * calls the appropriate method
 */
static uint8_t
sun_load_table(MM_INFO * mm, uint8_t test)
{
/* this will need to change if any of the disk label structures change */
#define LABEL_BUF_SIZE	512

    sun_dlabel_sparc *dlabel_sp;
    sun_dlabel_i386 *dlabel_x86;
    char buf[LABEL_BUF_SIZE];


    /* Sanity check in case label sizes change */
    if ((sizeof(*dlabel_sp) > LABEL_BUF_SIZE) ||
	(sizeof(*dlabel_x86) > LABEL_BUF_SIZE)) {
	error("sun_load_table: Buffer smaller than label sizes");
    }

    if (verbose)
	fprintf(stderr, "sun_load_table: Trying sector: %" PRIuDADDR "\n",
		mm->sect_offset);

    /* Try the given offset */
    if (mm_read_block_nobuf
	(mm, (char *) &buf, LABEL_BUF_SIZE,
	 mm->sect_offset) != LABEL_BUF_SIZE) {
	if (test) {
	    fprintf(stderr,
		    "Error reading SUN Disk Label in Sector: %" PRIuDADDR
		    "\n", mm->sect_offset);
	    return 1;
	}
	else {
	    error("Error reading SUN Disk Label in Sector: %" PRIuDADDR
		  "\n", mm->sect_offset);
	}
    }


    /* Check the magic value 
     * Both intel and sparc have the magic value in the same location
     *
     * We try both in case someone specifies the exact location of the
     * intel disk label.
     * */
    dlabel_sp = (sun_dlabel_sparc *) buf;
    dlabel_x86 = (sun_dlabel_i386 *) buf;
    if (guessu16(mm, dlabel_sp->magic, SUN_MAGIC) == 0) {

	if (getu32(mm, dlabel_sp->sanity) == SUN_SANITY) {
	    return sun_load_table_sparc(mm, dlabel_sp, test);
	}
	else if (getu32(mm, dlabel_x86->sanity) == SUN_SANITY) {
	    return sun_load_table_i386(mm, dlabel_x86, test);
	}
    }


    /* Now try the next sector, which is where the intel 
     * could be stored */
    mm->sect_offset++;

    if (verbose)
	fprintf(stderr, "sun_load_table: Trying sector: %" PRIuDADDR "\n",
		mm->sect_offset);

    if (mm_read_block_nobuf
	(mm, (char *) &buf, LABEL_BUF_SIZE,
	 mm->sect_offset) != LABEL_BUF_SIZE) {
	if (test) {
	    fprintf(stderr,
		    "Error reading SUN (Intel) Disk Label in Sector: %"
		    PRIuDADDR "\n", mm->sect_offset);
	    return 1;
	}
	else {
	    error("Error reading SUN (Intel) Disk Label in Sector: %"
		  PRIuDADDR "\n", mm->sect_offset);
	}

    }

    dlabel_x86 = (sun_dlabel_i386 *) buf;
    if (guessu16(mm, dlabel_x86->magic, SUN_MAGIC)) {
	if (test)
	    return 1;
	else
	    error("Invalid SUN (intel) partition table (Sector: %"
		  PRIuDADDR ") %x\n", mm->sect_offset, getu16(mm,
							      dlabel_sp->
							      magic));
    }

    if (getu32(mm, dlabel_x86->sanity) != SUN_SANITY) {
	if (test)
	    return 1;
	else
	    error("Invalid SUN (intel) sanity value (Sector: %" PRIuDADDR
		  ") %x\n", mm->sect_offset, getu16(mm, dlabel_sp->magic));
    }

    return sun_load_table_i386(mm, dlabel_x86, test);
}


/* 
 * Walk the partitions that have already been loaded during _open
 *
 */
void
sun_part_walk(MM_INFO * mm, PNUM_T start, PNUM_T last, int flags,
	      MM_PART_WALK_FN action, char *ptr)
{
    MM_PART *part;
    unsigned int cnt = 0;

    if (start < mm->first_part || start > mm->last_part)
	error("Invalid starting partition: %d", start);

    if (last < mm->first_part || last > mm->last_part)
	error("Invalid ending partition: %d", last);

    part = mm->part_list;
    while ((part != NULL) && (cnt <= last)) {

	if (cnt >= start)
	    action(mm, cnt, part, 0, ptr);

	part = part->next;
	cnt++;
    }

    return;
}


void
sun_close(MM_INFO * mm)
{
    mm_part_free(mm);
    free(mm);
}

MM_INFO *
sun_open(IMG_INFO * img_info, DADDR_T offset, uint8_t test)
{
    MM_INFO *mm = (MM_INFO *) mymalloc(sizeof(*mm));

    mm->img_info = img_info;
    mm->mmtype = MM_SUN;
    mm->str_type = "Sun VTOC";

    /* If an offset was given, then use that too */
    mm->sect_offset = offset + SUN_SPARC_PART_OFFSET;

    /* inititialize settings */
    mm->part_list = NULL;
    mm->first_part = mm->last_part = 0;
    mm->endian = 0;
    mm->dev_bsize = 512;
    mm->block_size = 512;

    /* Assign functions */
    mm->part_walk = sun_part_walk;
    mm->close = sun_close;

    /* Load the partitions into the sorted list */
    if (sun_load_table(mm, test)) {
	sun_close(mm);
	return NULL;
    }

    /* fill in the sorted list with the 'unknown' values */
    mm_part_unused(mm);

    return mm;
}
