/*
 * libewf file handling
 *
 * Copyright (c) 2006, Joachim Metz <forensics@hoffmannbv.nl>,
 * Hoffmann Investigations. All rights reserved.
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the creator, related organisations, nor the names of
 *   its contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 * - All advertising materials mentioning features or use of this software
 *   must acknowledge the contribution by people stated in the acknowledgements.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER, COMPANY AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "libewf_common.h"

#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>

#include "libewf_definitions.h"
#include "libewf_endian.h"
#include "libewf_notify.h"
#include "libewf_md5.h"

#include "ewf_compress.h"
#include "ewf_crc.h"
#include "ewf_md5hash.h"
#include "ewf_file_header.h"
#include "ewf_header.h"
#include "ewf_section.h"
#include "ewf_volume.h"
#include "ewf_table.h"
#include "libewf_file.h"
#include "libewf_offset_table.h"
#include "libewf_read.h"
#include "libewf_section_list.h"
#include "libewf_segment_table.h"

/* Detects if a file is an EWF file
 * Returns 1 if true, 0 otherwise
 */
uint8_t libewf_check_file_signature( const char *filename )
{
	uint8_t signature[ 8 ];

	int file_descriptor = 0;
	int32_t count       = 0;

	if( filename == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_check_file_signature: invalid filename.\n" );

		return( 0 );
	}
	file_descriptor = libewf_common_open( filename, O_RDONLY, 0644 );

	if( file_descriptor < 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_check_file_signature: unable to open file: %s.\n", filename );

		return( 0 );
	}
	count = libewf_common_read( file_descriptor, signature, 8 );

	libewf_common_close( file_descriptor );

	if( count <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_check_file_signature: error reading signature from file: %s.\n", filename );

		return( 0 );
	}
	else if( count != 8 )
	{
		LIBEWF_WARNING_PRINT( "libewf_check_file_signature: unable to read signature from file: %s.\n", filename );

		return( 0 );
	}
	return( ewf_file_header_check_signature( signature ) );
}

/* Opens an EWF file
 * For reading files should contain all filenames that make up an EWF image
 * For writing files should contain the base of the filename, extentions like .e01 will be automatically added
 * Returns a pointer to the new instance of handle, NULL on error
 */
LIBEWF_HANDLE *libewf_open( const char **filenames, uint32_t file_amount, uint8_t flags )
{
	LIBEWF_HANDLE *handle   = NULL;
	LIBEWF_HANDLE *data_set = NULL;
	uint32_t iterator       = 0;

	if( flags == LIBEWF_OPEN_READ )
	{
		/* 1 additional entry required because
		 * entry [ 0 ] is not used for reading
		 */
		handle = libewf_handle_alloc( file_amount + 1 );

		if( handle == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_open: unable to create handle.\n" );

			return( NULL );
		}
		for( iterator = 0; iterator < file_amount; iterator++ )
		{
			data_set = libewf_open_read( handle, filenames[ iterator ] );

			if( data_set == NULL )
			{
				LIBEWF_WARNING_PRINT( "libewf_open: unable to open file(s).\n" );

				libewf_handle_free( handle );

				return( NULL );
			}
			handle = data_set;
		}
		data_set = libewf_build_index( handle );

		if( data_set == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_open: unable to create index.\n" );

			libewf_handle_free( handle );

			return( NULL );
		}
		handle = data_set;
	}
	else if( flags == LIBEWF_OPEN_WRITE )
	{
		/* Allocate 2 entries
		 * entry [ 0 ] is used for the base filename
		 */
		handle = libewf_handle_alloc( 2 );

		if( handle == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_open: unable to create handle.\n" );

			return( NULL );
		}
		data_set = libewf_open_write( handle, filenames[ 0 ] );

		if( data_set == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_open: unable to open file.\n" );

			libewf_handle_free( handle );

			return( NULL );
		}
		handle = data_set;
	}
	else
	{
		LIBEWF_WARNING_PRINT( "libewf_open: unsupported flags.\n" );

		return( NULL );
	}
	LIBEWF_VERBOSE_PRINT( "libewf_open: open successful.\n" );

	return( handle );
}

/* Opens an EWF file for reading
 * Returns a pointer to the instance of handle, NULL on error
 */
LIBEWF_HANDLE *libewf_open_read( LIBEWF_HANDLE *handle, const char *filename )
{
	LIBEWF_SEGMENT_TABLE* segment_table = NULL;
	EWF_FILE_HEADER *file_header        = NULL;
	uint16_t fields_segment             = 0;
	int file_descriptor                 = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_open_read: invalid handle.\n" );

		return( NULL );
	}
	file_descriptor = libewf_common_open( filename, O_RDONLY, 0644 );

	if( file_descriptor == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_open: unable to open file: %s.\n", filename );

		return( NULL );
	}
	file_header = ewf_file_header_read( file_descriptor );

	if( file_header == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_open_read: invalid file header in: %s.\n", filename );

		return( NULL );
	}
	fields_segment = libewf_endian_convert_16bit( file_header->fields_segment );

	LIBEWF_VERBOSE_PRINT( "libewf_open_read: added segment file: %s with file descriptor: %d with segment number: %" PRIu16 ".\n", filename, file_descriptor, fields_segment );

	segment_table = libewf_segment_table_set_values( handle->segment_table, fields_segment, filename, file_descriptor );

	if( segment_table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_open_read: unable to set value of segment table.\n" );

		ewf_file_header_free( file_header );

		return( NULL );
	}
	handle->segment_table = segment_table;

	ewf_file_header_free( file_header );

	LIBEWF_VERBOSE_PRINT( "libewf_open_read: open successful.\n" );

	return( handle );
}

/* Opens an EWF file for writing
 * Returns a pointer to the instance of handle, NULL on error
 */
LIBEWF_HANDLE *libewf_open_write( LIBEWF_HANDLE *handle, const char *filename )
{
	LIBEWF_SEGMENT_TABLE* segment_table = NULL;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_open_write: invalid handle.\n" );

		return( NULL );
	}
	segment_table = libewf_segment_table_set_values( handle->segment_table, 0, filename, -1 );

	if( segment_table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_open_write: unable to set value of segment table.\n" );

		return( NULL );
	}
	handle->segment_table = segment_table;

	LIBEWF_VERBOSE_PRINT( "libewf_open_write: open successful.\n" );

	return( handle );
}

/* Builds the index from the files
 * Returns a pointer to the instance of handle, NULL on error
 */
LIBEWF_HANDLE *libewf_build_index( LIBEWF_HANDLE *handle )
{
	EWF_SECTION *last_section = NULL;
	uint32_t segment          = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_build_index: invalid handle.\n" );

		return( NULL );
	}
	if( handle->index_build != 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_build_index: index has already been build.\n" );

		return( NULL );
	}
	for( segment = 1; segment < handle->segment_table->amount; segment++ )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_build_index: building index for segment: %" PRIu32 ".\n", segment );

		last_section = libewf_read_sections_from_segment( handle, segment );
	}
	/* Check to see if the done section has been found
	 */
	if( last_section == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_build_index: invalid last section.\n" );

		return( NULL );
	}
	else if( !ewf_section_is_type_done( last_section ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_build_index: unable to find the last segment file (done section).\n" );

		ewf_section_free( last_section );

		return( NULL );
	}
	ewf_section_free( last_section );

	LIBEWF_VERBOSE_PRINT( "libewf_build_index: index successful build.\n" );

	handle->index_build = 1;

	return( handle );
}

/* Closes the EWF handle
 */
void libewf_close( LIBEWF_HANDLE *handle )
{
	uint32_t iterator = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_close: invalid handle.\n" );

		return;
	}
	for( iterator = 0; iterator < handle->segment_table->amount; iterator++ )
	{
		if( handle->segment_table->file_descriptor[ iterator ] > 0 )
		{
			libewf_common_close( handle->segment_table->file_descriptor[ iterator ] );
		}
	}
	libewf_handle_free( handle );
}

/* Returns the size of the contained media data, or 0 on error
 */
uint64_t libewf_data_size( LIBEWF_HANDLE *handle )
{
	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_data_size: invalid handle.\n" );

		return( 0 );
	}
	return( libewf_handle_media_size( handle ) );
}

/* Returns a printable string of the stored md5 hash
 * Returns an NULL if no hash was stores in the file
 */
char *libewf_data_md5hash( LIBEWF_HANDLE *handle )
{
	LIBEWF_STRING *md5hash_string = NULL;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_data_md5hash: invalid handle.\n" );

		return( NULL );
	}
	if( handle->md5hash == NULL )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_data_md5hash: MD5 hash was not set.\n" );

		return( NULL );
	}
	md5hash_string = ewf_md5hash_to_string( handle->md5hash );

	if( md5hash_string == NULL )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_data_md5hash: unable to create MD5 hash string.\n" );

		return( NULL );
	}
	return( md5hash_string );
}

/* Calculates the MD5 hash and compares this with the MD5 hash stored in the image
 */
char *libewf_calculate_md5hash( LIBEWF_HANDLE *handle )
{
	LIBEWF_MD5_CTX md5;

	EWF_MD5HASH *calculated_md5hash          = NULL;
	LIBEWF_STRING *calculated_md5hash_string = NULL;
	uint8_t *data                            = NULL;
	uint64_t iterator                        = 0;
	uint64_t offset                          = 0;
	int64_t count                            = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_check_md5hash: invalid handle.\n" );

		return( NULL );
	}
	if( handle->index_build == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_check_md5hash: index was not build.\n" );

		return( NULL );
	}
	data = (uint8_t *) libewf_common_alloc( handle->chunk_size * sizeof( uint8_t ) );

	if( data == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_check_md5hash: unable to allocate data.\n" );

		return( NULL );
	}
	LIBEWF_MD5_INIT( &md5 );

	for( iterator = 0; iterator <= handle->offset_table->last; iterator++ )
	{
		offset = iterator * handle->chunk_size;
		count  = libewf_read_random( handle, data, handle->chunk_size, offset );

		if( count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_check_md5hash: unable to read chunk.\n" );

			libewf_common_free( data );

			return( NULL );
		}
		LIBEWF_MD5_UPDATE( &md5, data, count );
  	}
	libewf_common_free( data ) ;

	calculated_md5hash = ewf_md5hash_alloc();

	if( calculated_md5hash == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_check_md5hash: unable to create MD5 hash.\n" );

		return( NULL );
	}
  	LIBEWF_MD5_FINAL( calculated_md5hash, &md5 );

	calculated_md5hash_string = ewf_md5hash_to_string( calculated_md5hash );

	ewf_md5hash_free( calculated_md5hash );

	if( calculated_md5hash_string == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_check_md5hash: unable to create MD5 hash string.\n" );

		return( NULL );
	}
	return( (char *) calculated_md5hash_string );
}

