/*
 * libewf file writing
 *
 * Copyright (c) 2006, Joachim Metz <forensics@hoffmannbv.nl>,
 * Hoffmann Investigations. All rights reserved.
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the creator, related organisations, nor the names of
 *   its contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 * - All advertising materials mentioning features or use of this software
 *   must acknowledge the contribution by people stated in the acknowledgements.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER, COMPANY AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "libewf_common.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "ewf_compress.h"
#include "ewf_crc.h"
#include "ewf_data.h"
#include "ewf_file_header.h"
#include "ewf_header.h"

#include "libewf_chunk_cache.h"
#include "libewf_definitions.h"
#include "libewf_endian.h"
#include "libewf_notify.h"
#include "libewf_md5.h"
#include "libewf_offset_table.h"
#include "libewf_section.h"
#include "libewf_section_list.h"
#include "libewf_segment_table.h"
#include "libewf_write.h"

/* Create the header strings from the header values
 * Returns 1 on success, -1 on error
 */
int8_t libewf_write_create_headers( LIBEWF_HANDLE *handle, LIBEWF_HEADER_VALUES *header_values )
{
	time_t timestamp = time( NULL );

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_headers: invalid handle.\n" );

		return( -1 );
	}
	if( ( handle->format == LIBEWF_FORMAT_ENCASE1 ) || ( handle->format == LIBEWF_FORMAT_ENCASE2 ) || ( handle->format == LIBEWF_FORMAT_ENCASE3 ) )
	{
		handle->header = libewf_header_values_generate_header_string_encase3( header_values, timestamp, handle->compression_level );

		if( handle->header == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_create_headers: unable to create header values.\n" );

			return( -1 );
		}
	}
	else if( ( handle->format == LIBEWF_FORMAT_FTK ) || ( handle->format == LIBEWF_FORMAT_SMART ) )
	{
		handle->header = libewf_header_values_generate_header_string_ftk( header_values, timestamp, handle->compression_level );

		if( handle->header == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_create_headers: unable to create header values.\n" );

			return( -1 );
		}
	}
	else if( handle->format == LIBEWF_FORMAT_ENCASE4 )
	{
		handle->header = libewf_header_values_generate_header_string_encase4( header_values, timestamp );

		if( handle->header == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_create_headers: unable to create header values.\n" );

			return( -1 );
		}
		handle->header2 = libewf_header_values_generate_header2_string_encase4( header_values, timestamp );

		if( handle->header2 == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_create_headers: unable to create header2 values.\n" );

			ewf_header_free( handle->header );

			return( -1 );
		}
	}
	else if( handle->format == LIBEWF_FORMAT_ENCASE5 )
	{
		handle->header = libewf_header_values_generate_header_string_encase4( header_values, timestamp );

		if( handle->header == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_create_headers: unable to create header values.\n" );

			return( -1 );
		}
		handle->header2 = libewf_header_values_generate_header2_string_encase5( header_values, timestamp );

		if( handle->header2 == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_create_headers: unable to create header2 values.\n" );

			ewf_header_free( handle->header );

			return( -1 );
		}
	}
	return( 1 );
}

/* Sets file writing parameters
 * Returns a pointer to the instance, NULL on error
 */
LIBEWF_HANDLE *libewf_write_set_parameters( LIBEWF_HANDLE *handle, uint64_t input_file_size, uint32_t sectors_per_chunk, uint32_t bytes_per_sector, uint32_t error_granularity_sectors, uint64_t ewf_file_size, int8_t compression_level, uint8_t format, uint8_t read_error_retry, uint8_t wipe_block_on_read_error, uint8_t compress_empty_block )
{
	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_parameters: invalid handle.\n" );

		return( NULL );
	}
	if( input_file_size <= 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_parameters: invalid value for parameter: input_file_size.\n" );

		return( NULL );
	}
	if( sectors_per_chunk <= 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_parameters: invalid value for parameter: sectors_per_chunk.\n" );

		return( NULL );
	}
	if( bytes_per_sector <= 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_parameters: invalid value for parameter: bytes_per_sector.\n" );

		return( NULL );
	}
	if( error_granularity_sectors <= 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_parameters: invalid value for parameter: error_granularity_sectors.\n" );

		return( NULL );
	}
	if( ( ewf_file_size <= 0 ) || ( ewf_file_size > INT32_MAX ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_parameters: invalid value for parameter: ewf_file_size.\n" );

		return( NULL );
	}
	handle->input_file_size           = input_file_size;
	handle->sectors_per_chunk         = sectors_per_chunk;
	handle->bytes_per_sector          = bytes_per_sector;
	handle->error_granularity_sectors = error_granularity_sectors;
	handle->ewf_file_size             = ewf_file_size;
	handle->compression_level         = compression_level;
	handle->compress_empty_block      = compress_empty_block;
	handle->wipe_block_on_read_error  = wipe_block_on_read_error;
	handle->format                    = format;
	handle->read_error_retry          = read_error_retry;
	handle->chunk_size                = sectors_per_chunk * bytes_per_sector;

	LIBEWF_VERBOSE_PRINT( "libewf_write_set_parameters: input file size: %" PRIu64 ".\n", handle->input_file_size );
	LIBEWF_VERBOSE_PRINT( "libewf_write_set_parameters: requested ewf segment file size: %" PRIu64 ".\n", handle->ewf_file_size );

	return( handle );
}

/* Sets the header values to write
 * Returns a pointer to the instance, NULL on error
 */
LIBEWF_HANDLE *libewf_write_set_header_values( LIBEWF_HANDLE *handle, LIBEWF_HEADER_VALUES *header_values )
{
	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_header_values: invalid handle.\n" );

		return( NULL );
	}
	if( header_values == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_header_values: invalid value for parameter: header values.\n" );

		return( NULL );
	}
	if( handle->format == LIBEWF_FORMAT_UNKNOWN )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_header_values: file and header format has not been set.\n" );

		return( NULL );
	}
	if( handle->compression_level == EWF_COMPRESSION_UNKNOWN )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_header_values: compression level has not been set.\n" );

		return( NULL );
	}
	if( handle->header != NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_header_values: header already created.\n" );

		ewf_header_free( handle->header );
	}
	if( handle->header2 != NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_header_values: header2 already created.\n" );

		ewf_header_free( handle->header2 );
	}
	if( libewf_write_create_headers( handle, header_values ) == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_set_header_values: unable to create header(s).\n" );

		return( NULL );
	}
	return( handle );
}

/* Determines the amount of maximum supported segment files
 * Returns the amount, or -1 on error
 */
int32_t libewf_write_maximum_segment_files( LIBEWF_HANDLE *handle )
{
	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_maximum_segment_files: invalid handle.\n" );

		return( -1 );
	}
	if( handle->ewf_format == EWF_FORMAT_S01 )
	{
		/* = 4831
		 */
		return( ( ( 'z' - 's' ) * 26 * 26 ) + 99 );
	}
	else if( handle->ewf_format == EWF_FORMAT_E01 )
	{
		/* = 14295
		 */
		return( ( ( 'Z' - 'E' ) * 26 * 26 ) + 99 );
	}
	else
	{
		LIBEWF_WARNING_PRINT( "libewf_write_maximum_segment_files: unsupported EWF format.\n" );

		return( -1 );
	}
}

/* Determine the extension for a certain segment file
 * For EWF-E01, EWF-S01 segment file extension naming scheme
 * Returns 1 on success, -1 on error
 */
int8_t libewf_write_determine_segment_file_extension( LIBEWF_HANDLE *handle, char* extension, uint32_t segment )
{
	int32_t maximum_segment_files        = 0;
	char extension_first_character       = 0;
	char extension_additional_characters = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_determine_segment_file_extension: invalid handle.\n" );

		return( -1 );
	}
	if( extension == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_determine_segment_file_extension: invalid extension.\n" );

		return( -1 );
	}
	if( segment == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_determine_segment_file_extension: invalid segment 0.\n" );

		return( -1 );
	}
	maximum_segment_files = libewf_write_maximum_segment_files( handle );

	if( maximum_segment_files == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_determine_segment_file_extension: unable to determine the maximum amount of segment files.\n" );

		return( -1 );
	}
	if( segment > maximum_segment_files )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_determine_segment_file_extension: segment exceeds the maximum amount of segment files.\n" );

		return( -1 );
	}
	if( handle->ewf_format == EWF_FORMAT_S01 )
	{
		extension_first_character       = 's';
		extension_additional_characters = 'a';
	}
	else if( handle->ewf_format == EWF_FORMAT_E01 )
	{
		extension_first_character       = 'E';
		extension_additional_characters = 'A';
	}
	else
	{
		LIBEWF_WARNING_PRINT( "libewf_write_determine_segment_file_extension: unsupported EWF format.\n" );

		return( -1 );
	}
	extension[ 0 ] = extension_first_character;

	if( segment <= 99 )
	{
		extension[ 2 ] = '0' + ( segment % 10 );
		extension[ 1 ] = '0' + ( segment / 10 );
	}
	else if( segment >= 100 )
	{
		segment        -= 100;
		extension[ 2 ]  = extension_additional_characters + ( segment % 26 );
		segment        /= 26;
		extension[ 1 ]  = extension_additional_characters + ( segment % 26 );
		segment        /= 26;

		if( segment >= 26 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_determine_segment_file_extension: unable to support for more segment files.\n" );

			return( -1 );
		}
		extension[ 0 ] += segment;
	}
	/* Safety check
	 */
	if( ( extension[ 0 ] > 'z' ) || ( ( extension[ 0 ] > 'Z' ) && ( extension[ 0 ] < 'a' ) ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_determine_segment_file_extension: unable to support for more segment files.\n" );

		return( -1 );
	}
	extension[ 3 ] = '\0';

	return( 1 );
}

/* Creates the filename for a certain segment file
 * Returns the pointer to the filename, NULL on error
 */
char *libewf_write_create_segment_filename( LIBEWF_HANDLE *handle, char* basename, uint32_t segment )
{
	LIBEWF_STRING *filename = NULL;
	void *data_set          = NULL;
	uint32_t size           = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_filename: invalid handle.\n" );

		return( NULL );
	}
	if( basename == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_filename: invalid basename.\n" );

		return( NULL );
	}
	if( segment == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_filename: invalid segment 0.\n" );

		return( NULL );
	}
	size = libewf_common_strlen( basename );

	if( size == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_filename: an emtpy basename is not supported.\n" );

		return( NULL );
	}
	/* The actual filename also contain a . 3 character extension and a end of string byte
	 */
	filename = libewf_string_alloc( size + 5 );

	if( filename == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_filename: unable to allocate filename.\n" );

		return( NULL );
	}
	data_set = libewf_common_memcpy( (void *) filename, (void *) basename, size );

	if( data_set == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_filename: unable to copy basename.\n" );

		libewf_string_free( filename );

		return( NULL );
	}
	filename[ size ] = '.';

	libewf_write_determine_segment_file_extension( handle, &filename[ size + 1 ], segment );

	return( (char *) filename );
}

/* Creates a new segment file within the segment table
 * Returns 1 on success, -1 on error
 */
int8_t libewf_write_create_segment_file( LIBEWF_HANDLE *handle, uint32_t segment )
{
	LIBEWF_SEGMENT_TABLE *segment_table = NULL;
	char *filename                      = NULL;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_file: invalid handle.\n" );

		return( -1 );
	}
	if( segment == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_file: invalid segment 0.\n" );

		return( -1 );
	}
	/* Check if one additional entry in the segment table is needed
	 */
	if( handle->segment_table->amount <= segment )
	{
		/* Add one additional entry because the 0 entry is used for the basename
		 */
		segment_table = libewf_segment_table_realloc( handle->segment_table, ( segment + 1 ) );

		if( segment_table == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_create_segment_file: unable to reallocate segment table.\n" );

			return( -1 );
		}
		handle->segment_table = segment_table;

	}
	/* Check if the entry has already been filled
	 */
	else if( libewf_segment_table_values_is_set( handle->segment_table, segment ) == 1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_file: segment file has already been created.\n" );

		return( -1 );
	}
	filename = libewf_write_create_segment_filename( handle, handle->segment_table->filename[ 0 ], segment );

	if( filename == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_file: unable to create filename.\n" );

		return( -1 );
	}
	segment_table = libewf_segment_table_set_values( handle->segment_table, segment, filename, -1 );

	if( segment_table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_create_segment_file: unable to set value of segment table.\n" );

		libewf_common_free( filename );

		return( -1 );
	}
	handle->segment_table = segment_table;

	LIBEWF_VERBOSE_PRINT( "libewf_write_create_segment_file: segment file created: %" PRIu32 " with name: %s.\n", segment, filename );

	libewf_common_free( filename );

	return( 1 );
}

/* Write the headers to file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_write_headers( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t start_offset )
{
	int32_t write_count          = 0;
	int32_t total_count          = 0;
	uint32_t header_size         = 0;
	uint32_t header2_size        = 0;
	uint32_t segment_file_offset = start_offset;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_headers: invalid handle.\n" );

		return( -1 );
	}
	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_headers: invalid file descriptor.\n" );

		return( -1 );
	}
	if( handle->header == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_headers: invalid header.\n" );

		return( -1 );
	}
	header_size = ewf_header_length( handle->header );

	if( handle->format == LIBEWF_FORMAT_SMART )
	{
		/* The header should be written only once
		 * and using the compression used in the file
		 */
		write_count = libewf_section_header_write( handle, file_descriptor, segment_file_offset, handle->header, header_size, handle->compression_level );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_headers: unable to write single header section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_count         += write_count;
	}
	if( ( handle->format == LIBEWF_FORMAT_ENCASE1 ) || ( handle->format == LIBEWF_FORMAT_ENCASE2 ) || ( handle->format == LIBEWF_FORMAT_ENCASE3 ) || ( handle->format == LIBEWF_FORMAT_FTK ) )
	{
		/* The header should be written twice
		 * the default compression is used
		 */
		write_count = libewf_section_header_write( handle, file_descriptor, segment_file_offset, handle->header, header_size, EWF_COMPRESSION_DEFAULT );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_headers: unable to write first header section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_count         += write_count;

		write_count = libewf_section_header_write( handle, file_descriptor, segment_file_offset, handle->header, header_size, EWF_COMPRESSION_DEFAULT );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_headers: unable to write second header section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_count         += write_count;
	}
	else if( ( handle->format == LIBEWF_FORMAT_ENCASE4 ) || ( handle->format == LIBEWF_FORMAT_ENCASE5 ) )
	{
		if( handle->header2 == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_headers: invalid header2.\n" );

			return( -1 );
		}
		header2_size = ewf_header_length( handle->header2 );

		/* The header2 should be written twice
		 * the default compression is used
		 */
		write_count = libewf_section_header2_write( handle, file_descriptor, segment_file_offset, handle->header2, header2_size, EWF_COMPRESSION_DEFAULT );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_headers: unable to write first header2 section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_count         += write_count;

		write_count = libewf_section_header2_write( handle, file_descriptor, segment_file_offset, handle->header2, header2_size, EWF_COMPRESSION_DEFAULT );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_headers: unable to write second header2 section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_count         += write_count;

		/* The header should be written once
		 * the default compression is used
		 */
		write_count = libewf_section_header_write( handle, file_descriptor, segment_file_offset, handle->header, header_size, EWF_COMPRESSION_DEFAULT );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_headers: unable to write third header section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_count         += write_count;
	}
	return( total_count );
}

/* Write the necessary sections at the start of the segment file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_write_segment_file_start( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t segment )
{
	EWF_FILE_HEADER *file_header = NULL;
	int32_t total_write_count    = 0;
	int32_t write_count          = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_start: invalid handle.\n" );

		return( -1 );
	}
	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_start: invalid file descriptor.\n" );

		return( -1 );
	}
	if( segment == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_start: invalid segment 0.\n" );

		return( -1 );
	}
	file_header = ewf_file_header_alloc();

	if( file_header == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_start: unable to create file header.\n" );

		return( -1 );
	}
	libewf_endian_revert_16bit( segment, file_header->fields_segment );

	write_count = ewf_file_header_write( file_header, file_descriptor );

	if( write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_start: unable to write file header to file.\n" );

		ewf_file_header_free( file_header );

		return( -1 );
	}
	total_write_count += write_count;

	ewf_file_header_free( file_header );

	if( segment == 1 )
	{
		/* Write header section(s)
		 */
		write_count = libewf_write_headers( handle, file_descriptor, total_write_count );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_start: unable to write header sections.\n" );

			ewf_file_header_free( file_header );

			return( -1 );
		}
		total_write_count += write_count;

		if( handle->ewf_format == EWF_FORMAT_E01 )
		{
			/* Write volume section
			 */
			write_count = libewf_section_volume_e01_write( handle, file_descriptor, total_write_count );
		}
		else if( handle->ewf_format == EWF_FORMAT_S01 )
		{
			/* Write volume (SMART) section
			 */
			write_count = libewf_section_volume_s01_write( handle, file_descriptor, total_write_count );
		}
		else
		{
			/* Fail safe
			 */
			write_count = -1;
		}
		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_start: unable to write volume section.\n" );

			return( -1 );
		}
		total_write_count += write_count;
	}
	else if( handle->ewf_format == EWF_FORMAT_E01 )
	{
		/* Write data section
		 */
		write_count = libewf_section_data_write( handle, file_descriptor, total_write_count );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_start: unable to write data section.\n" );

			return( -1 );
		}
		total_write_count += write_count;
	}
	return( total_write_count );
}

/* Write the necessary sections before the actual data chunks to file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_write_segment_file_chunks_section_start( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t segment_file_offset, uint32_t chunk_amount, uint8_t section_number, EWF_TABLE_OFFSET *offsets )
{
	char *table_section_string = NULL;
	int32_t total_write_count  = 0;
	int32_t write_count        = 0;
	uint32_t section_size      = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_start: invalid handle.\n" );

		return( -1 );
	}
	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_start: invalid file descriptor.\n" );

		return( -1 );
	}
	if( handle->ewf_format == EWF_FORMAT_E01 )
	{
		if( section_number > 1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_start: no more than 1 sectors section supported per segment file.\n" );

			return( -1 );
		}
		section_size = chunk_amount * ( handle->chunk_size + EWF_CRC_SIZE );

		/* Write sectors section start
		 */
		write_count = libewf_section_start_write( handle, file_descriptor, "sectors", section_size, segment_file_offset );

		if( write_count == -1 )
		{
			LIBEWF_VERBOSE_PRINT( "libewf_write_segment_file_chunks_section_start: unable to write sectors section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_write_count   += write_count;
	}
	else if( handle->ewf_format == EWF_FORMAT_S01 )
	{
		if( section_number == 1 )
		{
			table_section_string = "table";
		}
		else if( section_number == 2 )
		{
			table_section_string = "table2";
		}
		else
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_start: no more than 2 table sections supported per segment file.\n" );

			return( -1 );
		}
		if( offsets == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_start: invalid offsets.\n" );

			return( -1 );
		}
		/* Write table or table2 section start
		 */
		write_count = libewf_section_table_write( handle, file_descriptor, segment_file_offset, offsets, chunk_amount, table_section_string, 0 );

		if( write_count == -1 )
		{
			LIBEWF_VERBOSE_PRINT( "libewf_write_segment_file_chunks_section_start: unable to write %s section.\n", table_section_string );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_write_count   += write_count;
	}
	return( total_write_count );
}

/* Correct the sections before the actual data chunks
 * Also write the necessary sections after the actual data chunks to file (like table and table2 sections for EWF-E01 format)
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_write_segment_file_chunks_section_correction( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t segment_file_offset, uint32_t chunk_amount, uint8_t section_number, EWF_TABLE_OFFSET *offsets, uint32_t chunks_section_offset, uint32_t chunks_section_size )
{
	char *table_section_string = NULL;
	int32_t total_write_count  = 0;
	int32_t write_count        = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_correction: invalid handle.\n" );

		return( -1 );
	}
	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_correction: invalid file descriptor.\n" );

		return( -1 );
	}
	if( offsets == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_correction: invalid offsets.\n" );

		return( -1 );
	}
	/* Seek the start of the data chunks
	*/
	LIBEWF_VERBOSE_PRINT( "libewf_write_segment_file_chunks_section_correction: setting file descriptor to start of chunks section offset: %" PRIu32 ".\n", chunks_section_offset );

	if( libewf_common_lseek( file_descriptor, chunks_section_offset, SEEK_SET ) == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_correction: unable to find offset to correct sectors size.\n" );

		return( -1 );
	}
	if( handle->ewf_format == EWF_FORMAT_E01 )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_write_segment_file_chunks_section_correction: correcting sectors section size: %" PRIu32 " offset: %" PRIu32 ".\n", chunks_section_size, chunks_section_offset );

		/* Rewrite sectors section start
		 */
		write_count = libewf_section_start_write( handle, file_descriptor, "sectors", chunks_section_size, chunks_section_offset );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_correction: unable to rewrite sectors section.\n" );

			return( -1 );
		}
	}
	else if( handle->ewf_format == EWF_FORMAT_S01 )
	{
		if( section_number == 1 )
		{
			table_section_string = "table";
		}
		else if( section_number == 2 )
		{
			table_section_string = "table2";
		}
		else
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_correction: no more than 2 table sections supported per segment file.\n" );

			return( -1 );
		}
		LIBEWF_VERBOSE_PRINT( "libewf_write_segment_file_chunks_section_correction: correcting %s section size: %" PRIu32 " offset: %" PRIu32 ".\n", table_section_string, chunks_section_size, chunks_section_offset );

		/* Rewrite table or table2 section start
		 */
		libewf_section_table_write( handle, file_descriptor, chunks_section_offset, offsets, chunk_amount, table_section_string, chunks_section_size );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_correction: unable to rewrite %s section.\n", table_section_string );

			return( -1 );
		}
	}
	/* Seek the end of the chunks section
	*/
	LIBEWF_VERBOSE_PRINT( "libewf_write_segment_file_chunks_section_correction: setting file descriptor back to end of data at offset: %" PRIu32 ".\n", segment_file_offset );

	if( libewf_common_lseek( file_descriptor, segment_file_offset, SEEK_SET ) == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_correction: unable to find offset to continue.\n" );

		return( -1 );
	}
	if( handle->ewf_format == EWF_FORMAT_E01 )
	{
		/* Write table section start
		 */
		write_count = libewf_section_table_write( handle, file_descriptor, segment_file_offset, offsets, chunk_amount, "table", 0 );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_correction: unable to rewrite table section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_write_count   += write_count;

		/* Write table2 section start
		 */
		write_count = libewf_section_table_write( handle, file_descriptor, segment_file_offset, offsets, chunk_amount, "table2", 0 );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_chunks_section_correction: unable to rewrite table2 section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_write_count   += write_count;
	}
	return( total_write_count );
}

/* Write the necessary sections at the end of the segment file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_write_segment_file_end( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t segment_file_offset, uint8_t last_segment_file )
{
	int32_t total_write_count = 0;
	int32_t write_count       = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_end: invalid handle.\n" );

		return( -1 );
	}
	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_segment_file_end: invalid file descriptor.\n" );

		return( -1 );
	}
	if( last_segment_file == 0 )
	{
		/* Write next section
		 */
		write_count = libewf_section_last_write( handle, file_descriptor, "next", segment_file_offset );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_end: unable to write next section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_write_count   += write_count;
	}
	else
	{
		if( handle->md5hash != NULL )
		{
			/* Write the hash section
			 */
			write_count = libewf_section_hash_write( handle, file_descriptor, segment_file_offset, handle->md5hash );

			if( write_count == -1 )
			{
				LIBEWF_WARNING_PRINT( "libewf_write_segment_file_end: unable to write hash section.\n" );

				return( -1 );
			}
			segment_file_offset += write_count;
			total_write_count   += write_count;
		}

		/* Write the done section
		 */
		write_count = libewf_section_last_write( handle, file_descriptor, "done", segment_file_offset );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_segment_file_end: unable to write done section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_write_count   += write_count;
	}
	return( total_write_count );
}

/* Writes data in EWF format from a file descriptor
 * Returns the amount of bytes written, or -1 on error
 */
int64_t libewf_write_from_file_descriptor( LIBEWF_HANDLE *handle, int input_file_descriptor, uint64_t write_size, uint64_t write_offset, void (*callback)( uint64_t bytes_read, uint64_t bytes_total ) )
{
	LIBEWF_MD5_CTX md5;

	EWF_TABLE_OFFSET *offsets                = NULL;
	EWF_CRC *zlib_crc                        = NULL;
	EWF_CRC *calculated_crc                  = NULL;
	EWF_ERROR2_SECTOR *error2_sectors        = NULL;
	LIBEWF_CHUNK_CACHE *chunk_cache          = NULL;
	LIBEWF_STRING *calculated_md5hash_string = NULL;
	void *data_set                           = NULL;

	int64_t total_write_count                = 0;
	int64_t total_read_count                 = 0;
	int64_t sectors_read_count               = 0;
	int64_t sectors_write_count              = 0;
	int64_t current_read_offset              = 0;
	int64_t read_error_count                 = 0;
	uint64_t sectors_size                    = 0;
	uint64_t additional_size                 = 0;
	int32_t read_count                       = 0;
	int32_t write_count                      = 0;
	int32_t total_read_error_count           = 0;
	int32_t maximum_segment_amount           = 0;
	uint32_t compressed_data_size            = 0;
	uint32_t maximum_compressed_data_size    = 0;
	uint32_t total_chunk_write_count         = 0;
	uint32_t segment                         = 1;
	uint32_t chunk_amount                    = 0;
	uint32_t chunks_per_file                 = 0;
	uint32_t bytes_to_read                   = 0;
	uint32_t read_error_offset               = 0;
	uint32_t error_granularity_bytes         = 0;
	uint32_t error_granularity_offset        = 0;
	uint32_t error_remaining_bytes           = 0;
	uint32_t read_remaining_bytes            = 0;
	uint32_t error_skip_bytes                = 0;
	uint32_t sectors_chunk_amount            = 0;
	uint32_t remaining_sectors_chunk_amount  = 0;
	uint32_t remaining_chunk_amount          = 0;
	uint32_t error2_sector                   = 0;
	uint32_t error2_sector_count             = 0;
	uint32_t chunks_section_offset           = 0;
	uint32_t segment_file_offset             = 0;
	uint32_t data_write_size                 = 0;
	uint8_t maximum_chunk_write_iterations   = 1;
	uint8_t chunks_section_iterator          = 0;
	int result                               = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: invalid handle.\n" );

		return( -1 );
	}
	if( input_file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: invalid file descriptor.\n" );

		return( -1 );
	}
	if( ( write_size == 0 ) || ( write_size > handle->input_file_size ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: invalid size to write.\n" );

		return( -1 );
	}
	if( write_offset >= handle->input_file_size )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: invalid offset to write.\n" );

		return( -1 );
	}
	if( ( write_size + write_offset ) > handle->input_file_size )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to acquire beyond size of media.\n" );

		return( -1 );
	}
	if( ( write_offset > 0 ) && ( libewf_common_lseek( input_file_descriptor, write_offset, SEEK_SET ) != write_offset ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to find write offset.\n" );

		return( -1 );
	}
	if( handle->chunk_cache == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: invalid chunk cache.\n" );

		return( -1 );
	}
	if( handle->chunk_size <= 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: invalid chunk size.\n" );

		return( -1 );
	}
	handle->chunk_count = write_size / handle->chunk_size;

	if( write_size % handle->chunk_size != 0 )
	{
		handle->chunk_count += 1;
	}
	handle->sector_count    = write_size / handle->bytes_per_sector;
	error_granularity_bytes = handle->error_granularity_sectors * handle->bytes_per_sector;
	chunks_per_file         = ( handle->ewf_file_size - EWF_FILE_HEADER_SIZE - EWF_DATA_SIZE ) / handle->chunk_size;

	if( handle->format == LIBEWF_FORMAT_SMART )
	{
		handle->ewf_format = EWF_FORMAT_S01;
	}
	else
	{
		handle->ewf_format = EWF_FORMAT_E01;
	}
	maximum_segment_amount = libewf_write_maximum_segment_files( handle );

	if( maximum_segment_amount == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to determine the maximum amount of allowed segment files.\n" );

		return( -1 );
	}
	if( ( write_size / handle->ewf_file_size ) > maximum_segment_amount )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: the settings exceed the maximum amount of allowed segment files.\n" );

		return( -1 );
	}
	/* Make sure the compressed data size buffer is large enough
	 * zlib compression can enlarge the data
	 * about 1024 bytes should be enough
	 */
	maximum_compressed_data_size = handle->chunk_size + 1024;

	chunk_cache = libewf_chunk_cache_realloc( handle->chunk_cache, maximum_compressed_data_size );

	if( chunk_cache == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to reallocate chunk cache.\n" );

		return( -1 );
	}
	handle->chunk_cache = chunk_cache;

	LIBEWF_MD5_INIT( &md5 );

	while( total_read_count < write_size )
	{
		/* Finish and close the previous segment file
		 */
		if( segment > 1 )
		{
			/* Write the end of the segment file
			 */
			write_count = libewf_write_segment_file_end( handle, handle->segment_table->file_descriptor[ segment - 1 ], segment_file_offset, 0 );

			if( write_count == -1 )
			{
				LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to write end of segment file.\n" );

				return( -1 );
			}
			total_write_count += write_count;

			libewf_common_close( handle->segment_table->file_descriptor[ segment - 1 ] );
		}
		/* Create a new segment file
		 */
		if( libewf_write_create_segment_file( handle, segment ) == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to create segment file for segment: %" PRIu32 ".\n", segment );

			return( -1 );
		}
		segment_file_offset = 0;

		handle->segment_table->file_descriptor[ segment ] = libewf_common_open( handle->segment_table->filename[ segment ], O_WRONLY | O_CREAT | O_TRUNC, 0644 );

		/* Write the start of the segment file
		 * like the file header, the header, volume and/or data section, etc.
		 */
		write_count = libewf_write_segment_file_start( handle, handle->segment_table->file_descriptor[ segment ], segment );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to write segment file start.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_write_count   += write_count;

		sectors_size  = handle->ewf_file_size;
		sectors_size -= segment_file_offset;

		/* Leave space for at least the sectors, table and table 2 and next or done sections
		 */
		sectors_size -= ( 4 * EWF_SECTION_SIZE ) + ( 2 * chunks_per_file * sizeof( EWF_TABLE_OFFSET ) );

		/* Determine how many chunks will fit in the remaining space
		 * This will be an estimate
		 */
		sectors_chunk_amount   = sectors_size / ( handle->chunk_size + EWF_CRC_SIZE );
		remaining_chunk_amount = handle->chunk_count - total_chunk_write_count;

		LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: calculated amount of chunks: %d.\n", sectors_chunk_amount );

		if( remaining_chunk_amount < chunks_per_file )
		{
			sectors_chunk_amount = remaining_chunk_amount;
		}
		/* Check if the maximum allowed amounts of chunks per segment file is not exceeded
		 * Only required for EWF-S01 format
		 */
		maximum_chunk_write_iterations = 1;

		if( ( handle->ewf_format == EWF_FORMAT_S01 ) && ( sectors_chunk_amount > 16375 ) )
		{
			remaining_sectors_chunk_amount = sectors_chunk_amount - 16375;
			sectors_chunk_amount           = 16375;
			maximum_chunk_write_iterations = 2;

			if( remaining_sectors_chunk_amount > 16375 )
			{
				remaining_sectors_chunk_amount = 16375;
			}
		}
		/* This loop is needed to write the table2 section for the EWF-S01 format
		 */
		for( chunks_section_iterator = 1; chunks_section_iterator <= maximum_chunk_write_iterations; chunks_section_iterator++ )
		{
			if( chunks_section_iterator == 2 )
			{
				sectors_chunk_amount = remaining_sectors_chunk_amount;
			}
			offsets = ewf_table_offsets_alloc( sectors_chunk_amount );

			if( offsets == NULL )
			{
				LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to create table offsets.\n" );

				return( -1 );
			}
			chunks_section_offset = segment_file_offset;

			sectors_read_count  = 0;
			sectors_write_count = 0;
			chunk_amount        = 0;

			/* Calculate additional space for the sectors, table and table 2 and next or done sections
			 */
			additional_size = ( 4 * EWF_SECTION_SIZE ) + ( 2 * chunks_per_file * sizeof( EWF_TABLE_OFFSET ) );

			/* Write the section start of the chunks section
			 */
			write_count = libewf_write_segment_file_chunks_section_start( handle, handle->segment_table->file_descriptor[ segment ], segment_file_offset, sectors_chunk_amount, chunks_section_iterator, offsets );

			if( write_count == -1 )
			{
				LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: unable to write section start for chunks.\n" );

				ewf_table_offsets_free( offsets );

				return( -1 );
			}
			segment_file_offset += write_count;
			total_write_count   += write_count;

			/* Check if another chunk would fit
			 */
			while( 1 )
			{
				/* Fill the segment file for EWF-E01
				 */
				if( handle->ewf_format == EWF_FORMAT_E01 )
				{
					if( ( segment_file_offset + handle->chunk_size + EWF_CRC_SIZE + additional_size ) >= handle->ewf_file_size )
					{
						break;
					}
				}
				/* Use the calculated amount of sectors for EWF-S01
				 */
				else if( handle->ewf_format == EWF_FORMAT_S01 )
				{
					if( chunk_amount >= sectors_chunk_amount )
					{
						break;
					}
				}
				/* Unsupported format
				 */
				else
				{
					LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: specified EWF format is not supported.\n" );

					ewf_table_offsets_free( offsets );

					return( -1 );
				}
				/* At the end of the input
				 */
				if( total_read_count == (int64_t) write_size )
				{
					break;
				}

				/* Make sure there is no data contamination whatsoever
				 */
				chunk_cache = libewf_chunk_cache_wipe( handle->chunk_cache );

				if( chunk_cache == NULL )
				{
					LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to wipe chunk cache.\n" );

					ewf_table_offsets_free( offsets );

					return( -1 );
				}
				handle->chunk_cache = chunk_cache;

				LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: reading chunk: %d with size: %" PRIu32 ".\n", chunk_amount, handle->chunk_size );

				bytes_to_read     = handle->chunk_size;
				read_error_offset = 0;
				read_error_count  = 0;

				while( 1 )
				{
					read_count = libewf_common_read( input_file_descriptor, &handle->chunk_cache->read[ read_error_offset ], bytes_to_read );

					LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: read chunk: %d with size: %" PRIi32 ".\n", chunk_amount, read_count );

					if( read_count <= -1 )
					{
#if HAVE_STRERROR
						LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: error reading data: %s.\n", strerror( errno ) );
#endif
						if( errno == ESPIPE )
						{
#if !HAVE_STRERROR
							LIBEWF_WARNING_PRINT( "libewf_read_chunk: error reading data: Invalid seek.\n" );
#endif
							ewf_table_offsets_free( offsets );

							return( -1 );
						}
						else if( errno == EPERM )
						{
#if !HAVE_STRERROR
							LIBEWF_WARNING_PRINT( "libewf_read_chunk: error reading data: Operation not permitted.\n" );
#endif
							ewf_table_offsets_free( offsets );

							return( -1 );
						}
						else if( errno == ENXIO )
						{
#if !HAVE_STRERROR
							LIBEWF_WARNING_PRINT( "libewf_read_chunk: error reading data: No such device or address.\n" );
#endif
							ewf_table_offsets_free( offsets );

							return( -1 );
						}
						else if( errno == ENODEV )
						{
#if !HAVE_STRERROR
							LIBEWF_WARNING_PRINT( "libewf_read_chunk: error reading data: No such device.\n" );
#endif
							ewf_table_offsets_free( offsets );

							return( -1 );
						}
						current_read_offset = libewf_common_lseek( input_file_descriptor, 0, SEEK_CUR );

						if( current_read_offset != ( total_read_count + read_error_offset ) )
						{
							LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: correcting offset drift current: %" PRIi64 ", calculated: %" PRIi64 ".\n", current_read_offset, ( total_read_count + read_error_offset ) );

							if( current_read_offset < ( total_read_count + read_error_offset ) )
							{
								LIBEWF_WARNING_PRINT( "libewf_read_chunk: unable to correct offset drift.\n" );

								ewf_table_offsets_free( offsets );

								return( -1 );
							}
							read_count         = (int32_t) ( current_read_offset - ( total_read_count + read_error_offset ) );
							read_error_offset += read_count;
							bytes_to_read     -= read_count;
						}
					}
					else
					{
						/* The last read is OK, correct read_count
						 */
						if( read_count == bytes_to_read )
						{
							read_count = read_error_offset + bytes_to_read;
						}
						/* The entire read is OK
						 */
						if( read_count == handle->chunk_size )
						{
							break;
						}
						/* At the end of the input
						 */
						if( ( total_read_count + read_count ) >= (int64_t) write_size )
						{
							break;
						}
						/* No bytes were read
						 */
						if( read_count == 0 )
						{
							LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: error reading data: unexpected end of data bytes read: %" PRIu64 " total bytes to read: %" PRIu64 ".\n", ( total_read_count + read_error_offset ), write_size );

							ewf_table_offsets_free( offsets );

							return( -1 );
						}
						/* There was a read error at a certain offset
						 */
						read_error_offset += read_count;
						bytes_to_read     -= read_count;

						LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: read error at offset %" PRIu64 ".\n", ( total_read_count + read_error_offset ) );
					}
					read_error_count++;

					if( read_error_count >= handle->read_error_retry )
					{
						if( handle->error2_sectors == NULL )
						{
							 error2_sectors = ewf_error2_sectors_alloc( ( total_read_error_count + 1 ) );
						}
						else
						{
							 error2_sectors = ewf_error2_sectors_realloc( handle->error2_sectors, total_read_error_count, ( total_read_error_count + 1 ) );
						}
						if( error2_sectors == NULL )
						{
							LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to create error2 sectors.\n" );

							ewf_table_offsets_free( offsets );

							return( -1 );
						}
						handle->error2_sectors = error2_sectors;

						total_read_error_count++;

						/* Check if last chunk is smaller than the chunk size and take corrective measures
						 */
						if( ( total_read_count + handle->chunk_size ) > write_size )
						{
							read_remaining_bytes = write_size - total_read_count;
						}
						else
						{
							read_remaining_bytes = handle->chunk_size;
						}
						error_remaining_bytes    = read_remaining_bytes - read_error_offset;
						error2_sector            = total_read_count;
						error_granularity_offset = ( read_error_offset / error_granularity_bytes ) * error_granularity_bytes;
						error_skip_bytes         = ( error_granularity_offset + error_granularity_bytes ) - read_error_offset;

						if( handle->wipe_block_on_read_error == 1 )
						{
							LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: wiping block of %" PRIu32 " bytes at offset %" PRIu32 ".\n", error_granularity_bytes, error_granularity_offset );

							data_set = libewf_common_memset( &handle->chunk_cache->read[ error_granularity_offset ], 0, error_granularity_bytes );

							if( data_set == NULL )
							{
								LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to wipe data in chunk on error.\n" );

								ewf_table_offsets_free( offsets );

								return( -1 );
							}
							error2_sector      += error_granularity_offset;
							error2_sector_count = error_granularity_bytes;
						}
						else
						{
							error2_sector      += read_error_offset;
							error2_sector_count = error_skip_bytes;
						}
						error2_sector       /= handle->bytes_per_sector;
						error2_sector_count /= handle->bytes_per_sector;

						LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: adding error2: %" PRIi32 " sector: %" PRIu32 ", count: %" PRIu32 ".\n", total_read_error_count, error2_sector, error2_sector_count );

						libewf_endian_revert_32bit( error2_sector, handle->error2_sectors[ total_read_error_count - 1 ].sector );
						libewf_endian_revert_32bit( error2_sector_count, handle->error2_sectors[ total_read_error_count - 1 ].sector_count );

						LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: skipping %" PRIu32 " bytes.\n", error_skip_bytes );

						/* At the end of the input
						 */
						if( ( total_read_count + read_remaining_bytes ) >= write_size )
						{
							LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: at end of input no remaining bytes to read from chunk.\n" );

							read_count = read_remaining_bytes;

							break;
						}
						if( libewf_common_lseek( input_file_descriptor, error_skip_bytes, SEEK_CUR ) == -1 )
						{
							LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable skip sector with error.\n" );

							ewf_table_offsets_free( offsets );

							return( -1 );
						}
						/* If error granularity skip is still within the chunk
						 */
						if( error_remaining_bytes > error_granularity_bytes )
						{
							bytes_to_read      = error_remaining_bytes - error_skip_bytes;
							read_error_offset += error_skip_bytes;

							LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: remaining to read from chunk %" PRIu32 " bytes.\n", bytes_to_read );
						}
						else
						{
							read_count = read_remaining_bytes;

							LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: no remaining bytes to read from chunk.\n" );
							break;
						}
					}
				}
#ifndef _TEST_
				if( read_count <= -1 )
				{
					LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable read chunk from file.\n" );

					ewf_table_offsets_free( offsets );

					return( -1 );
				}
#endif
				total_read_count += read_count;

				if( handle->swap_byte_pairs == 1 )
				{
					libewf_endian_swap_byte_pairs( handle->chunk_cache->read, read_count );
				}
				/* Callback for status update
				 */
				if( callback != NULL )
				{
					callback( total_read_count, write_size );
				}
				LIBEWF_MD5_UPDATE( &md5, handle->chunk_cache->read, read_count );

				sectors_read_count  += read_count;
				data_write_size      = 0;
				compressed_data_size = maximum_compressed_data_size;

				if( ( handle->ewf_format == EWF_FORMAT_S01 ) || ( handle->compression_level != EWF_COMPRESSION_NONE ) )
				{
					result = ewf_sectors_chunk_compress( handle->chunk_cache->data, &compressed_data_size, handle->chunk_cache->read, read_count, handle->compression_level );

					if( result != 1 )
					{
						LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to compress chunk: %d.\n", chunk_amount );

						ewf_table_offsets_free( offsets );

						return( -1 );
					}
				}
				else if( ( handle->compress_empty_block == 1 ) && ( libewf_common_test_empty_block( handle->chunk_cache->read, read_count ) == 1 ) )
				{
					result = ewf_sectors_chunk_compress( handle->chunk_cache->data, &compressed_data_size, handle->chunk_cache->read, read_count, EWF_COMPRESSION_DEFAULT );

					if( result != 1 )
					{
						LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to compress empty chunk: %d.\n", chunk_amount );

						ewf_table_offsets_free( offsets );

						return( -1 );
					}
				}
				if( ( handle->ewf_format == EWF_FORMAT_S01 ) || ( compressed_data_size < handle->chunk_size ) )
				{
					/* No additional CRC required, zlib creates its own CRC
					 */
					data_write_size = compressed_data_size;
					write_count     = ewf_sectors_chunk_write( handle->chunk_cache->data, handle->segment_table->file_descriptor[ segment ], data_write_size );

					zlib_crc = ewf_crc_alloc();

					if( zlib_crc == NULL )
					{
						LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to create CRC.\n" );

						ewf_table_offsets_free( offsets );

						return( -1 );
					}
					data_set = libewf_common_memcpy( (void *) zlib_crc, (void *) &handle->chunk_cache->data[ compressed_data_size - EWF_CRC_SIZE ], EWF_CRC_SIZE );

					if( data_set == NULL )
					{
						LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to set CRC.\n" );

						ewf_crc_free( zlib_crc );
						ewf_table_offsets_free( offsets );

						return( -1 );
					}
					LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: writing COMPRESSED chunk: %" PRIu32 " at offset: %" PRIu32 " with size: %" PRIu32 ", with CRC: %" PRIu32 ".\n", chunk_amount, segment_file_offset, data_write_size, zlib_crc );

					ewf_crc_free( zlib_crc );

					if( sectors_chunk_amount <= chunk_amount )
					{
						LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: enlarging offsets size: %" PRIu32 ", required: %" PRIu32 ".\n", sectors_chunk_amount, chunk_amount );

						offsets = ewf_table_offsets_realloc( offsets, sectors_chunk_amount, ( chunk_amount + 1 ) );

						if( offsets == NULL )
						{
							LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to reallocate offsets.\n" );

							ewf_table_offsets_free( offsets );
	
							return( -1 );
						}
						sectors_chunk_amount = chunk_amount + 1;
					}
					libewf_endian_revert_32bit( ( segment_file_offset | EWF_OFFSET_COMPRESSED_WRITE_MASK ), offsets[ chunk_amount ].offset );
				}
				else
				{
					data_write_size = read_count + EWF_CRC_SIZE;
					calculated_crc  = ewf_crc_calculate( (void *) handle->chunk_cache->read, read_count, 1 );

					if( calculated_crc == NULL )
					{
						LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: unable to calculate CRC.\n" );

						ewf_table_offsets_free( offsets );
	
						return( -1 );
					}
					libewf_endian_revert_32bit( *calculated_crc, &handle->chunk_cache->read[ read_count ] );

					LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: writing UNCOMPRESSED chunk: %" PRIu32 " at offset: %" PRIu32 " with size: %" PRIu32 ", with CRC: %" PRIu32 ".\n", chunk_amount, segment_file_offset, data_write_size, *calculated_crc );

					ewf_crc_free( calculated_crc );

					write_count = ewf_sectors_chunk_write( handle->chunk_cache->read, handle->segment_table->file_descriptor[ segment ], data_write_size );

					if( sectors_chunk_amount <= chunk_amount )
					{
						LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: enlarging offsets size: %" PRIu32 ", required: %" PRIu32 ".\n", sectors_chunk_amount, chunk_amount );

						offsets = ewf_table_offsets_realloc( offsets, sectors_chunk_amount, ( chunk_amount + 1 ) );

						if( offsets == NULL )
						{
							LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to reallocate offsets.\n" );

							ewf_table_offsets_free( offsets );
	
							return( -1 );
						}
						sectors_chunk_amount = chunk_amount + 1;
					}
					libewf_endian_revert_32bit( segment_file_offset, offsets[ chunk_amount ].offset );
				}

				if( write_count != data_write_size )
				{
					LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to write data.\n" );

					ewf_table_offsets_free( offsets );
	
					return( -1 );
				}
				segment_file_offset += write_count;
				total_write_count   += write_count;
				sectors_write_count += write_count;

				total_chunk_write_count++;
				chunk_amount++;
			}
			LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: written sectors size: %" PRIi64 ".\n", sectors_write_count );

			/* Correct the offset, size in the chunks section
			 */
			write_count = libewf_write_segment_file_chunks_section_correction( handle, handle->segment_table->file_descriptor[ segment ], segment_file_offset, sectors_chunk_amount, chunks_section_iterator, offsets, chunks_section_offset, sectors_write_count );

			if( write_count == -1 )
			{
				LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to correct chunks section.\n" );

				ewf_table_offsets_free( offsets );

				return( -1 );
			}
			segment_file_offset += write_count;
			total_write_count   += write_count;

			ewf_table_offsets_free( offsets );
		}
		segment++;
	}
	/* Complete the MD5 hash calculation
	 */
	if( handle->md5hash == NULL )
	{
		handle->md5hash = ewf_md5hash_alloc();

		if( handle->md5hash == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to create MD5 hash.\n" );

			return( -1 );
		}
	}
  	LIBEWF_MD5_FINAL( handle->md5hash, &md5 );

	/* Write the data section for a single segment file
	 * only for EWF-E01 and the segment count must be 2 (one segment file)
	 */
	if( ( handle->ewf_format == EWF_FORMAT_E01 ) && ( segment == 2 ) )
	{
		write_count = libewf_section_data_write( handle, handle->segment_table->file_descriptor[ segment - 1 ], segment_file_offset );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to write data section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_write_count   += write_count;
	}
	/* Write the error2 section if required 
	 */
	if( ( total_read_error_count > 0 ) && ( ( handle->format == LIBEWF_FORMAT_ENCASE3 ) || ( handle->format == LIBEWF_FORMAT_ENCASE4 ) || ( handle->format == LIBEWF_FORMAT_ENCASE5 ) ) )
	{
		write_count = libewf_section_error2_write( handle, handle->segment_table->file_descriptor[ segment - 1 ], segment_file_offset, handle->error2_sectors, total_read_error_count );

		if( write_count == -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to write error2 section.\n" );

			return( -1 );
		}
		segment_file_offset += write_count;
		total_write_count   += write_count;
	}

	/* Write the end of the segment file
	 * like the next, done and/or hash section
	 */
	write_count = libewf_write_segment_file_end( handle, handle->segment_table->file_descriptor[ segment - 1 ], segment_file_offset, 1 );

	if( write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_write_from_file_descriptor: unable to write end of segment file.\n" );

		return( -1 );
	}
	segment_file_offset += write_count;
	total_write_count   += write_count;

	libewf_common_close( handle->segment_table->file_descriptor[ segment - 1 ] );

	/* Print the MD5 hash
	 */
	calculated_md5hash_string = ewf_md5hash_to_string( handle->md5hash );

	if( calculated_md5hash_string == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: unable to create MD5 hash string.\n" );

		return( -1 );
	}
	LIBEWF_VERBOSE_PRINT( "libewf_write_from_file_descriptor: MD5 calculated: %s.\n", (char *) calculated_md5hash_string );

	libewf_string_free( calculated_md5hash_string );

	return( total_write_count );
}

