/*
    This file is part of the sly ident daemon (slidentd).  slidentd 
    was written by Sean Hunter <sean@uncarved.com> as a minimal 
    RFC1413 (ident) daemon.

    slidentd is copyright (c) 2001 Uncarved Systems Ltd.

    slidentd is free software; you can redistribute it and/or modify
    it under the terms of version 2 of the GNU General Public License 
    as published by the Free Software Foundation.

    slidentd is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    in the file COPYING along with slidentd; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "slid_datestamp.h"
#include "slid_config.h"
#include "slid_remote_info.h"
#include "slidentd.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <time.h>
#include <buffer.h>

void
slid_put_datestamp(buffer * buffer)
{
	stralloc datebuf = { 0 };
	slid_get_datestamp(&datebuf);
	buffer_put(buffer, datebuf.s, datebuf.len);
	stralloc_free(&datebuf);
}

/* create an ISO-8601 datestamp in UTC.  String must be destroyed by caller */
void
slid_get_datestamp(stralloc * buffer)
{
#if SLIDENTD_USE_SYSLOG != 0
	/*Rather than just call syslog(3) directly, I hand-roll a message header of 
	   the kind syslog expects and write directly to the syslog device.  I do this
	   to try to avoid format string problems */
	pid_t pid;
	time_t now;
	struct tm now_tm;
	char time_buf[20];
	int priority;

	pid = getpid();
	time(&now);
	strftime(time_buf, 20, "%h %e %T", localtime_r(&now, &now_tm));
	priority = LOG_MAKEPRI(LOG_FAC(SLIDENTD_SYSLOG_FACILITY),
			       LOG_PRI(SLIDENTD_SYSLOG_PRIORITY));
	stralloc_copys(buffer, "<");
	stralloc_catlong(buffer, (long) priority);
	stralloc_cats(buffer, ">");
	stralloc_cats(buffer, time_buf);
	stralloc_cats(buffer, " slidentd [");
	stralloc_catlong(buffer, (long) pid);
	stralloc_cats(buffer, "]: ");
#else
	struct timeval tv;
	struct tm *utctime;

	if (slid_running_tcpserver) {
		return;
	}
	gettimeofday(&tv, NULL);
	utctime = gmtime(&tv.tv_sec);

	/* tm_year is the number of years since 1900.  This is not a y2k bug- see gmtime(3) */
	utctime->tm_year += 1900;

	/* tm_mon is in the range 0 to 11 */
	utctime->tm_mon++;

	stralloc_readyplus(buffer, SLID_DATESTAMP_LEN);
	stralloc_catuint0(buffer, utctime->tm_year, 4);
	stralloc_cats(buffer, "-");
	stralloc_catuint0(buffer, utctime->tm_mon, 2);
	stralloc_cats(buffer, "-");
	stralloc_catuint0(buffer, utctime->tm_mday, 2);
	stralloc_cats(buffer, " ");
	stralloc_catuint0(buffer, utctime->tm_hour, 2);
	stralloc_cats(buffer, ":");
	stralloc_catuint0(buffer, utctime->tm_min, 2);
	stralloc_cats(buffer, ":");
	stralloc_catuint0(buffer, utctime->tm_sec, 2);
	stralloc_cats(buffer, ".");
	stralloc_catulong0(buffer, tv.tv_usec, 2);
#endif
}
