package SliMP3::Buttons::Playlist;

# $Id: Playlist.pm,v 1.72 2003/08/04 17:22:32 blackketter Exp $

# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License, 
# version 2.

use strict;
use File::Spec::Functions qw(:ALL);
use File::Spec::Functions qw(updir);
use SliMP3::Buttons::Common;
use SliMP3::Buttons::Browse;
use SliMP3::Misc;
use SliMP3::Strings qw(string);

# Each button on the remote has a function:

my %functions = (
	'playdisp' => sub {
		# toggle display mod for now playing...
		my $client = shift;
		my $button = shift;
		my $buttonarg = shift;
		my $pdm = SliMP3::Prefs::clientGet($client, "playingDisplayMode");
		unless (defined $pdm) { $pdm = 1; };
		unless (defined $buttonarg) { $buttonarg = 'toggle'; };
		if ($button eq 'playdisp_toggle') {
			my $playlistlen = SliMP3::Playlist::count($client);
			# playingDisplayModes are
			# 0 show nothing
			# 1 show elapsed time
			# 2 show remaining time
			# 3 show progress bar
			# 4 show elapsed time and progress bar
			# 5 show remaining time and progress bar
			if (($playlistlen > 0) && (showingNowPlaying($client))) {
				$pdm = ($pdm + 1) % 6;
			} elsif ($playlistlen > 0) {
				browseplaylistindex($client,SliMP3::Playlist::currentSongIndex($client));
			}
		} else {
			if ($buttonarg =~ /[0-5]$/) {
				$pdm = $buttonarg;
			}
		}
		SliMP3::Prefs::clientSet($client, "playingDisplayMode", $pdm);
		SliMP3::Display::update($client);
	},
	'up' => sub  {
		my $client = shift;
		my $button = shift;
		my $inc = shift || 1;
		my($songcount) = SliMP3::Playlist::count($client);
		if ($songcount < 2) {
			SliMP3::Animation::bumpUp($client);
		} else {
			$inc = ($inc =~ /\D/) ? -1 : -$inc;
			my $newposition = SliMP3::Buttons::Common::scroll($client, $inc, $songcount, browseplaylistindex($client));
			browseplaylistindex($client, $newposition);
			SliMP3::Display::update($client);
		}
	},
	'down' => sub  {
		my $client = shift;
		my $button = shift;
		my $inc = shift || 1;
		my($songcount) = SliMP3::Playlist::count($client);
		if ($songcount < 2) {
			SliMP3::Animation::bumpDown($client);
		} else {
			if ($inc =~ /\D/) {$inc = 1}
			my $newposition = SliMP3::Buttons::Common::scroll($client, $inc, $songcount, browseplaylistindex($client));
			browseplaylistindex($client,$newposition);
			SliMP3::Display::update($client);
		}
	},
	'left' => sub  {
		my $client = shift;
		my @oldlines = SliMP3::Display::curLines($client);
		SliMP3::Buttons::Home::jump($client, 'NOW_PLAYING');
		SliMP3::Buttons::Common::setMode($client, 'home');
		SliMP3::Animation::pushRight($client, @oldlines, SliMP3::Display::curLines($client));
	},
	'right' => sub  {
		my $client = shift;
		my $playlistlen = SliMP3::Playlist::count($client);
		if ($playlistlen < 1) {
			SliMP3::Animation::bumpRight($client);
		} else {
			my @oldlines = SliMP3::Display::curLines($client);
			SliMP3::Buttons::Common::pushMode($client, 'trackinfo', {'track' => SliMP3::Playlist::song($client, browseplaylistindex($client)) } );
			SliMP3::Animation::pushLeft($client, @oldlines, SliMP3::Display::curLines($client));
		}
	},
	'numberScroll' => sub  {
		my $client = shift;
		my $button = shift;
		my $digit = shift;
		my $newposition;
		# do an unsorted jump
		$newposition = SliMP3::Buttons::Common::numberScroll($client, $digit, SliMP3::Playlist::shuffleList($client), 0);
		browseplaylistindex($client,$newposition);
		SliMP3::Display::update($client);	
	},
	'add' => sub  {
		my $client = shift;
		if (SliMP3::Playlist::count($client) > 0) {
			# rec button deletes an entry if you are browsing the playlist...
			SliMP3::Animation::showBriefly($client, 
					string('REMOVING_FROM_PLAYLIST'), 
					SliMP3::Info::standardTitle($client, SliMP3::Playlist::song($client, browseplaylistindex($client))), undef, 1);
		
			SliMP3::Command::execute($client, ["playlist", "delete", browseplaylistindex($client)]);	
		}
	},
	
 	'zap' => sub {
 		my $client = shift;
 		my $zapped=catfile(SliMP3::Prefs::get('playlistdir'), string('ZAPPED_SONGS') . '.m3u');
 
 		my $currsong = SliMP3::Playlist::song($client);
 		my $currindex = SliMP3::Playlist::currentSongIndex($client);
 		$::d_stdio && msg("ZAP SONG!! $currsong, Index: $currindex\n");
 		
 		#Remove from current playlist
 		if (SliMP3::Playlist::count($client) > 0) {
 			# rec button deletes an entry if you are browsing the playlist...
 			SliMP3::Animation::showBriefly($client,
 					string('ZAPPING_FROM_PLAYLIST'),
 					SliMP3::Info::standardTitle($client, $currsong), undef, 1);
 			SliMP3::Command::execute($client, ["playlist", "delete", $currindex]);
 		}
 		# Append the zapped song to the zapped playlist
 		# This isn't as nice as it should be, but less work than loading and rewriting the whole list
 		my $zapref = new FileHandle $zapped, "a";
 		if (!$zapref) {
 			msg("Could not open $zapped for writing.\n");
 		}
 		my @zaplist = ($currsong);
 		my $zapitem = SliMP3::Parse::writeM3U(\@zaplist);
 		print $zapref $zapitem;
 		close $zapref;
 	},

	'play' => sub  {
		my $client = shift;
		if (showingNowPlaying($client)) {
			if (SliMP3::Playlist::playmode($client) eq 'pause') {
				SliMP3::Command::execute($client, ["pause"]);
			} elsif (SliMP3::Playlist::rate($client) != 1) {
				SliMP3::Command::execute($client, ["rate", 1]);
			} else {
				SliMP3::Command::execute($client, ["playlist", "jump", browseplaylistindex($client)]);
			}	
		} else {
			SliMP3::Command::execute($client, ["playlist", "jump", browseplaylistindex($client)]);
		}
		SliMP3::Display::update($client);
	}
);

sub getFunctions {
	return \%functions;
}

sub setMode {
	my $client = shift;
	my $how = shift;
	$client->lines(\&lines);
	if ($how ne 'pop') { jump($client); }
}

sub jump {
	my $client = shift;
	my $pos = shift;
	if (SliMP3::Buttons::Common::mode($client) eq 'playlist') {
		if (!defined($pos)) { 
			$pos = SliMP3::Playlist::currentSongIndex($client);
		}
		browseplaylistindex($client,$pos);
	}
}

#
# Display the playlist browser
#		
sub lines {
	my $client = shift;
	my ($line1, $line2, $overlay2);
	
	if (showingNowPlaying($client) || (SliMP3::Playlist::count($client) < 1)) {
		return currentSongLines($client);
	} else {
		$line1 = sprintf "%s (%d %s %d) ", string('PLAYLIST'), browseplaylistindex($client) + 1, string('OUT_OF'), SliMP3::Playlist::count($client);
		$line2 = SliMP3::Info::standardTitle($client, SliMP3::Playlist::song($client, browseplaylistindex($client)));
		$overlay2 = SliMP3::VFD::symbol('notesymbol');
		return ($line1, $line2, undef, $overlay2);
	}
}

sub showingNowPlaying {
	my $client = shift;
	return ((SliMP3::Buttons::Common::mode($client) eq 'playlist') && (browseplaylistindex($client) == SliMP3::Playlist::currentSongIndex($client)));
}

sub currentSongLines {
	my $client = shift;
	my ($line1, $line2, $overlay2);

	my $overlay1 = "";
	if ($::d_usage) {
		$overlay1 = $client->usage() ? " " . int($client->usage() * 10) : "";
		$client->usage() && msg(SliMP3::Client::id($client) . " Usage: ". int($client->usage() * 100) . "%\n");
	#	bt();  todo = use this to figure out why we call this function three times a second
	}
	
	my $playlistlen = SliMP3::Playlist::count($client);
	if ($playlistlen < 1) {
		$line1 = string('NOW_PLAYING');
		$line2 = string('NOTHING');
	} else {
		if (SliMP3::Playlist::playmode($client) eq "pause") {
			$line1 = sprintf string('PAUSED')." (%d %s %d) ", SliMP3::Playlist::currentSongIndex($client) + 1, string('OUT_OF'), $playlistlen;

		# for taking photos of the SliMP3 display, comment out the line above, and use this one instead.
		# this will cause the display to show the "Now playing" screen to show when paused.
		#	$line1 = "Now playing" . sprintf " (%d %s %d) ", SliMP3::Playlist::currentSongIndex($client) + 1, string('OUT_OF'), $playlistlen;

		} elsif (SliMP3::Playlist::playmode($client) eq "stop") {
			$line1 = sprintf string('STOPPED')." (%d %s %d) ", SliMP3::Playlist::currentSongIndex($client) + 1, string('OUT_OF'), $playlistlen;
		} else {
			if (SliMP3::Playlist::rate($client) != 1) {
				$line1 = string('NOW_SCANNING') . ' ' . SliMP3::Playlist::rate($client) . 'x';	
		#		$line1 = SliMP3::Playlist::rate($client) > 0 ? string('NOW_SCANNING') : string('NOW_REWINDING');
			} elsif (SliMP3::Playlist::shuffle($client)) {
				$line1 = string('PLAYING_RANDOMLY');
			} else {
				$line1 = string('PLAYING');
			}
				
			if (SliMP3::Prefs::clientGet($client, "volume") < 0) {
				$line1 .= " ".string('LCMUTED')
			}
			$line1 = $line1 . sprintf " (%d %s %d) ", SliMP3::Playlist::currentSongIndex($client) + 1, string('OUT_OF'), $playlistlen;
		}

		my $playingDisplayMode = SliMP3::Prefs::clientGet($client, "playingDisplayMode");
		my $fractioncomplete = 0;
		
		if (!defined($playingDisplayMode)) { $playingDisplayMode = 1; };
		
		# check if we're streaming...
		if (SliMP3::Info::isHTTPURL(SliMP3::Playlist::song($client))) {
		        # no progress bar, remaining time is meaningless
			$playingDisplayMode = ($playingDisplayMode % 3) ? 1 : 0;
		} else {
			if (SliMP3::Playlist::playmode($client) eq "stop") {
				$fractioncomplete = 0;		
			} else {
				my $length = SliMP3::Info::size(SliMP3::Playlist::song($client));
				my $realpos = SliMP3::Playlist::songRealPos($client);
	
				$fractioncomplete = $realpos / $length if ($length && $realpos);
			}
		}

		my $songtime = songTime($client, $playingDisplayMode);

		if ($playingDisplayMode == 1 || $playingDisplayMode == 2) {
			# just show the song time
			$line1 .= " " x 40;
			$line1 = SliMP3::VFD::subString($line1, 0, 40 - SliMP3::VFD::lineLength($songtime) - SliMP3::VFD::lineLength($overlay1)) . $songtime;
		} elsif ($playingDisplayMode == 3) {
			# just show the bar
			my $barlen = 40 - SliMP3::VFD::lineLength($line1) - SliMP3::VFD::lineLength($overlay1);
			my $bar = SliMP3::Display::progressBar($client, $barlen, $fractioncomplete);	
			$line1 .= " " x 40;
			$line1 = SliMP3::VFD::subString($line1, 0, 40 - $barlen - SliMP3::VFD::lineLength($overlay1)) . $bar;
		} elsif ($playingDisplayMode == 4 || $playingDisplayMode == 5) {
			# show both the bar and the time
			my $barlen = 40 - SliMP3::VFD::lineLength($line1) - SliMP3::VFD::lineLength($songtime) - 1 - SliMP3::VFD::lineLength($overlay1);
			my $bar = SliMP3::Display::progressBar($client, $barlen, $fractioncomplete);	
			$line1 .= " " x 40;
			$line1 = SliMP3::VFD::subString($line1, 0, 40 - SliMP3::VFD::lineLength($bar) - SliMP3::VFD::lineLength($songtime) - 1 - SliMP3::VFD::lineLength($overlay1))
					. $bar . " " . $songtime;
		} 

		$line2 = SliMP3::Info::standardTitle($client, SliMP3::Playlist::song($client));
		$overlay2 = SliMP3::VFD::symbol('notesymbol');
	}
	
	return ($line1, $line2, $overlay1, $overlay2);
}

sub songTime {
	my $client = shift;
	my $playingDisplayMode = shift;

	my ($time, $delta);	

	if (SliMP3::Playlist::playmode($client) eq "stop") {
		$delta = 0;
	} else {	
		$delta = SliMP3::Playlist::songTime($client);
	}
	
	# 2 and 5 display remaining time, not elapsed
	if ($playingDisplayMode % 3 == 2) {
		my $duration = SliMP3::Info::durationSeconds(SliMP3::Playlist::song($client)) || 0;
		$delta = $duration - $delta;
	}	    

	$time = sprintf("%s%02d:%02d", ($playingDisplayMode % 3 == 2 ? '-' : ''), $delta / 60, $delta % 60);

	return $time;	
}

sub browseplaylistindex {
	my $client = shift;
	my $playlistindex = shift;
	
	# get (and optionally set) the browseplaylistindex parameter that's kept in param stack
	return SliMP3::Buttons::Common::param($client, 'browseplaylistindex', $playlistindex);
}
1;

__END__
