-- See the Copyright notice at the end of this file.
--
class XML_DTD_ELEMENT

inherit
	BACKTRACKING_NODE
		redefine
			out_in_tagged_out_memory
		end

insert
	RECYCLABLE
		redefine
			out_in_tagged_out_memory
		end

creation {XML_DTD_VALIDATOR}
	make

feature {ANY}
	name: STRING

	out_in_tagged_out_memory is
		do
			tagged_out_memory.extend('<')
			tagged_out_memory.append(name)
			tagged_out_memory.append(once ": ")
			if structure = Void then
				tagged_out_memory.append(once "unknown")
			else
				structure.fill_tagged_out_memory
			end
			tagged_out_memory.extend('>')
		end

feature {XML_DTD_VALIDATOR}
	build is
			-- Prepare to build the element's structure
		do
			open_lists.clear_count
		end

feature {XML_DTD_VALIDATOR, XML_DTD_NODE}
	is_built: BOOLEAN

feature {XML_DTD_VALIDATOR} -- Data validation
	is_valid_attributes (xml_attributes: DICTIONARY[STRING, STRING]): BOOLEAN is
		local
			i: INTEGER
		do
			from
				Result := True
				i := attributes.lower
			until
				not Result or else i > attributes.upper
			loop
				Result := attributes.item(i).is_valid(xml_attributes)
				i := i + 1
			end
		end

feature {XML_DTD_NODE} -- Tree validation
	is_valid_child (explorer: XML_DTD_VALIDATOR; node_name: STRING; children: FAST_ARRAY[XML_DTD_NODE]): BOOLEAN is
		require
			explorer /= Void
			children /= Void
			is_built
		do
			Result := explorer.backtrack_is_valid(children, structure, node_name)
		end

	is_valid_data (explorer: XML_DTD_VALIDATOR; data: STRING; children: FAST_ARRAY[XML_DTD_NODE]): BOOLEAN is
		require
			explorer /= Void
			children /= Void
			is_built
		do
			Result := explorer.backtrack_valid_data(children, structure, data)
		end

feature {ANY} -- Tree structure validation
	explore (explorer: XML_DTD_VALIDATOR) is
		do
			explorer.backtrack_valid_child(Current)
		end

feature {} -- The element's structure
	structure: BACKTRACKING_NODE
			-- the element's structure

	open_lists: FAST_ARRAY[BACKTRACKING_NODE_AND_LIST] is
			-- used during construction
			-- a once array is enough since we build one element at a time
		once
			create Result.make(0)
		end

	current_list_root_memory: REFERENCE[BACKTRACKING_NODE_AND_LIST] is
		once
			create Result
		end

	current_list_root: BACKTRACKING_NODE_AND_LIST is
			-- the last link of the current list
		do
			Result := current_list_root_memory.item
		end

	set_current_list_root (a_current_list_root: like current_list_root) is
		do
			current_list_root_memory.set_item(a_current_list_root)
		ensure
			current_list_root = a_current_list_root
		end

	current_list_memory: REFERENCE[BACKTRACKING_NODE_AND_LIST] is
		once
			create Result
		end

	current_list: BACKTRACKING_NODE_AND_LIST is
			-- the last link of the current list
		do
			Result := current_list_memory.item
		end

	set_current_list (a_current_list: like current_list) is
		do
			current_list_memory.set_item(a_current_list)
		ensure
			current_list = a_current_list
		end

	current_alt_memory: REFERENCE[BACKTRACKING_NODE_OR_LIST] is
		once
			create Result
		end

	current_alt: BACKTRACKING_NODE_OR_LIST is
		do
			Result := current_alt_memory.item
		ensure
			Result /= Void implies current_list /= Void and then alt_is_last_of(current_list.node, Result)
		end

	set_current_alt (a_current_alt: like current_alt) is
		do
			current_alt_memory.set_item(a_current_alt)
		ensure
			current_alt = a_current_alt
		end

	alt_is_last_of (node: BACKTRACKING_NODE; alt: BACKTRACKING_NODE_OR_LIST): BOOLEAN is
		local
			r_alt: like alt
		do
			if (r_alt ?:= node) and then alt.next = Void then
				r_alt ::= node
				from
				until
					r_alt = Void or else r_alt = alt
				loop
					r_alt := r_alt.next
				end
				Result := r_alt /= Void
				check
					Result = (r_alt = alt) -- but the other formula is more efficient
				end
			end
		end

	node_in_current_list_or_alt: BACKTRACKING_NODE is
		do
			if current_alt /= Void then
				Result := current_alt.node
			else
				Result := current_list.node
			end
		end

	set_node_in_current_list_or_alt (new_node: BACKTRACKING_NODE) is
		do
			if current_alt /= Void then
				current_alt.set_node(new_node)
			else
				current_list.set_node(new_node)
			end
		end

feature {XML_DTD_VALIDATOR} -- Building element's structure
	list_depth: INTEGER is
		do
			Result := open_lists.count
		end

	open_list is
		do
			check
				list_depth = 0 implies current_list = Void
			end
			open_lists.add_last(current_list)
			set_current_list(Void)
			set_current_alt(Void)
		ensure
			list_depth = old list_depth + 1
			current_list = Void
		end

	close_list is
		require
			list_depth > 0
		local
			parent_list, list: like current_list
			parent_alt, alt: like current_alt
		do
			parent_list := open_lists.last
			if parent_list = Void and then list_depth = 1 then
				if structure = Void then
					structure := current_list
				else
					check
						predefined_nodes: structure = empty_node or else structure = any_node
					end
				end
				check
					structure /= Void
				end
				set_current_list(Void)
				set_current_alt(Void)
			else
				create list.set_node(current_list_root)
				if parent_list = Void then
					set_current_list_root(list)
				else
					parent_list.set_next(list)
					set_current_list_root(parent_list)
				end
				from
					-- find again the last added link
				until
					parent_list = Void
				loop
					list := parent_list
					parent_list := parent_list.next
				end
				check
					list /= Void
				end
				set_current_list(list)
				if alt ?:= list.node then
					from
						-- find again the last alternative node
						parent_alt ::= list.node
					until
						parent_alt = Void
					loop
						alt := parent_alt
						parent_alt := alt.next
					end
				end
				set_current_alt(alt)
			end
			open_lists.remove_last
		ensure
			list_depth = old list_depth - 1
			old current_list /= Void implies current_list /= old current_list
			list_depth = 0 implies structure /= Void
		end

	child (element: XML_DTD_ELEMENT) is
		local
			list: BACKTRACKING_NODE_AND_LIST
		do
			create list.set_node(element)
			if current_list = Void then
				set_current_list_root(list)
			else
				current_list.set_next(list)
			end
			set_current_list(list)
			set_current_alt(Void)
		end

	alternative_child (element: XML_DTD_ELEMENT) is
		local
			alt, p_alt: BACKTRACKING_NODE_OR_LIST
		do
			check
				current_list /= Void
			end
			create alt.set_node(element)
			if current_alt /= Void then
				current_alt.set_next(alt)
			else
				create p_alt.make(current_list.node, alt)
				current_list.set_node(p_alt)
			end
			set_current_alt(alt)
		end

	one_or_more is
		do
			if open_lists.is_empty then
				structure := node_and_end(one_or_more_node(structure))
			else
				set_node_in_current_list_or_alt(one_or_more_node(node_in_current_list_or_alt))
			end
		end

	zero_or_more is
		do
			if open_lists.is_empty then
				check
					structure /= Void
				end
				structure := node_and_end(zero_or_more_node(structure))
			else
				set_node_in_current_list_or_alt(zero_or_more_node(node_in_current_list_or_alt))
			end
		end

	zero_or_one is
		do
			if open_lists.is_empty then
				check
					structure /= Void
				end
				structure := node_and_end(zero_or_one_node(structure))
			else
				set_node_in_current_list_or_alt(zero_or_one_node(node_in_current_list_or_alt))
			end
		end

	exactly_one is
		do
			if open_lists.is_empty then
				check
					structure /= Void
				end
				structure := node_and_end(structure)
			end
		end

	pcdata (alternative: BOOLEAN) is
		local
			list: BACKTRACKING_NODE_AND_LIST
			alt, p_alt: BACKTRACKING_NODE_OR_LIST
		do
			--|*** TODO: refactor with child/alternative_child
			if alternative then
				check
					current_list /= Void
				end
				create alt.set_node(pcdata_node)
				if current_alt /= Void then
					current_alt.set_next(alt)
				else
					create p_alt.make(current_list.node, alt)
					current_list.set_node(p_alt)
				end
				set_current_alt(alt)
			else
				create list.set_node(pcdata_node)
				if current_list = Void then
					set_current_list_root(list)
				else
					current_list.set_next(list)
				end
				set_current_list(list)
				set_current_alt(Void)
			end
		end

	any is
		do
			structure := any_node
		end

	empty is
		do
			structure := empty_node
		end

	commit is
		require
			not is_built
		do
			check
				structure /= Void
			end
			is_built := True
		ensure
			is_built
		end

feature {} -- Multiplicity helpers
	one_or_more_node (a_node: BACKTRACKING_NODE): BACKTRACKING_NODE is
		local
			a: BACKTRACKING_NODE_AND_PAIR; b: BACKTRACKING_NODE_OR_TRUE
		do
			create b.make(a_node)
			create a.make(a_node, b)
			b.set_node(a)
			Result := a
		end

	zero_or_more_node (a_node: BACKTRACKING_NODE): BACKTRACKING_NODE is
		local
			c: BACKTRACKING_NODE_OR_TRUE; d: BACKTRACKING_NODE_AND_PAIR
		do
			create c.make(a_node)
			create d.make(a_node, c)
			c.set_node(d)
			Result := c
		end

	zero_or_one_node (a_node: BACKTRACKING_NODE): BACKTRACKING_NODE is
		do
			create {BACKTRACKING_NODE_OR_TRUE} Result.make(a_node)
		end

	node_and_end (a_node: BACKTRACKING_NODE): BACKTRACKING_NODE is
		do
			create {BACKTRACKING_NODE_AND_PAIR} Result.make(a_node, end_node)
		end

feature {} -- Special nodes
	any_node: XML_DTD_ANY_NODE is
		once
			create Result.make
		end

	empty_node: XML_DTD_EMPTY_NODE is
		once
			create Result.make
		end

	end_node: XML_DTD_END_NODE is
		once
			create Result.make
		end

	pcdata_node: XML_DTD_PCDATA_NODE is
		once
			create Result.make
		end

feature {XML_DTD_VALIDATOR} -- Adding attributes
	attributes: HASHED_DICTIONARY[XML_DTD_ATTRIBUTE, STRING]

	current_attribute: XML_DTD_ATTRIBUTE

	building_attlist: BOOLEAN is
		do
			Result := current_attribute /= Void
		end

	adding_attlist (attribute_name: STRING): BOOLEAN is
		do
			Result := current_attribute /= Void and then current_attribute.name.is_equal(attribute_name)
		ensure
			Result implies building_attlist
		end

	has_attlist (attribute_name: STRING): BOOLEAN is
		do
			Result := attributes.has(attribute_name)
		end

	add_attlist (attribute_name: STRING) is
		require
			not has_attlist(attribute_name)
			not building_attlist
		local
			att: XML_DTD_ATTRIBUTE
		do
			att := attributes_pool.item
			if att = Void then
				create att.make(attribute_name)
			else
				att.make(attribute_name)
			end
			current_attribute := att
		ensure
			building_attlist
		end

	commit_attlist (attribute_name: STRING) is
		require
			adding_attlist(attribute_name)
		do
			attributes.add(current_attribute, current_attribute.name)
			current_attribute := Void
		ensure
			not building_attlist
		end

	attlist_list_value (value: STRING) is
		require
			building_attlist
		do
			current_attribute.list_value(value)
		end

	attlist_cdata is
		require
			building_attlist
		do
			current_attribute.cdata
		end

	attlist_id is
		require
			building_attlist
		do
			current_attribute.id
		end

	attlist_idref is
		require
			building_attlist
		do
			current_attribute.idref
		end

	attlist_idrefs is
		require
			building_attlist
		do
			current_attribute.idrefs
		end

	attlist_nmtoken is
		require
			building_attlist
		do
			current_attribute.nmtoken
		end

	attlist_nmtokens is
		require
			building_attlist
		do
			current_attribute.nmtokens
		end

	attlist_entity is
		require
			building_attlist
		do
			current_attribute.entity
		end

	attlist_entities is
		require
			building_attlist
		do
			current_attribute.entities
		end

	attlist_notation is
		require
			building_attlist
		do
			current_attribute.notation
		end

	attlist_required is
		require
			building_attlist
		do
			current_attribute.required
		end

	attlist_implied is
		require
			building_attlist
		do
			current_attribute.implied
		end

	attlist_fixed (value: STRING) is
		require
			building_attlist
		do
			current_attribute.fixed(value)
		end

	attlist_default_value (value: STRING) is
		require
			building_attlist
		do
			current_attribute.default_value(value)
		end

feature {XML_DTD_VALIDATOR} -- Constructor
	make (a_name: like name) is
		require
			not a_name.is_empty
		do
			-- the element's proper attributes:
			name := a_name
			if attributes = Void then
				create attributes.make
			end
		ensure
			name = a_name
		end

feature {WEAK_ARRAY}
	recycle is
		do
			recycle_attributes
		end

feature {} -- Memory management
	attributes_pool: WEAK_ARRAY[XML_DTD_ATTRIBUTE] is
		once
			create Result.make
		end

	recycle_attributes is
		local
			i: INTEGER
		do
			from
				i := attributes.lower
			until
				i > attributes.upper
			loop
				attributes_pool.recycle(attributes.item(i))
				i := i + 1
			end
			attributes.clear_count
		ensure
			attributes.is_empty
		end

invariant
	not name.is_empty
	attributes /= Void

end -- class XML_DTD_ELEMENT
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
