-- See the Copyright notice at the end of this file.
--
class XML_DTD_PARSER
	--
	-- Parses a validation instruction in an XML file:
	--  <!DOCTYPE . . .>
	--

insert
	XML_PARSER_TOOLS
	XML_DTD_MEMORY
	SINGLETON

creation {XML_PARSER}
	make

feature {XML_PARSER} -- Error reporting
	has_error: BOOLEAN is
		do
			Result := error /= Void
		end

	error_message: STRING is
		require
			has_error
		do
			Result := once ""
			Result.copy(error)
		end

feature {XML_PARSER}
	parse (a_line: like line; a_column: like column; a_stream: like stream): XML_DTD_VALIDATOR is
		local
			id: STRING
		do
			error := Void
			stream := a_stream
			line := a_line
			column := a_column
			skip_blanks
			if end_of_input then
				set_error(Error_end_of_file)
			else
				id := read_identifier
				if end_of_input then
					set_error(Error_end_of_file)
				else
					Result := new_dtd_validator(id.twin)
					skip_blanks
					id := read_identifier
					skip_blanks
					if id /= Void then
						inspect
							id
						when "SYSTEM" then
							parse_system_dtd(Result)
							stream := a_stream -- because it was changed by `parse_system_dtd'
							if skip('[') then
								parse_inline_dtd(Result)
							end
						when "PUBLIC" then
							parse_public_dtd(Result)
							stream := a_stream -- because it was changed by `parse_public_dtd'
							if skip('[') then
								parse_inline_dtd(Result)
							end
						else
							set_error(once "Unknown dtd location")
						end
					elseif skip('[') then
						parse_inline_dtd(Result)
					else
						set_error(once "'[' expected")
					end
				end
			end
		end

feature {}
	parse_system_dtd (validator: XML_DTD_VALIDATOR) is
		local
			path, err: STRING; l, c: INTEGER
		do
			path := read_string
			skip_blanks
			if path /= Void and then skip('>') then
				dtd_url := once ""
				dtd_url.copy(once "file://")
				dtd_url.append(path)
				dtd_file.connect_to(path)
				if dtd_file.is_connected then
					l := line
					c := column
					line := 1
					column := 0
					stream := dtd_file
					parse_dtd(validator, True)
					line := l
					column := c
					stream.disconnect
				else
					err := once ""
					err.copy(once "Unable to retrieve system DTD (")
					err.append(path)
					err.extend(')')
					set_error(err)
				end
			else
				set_error(once "Bad system specification")
			end
		end

	parse_public_dtd (validator: XML_DTD_VALIDATOR) is
		local
			pubid, url, err: STRING; l, c: INTEGER; repo: XML_DTD_PUBLIC_REPOSITORY
		do
			pubid := once ""
			pubid.copy(read_string)
			skip_blanks
			url := read_string
			skip_blanks
			if pubid /= Void and then url /= Void and then skip('>') then
				dtd_url := once ""
				dtd_url.copy(url)
				stream := repo.public_dtd(pubid, dtd_url)
				if stream = Void then
					err := once ""
					err.copy(once "Unable to retrieve public DTD (")
					err.append(url)
					err.append(once "): ")
					err.append(repo.last_error)
					set_error(err)
				else
					l := line
					c := column
					line := 1
					column := 0
					parse_dtd(validator, True)
					line := l
					column := c
					repo.free(pubid, stream)
				end
			else
				set_error(once "Bad public specification")
			end
		end

	parse_inline_dtd (validator: XML_DTD_VALIDATOR) is
		do
			dtd_url := Void
			parse_dtd(validator, False)
		end

	parse_dtd (validator: XML_DTD_VALIDATOR; standalone: BOOLEAN) is
		local
			done: BOOLEAN; id: STRING
		do
			from
			until
				done or else has_error
			loop
				skip_blanks
				if skip2('<', '!') then
					if end_of_input then
						set_error(Error_end_of_file)
					elseif skip_word(once "--") then
						skip_comment
					else
						id := read_identifier
						if id = Void then
							set_error(once "ELEMENT/ATTLIST/ENTITY expected")
						else
							inspect
								id
							when "ELEMENT" then
								skip_blanks
								parse_element(validator)
							when "ATTLIST" then
								skip_blanks
								parse_attlist(validator)
							when "ENTITY" then
								skip_blanks
								parse_entity(validator)
							else
								set_error(once "ELEMENT/ATTLIST/ENTITY expected")
							end
							if not has_error then
								skip_blanks
								if not skip('>') then
									set_error(once "Missing '>'")
								else
									skip_blanks
									if standalone then
										done := end_of_input
									else
										if end_of_input then
											set_error(Error_end_of_file)
										end
									end
								end
							end
						end
					end
				elseif not standalone and then skip2(']', '>') then
					done := True
				elseif standalone then
					set_error(once "Unexpected characters")
				else
					set_error(once "']>' expected")
				end
			end
			if not has_error then
				validator.parse_done
			end
		end

	skip_comment is
		local
			state: INTEGER
		do
			from
			until
				state < 0
			loop
				if end_of_input then
					set_error(Error_end_of_file)
					state := -1
				else
					inspect state
					when 0 then
						if current_character = '-' then
							state := 1
						end
						next
					when 1 then
						if current_character = '-' then
							state := 2
						else
							state := 0
						end
						next
					when 2 then
						if current_character = '>' then
							state := -1
						else
							state := 0
						end
						next
					end
				end
			end
		end

	parse_element (validator: XML_DTD_VALIDATOR) is
		local
			node: STRING
		do
			node := once ""
			node.copy(read_identifier)
			skip_blanks
			if validator.element_built(node) then
				set_error(once "Could not build element")
			else
				node := node.twin -- keep it
				validator.add_element(node)
				parse_element_children(validator)
				validator.commit_element(node)
			end
		end

	parse_element_children (validator: XML_DTD_VALIDATOR) is
		local
			alternative: BOOLEAN; nest: INTEGER; node: STRING
		do
			from
				validator.open_list
			until
				current_character = '>' or else has_error
			loop
				if skip('(') then
					nest := nest + 1
					validator.open_list
				else
					if skip('#') then
						if skip_word(once "PCDATA") then
							validator.pcdata(alternative)
						else
							set_error(once "#PCDATA expected")
						end
					else
						node := read_identifier
						if node = Void then
							set_error(once "Tag name expected")
						else
							inspect
								node
							when "ANY" then
								validator.any
							when "EMPTY" then
								validator.empty
							else
								if alternative then
									validator.alternative_child(node.twin)
								else
									validator.child(node.twin)
								end
								skip_blanks
								if skip('+') then
									validator.one_or_more
								elseif skip('*') then
									validator.zero_or_more
								elseif skip('?') then
									validator.zero_or_one
								else
									validator.exactly_one
								end
							end
						end
					end
					skip_blanks
					if skip(')') then
						if nest = 0 then
							set_error(once "Two many closing parentheses")
						else
							nest := nest - 1
							validator.close_list
							skip_blanks
							if skip('+') then
								validator.one_or_more
							elseif skip('*') then
								validator.zero_or_more
							elseif skip('?') then
								validator.zero_or_one
							else
								validator.exactly_one
							end
						end
					end
					skip_blanks
					if skip('|') then
						alternative := True
					elseif skip(',') then
						alternative := False
					else
						if current_character /= '>' then
							set_error(once "'>' expected")
						end
					end
					skip_blanks
				end
			end
			if nest > 0 then
				set_error(once "Missing closing parentheses")
			else
				validator.close_list
				validator.exactly_one
			end
		end

	parse_attlist (validator: XML_DTD_VALIDATOR) is
		local
			n, node, att, id, val: STRING; done: BOOLEAN
		do
			n := read_identifier
			if n = Void then
				set_error(once "Identifier expected")
			else
				node := once ""
				node.copy(n)
				if not validator.element_built(node) then
					set_error(once "Please define ELEMENT before ATTLIST")
				elseif end_of_input then
					set_error(Error_end_of_file)
				else
					skip_blanks
					n := read_identifier
					from
					until
						n = Void or else has_error
					loop
						att := n.twin
						skip_blanks
						if end_of_input then
							set_error(Error_end_of_file)
						else
							validator.add_attlist(node, att)
							if skip('(') then
								from
									skip_blanks
								until
									done or else has_error
								loop
									if end_of_input then
										set_error(Error_end_of_file)
									else
										id := read_identifier
										skip_blanks
										validator.addlist_list_value(id.twin)
										if skip(')') then
											done := True
										elseif not skip('|') then
											set_error(once "')' or '|' expected")
										end
										skip_blanks
									end
								end
							else
								id := read_identifier
								skip_blanks
								inspect
									id
								when "CDATA" then
									validator.attlist_cdata
								when "ID" then
									validator.attlist_id
								when "IDREF" then
									validator.attlist_idref
								when "IDREFS" then
									validator.attlist_idrefs
								when "NMTOKEN" then
									validator.attlist_nmtoken
								when "NMTOKENS" then
									validator.attlist_nmtokens
								when "ENTITY" then
									validator.attlist_entity
								when "ENTITIES" then
									validator.attlist_entities
								when "NOTATION" then
									validator.attlist_notation
								when "xml" then
									if skip(':') then
										not_yet_implemented
									else
										set_error(once "':' expected")
									end
								else
									set_error(once "Unknown attribute specification")
								end
							end
							skip_blanks
							if skip('#') then
								if end_of_input then
									set_error(Error_end_of_file)
								else
									id := read_identifier
									inspect
										id
									when "REQUIRED" then
										validator.attlist_required
									when "IMPLIED" then
										validator.attlist_implied
									when "FIXED" then
										val := read_string
										validator.attlist_fixed(val.twin)
									else
										set_error(once "Unknown attribute specification")
									end
								end
							elseif current_character /= '>' then
								val := read_string
								validator.attlist_default_value(val.twin)
							end
							validator.commit_attlist(node, att)
						end
						if not has_error then
							skip_blanks
							n := read_identifier
						end
					end
				end
			end
		end

	parse_entity (validator: XML_DTD_VALIDATOR) is
		local
			peid, id, val, pubid: STRING
		do
			skip_blanks
			if skip('%%') then
				skip_blanks
				peid := read_identifier.twin
			end
			id := read_identifier.twin
			skip_blanks
			if validator.has_entity(id) then
				--|*** TODO: it must be only a warning
				set_error(once "ENTITY already defined")
			elseif end_of_input then
				set_error(Error_end_of_file)
			elseif skip_word(once "SYSTEM") then
				skip_blanks
				val := read_string
				validator.add_entity(peid, id, entity_system_file(val))
			elseif skip_word(once "PUBLIC") then
				skip_blanks
				pubid := once ""
				pubid.copy(read_string)
				skip_blanks
				val := read_string
				validator.add_entity(peid, id, entity_public_file(pubid, val))
			else
				val := read_identifier.twin
				validator.add_entity(peid, id, val)
			end
			if skip_word(once "NDATA") then
				if peid /= Void then
					set_error(once "Unexpected NDATA with a PEDecl")
				else
					not_yet_implemented
				end
			end
		end

feature {}
	entity_file: TEXT_FILE_READ is
			-- Used for entity inclusion
		once
			create Result.make
		end

	entity_system_file (path: STRING): STRING is
		do
			Result := once ""
			Result.clear_count
			entity_file.connect_to(path)
			if not entity_file.is_connected then
				set_error(once "Cannot find entity system file")
			else
				read_stream_in(Result, entity_file)
				entity_file.disconnect
			end
		end

	entity_public_file (pubid, url: STRING): STRING is
		local
			s: INPUT_STREAM; repo: XML_DTD_PUBLIC_REPOSITORY
		do
			Result := once ""
			Result.clear_count
			s := repo.public_dtd(pubid, url)
			if s = Void then
				set_error(once "Cannot find entity public file")
			else
				read_stream_in(Result, s)
				repo.free(pubid, s)
			end
		end

	read_stream_in (buffer: STRING; s: INPUT_STREAM) is
		require
			s.is_connected
		do
			from
				s.read_line
			until
				s.end_of_input
			loop
				buffer.append(s.last_string)
				buffer.extend('%N')
				s.read_line
			end
			buffer.append(s.last_string)
		ensure
			s.end_of_input
		end

feature {}
	make is
		do
		end

	Error_end_of_file: STRING is "Unexpected end of file"

	error: STRING

	dtd_url: STRING

	set_error (a_error: STRING) is
		do
			error := once ""
			error.copy(a_error)
			if dtd_url /= Void then
				error.append(once " in ")
				error.append(dtd_url)
				error.append(once " at line ")
				line.append_in(error)
				error.append(once ", column ")
				column.append_in(error)
			end
		end

	dtd_file: TEXT_FILE_READ is
			-- Used when the DTD is in a standalone file
		once
			create Result.make
		end

end -- class XML_DTD_PARSER
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
