-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class E_LOOP
	--
	-- The Eiffel instruction : "from ... until ... loop ... end".
	--

inherit
	INSTRUCTION

creation {ANY}
	make

feature {ANY}
	start_position: POSITION
			-- Of letter 'f' of keyword "from".

	initialize: INSTRUCTION

	loop_invariant: LOOP_INVARIANT

	loop_variant: LOOP_VARIANT

	until_expression: EXPRESSION

	loop_body: INSTRUCTION

	side_effect_free (type: TYPE): BOOLEAN is
		do
			--|*** Could be better. ***
		end

	end_mark_comment: BOOLEAN is True

	safety_check (type: TYPE) is
		do
			--|*** VARIANT *** INVARIANT ***
			if initialize /= Void then
				initialize.safety_check(type)
			end
			until_expression.safety_check(type)
			if loop_body /= Void then
				loop_body.safety_check(type)
			end
		end

	simplify (type: TYPE): INSTRUCTION is
		local
			bc: BOOLEAN_CONSTANT; it: like initialize; ue: like until_expression; lb: like loop_body
		do
			if initialize /= Void then
				it := initialize.simplify(type)
			end
			ue := until_expression.simplify(type)
			bc ?= ue
			if bc /= Void and then bc.value then
				smart_eiffel.magic_count_increment
				Result := it
			else
				if loop_body /= Void then
					lb := loop_body.simplify(type)
				end
				if it = initialize and then ue = until_expression and then lb = loop_body then
					Result := Current
				else
					smart_eiffel.magic_count_increment
					create {E_LOOP} Result.make(start_position, it, loop_invariant, loop_variant, ue, lb)
				end
			end
		end

	compile_to_c (type: TYPE) is
		local
			variant_flag, invariant_flag: BOOLEAN
		do
			if loop_check and then loop_variant /= Void then
				cpp.pending_c_function_body.append(once "{int c=0;int v=0;%N")
				variant_flag := True
			end
			if initialize /= Void then
				initialize.compile_to_c_with_internal_c_local_saving(type)
			end
			if loop_check and then loop_invariant /= Void then
				loop_invariant.compile_to_c(type)
				invariant_flag := True
			end
			cpp.pending_c_function_body.append(once "while (!(")
			if ace.no_check then
				cpp.trace_boolean_expression(type, until_expression)
			else
				until_expression.compile_to_c_with_internal_c_local_saving(type)
			end
			cpp.pending_c_function_body.append(once "))%N{%N")
			if variant_flag then
				cpp.variant_check(type, loop_variant.expression)
			end
			if loop_body /= Void then
				loop_body.compile_to_c_with_internal_c_local_saving(type)
			end
			if invariant_flag then
				loop_invariant.compile_to_c(type)
			end
			cpp.pending_c_function_body.append(once "}%N")
			if variant_flag then
				cpp.pending_c_function_body.append(once "}%N")
			end
		end

	compile_to_jvm (type: TYPE) is
		local
			until_point, exit_point: INTEGER; ca: like code_attribute; counter, previous: INTEGER
		do
			ca := code_attribute
			if loop_check then
				counter := ca.extra_local_size1
				ca.opcode_iconst_0
				ca.opcode_istore(counter)
				previous := ca.extra_local_size1
				ca.opcode_iconst_0
				ca.opcode_istore(previous)
			end
			if initialize /= Void then
				initialize.compile_to_jvm(type)
			end
			if loop_check then
				if loop_invariant /= Void then
					loop_invariant.compile_to_jvm(type, True)
					ca.opcode_pop
				end
			end
			until_point := ca.program_counter
			exit_point := until_expression.jvm_branch_if_true(type)
			if loop_check then
				if loop_variant /= Void then
					ca.opcode_iload(counter)
					ca.opcode_iload(previous)
					ca.runtime_check_loop_variant(type, loop_variant.expression)
					ca.opcode_istore(previous)
				end
			end
			if loop_body /= Void then
				loop_body.compile_to_jvm(type)
			end
			if loop_check then
				if loop_invariant /= Void then
					loop_invariant.compile_to_jvm(type, True)
					ca.opcode_pop
				end
			end
			if loop_check then
				ca.opcode_iinc(counter, 1)
			end
			ca.opcode_goto_backward(until_point)
			ca.resolve_u2_branch(exit_point)
		end

	use_current (type: TYPE): BOOLEAN is
		do
			if loop_check and then loop_variant /= Void then
				Result := loop_variant.use_current(type)
			end
			if initialize /= Void then
				Result := Result or else initialize.use_current(type)
			end
			Result := Result or else until_expression.use_current(type)
			if loop_check and then loop_invariant /= Void then
				Result := Result or else loop_invariant.use_current(type)
			end
			if loop_body /= Void then
				Result := Result or else loop_body.use_current(type)
			end
		end

	specialize_in (type: TYPE): like Current is
		local
			it: like initialize; li: like loop_invariant; lv: like loop_variant; ue: like until_expression
			lb: like loop_body
		do
			if initialize /= Void then
				it := initialize.specialize_in(type)
			end
			if loop_invariant /= Void then
				li := loop_invariant.specialize_in(type)
			end
			if loop_variant /= Void then
				lv := loop_variant.specialize_in(type)
			end
			ue := until_expression.specialize_in(type)
			if loop_body /= Void then
				lb := loop_body.specialize_in(type)
			end
			if it = initialize and then li = loop_invariant and then lv = loop_variant and then ue = until_expression and then lb = loop_body then
				Result := Current
			else
				create Result.make(start_position, it, li, lv, ue, lb)
			end
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		local
			it: like initialize; li: like loop_invariant; lv: like loop_variant; ue: like until_expression
			lb: like loop_body
		do
			if initialize /= Void then
				it := initialize.specialize_thru(parent_type, parent_edge, new_type)
			end
			if loop_invariant /= Void then
				li := loop_invariant.specialize_thru(parent_type, parent_edge, new_type)
			end
			if loop_variant /= Void then
				lv := loop_variant.specialize_thru(parent_type, parent_edge, new_type)
			end
			ue := until_expression.specialize_thru(parent_type, parent_edge, new_type)
			if loop_body /= Void then
				lb := loop_body.specialize_thru(parent_type, parent_edge, new_type)
			end
			if it /= initialize or else li /= loop_invariant or else lv /= loop_variant or else ue /= until_expression or else lb /= loop_body then
				create Result.make(start_position, it, li, lv, ue, lb)
			else
				Result := Current
			end
		end

	specialize_2 (type: TYPE): like Current is
		local
			it: like initialize; li: like loop_invariant; lv: like loop_variant; ue: like until_expression
			lb: like loop_body
		do
			if initialize /= Void then
				it := initialize.specialize_2(type)
			end
			if loop_invariant /= Void then
				li := loop_invariant.specialize_2(type)
			end
			if loop_variant /= Void then
				lv := loop_variant.specialize_2(type)
			end
			ue := until_expression.specialize_2(type)
			if loop_body /= Void then
				lb := loop_body.specialize_2(type)
			end
			if it /= initialize or else li /= loop_invariant or else lv /= loop_variant or else ue /= until_expression or else lb /= loop_body then
				create Result.make(start_position, it, li, lv, ue, lb)
			else
				Result := Current
			end
			Result.specialize_2_checks(type)
		end

	has_been_specialized: BOOLEAN is
		do
			Result := initialize /= Void implies initialize.has_been_specialized
			if Result and loop_invariant /= Void then
				Result := loop_invariant.has_been_specialized
			end
			if Result and loop_variant /= Void then
				Result := loop_variant.has_been_specialized
			end
			Result := Result and until_expression.has_been_specialized
			if Result and loop_body /= Void then
				Result := loop_body.has_been_specialized
			end
		end

	accept (visitor: E_LOOP_VISITOR) is
		do
			visitor.visit_e_loop(Current)
		end

	pretty (indent_level: INTEGER) is
		local
			semi_colon_flag: BOOLEAN
		do
			pretty_printer.set_indent_level(indent_level)
			pretty_printer.put_string(once "from")
			pretty_printer.set_indent_level(0)
			if initialize /= Void then
				initialize.pretty(indent_level + 1)
			end
			if loop_invariant /= Void then
				loop_invariant.pretty(indent_level)
			end
			if loop_variant /= Void then
				pretty_printer.set_indent_level(indent_level)
				pretty_printer.put_string(once "variant")
				pretty_printer.set_indent_level(indent_level + 1)
				semi_colon_flag := pretty_printer.semi_colon_flag
				pretty_printer.set_semi_colon_flag(False)
				loop_variant.pretty(indent_level + 1)
				pretty_printer.set_semi_colon_flag(semi_colon_flag)
			end
			pretty_printer.set_indent_level(indent_level)
			pretty_printer.put_string(once "until")
			pretty_printer.set_indent_level(indent_level + 1)
			pretty_printer.set_semi_colon_flag(False)
			until_expression.pretty(indent_level + 1)
			pretty_printer.set_indent_level(indent_level)
			pretty_printer.put_string(once "loop")
			if loop_body /= Void then
				loop_body.pretty(indent_level + 1)
			end
			pretty_printer.set_indent_level(indent_level)
			pretty_printer.keyword(once "end")
			if pretty_printer.print_end_of_statement then
				pretty_printer.put_end_of(once "loop")
			end
			pretty_printer.set_indent_level(0)
		end

	collect (t: TYPE) is
		do
			if initialize /= Void then
				initialize.collect(t)
			end
			if loop_invariant /= Void then
				loop_invariant.collect(t)
			end
			if loop_variant /= Void then
				loop_variant.collect(t)
			end
			if until_expression.collect(t) = Void then
				check
					False
				end
			end
			if loop_body /= Void then
				loop_body.collect(t)
			end
		end

	adapt_for (t: TYPE): like Current is
		local
			it: like initialize; li: like loop_invariant; lv: like loop_variant; ue: like until_expression
			lb: like loop_body
		do
			if initialize /= Void then
				it := initialize.adapt_for(t)
			end
			if loop_invariant /= Void then
				li := loop_invariant.adapt_for(t)
			end
			if loop_variant /= Void then
				lv := loop_variant.adapt_for(t)
			end
			ue := until_expression.adapt_for(t)
			if loop_body /= Void then
				lb := loop_body.adapt_for(t)
			end
			if it /= initialize or else li /= loop_invariant or else lv /= loop_variant or else ue /= until_expression or else lb /= loop_body then
				create Result.make(start_position, it, li, lv, ue, lb)
			else
				Result := Current
			end
		end

feature {E_LOOP}
	specialize_2_checks (type: TYPE) is
		local
			dt: TYPE; variant_expression: EXPRESSION
		do
			dt := until_expression.declaration_type
			if not dt.is_boolean then
				error_handler.add_position(until_expression.start_position)
				error_handler.append("Expression of until must be of BOOLEAN type. (The actual ")
				error_handler.append(dt.name.to_string)
				error_handler.append(" type is not allowed.)")
				error_handler.print_as_error
				error_handler.add_position(start_position)
				error_handler.add_position(until_expression.start_position)
				error_handler.append("Error in until part of loop definition.")
				error_handler.print_as_fatal_error
			end
			if loop_variant /= Void then
				variant_expression := loop_variant.expression
				dt := variant_expression.declaration_type
				if not dt.is_integer then
					error_handler.add_position(variant_expression.start_position)
					error_handler.append("Expression of the loop variant must be of INTEGER type. (The actual ")
					error_handler.append(dt.name.to_string)
					error_handler.append(" type is not allowed as a variant type.)")
					error_handler.print_as_error
					error_handler.add_position(start_position)
					error_handler.add_position(variant_expression.start_position)
					error_handler.append("Error in variant part of loop definition.")
					error_handler.print_as_fatal_error
				end
			end
		end

feature {}
	loop_check: BOOLEAN is
			--|*** SHOULD BE DONE ON THE CONTEXT OF THE SPECIALIZE ***
		do
			Result := start_position.class_text.loop_check
		end

	make (sp: like start_position; i: like initialize; li: like loop_invariant; lv: like loop_variant
		ue: like until_expression; lb: like loop_body) is
		require
			not sp.is_unknown
			ue /= Void
		do
			start_position := sp
			initialize := i
			loop_invariant := li
			loop_variant := lv
			until_expression := ue
			loop_body := lb
		ensure
			start_position = sp
			initialize = i
			loop_invariant = li
			loop_variant = lv
			until_expression = ue
			loop_body = lb
		end

invariant
	not start_position.is_unknown

	until_expression /= Void

end -- class E_LOOP
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
