-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
deferred class PROC_CALL
	--
	-- For all sort of procedure calls.
	-- Does not include function calls (see CALL).
	--
	-- Classification: E_PROC_0 when 0 argument, PROC_CALL_1 when
	-- 1 argument and PROC_CALL_N when N arguments.
	--

inherit
	INSTRUCTION
	CALL_SUPPORT

feature {ANY}
	frozen pretty (indent_level: INTEGER) is
		do
			pretty_printer.set_indent_level(indent_level)
			target.pretty_target(indent_level)
			pretty_printer.put_string(feature_name.to_string)
			if arguments /= Void then
				arguments.pretty(indent_level)
			end
			if pretty_printer.semi_colon_flag then
				pretty_printer.put_character(';')
			end
			pretty_printer.set_indent_level(0)
		end

feature {ANY}
	frozen collect (type: TYPE) is
		local
			proc_type: TYPE
		do
			proc_type := call_support_collect(type)
			check
				is_procedure: proc_type = Void
			end
		end

	frozen has_been_specialized: BOOLEAN is
		do
			Result := target.has_been_specialized
			if Result and then arguments /= Void then
				Result := arguments.has_been_specialized
			end
			Result := Result and then target.has_been_specialized
			if target.is_current then
				Result := Result and then feature_stamp /= Void
			end
		end

	frozen compile_to_c (type: TYPE) is
		do
			if ace.no_check then
				cpp.put_trace_or_sedb_instruction(start_position)
			end
			call_proc_call_c2c(type)
		end

	end_mark_comment: BOOLEAN is False

feature {CALL_SUPPORT, CREATE_SUPPORT}
	creation_type: TYPE_MARK
			-- Used when the PROC_CALL is actually used in a creation instruction, the `creation_type' is then the
			-- type to be created.
			--|*** Is it a good idea to have this information here (i.e. for all PROC_CALL ?)
			--|    Should it be inside create instruction ?
			--|*** Dom: jan 9th 2004 ***

feature {CREATE_SUPPORT}
	set_creation_type (ct: like creation_type) is
		require
			ct.is_static
		do
			creation_type := ct
		ensure
			creation_type = ct
		end

	collect_create (t: TYPE; target_type: TYPE; target_pos: POSITION) is
			-- also given is the type of the *target* (that's the difference with collect)
			-- Used by CREATE_SUPPORT
			--|*** That's almost copy-pasted from CALL_SUPPORT.collect_exp :-(
			--|*** a voir pour supprimer cela et utiliser "writable"
		local
			proc_type: TYPE; fs: FEATURE_STAMP
		do
			--|*** Need to be checked (probably declaration_type should be
			--used, here or by the caller)
			if not target_type.valid_feature_name(feature_name) then
				error_handler.append("Unknown create feature name ")
				error_handler.append(feature_name.to_string)
				error_handler.append(" from ")
				error_handler.append(target_type.name.to_string)
				error_handler.append(".")
				error_handler.add_position(feature_name.start_position)
				error_handler.add_position(target_pos)
				error_handler.print_as_fatal_error
			end
			fs := target_type.lookup(feature_name)
			proc_type := smart_eiffel.collect(target_type, fs, True)
			check
				is_procedure: proc_type = Void
			end
			-- now the arguments
			if arguments /= Void then
				arguments.collect(t, fs, target_type)
			end
			check
				set_collected_for(t)
			end
		end

feature {CREATE_SUPPORT}
	simplify_arguments (type: TYPE): PROC_CALL is
		local
			args: like arguments
		do
			if arguments /= Void then
				args := arguments.simplify(type)
				if args /= arguments then
					Result := twin
					Result.set_arguments(args)
				else
					Result := Current
				end
			else
				Result := Current
			end
		ensure
			Result /= Void
		end

feature {CREATE_EXPRESSION}
	frozen short (type: TYPE) is
		require
			type /= Void
		local
			target_type: TYPE; fs: FEATURE_STAMP
		do
			-- Note: PROC_CALLs are instruction which are part of CREATE_EXPRESSION, hence this definition.
			target.short_target(type)
			target_type := target.resolve_in(type)
			fs := feature_stamp.resolve_static_binding_for(target.declaration_type, target_type)
			target_type.get_feature_name(fs).short(type)
			if arguments /= Void then
				arguments.short(type)
			end
		end

feature {}
	frozen afd_check_hook is
		do
		end

	frozen procedure_and_argument_count_check (af: ANONYMOUS_FEATURE; actual_args: like arguments) is
			-- Check that the feature found is really a procedure then launch `argument_count_check'.
		require
			af /= Void
		do
			if af.result_type /= Void then
				error_handler.add_position(af.start_position)
				error_handler.add_position(feature_name.start_position)
				error_handler.append("Feature found is not a procedure.")
				error_handler.print_as_error
				error_handler.add_position(feature_name.start_position)
				error_handler.append("This call has a result value (and you must use it).")
				error_handler.print_as_fatal_error
			end
			smart_eiffel.argument_count_check(feature_name.start_position, af, actual_args)
		end

end -- class PROC_CALL
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
