-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class ASSERTION
	--
	-- To store one assertion.
	--

inherit
	VISITABLE
insert
	GLOBALS

creation {ANY}
	make

feature {ANY}
	tag: TAG_NAME

	expression: EXPRESSION

	comment: COMMENT

	start_position: POSITION is
		do
			if tag /= Void then
				Result := tag.start_position
			elseif expression /= Void then
				Result := expression.start_position
			else
				Result := comment.start_position
			end
		end

	pretty (indent_level: INTEGER) is
		do
			pretty_printer.set_indent_level(indent_level)
			if tag /= Void then
				pretty_printer.put_string(tag.to_string)
				pretty_printer.put_string(once ": ")
			end
			if expression /= Void then
				expression.pretty(indent_level)
				if pretty_printer.semi_colon_flag then
					pretty_printer.put_character(';')
				end
			end
			if comment /= Void then
				comment.pretty(indent_level + 1)
			end
			pretty_printer.set_indent_level(0)
		end

	short (type: TYPE
		h01, r01, h02, r02, h03, r03, h04, r04, h05, r05, h06, r06, h07, r07, h08, r08, h09, r09, h10, r10, h11
		r11, h12, r12, h13, r13: STRING) is
		do
			short_printer.hook_or(h01, r01)
			if tag = Void then
				short_printer.hook_or(h02, r02)
			else
				short_printer.hook_or(h03, r03)
				tag.short
				short_printer.hook_or(h04, r04)
			end
			if expression = Void then
				short_printer.hook_or(h05, r05)
			else
				short_printer.hook_or(h06, r06)
				expression.short(type)
				short_printer.hook_or(h07, r07)
			end
			if comment = Void then
				short_printer.hook_or(h08, r08)
			else
				short_printer.hook_or(h09, r09)
				comment.short(h10, r10, h11, r11)
				short_printer.hook_or(h12, r12)
			end
			short_printer.hook_or(h13, r13)
		end

	use_current (type: TYPE): BOOLEAN is
		do
			if expression /= Void then
				Result := expression.use_current(type)
			end
		end

	compile_to_c (type: TYPE) is
		local
			s: STRING
		do
			if expression /= Void then
				if tag /= Void then
					s := tag.to_string
				elseif expression /= Void then
					s := once ""
					s.clear_count
					pretty_printer.expression_in(s, expression)
				end
				cpp.check_assertion(type, expression, s)
			end
		end

	is_always_true: BOOLEAN is
		local
			bc: BOOLEAN_CONSTANT
		do
			if expression = Void then
				Result := True
			else
				bc ?= expression
				if bc /= Void then
					Result := bc.value
				end
			end
		end

	safety_check (type: TYPE) is
		do
			if expression /= Void then
				expression.safety_check(type)
			end
		end
			
	accept (visitor: ASSERTION_VISITOR) is
		do
			visitor.visit_assertion(Current)
		end

feature {ASSERTION_LIST}
	compile_to_jvm (type: TYPE; last_chance: BOOLEAN) is
			-- The boolean result of the expression is pushed.
			-- When `last_chance' is True, the generated code includes an
			-- error message to be printed when assertion is False.
			--
		local
			point1: INTEGER; ca: like code_attribute
		do
			ca := code_attribute
			if expression = Void then
				ca.opcode_iconst_1
			else
				expression.compile_to_jvm(type)
				if last_chance then
					point1 := code_attribute.opcode_ifne
					ca.runtime_error(expression.start_position, Void, once "Assertion failed.")
					ca.resolve_u2_branch(point1)
					ca.opcode_iconst_1
				end
			end
		end

	collect (type: TYPE) is
		require
			has_been_specialized
			type.feature_collection_done
			smart_eiffel.status.is_collecting
			not smart_eiffel.status.is_specializing
		local
			rt: TYPE
		do
			if expression /= Void then
				rt := expression.collect(type)
			end
		end

	adapt_for (t: TYPE): like Current is
		local
			exp: EXPRESSION
		do
			if expression /= Void then
				exp := expression.adapt_for(t)
			end
			Result := current_or_twin_init(exp)
		end

feature {ASSERTION_LIST}
	side_effect_free (type: TYPE): BOOLEAN is
		do
			Result := True
			if expression /= Void then
				Result := expression.side_effect_free(type)
			end
		end

	specialize_in (type: TYPE): like Current is
		require
			type /= Void
		local
			exp: EXPRESSION
		do
			if expression /= Void then
				exp := expression.specialize_in(type)
			end
			Result := current_or_twin_init(exp)
		ensure
			Result.has_been_specialized
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		require
			new_type.direct_thru_step(parent_type, parent_edge)
			has_been_specialized
		local
			exp: EXPRESSION
		do
			if expression /= Void then
				exp := expression.specialize_thru(parent_type, parent_edge, new_type)
			end
			Result := current_or_twin_init(exp)
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	specialize_2 (type: TYPE): like Current is
		require
			has_been_specialized
			not smart_eiffel.status.is_specializing
		local
			exp: EXPRESSION
		do
			if expression /= Void then
				exp := expression.specialize_2(type)
			end
			Result := current_or_twin_init(exp)
			Result.specialize_2_check(type)
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

feature {ASSERTION, ASSERTION_LIST}
	has_been_specialized: BOOLEAN is
		do
			Result := expression /= Void implies expression.has_been_specialized
		ensure
			Result
		end

feature {ASSERTION}
	specialize_2_check (type: TYPE) is
		local
			rt: TYPE
		do
			if expression /= Void then
				rt := expression.resolve_in(type)
				if not rt.is_boolean then
					error_handler.append("An assertion must be a BOOLEAN expression.")
					error_handler.add_position(expression.start_position)
					error_handler.append("(The type of this expression is actually ")
					error_handler.append(rt.name.to_string)
					error_handler.append(".)")
					error_handler.print_as_fatal_error
				end
			end
		end

feature {ASSERTION}
	init (exp: like expression) is
		require
			exp /= Void
		do
			expression := exp
		ensure
			expression = exp
		end

feature {}
	current_or_twin_init (exp: like expression): like Current is
		do
			if exp = expression then
				Result := Current
			else
				Result := twin
				Result.init(exp)
			end
		end

	make (t: like tag; exp: like expression; c: like comment) is
		require
			t /= Void or exp /= Void or c /= Void
		do
			tag := t
			expression := exp
			comment := c
		ensure
			tag = t
			expression = exp
			comment = c
		end

	tmp_string: STRING is
		once
			create Result.make(128)
		end

invariant
	tag /= Void or expression /= Void or comment /= Void

end -- class ASSERTION
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
