/***************************************************************************
    smb4kipaddressscanner  -  This class scans for IP addresses. It
    belongs to the core classes of Smb4K.
                             -------------------
    begin                : Di Apr 22 2008
    copyright            : (C) 2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KIPADDRESSSCANNER_H
#define SMB4KIPADDRESSSCANNER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <QObject>
#include <QTimerEvent>
#include <QList>

// KDE includes
#include <kdemacros.h>
#include <kprocess.h>

// forward declarations
class Smb4KHost;
class Smb4KWorkgroup;

/**
 * This class scans for IP addresses of hosts found in the network. It
 * operates on the global host list.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class KDE_EXPORT Smb4KIPAddressScanner : public QObject
{
  Q_OBJECT

  friend class Smb4KIPAddressScannerPrivate;

  public:
    /**
     * Returns a static pointer to this class.
     */
    static Smb4KIPAddressScanner *self();

    /**
     * Triggers a scan for IP addresses. This function calls lookupIPAddresses(),
     * except the scanner is in auto scan mode. In this case it will return immediately
     * without doing anything.
     */
    void triggerScan();

    /**
     * Sets the scanner in automatic scan mode, if @p interval is non-zero.
     *
     * @param interval        The interval between the scans in milliseconds.
     */
    void setAutoScan( int interval );

    /**
     * Returns TRUE if the scanner is in auto scan mode and FALSE otherwise.
     *
     * @returns TRUE if the scanner is in auto scan mode.
     */
    bool autoScan() { return m_auto_scan; }

    /**
     * Lookup the IP address of a certain host. You need to connect to the
     * ipAddress() signal to get the result. The IP address scan will be
     * executed when the scanner is idle the next time.
     *
     * Please make sure that the pointer to @p host stays valid until the
     * search for the IP address finished and the ipAddress() signal is
     * emitted.
     *
     * @param host            The Smb4KHost item for which the IP address
     *                        should be looked up.
     */
    void getIPAddress( Smb4KHost *host );

  signals:
    /**
     * This signal is emitted, when an IP address was found for a host. It passed
     * the associated Smb4KHost object.
     *
     * @param host            The Smb4KHost object that carries the discovered
     *                        IP address.
     */
    void ipAddress( Smb4KHost *host );

  protected:
    /**
     * Reimplemented from QObject.
     */
    void timerEvent( QTimerEvent *e );

  protected slots:
    /**
     * This slot is connected to the KProcess::readyReadStandardOutput()
     * signal and collects the IP addresses from the output.
     */
    void slotReceivedIPAddresses();

    /**
     * This slot is called, when the process exited.
     *
     * @param exitCode            The exit code of the process
     *
     * @param exitStatus          The exit status of the process (@see QProcess::ExitStatus)
     */
    void slotProcessFinished( int exitCode, QProcess::ExitStatus exitStatus );

    /**
     * This slot is invoked when an error occurred with the process.
     *
     * @param errorCode       The error code returned by the process
     */
    void slotProcessError( QProcess::ProcessError errorCode );

  private:
    /**
     * The constructor
     */
    Smb4KIPAddressScanner();

    /**
     * The destructor
     */
    ~Smb4KIPAddressScanner();

    /**
     * This function loops through the global hosts list and initiates
     * an IP address lookup process for every host, that has not been
     * checked before.
     */
    void lookupIPAddresses();

    /**
     * Auto scan mode
     */
    bool m_auto_scan;

    /**
     * The id of the timer
     */
    int m_timer_id;

    /**
     * The scanner is working
     */
    bool m_working;

    /**
     * The process object
     */
    KProcess *m_proc;

    /**
     * The process error code
     */
    QProcess::ProcessError m_process_error;

    /**
     * The queue for buffering scan requests that were received
     * through the getIPAddress() function.
     */
    QList<Smb4KHost> m_queue;
};

#endif
