/***************************************************************************
    smb4kpasswordhandler  -  This class handles the passwords for Smb4K.
                             -------------------
    begin                : So Jan 16 2005
    copyright            : (C) 2005 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KPASSWORDHANDLER_H
#define SMB4KPASSWORDHANDLER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qcheckbox.h>

// KDE includes
#include <kwallet.h>
#include <kdialogbase.h>
#include <kprocess.h>
#include <kconfig.h>

// application specific includes
#include "smb4kauthinfo.h"
#include "smb4khomesshareshandler.h"

#ifdef __FreeBSD__
#include "smb4ksambaoptionshandler.h"
#endif



/**
 * This class handles the passwords used by Smb4K to authenticate to
 * the network shares.
 *
 * @author                 Alexander Reinholdt <dustpuppy@mail.berlios.de>
 */

class Smb4KPasswordHandler : public QObject
{
  Q_OBJECT

  public:
#ifndef __FreeBSD__
    /**
     * The constructor.
     *
     * @param config        The KConfig object that should be used
     *
     * @param handler       The Smb4KHomesSharesHandler object that is needed to retrieve
     *                      infos in case we need to look up authentication information for
     *                      'homes' shares.
     *
     * @param parent        The parent object.
     *
     * @param name          The name this class should carry.
     */
    Smb4KPasswordHandler( KConfig *config,
                          Smb4KHomesSharesHandler *handler,
                          QObject *parent = 0,
                          const char *name = 0 );
#else
    /**
     * The constructor (FreeBSD).
     *
     * @param config        The KConfig object that should be used
     *
     * @param s_handler     The Smb4KHomesSharesHandler object that is needed to retrieve
     *                      infos in case we need to look up authentication information for
     *                      'homes' shares.
     *
     * @param o_handler     The Smb4KSambaOptionsHandler object. It is needed to write the
     *                      ~/.nsmbrc file correctly.
     *
     * @param parent        The parent object.
     *
     * @param name          The name this class should carry.
     */
    Smb4KPasswordHandler( KConfig *config,
                          Smb4KHomesSharesHandler *s_handler,
                          Smb4KSambaOptionsHandler *o_handler,
                          QObject *parent = 0,
                          const char *name = 0 );
#endif

    /**
     * The destructor.
     */
    ~Smb4KPasswordHandler();

    /**
     * Opens the askpass dialog.
     *
     * @param workgroup     The workgroup where the share is located in (may be empty).
     *
     * @param host          The host the share belongs to.
     *
     * @param name          The name of the share.
     *
     * @param desc          Determines which descriptive label the askpass dialog should carry.
     *                      Values are taken from the Smb4KPasswordHandler::AskPass::Description
     *                      enumeration.
     *
     * @param parent        The parent of this dialog.
     *
     * @param name          The name of this dialog.
     *
     * @returns             TRUE if a new password has been supplied and FALSE otherwise.
     */
    bool askpass( const QString &workgroup, const QString &host, const QString &share, int desc, QWidget *parent = 0, const char *name = 0 );

    /**
     * Returns the authentication data that was found for the defined share. You need to
     * provide the workgroup, host name, and share name with @p authInfo.
     *
     * @param authInfo      The authentication information object.
     *
     * @returns             a pointer to @p authInfo. This may be NULL if @p authInfo was
     * also NULL.
     */
    Smb4KAuthInfo *readAuth( Smb4KAuthInfo *authInfo );

    /**
     * Writes the provided authentication data to the wallet.
     *
     * @param auth          The Smb4KAuthInfo object that holds the authentication information.
     */
    void writeAuth( Smb4KAuthInfo *auth );

    /**
     * Reads the default authentication from the wallet. This function seems only to
     * be useful for the configuration dialog.
     *
     * @returns             pointer to the authentication data object if a wallet is used or NULL otherwise
     */
    Smb4KAuthInfo *readDefaultAuth();

    /**
     * Writes the default authentication data to the wallet. This function seems only to
     * be useful for the configuration dialog.
     *
     * @param auth          The authentication data for the default login if a wallet is
     *                      used or NULL otherwise.
     */
    void writeDefaultAuth( Smb4KAuthInfo *auth );

    /**
     * This enumeration destermines which destriptive label the askpass
     * dialog should carry.
     */
    enum Description{ NewData, AccessDenied, BadPassword, PermDenied, AuthError, LogonFailure, None };

  protected slots:
    /**
     * Get the password for a specific share and user name.
     */
    void slotGetPassword( const QString &username );

    /**
     * This slot is used to enable the OK button of the askpass
     * dialog when text has been entered in the text boxes and/or the
     * combo box.
     *
     * @param text          The text that has been entered.
     */
    void slotEnableOKButton( const QString &text );

    /**
     * FreeBSD specific: This slot receives output from the process that encrypts
     * the passwords.
     *
     * @param proc          The process that sends the output
     *
     * @param buffer        The buffer that contains the output
     *
     * @param buflen        The length of the buffer
     */
    void slotReceivePassword( KProcess *proc, char *buffer, int buflen );

    /**
     * FreeBSD specific: This slot is invoked if the process exited.
     *
     * @param proc          The process that exited
     */
    void slotWritePassword( KProcess *proc );

  private:
    /**
     * This function opens the wallet in which the passwords should be
     * stored.
     */
    void open_close_wallet();

    /**
     * The KWallet object.
     */
    KWallet::Wallet *m_wallet;

    /**
     * The Smb4KAuthInfo object. For use with the askpass dialog.
     */
    Smb4KAuthInfo *m_auth;

    /**
     * The askpass dialog.
     */
    KDialogBase *m_dlg;

    /**
     * This funtion imports the authentication data from the old password file
     * into the wallet.
     */
    void import();

    /**
     * This list holds the authentication data the user supplied if he/she
     * does not want to use KWallet.
     */
    QValueList<Smb4KAuthInfo *> m_auth_list;

    /**
     * If the user neither wants to store the passwords in a wallet nor in a
     * temporary list, the authentication data received from the askpass dialog
     * will the stored in this object.
     */
    Smb4KAuthInfo *m_temp_auth;

    /**
     * The KConfig object
     */
    KConfig *m_config;

    /**
     * The Smb4KHomesSharesHandler object
     */
    Smb4KHomesSharesHandler *m_handler;

#ifdef __FreeBSD__

    /**
     * FreeBSD specific: This function writes authentication information
     * to the ~/.nsmbrc file, which is used by mount_smbfs.
     *
     * @param auth          The authentication information that should be written
     *                      to ~/.nsmbrc file.
     */
    void writeToSMBConfFile( Smb4KAuthInfo *auth );

    /**
     * FreeBSD specific: Holds the authentication information that should
     * be written to ~/.nsmbrc.
     */
    Smb4KAuthInfo m_nsmbrc_auth;

    /**
     * The buffer for the process that encrypts the password.
     */
    QString m_buffer;

    /**
     * The Smb4KSambaOptionsHandler object.
     */
    Smb4KSambaOptionsHandler *m_options_handler;
#endif

};

#endif
