/***************************************************************************
    smb4ksharewidget.cpp  -  The widget, that displays the mounted shares.
                             -------------------
    begin                : Sam M� 1 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qlabel.h>
#include <qlayout.h>
#include <qdir.h>

// KDE includes
#include <klocale.h>
#include <kapplication.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kurl.h>
#include <kdebug.h>
#include <kio/job.h>

// application specific includes
#include "smb4ksharewidget.h"
#include "smb4ksharewidgetitem.h"
// #include "smb4kmountdialog.h"
#include "../core/smb4kcore.h"
#include "../core/smb4kglobal.h"
using namespace Smb4KGlobal;



Smb4KShareWidget::Smb4KShareWidget( QWidget *parent, const char *name ) : KIconView( parent, name )
{
  setFrameShape( KIconView::PopupPanel );
  setFrameShadow( KIconView::Sunken );
  setSelectionMode( KIconView::Single ); // If this is changed, revise dragObject() function.
  setResizeMode( KIconView::Adjust );
  setShowToolTips( false );
  setAutoArrange( true );
  setSorting( true, true );
  setItemsMovable( false );
  // This is set by readOptions()
  setAcceptDrops( false );

  m_collection = new KActionCollection( this, "SharesView_ActionCollection", KGlobal::instance() );

  m_menu = new Smb4KShareActionMenu( Smb4KShareActionMenu::Full, m_collection, QString::null, QIconSet(), this, "SharesView_ActionMenu" );
  m_menu->popupMenu()->insertTitle( SmallIcon( "hdd_mount" ), i18n( "Shares" ), 0, 0 );

  m_tooltip = NULL;

  initActions();

  // Internal connections
  connect( this,                  SIGNAL( mouseButtonPressed( int, QIconViewItem *, const QPoint & ) ),
           this,                  SLOT( slotMouseButtonPressed( int, QIconViewItem *, const QPoint & ) ) );
  connect( this,                  SIGNAL( executed( QIconViewItem * ) ),
           this,                  SLOT( slotOpenFilemanager() ) );
  connect( this,                  SIGNAL( selectionChanged( QIconViewItem * ) ),
           this,                  SLOT( slotSelectionChanged( QIconViewItem * ) ) );

  // External connections
  connect( smb4k_core->mounter(), SIGNAL( updated() ),
           this,                  SLOT( slotMountedShares() ) );
  connect( kapp,                  SIGNAL( iconChanged( int ) ),
           this,                  SLOT( slotIconChanged( int ) ) );
}


Smb4KShareWidget::~Smb4KShareWidget()
{
  delete m_menu;
  delete m_tooltip;

  uint index = 0;

  while ( index < m_collection->count() )
  {
    delete m_collection->action( index++ );
  }

  m_collection->clear();
}


/****************************************************************************
   Initialize the actions needed by this class.
****************************************************************************/

void Smb4KShareWidget::initActions()
{
  m_unmount_action = m_menu->unmountAction();
  m_unmount_action->setGroup( "ShareWidget" );
  m_unmount_action->setEnabled( false );
  connect( m_unmount_action, SIGNAL( activated() ), this, SLOT( slotUnmountShare() ) );

#ifdef __linux__
  m_force_action = m_menu->forcedUnmountAction();
  m_force_action->setGroup( "ShareWidget" );
  m_force_action->setEnabled( false );
  connect( m_force_action, SIGNAL( activated() ), this, SLOT( slotForceUnmountShare() ) );
#endif

  m_unmount_all_action = m_menu->unmountAllAction();
  m_unmount_all_action->setGroup( "ShareWidget" );
  m_unmount_all_action->setEnabled( false );
  connect( m_unmount_all_action, SIGNAL( activated() ), this, SLOT( slotUnmountAllShares() ) );

  m_filemanager_action = m_menu->filemanagerAction();
  m_filemanager_action->setGroup( "ShareWidget" );
  m_filemanager_action->setEnabled( false );
  connect( m_filemanager_action, SIGNAL( activated() ), this, SLOT( slotOpenFilemanager() ) );

  m_sync_action = m_menu->syncAction();
  m_sync_action->setGroup( "ShareWidget" );
  m_sync_action->setEnabled( false );
  connect( m_sync_action, SIGNAL( activated() ), this, SLOT( slotSynchronize() ) );
}


/****************************************************************************
   Reads the options for the share view widget.
****************************************************************************/

void Smb4KShareWidget::readOptions()
{
  // Which appearance does the user want?
  config()->setGroup( "User Interface" );
  m_display = config()->readEntry( "Show Shares", "icons" );

  if ( QString::compare( m_display, "icons" ) == 0 )
  {
    setItemTextPos( KIconView::Bottom );
    setMaxItemWidth( 150 );
    setArrangement( KIconView::LeftToRight );
    setWordWrapIconText( true );
  }
  else if ( QString::compare( m_display, "list" ) == 0 )
  {
    setItemTextPos( KIconView::Right );
    setMaxItemWidth( 500 );
    setArrangement( KIconView::TopToBottom );
    setWordWrapIconText( false );
  }

  m_mountpoint = config()->readBoolEntry( "Show Mount Point", false );
  m_showExternal = config()->readBoolEntry( "Show All Shares", false );
  m_allow_dragging = config()->readBoolEntry( "Allow Dragging", false );
  m_allow_dropping = config()->readBoolEntry( "Allow Dropping", false );

  changeIcons();

  slotMountedShares();

  adjustItems();

  setAcceptDrops( m_allow_dropping );

  // Mount options.
  config()->setGroup( "Programs" );
  QString super = config()->readPathEntry( "super" );
  QString sudo = config()->readPathEntry( "sudo" );
  m_rsync = (config()->readPathEntry( "rsync" )).isEmpty() ? false : true;

  config()->setGroup( "Super User Privileges" );

#ifdef __linux__
  if ( !super.isEmpty() || !sudo.isEmpty() )
  {
    m_forcedUnmount = config()->readBoolEntry( "Force Unmount", false );
  }
  else
  {
    m_forcedUnmount = false;
  }
#endif
}


void Smb4KShareWidget::contentsMouseMoveEvent( QMouseEvent *e )
{
  m_pos = e->globalPos();

  Smb4KShareWidgetItem *item = (Smb4KShareWidgetItem *)findItem( e->pos() );

  if ( item )
  {
    config()->setGroup( "User Interface" );

    if ( !m_tooltip && hasMouse() && config()->readBoolEntry( "Show Share Tooltip", true ) )
    {
      m_tooltip = new Smb4KShareTooltip( item, e->globalPos() );

      QTimer::singleShot( 2000, this, SLOT( slotShowTooltip() ) );
    }
  }
  else
  {
    if ( m_tooltip )
    {
      delete m_tooltip;
      m_tooltip = NULL;
    }
    else
    {
      m_tooltip = NULL;
    }
  }

  QIconView::contentsMouseMoveEvent( e );
}


KURLDrag *Smb4KShareWidget::dragObject()
{
  // Get the KURL of the item that is to be dragged:
  KURL url = KURL( ((Smb4KShareWidgetItem *)currentItem())->shareObject()->canonicalPath() );

  KURLDrag *drag = new KURLDrag( KURL::List::List( url ), this );
  drag->setPixmap( DesktopIcon( "folder" ) );
//   drag->dragCopy();

  return drag;
}


void Smb4KShareWidget::startDrag()
{
  if ( !m_allow_dragging )
  {
    return;
  }

  // Kill the tooltip, so it is not in the way:
  if ( m_tooltip )
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }

  KIconView::startDrag();
}


void Smb4KShareWidget::contentsDropEvent( QDropEvent *e )
{
  QIconViewItem *item = findItem( e->pos() );

  // Do we have to disallow the dropping?
  if ( !m_allow_dropping || !item )
  {
    e->ignore();
    KIconView::contentsDropEvent( e );
    return;
  }

  KURL::List src;

  // If we cannot decode it, we do not want it.
  if ( !KURLDrag::decode( e, src ) )
  {
    e->ignore();
    KIconView::contentsDropEvent( e );
    return;
  }

  KURL dest;
  dest.setPath( ((Smb4KShareWidgetItem *)item)->shareObject()->canonicalPath() );

  // Deny dropping if we dropped something on itself.
  // This was inspired by KonqOperations::doDrop() function.
  for ( KURL::List::Iterator it = src.begin(); it != src.end(); ++it )
  {
    if ( dest.equals( *it, true ) )
    {
      if ( e->source() == this || e->source()->parent() == this )
      {
        e->ignore();
        KIconView::contentsDropEvent( e );
        return;
      }
    }
  }

  // We only allow copying:
  KIO::CopyJob *job = KIO::copy( src, dest, true );
  job->setAutoErrorHandlingEnabled( true, NULL );
#if KDE_VERSION_MAJOR >= 3 && KDE_VERSION_MINOR >= 5
  job->setAutoWarningHandlingEnabled( true );
#endif

  KIconView::contentsDropEvent( e );
}


void Smb4KShareWidget::changeIcons()
{
  int alignment = (QString::compare( m_display, "list" ) == 0 ?
                  Smb4KShareWidgetItem::List :
                  Smb4KShareWidgetItem::Icons);

  Smb4KShareWidgetItem *item = (Smb4KShareWidgetItem *)firstItem();
  Smb4KShareWidgetItem *next_item = NULL;

  while ( item )
  {
    next_item = (Smb4KShareWidgetItem *)item->nextItem();

    // Reload icons:
    ((Smb4KShareWidgetItem *)item)->setupItem( alignment, m_mountpoint );

    item = next_item;
  }
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


/****************************************************************************
   Internal slots.
****************************************************************************/

void Smb4KShareWidget::slotMouseButtonPressed( int button, QIconViewItem *item, const QPoint &pos )
{
  if ( m_tooltip )
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }

  if ( item )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)item;

#ifdef __linux__
    // Enable the actions:
    if ( m_forcedUnmount )
    {
      m_force_action->setEnabled( true );
    }
#endif

    if ( !i->isBroken() )
    {
      m_filemanager_action->setEnabled( true );
      m_sync_action->setEnabled( m_rsync ? true : false );
    }
    else
    {
      m_filemanager_action->setEnabled( false );
      m_sync_action->setEnabled( false );
    }

    m_unmount_action->setEnabled( true );
  }
  else
  {
    m_unmount_action->setEnabled( false );
    m_filemanager_action->setEnabled( false );
    m_sync_action->setEnabled( false );
#ifdef __linux__
    m_force_action->setEnabled( false );
#endif

    clearSelection();
  }

  if ( button == KIconView::RightButton )
  {
    if ( item )
    {
      m_menu->popupMenu()->changeTitle( 0, SmallIcon( "hdd_mount" ), ((Smb4KShareWidgetItem *)item)->name() );
    }
    else
    {
      m_menu->popupMenu()->changeTitle( 0, SmallIcon( "hdd_mount" ), i18n( "Shares" ) );
    }

    m_menu->popupMenu()->exec( pos, 0 );
  }
}


void Smb4KShareWidget::slotSelectionChanged( QIconViewItem *item )
{
  if ( item )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)item;

    if ( !i->isBroken() )
    {
      m_filemanager_action->setEnabled( true );
      m_sync_action->setEnabled( m_rsync ? true : false );
    }
    else
    {
      m_filemanager_action->setEnabled( false );
      m_sync_action->setEnabled( false );
    }

#ifdef __linux__
    if ( m_forcedUnmount )
    {
      m_force_action->setEnabled( true );
    }
#endif

    m_unmount_action->setEnabled( true );
  }
}


void Smb4KShareWidget::slotShowTooltip()
{
  if ( m_tooltip && m_tooltip->item() == (Smb4KShareWidgetItem*)findItem( viewport()->mapFromGlobal( m_pos ) ) )
  {
    if ( hasMouse() )
    {
      m_tooltip->showTip( m_pos );
    }
    else
    {
      delete m_tooltip;
      m_tooltip = NULL;
    }
  }
  else
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }
}


/****************************************************************************
   External slots.
****************************************************************************/

void Smb4KShareWidget::slotMountedShares()
{
  QValueList<Smb4KShare *> shares = smb4k_core->mounter()->getShares();
  Smb4KShareWidgetItem *current = (Smb4KShareWidgetItem *)currentItem();

  if ( !shares.isEmpty() )
  {
    Smb4KShareWidgetItem *test_item = (Smb4KShareWidgetItem *)firstItem();
    Smb4KShareWidgetItem *next_item = NULL;

    // First remove those, that are not mounted anymore:
    while ( test_item )
    {
      Smb4KShare *share = smb4k_core->mounter()->findShareByPath( test_item->mountpoint() );
      next_item = (Smb4KShareWidgetItem *)test_item->nextItem();

      if( !share || (test_item->shareObject()->isForeign() && !m_showExternal) )
      {
        delete test_item;
        test_item = NULL;

        sort( sortDirection() );
      }

      test_item = next_item;
    }

    // Now process the entries in the list:
    for ( QValueListIterator<Smb4KShare *> it = shares.begin(); it != shares.end(); ++it )
    {
      // Check, whether the share is already in the list. Look for the
      // mount point.
      bool alreadyInView = false;

      for ( Smb4KShareWidgetItem *item = (Smb4KShareWidgetItem *)this->firstItem(); item; item = (Smb4KShareWidgetItem *)item->nextItem() )
      {
        if ( QString::compare( item->mountpoint(), (*it)->path() ) == 0 ||
             QString::compare( item->mountpoint(), (*it)->canonicalPath() ) == 0 )
        {
          if ( *it != item->shareObject() )
          {
            item->replaceShareObject( *it );
          }

          alreadyInView = true;
          break;
        }
        else
        {
          continue;
        }
      }

      if ( !alreadyInView )
      {
        if ( QString::compare( m_display, "icons" ) == 0 )
        {
          if ( !m_showExternal && (*it)->isForeign() )
          {
            continue;
          }
          else
          {
            new Smb4KShareWidgetItem( *it, this, Smb4KShareWidgetItem::Icons, m_mountpoint );

            continue;
          }
        }
        else if ( QString::compare( m_display, "list" ) == 0 )
        {
          if ( !m_showExternal && (*it)->isForeign() )
          {
            continue;
          }
          else
          {
            new Smb4KShareWidgetItem( *it, this, Smb4KShareWidgetItem::List, m_mountpoint );

            continue;
          }
        }

        // Do the sorting:
        this->sort( this->sortDirection() );
      }
    }

    if ( current != (Smb4KShareWidgetItem *)currentItem() )
    {
      m_unmount_action->setEnabled( false );
#ifdef __linux__
      m_force_action->setEnabled( false );
#endif
      m_filemanager_action->setEnabled( false );
      m_sync_action->setEnabled( false );

      clearSelection();
    }

    m_unmount_all_action->setEnabled( true );
  }
  else
  {
    delete current;
    clear();

    m_unmount_action->setEnabled( false );
#ifdef __linux__
    m_force_action->setEnabled( false );
#endif
    m_unmount_all_action->setEnabled( false );
    m_filemanager_action->setEnabled( false );
    m_sync_action->setEnabled( false );
  }
}


/***************************************************************************
   Slots for the menu.
***************************************************************************/


void Smb4KShareWidget::slotOpenFilemanager()
{
  if ( currentItem() && !((Smb4KShareWidgetItem *)currentItem())->isBroken() )
  {
    smb4k_core->open( ((Smb4KShareWidgetItem *)currentItem())->shareObject() );
  }
}


void Smb4KShareWidget::slotUnmountShare()
{
  if ( currentItem() )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)currentItem();
    smb4k_core->mounter()->unmountShare( i->shareObject() );
  }
}


void Smb4KShareWidget::slotUnmountAllShares()
{
  smb4k_core->mounter()->unmountAllShares();
}


void Smb4KShareWidget::slotForceUnmountShare()
{
  if ( currentItem() )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)currentItem();
    smb4k_core->mounter()->unmountShare( i->shareObject(), true );
  }
}


void Smb4KShareWidget::slotSynchronize()
{
  smb4k_core->synchronizer()->synchronize( ((Smb4KShareWidgetItem *)currentItem())->shareObject(), isShown() ? this : 0 );
}


void Smb4KShareWidget::slotIconChanged( int )
{
  changeIcons();

  adjustItems();
}

#include "smb4ksharewidget.moc"
