/*
 * smil2yuv.cc -- Converts a Kino smil project to a YUV4MPEG Stream
 * Copyright (C) 2002 Raphael Amaury Jacquot <sxpert@esitcom.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <iostream>
using std::cerr;
using std::endl;

#include <extensions.h>
#include <Diagnostics.h>
#include <PlayListDVProvider.h>
#include <Threader.h>
#include <unistd.h>

class SMIL2YUV : public Threader
{
	private:
		ExtendedPlayList playlist;
		int offset;
		int frames;
		int deinterlace_type;
		string audio_file;
		double speed;
		int scene;

	public: 
		SMIL2YUV( ) : 
			offset( 0 ), 
			frames( -1 ), 
			deinterlace_type( 0 ), 
			audio_file( "" ),
			speed( 1 ),
			scene( -1 )
		{
		}

		virtual ~SMIL2YUV( )
		{
		}

		virtual string LogId( )
		{
			return "SMIL2YUV";
		}

		void SetOffset( int _offset )
		{
			offset = _offset;
		}

		void SetFrames( int _frames )
		{
			frames = _frames - 1;
		}

		void SetDeinterlaceType( int _deinterlace_type )
		{
			deinterlace_type = _deinterlace_type;
		}

		void SetAudioFile( string _audio_file )
		{
			audio_file = _audio_file;
		}

		bool Append( string file_name )
		{
			return playlist.Append( file_name );
		}

		void SetSpeed( double _speed )
		{
			speed = _speed;
		}

		bool IsRunnable( )
		{
			return playlist.GetNumFrames( ) != 0;
		}

		void SetScene( int _scene )
		{
			scene = _scene;
		}

	protected:
		void Thread( )
		{
			bool first = true;
			bool running = true;

			PlayListDVProvider input( playlist );
			input.SetPumpSize( 50 );
			input.SetSpeed( speed );

			if ( scene != -1 )
			{
				PlayList copy;
				int index = 0;
				int begin = 0;
				int end = input.FindEndOfScene( begin );
				while ( index != scene )
				{
					index ++;
					begin = end + 1;
					end = input.FindEndOfScene( begin );
				}
				input.GetPlayList( begin, end, copy );
				input.SetPlayList( copy );
			}

			if ( frames != -1 || offset != 0 )
			{
				PlayList copy;
				int end = frames == -1 ? input.GetNumFrames( ) - offset - 1 : offset + frames;
				input.GetPlayList( offset, end, copy );
				input.SetPlayList( copy );
			}

			input.ThreadStart( );

			AudioExtractor *audio = NULL;
			YUV420Extractor *yuv = NULL;

			while( running && ThreadIsRunning( ) )
			{
				if ( input.GetOutputAvailable( ) > 0 )
				{
					Frame &frame = input.GetOutputFrame( );

					if ( first )
					{
						audio = AudioExtractor::GetExtractor( audio_file );
						yuv = YUV420Extractor::GetExtractor( deinterlace_type );
						audio->Initialise( frame );
						yuv->Initialise( frame );
						first = false;
					}
					audio->Output( frame );
					yuv->Output( frame );
					//fflush( stdout );
					input.QueueInputFrame( );
				}
				else
				{
					running = input.ThreadIsRunning( ) && input.PumpIsNotCleared( );
				}
			}

			audio->Flush( );
			yuv->Flush( );

			delete yuv;
			delete audio;

			input.ThreadStop( );
		}
};


void Usage( )
{
	cerr << "Usage: smil2yuv [ -i type ] [ -a audio-file ] file [ file ... ]" << endl;
	cerr << "Where: file is smil, dv avi (1 or 2) or raw dv" << endl;
	cerr << "       -i type   - 0 no deinterlacing, 1 bad deinterlacing" << endl;
	cerr << "                   2 4:1:1 subsampling" << endl;
	cerr << "       -a file   - mp2 or wav file to write to" << endl;
	cerr << "       -o offset - frame offset (default: 0)" << endl;
	cerr << "       -f count  - frame count (default: all)" << endl;
	cerr << "       -s speed  - floating point speed (default: 1)" << endl;
	cerr << "       -c scene  - scene to output (default: all)" << endl;
	exit( 0 );
}

int main( int argc, char **argv )
{
	bool result = true;
	int index;
	SMIL2YUV output;

	if ( isatty( fileno( stdout ) ) )
	{
		cerr << "Output be must redirected to a pipe or a file." << endl;
		Usage( );
	}

	Diagnostics::SetApp( "smil2yuv" );

	try
	{
		for ( index = 1; result && index < argc; index ++ )
		{
			if ( !strcmp( argv[ index ], "--help" ) )
				Usage( );
			else if ( !strcmp( argv[ index ], "-i" ) )
				output.SetDeinterlaceType( atoi( argv[ ++ index  ] ) );
			else if ( !strcmp( argv[ index ], "-a" ) )
				output.SetAudioFile( argv[ ++ index ] );
			else if ( !strcmp( argv[ index ], "-o" ) )
				output.SetOffset( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-f" ) )
				output.SetFrames( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-s" ) )
				output.SetSpeed( atof( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-c" ) )
				output.SetScene( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-v" ) )
				Diagnostics::SetLevel( atoi( argv[ ++ index ] ) );
			else
				result = output.Append( argv[ index ] );
		}

		if ( result && output.IsRunnable( ) )
		{
			output.ThreadExecute( );
		}
		else if ( argc > 1 )
		{
			cerr << "Failed to load " << argv[ index - 1 ] << endl;
			Usage( );
		}
		else
		{
			Usage( );
		}
	}
	catch ( string exc )
	{
		cerr << "Exception caught: " << exc << endl;
	}

	return !result;
}
