/*==================================================================
 * sample.c - Audio sample routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/

#include "config.h"

#include <stdio.h>
#include <string.h>
#include "uif_sfont.h"
#include "sfont.h"
#include "sample.h"
#include "util.h"
#include "smurfcfg.h"
#include "i18n.h"

/* use libsndfile as default, use audiofile only if explicitly requested */
#ifndef WITH_AUDIOFILE
#include <sndfile.h>
#else
#include <audiofile.h>

/* using audiofile 0.2.0 or newer? define AUDIOFILE_ZERO_TWO if so */
#if defined (LIBAUDIOFILE_MAJOR_VERSION) && (LIBAUDIOFILE_MAJOR_VERSION > 0 || LIBAUDIOFILE_MINOR_VERSION >= 2)
#define AUDIOFILE_ZERO_TWO 1
#endif

#endif

/* sample data policy:
Sample data info is stored in a linked list of SFSamDataInfo structures,
sam_datainfo_list. Sample data resides in an open sound font or the sample
buffer, which is a temporary file used for storing unsaved and undo sample
data. Multiple samples can reference the same sample data (SFSamDataInfo
structure), this occurs if a sample is copied from one sound font to another.
The default policy is to try to use sample data in open sound fonts to reduce
the amount of temporary buffer storage. So if 2 or more unsaved samples from
different sound fonts refer to the same sample data in the sample buffer, then
the first sound font saved will cause those samples to then refer to the sample
data in that sound font file. A subsequent save of one of the other SF files
will cause a new SFSamDataInfo structure to be created for that sample.
Closing a sound font involves making sure that sample data referenced
by samples in other sound fonts gets duplicated into the sample buffer, and
those samples pointed to the new data.
*/

GSList *sam_datainfo_list = NULL;  /* linked list of SFSamDataInfo structs */

FILE *sambuf_fd = NULL;
gint sambuf_length;		/* sample buffer length in samples */

SFSample *sam_in_view = NULL;	/* ptr to sample in samview or NULL */
gint16 *sam_data_in_view;	/* data of sample in samview */

static gint sam_buf_write (void *data, SFSample * sam);
static void sam_buf_clean (void);

/* initialize the sample buffer */
int
sam_buf_init (void)
{
  if (sambuf_fd)
    return (OK);

  if (!(sambuf_fd = tmpfile ()))
    return (logit (LogFubar | LogErrno,
	_("Failed to create temporary file for sample buffer")));

  sambuf_length = 0;

  return (OK);
}

/* destroy sample buffer (closing a file opened by tmpfile() destroys it) */
void
sam_buf_destroy (void)
{
  if (!sambuf_fd)
    return;

  fclose (sambuf_fd);
  sambuf_fd = NULL;
}

/* allocates a SFSamDataInfo structure and adds it to sam_datainfo_list */
SFSamDataInfo *
sam_datainfo_new (void)
{
  SFSamDataInfo *datainfo;

  datainfo = sfont_samdatainfo_alloc ();
  sam_datainfo_list = g_slist_append (sam_datainfo_list, datainfo);
  return (datainfo);
}

/* removes a SFSamDataInfo structure from sam_datainfo_list and frees it */
void
sam_datainfo_destroy (SFSamDataInfo *datainfo)
{
  sam_datainfo_list = g_slist_remove (sam_datainfo_list, datainfo);
  sfont_samdatainfo_free (datainfo);
}

/* relocates sample data from a sound font into the sample buffer */
gint
sam_datainfo_move2sambuf (SFSample *sam, SFData *sf)
{
  gpointer buf;

  if (sam->datainfo->samfile) return (OK); /* already in sample buffer? */

  if (!(buf = sam_load_sample (sam, sam->datainfo->size, 0, NULL)))
    return (FAIL);

  sam_buf_write (buf, sam);

  return (OK);
}

/* write sample data to sample buffer and update sample */
static gint
sam_buf_write (void *data, SFSample * sam)
{
  if (!sambuf_fd)
    if (!sam_buf_init ())
      return (FAIL);

  if (!safe_fseek (sambuf_fd, (long int) (sambuf_length * 2), SEEK_SET))
    return (logit (LogFubar,
	_("Failed to seek to end of sample buffer file")));

  /* save the sample to the sample file buffer */
  if (!safe_fwrite (data, (sam->end + 1) * 2, sambuf_fd))
    return (FAIL);

  sam->datainfo->samfile = 1;
  sam->datainfo->sf = NULL;
  sam->datainfo->start = sambuf_length;

  sambuf_length += sam->end + 1;

  return (OK);
}

/* "cleans" up sample buffer if more than SMURFCFG_SAM_BUF_WASTE bytes are
wasted by unused samples, opens new file and writes used samples to it and
deletes old */
static void
sam_buf_clean (void)
{
  struct st_saminfo  /* structure stores old sample info */
  {
    SFSamDataInfo *datainfo;
    gint start;
  };

  gint maxwaste;
  FILE *temp_fd;	/* temp file descriptor for new sample buffer */
  GSList *p;
  gint used = 0;		/* amount of used sample buffer space */
  gint size, i;
  gboolean error = FALSE;
  gint newpos = 0;		/* current position in new sample buffer */
  guint8 *buf;
  GArray *backup_saminfo;  /* array of old samdata info (in case of error) */
  SFSamDataInfo *datainfo;
  struct st_saminfo bakinfo;	/* structure stores old sample info */
  struct st_saminfo *reinfo;

  if (!sambuf_fd)
    return;

  /* calc used space in sample buffer and remove unused data info structs */
  p = sam_datainfo_list;
  while (p)  /* loop over sample data information */
    {
      datainfo = (SFSamDataInfo *)(p->data);
      p = g_slist_next (p);

      if (!datainfo->refcount && !datainfo->dorefcount)  /* no refs? */
	sam_datainfo_destroy (datainfo);  /* destroy unused sam data info */
      else if (datainfo->samfile)
	used += datainfo->size;  /* add to used size accumulator */
    }

  /* SAM_BUF_WASTE is in megabytes */
  maxwaste = smurfcfg_get_int (SMURFCFG_SAM_BUF_WASTE);

  /* if wasted space is less than max tolerated, then return */
  if ((sambuf_length - used) * 2 <= maxwaste * 1024 * 1024)
    return;

  if (!(temp_fd = tmpfile ()))
    {
      logit (LogFubar | LogErrno, _("Failed to create temporary file for sample buffer"));
      return;
    }

  backup_saminfo = g_array_new (FALSE, FALSE, sizeof (bakinfo));

  /* copy used samples from old sample buffer to new */
  p = sam_datainfo_list;
  while (p)  /* loop over sample data info */
    {
      datainfo = (SFSamDataInfo *) (p->data);
      p = g_slist_next (p);
      if (!datainfo->samfile)
	continue;		/* only samples in sam buffer */

      /* save sample info (in case of error) */
      bakinfo.datainfo = datainfo;
      bakinfo.start = datainfo->start;
      g_array_append_val (backup_saminfo, bakinfo);

      size = datainfo->size * 2;

      /* copy sample to new sample buffer */
      if (!(buf = safe_malloc (size)))
	error = TRUE;

      /* seek to start of sample buffer */
      if (!error && !safe_fseek (sambuf_fd, datainfo->start * 2, SEEK_SET))
	error = TRUE;
      
      if (!error && !safe_fread (buf, size, sambuf_fd))
	error = TRUE;
      if (!error && !safe_fwrite (buf, size, temp_fd))
	error = TRUE;

      g_free (buf);
      
      datainfo->start = newpos;
      newpos += datainfo->size;

      if (error)
	{		/* if error occured, then restore sample info */
	  i = backup_saminfo->len - 1;
	  while (i >= 0)
	    {
	      reinfo = &g_array_index (backup_saminfo, struct st_saminfo, i);
	      datainfo = reinfo->datainfo;
	      datainfo->start = reinfo->start;
	      i--;
	    }
	  g_array_free (backup_saminfo, TRUE);
	  fclose (temp_fd);
	  return;
	}
    }

  g_array_free (backup_saminfo, TRUE);
  fclose (sambuf_fd);		/* close defunct sample buffer */
  sambuf_fd = temp_fd;		/* set temp buffer to new one */
  sambuf_length = newpos;	/* set length of buffer to new length */
}

/*  sam_load_sample() - load a sound font sample (possibly "cached" in samview)
    sam = the sample to load
    size = size to load (in samples)
    ofs = offset from start of sample (in samples)
    bufptr is an optional pointer to place sample data (NULL to allocate one)
    returns: pointer to sample data which should be free'd (if !bufptr) */
gpointer
sam_load_sample (SFSample * sam, gint size, gint ofs, gpointer bufptr)
{
  guint32 pos;
  FILE *fd;
  gpointer buf;

  /* if sample is a ROM sample, can't load it :( */
  if (sam->sampletype & SF_SAMPLETYPE_ROM)
    return (NULL);

  if (!bufptr)
    {  /* allocate buffer space if no bufptr supplied */
      if (!(buf = safe_malloc (size * 2)))
	return (NULL);
    }
  else				/* use supplied buffer ptr */
    buf = bufptr;

  /* if sample is already loaded (for viewing purposes) then memcopy it */
  if (sam == sam_in_view && sam_data_in_view)
    {
      memcpy (buf, (gint16 *) sam_data_in_view + ofs, size * 2);
      return (buf);
    }

  pos = (sam->datainfo->start + ofs) * 2;
  if (sam->datainfo->samfile == 0)
    {
      fd = sam->datainfo->sf->sffd;
      pos += sam->datainfo->sf->samplepos;
    }
  else
    fd = sambuf_fd;

  if (!safe_fseek (fd, pos, SEEK_SET))
    {
      if (!bufptr) g_free (buf);
      return (NULL);
    }

  if (!safe_fread (buf, size * 2, fd))
    {
      if (!bufptr) g_free (buf);
      return (NULL);
    }

  return (buf);
}

/* creates a new sample and adds it to a sound font, name is name of sample,
   rate is sampling rate, samdata is the audio data for new sample, size is
   the # of samples in samdata */
SFSample *
sam_create_sample (gchar *name, gint rate, gint16 *samdata, gint size,
  SFData *sf)
{
  SFSample *sam;

  sam = sfont_sample_alloc ();  /* allocate SFSample structure */

  /* create SFSamDataInfo structure and assign it to new sample */
  sam->datainfo = sam_datainfo_new ();
  sam->datainfo->size = size;
  sam->datainfo->refcount++;  /* increment reference count */

  sfont_set_namestr (sf, sam->name, name);
  sam->end = size - 1;	/* end is last point of sample (ofs from start) */
  sam->loopstart = 8;		/* offset from start to start of loop */
  sam->loopend = size - 8;	/* ofs from start to loop end */
  sam->samplerate = rate;
  sam->origpitch = 60;
  sam->pitchadj = 0;
  sam->sampletype = SF_SAMPLETYPE_MONO;	/* from sfont.h !libsndfile! */

  /* write to sample file buffer */
  if (!sam_buf_write (samdata, sam))
      return (NULL);

  sfont_add_sample (sf, sam);

  return(sam);
}

/* clone a sample */
SFSample *
sam_clone_sample (SFData * dst, SFData * src, SFSample * sam)
{
  SFSample *dupsam;

  dupsam = sfont_sample_dup (sam, SFDUP_NORMAL, TRUE);
  sfont_add_sample (dst, dupsam);

  return (dupsam);
}

/* cut a portion of a sample */
gint
sam_cut_sample (SFData * sf, SFSample * sam, guint spos, guint epos)
{
  gint after;		/* size of sample data after cut */
  gint newsize;		/* size of sample minus cut data */
  gint oldend;
  gint16 *buf;
  SFSamDataInfo *oldinfo;

  if (spos > epos || epos > sam->end)
    return (FAIL);

  newsize = sam->end + 1 - (epos - spos + 1);	/* size of sample - cut */

  if (newsize < smurfcfg_get_int (SMURFCFG_SAM_MINLOOP)
      + smurfcfg_get_int (SMURFCFG_SAM_MINLOOPPAD) * 2)
    {
      logit (LogFubar,
	_("Cutting requested area of sample would make it too small"));
      return (FAIL);
    }

  /* load the sample */
  if (!(buf = sam_load_sample (sam, sam->end + 1, 0, NULL)))
      return (FAIL);

  oldinfo = sam->datainfo;  /* save old sample data info structure */
  sam->datainfo = sam_datainfo_new ();  /* create new data info struct */
  sam->datainfo->size = newsize;
  sam->datainfo->refcount++;  /* increment reference count */

  after = (sam->end - epos) * 2;	/* size of data after cut */
  if (after)			/* join portions before and after cut */
    memcpy (buf + spos, buf + epos + 1, after);

  oldend = sam->end;		/* save old end position in case of error */
  sam->end = newsize - 1;	/* set sample end position */

  if (!sam_buf_write (buf, sam))  /* write new sample to sample buffer */
    {
      sam_datainfo_destroy (sam->datainfo);
      sam->datainfo = oldinfo;
      sam->end = oldend;
      g_free (buf);
      return (FAIL);
    }
  g_free (buf);

  oldinfo->refcount--;  /* decrement old sample data reference counter */

  /* update loop pointers if needed */

  /* if loop start within cut, then reset to beginning of sample */
  if (sam->loopstart >= spos && sam->loopstart <= epos)
    sam->loopstart = smurfcfg_get_int (SMURFCFG_SAM_MINLOOPPAD);
  else if (sam->loopstart > epos)  /* if loop start after cut.. */
    sam->loopstart -= epos - spos + 1;  /* subtract cut length */

  /* if loop end within cut, then reset to end of sample */
  if (sam->loopend >= spos && sam->loopend <= epos)
    sam->loopend = sam->end - smurfcfg_get_int (SMURFCFG_SAM_MINLOOPPAD);
  else if (sam->loopend > epos)  /* if loop end after cut.. */
    sam->loopend -= epos - spos + 1;  /* subtract cut length */

  return (OK);
}

/* Load audio file data as an SFSample (2 samples in the case of stereo) */
SFSample *
sam_load_file_as_sample (AudioFileRef * afref, SFData * sf)
{
  gint16 *buf;
  gint16 *och;			/* other channel ptr, for stereo samples */
  gint samples;
  gint i, i2;
  SFSample *samptr;

  /* clean up sample buffer if excessive unused sample waste */
  sam_buf_clean ();

  /* allocate space for sample data (3 times for stereo data to separate) */
  if (!(buf = safe_malloc (afref->info.frames
	* ((afref->info.channels == 2) ? 3 : 1) * 2)))
    return (NULL);

  /* read the sample into memory */
  if (!sam_read_file (afref, buf, afref->info.frames))
    {
      g_free (buf);
      return (NULL);
    }

  samples = afref->info.frames * afref->info.channels;
  if (afref->info.width == 16 && !afref->info.signd)
    {				/* u16 > s16 */
      for (i = 0; i < samples; i++)
	buf[i] ^= 0x8000;
    }
  else if (afref->info.width == 8 && !afref->info.signd)
    {				/* u8 -> s16 */
      for (i = 0; i < samples; i++)
	buf[i] = (buf[i] << 8) ^ 0x8000;
    }
  else if (afref->info.width == 8)
    {				/* signed 8  -> signed 16 */
      for (i = 0; i < samples; i++)
	buf[i] = buf[i] << 8;
    }

  /* sample data is now signed 16 bit, check if data needs separation */
  if (afref->info.channels == 2)
    {
      och = buf + afref->info.frames * 2;	/* och points after sample data */
      i2 = afref->info.frames;
      for (i = 0; i < i2; i++)
	och[i] = buf[(i << 1) + 1];

      for (i = 1; i < i2; i++)
	buf[i] = buf[i << 1];
    }

  samptr = sam_create_sample ("", afref->info.rate, buf, afref->info.frames,
			      sf);

  if (!samptr)
    {
      g_free (buf);
      return (NULL);
    }

  if (afref->info.channels == 2)
    {
      if (!sam_create_sample ("", afref->info.rate, och, afref->info.frames,
			      sf))
	{
	  g_free (buf);
	  sfont_remove_sample (sf, samptr);	/* remove already loaded ch */
	  return (NULL);
	}
    }

  g_free (buf);

  return (samptr);
}

/* check if audio is okay for loading as sample data */
gint
sam_check_file (AudioFileRef * afref)
{
  if (afref->info.channels != 1 && afref->info.channels != 2)
    {
      logit (LogFubar, _("Sample must be mono or stereo"));
      return (FAIL);
    }

#ifndef AUDIOFILE_ZERO_TWO
  if (afref->info.width != 16 && afref->info.width != 8)
    {				/* check width */
      logit (LogFubar, _("Sample must be 8 or 16 bits, sample is %d bits"),
	afref->info.width);
      return (FAIL);
    }
#endif

  if (afref->info.rate < SF_MIN_SAMPLERATE
    || afref->info.rate > SF_MAX_SAMPLERATE)
    {
      logit (LogFubar, _("Sample rate is out of bounds, sample rate is %d"),
	afref->info.rate);
      return (FAIL);
    }

  if (afref->info.frames < SF_MIN_SAMPLE_LENGTH)
    {				/* sample is too small? */
      logit (LogFubar, _("Sample is less than the minimum of %d samples"),
	SF_MIN_SAMPLE_LENGTH);
      return (FAIL);
    }

  /* sample is too big? */
  if (afref->info.frames * 2 > MAX_SAMPLE_SIZE)
    {
      logit (LogFubar, _("Sample is larger than %d bytes"), MAX_SAMPLE_SIZE);
      return (FAIL);
    }

  return (OK);
}

gint
sam_export_sample (SFSample * sam, SFData * sf, gchar * fname,
  enum AudioFileType filetype)
{
  AudioInfo afnfo;
  AudioFileRef *afref;
  void *buf;

  /* load buffer with sample data */
  if (!(buf = sam_load_sample (sam, sam->end + 1, 0, NULL)))
      return (FAIL);

  afnfo.filetype = filetype;	/* enum in sample.h */
  afnfo.rate = sam->samplerate;
  afnfo.channels = 1;
  afnfo.width = 16;

  /* open audio file for writing */
  if (!(afref = sam_open_file (fname, FALSE, &afnfo)))
    {
      g_free (buf);
      return (FAIL);
    }

  if (!sam_write_file (afref, buf, sam->end + 1))
    {
      sam_close_file (afref);
      g_free (buf);
      return (FAIL);
    }

  sam_close_file (afref);
  g_free (buf);

  return (OK);
}


/* +++ audio library specific routines +++ */

#ifndef WITH_AUDIOFILE		/*---- libsndfile routines ----*/

/* libsndfile: open audio file */
AudioFileRef *
sam_open_file (gchar * fname, gboolean read, AudioInfo * afmt)
{
  AudioFileRef *afref;
  SNDFILE *afh;
  SF_INFO sfnfo;
  gboolean raw = FALSE;
  gchar errbuf[100];

  if (read && afmt)
    raw = TRUE;

  if (raw || !read)
    {				/* raw file load or write mode requested? */
      if (!afmt)
	return (NULL);		/* afmt must be specified */

      sfnfo.samplerate = afmt->rate;
      sfnfo.channels = afmt->channels;
      sfnfo.pcmbitwidth = afmt->width;

      if (!read)
	{	       	/* if write mode, then see what format to write in */
	  switch (afmt->filetype)
	    {
	    case AUDIO_TYPE_AIFF:
	      sfnfo.format = SF_FORMAT_AIFF;
	      break;
	    case AUDIO_TYPE_AU:
	      sfnfo.format = SF_FORMAT_AU;
	      break;
	    default:
	      sfnfo.format = SF_FORMAT_WAV;
	      break;
	    }
	  sfnfo.format |= SF_FORMAT_PCM;
	}
      else
	{			/* raw mode read */
	  sfnfo.format = SF_FORMAT_RAW;
	  if (afmt->width > 8)
	    sfnfo.format |=
	      (afmt->lendian ? SF_FORMAT_PCM_LE : SF_FORMAT_PCM_BE);
	  else
	    sfnfo.format |=
	      (afmt->signd ? SF_FORMAT_PCM_S8 : SF_FORMAT_PCM_U8);
	}
    }

  if (read)
    {				/* read mode requested? */
      if (!(afh = sf_open_read (fname, &sfnfo)))
	{
	  sf_error_str (afh, errbuf, 100);
	  logit (LogFubar, "libsndfile: %s: %s",
	    _("Failed to open audio file for reading"), errbuf);
	  return (NULL);
	}
    }
  else
    {				/* write mode requested? */
      if (!(afh = sf_open_write (fname, &sfnfo)))
	{
	  sf_error_str (afh, errbuf, 100);
	  logit (LogFubar, "libsndfile: %s: %s",
	    _("Failed to open audio file for writing"), errbuf);
	  return (NULL);
	}
    }

  if (!(afref = safe_malloc (sizeof (AudioFileRef))))
    {
      sf_close (afref->afh);
      return (NULL);
    }

  afref->afh = afh;

  /* set sample information */
  afref->info.frames = sfnfo.samples;
  afref->info.rate = sfnfo.samplerate;
  afref->info.channels = sfnfo.channels;
  afref->info.width = sfnfo.pcmbitwidth;
  afref->info.lendian = TRUE;
  afref->info.signd = TRUE;

  return (afref);
}

gint
sam_read_file (AudioFileRef * afref, void *buf, gint count)
{
  gchar errbuf[100];
  gint c;

  if ((c = sf_readf_short (afref->afh, buf, count)) != count)
    {
      sf_error_str (afref->afh, errbuf, 100);
      logit (LogFubar, "libsndfile: %s (%d of %d frames): %s",
	_("Failed to read sample data"), c, count, errbuf);
      return (FAIL);
    }
  return (OK);
}

gint
sam_write_file (AudioFileRef * afref, void *buf, gint count)
{
  gchar errbuf[100];
  gint c;

  if ((c = sf_writef_short (afref->afh, buf, count)) != count)
    {
      sf_error_str (afref->afh, errbuf, 100);
      logit (LogFubar, "libsndfile: %s (%d of %d frames): %s",
	_("Failed to write sample data"), c, count, errbuf);
      return (FAIL);
    }
  return (OK);
}

/* libsndfile: close an open audio file */
void
sam_close_file (AudioFileRef * afref)
{
  sf_close (afref->afh);
  g_free (afref);
}


#else		/*---- AUDIOFILE routines ----*/


AudioFileRef *
sam_open_file (gchar * fname, gboolean read, AudioInfo * afmt)
{
  AudioFileRef *afref;
  AFfilehandle affd;
  AFfilesetup afsetup = NULL;
  gint fmt, wdth, chan, samrate, frame_count;
  gboolean raw = FALSE;

  if (read && afmt)
    raw = TRUE;

  if (raw || !read)
    {
      gint filefmt;

      if (!afmt) return (NULL);

      afsetup = afNewFileSetup ();
      afInitChannels (afsetup, AF_DEFAULT_TRACK, afmt->channels);
      afInitRate (afsetup, AF_DEFAULT_TRACK, afmt->rate);
      afInitSampleFormat (afsetup, AF_DEFAULT_TRACK,
			  afmt->signd ? AF_SAMPFMT_TWOSCOMP
			  : AF_SAMPFMT_UNSIGNED,
			  afmt->width);

      if (!read)	/* write file? */
	{
	  switch (afmt->filetype)
	    {
	    case AUDIO_TYPE_AIFF:
	      filefmt = AF_FILE_AIFFC;
	      break;
	    case AUDIO_TYPE_AU:
	      filefmt = AF_FILE_NEXTSND;
	      break;
	    default:
	      filefmt = AF_FILE_WAVE;
	      break;
	    }
	}
      else
	{
	  filefmt = AF_FILE_RAWDATA;
	  afInitByteOrder (afsetup, AF_DEFAULT_TRACK,
			   afmt->lendian ? AF_BYTEORDER_LITTLEENDIAN
			   : AF_BYTEORDER_BIGENDIAN);
	}

      afInitFileFormat (afsetup, filefmt);
    }

  /* open the audio file */
  affd = afOpenFile (fname, read ? "r" : "w", afsetup);
  if (!affd)
    {
      logit (LogFubar, "audiofile: %s",
	read ? _("Failed to open audio file for reading") :
	_("Failed to open audio file for writing"));
      return (NULL);
    }

#ifdef AUDIOFILE_ZERO_TWO
  /* set up the virtual sample format (audiofile will convert to this) */
  fmt = AF_SAMPFMT_TWOSCOMP;
  wdth = 16;
  afSetVirtualSampleFormat (affd, AF_DEFAULT_TRACK, fmt, wdth);

  /* convert to host byte order */
  if (G_BYTE_ORDER == G_LITTLE_ENDIAN)
    afSetVirtualByteOrder (affd, AF_DEFAULT_TRACK, AF_BYTEORDER_LITTLEENDIAN);
  else
    afSetVirtualByteOrder (affd, AF_DEFAULT_TRACK, AF_BYTEORDER_BIGENDIAN);
#else
  afGetSampleFormat (affd, AF_DEFAULT_TRACK, &fmt, &wdth);
#endif

  chan = afGetChannels (affd, AF_DEFAULT_TRACK);
  samrate = afGetRate (affd, AF_DEFAULT_TRACK);
  frame_count = afGetFrameCount (affd, AF_DEFAULT_TRACK);

  if (!(afref = safe_malloc (sizeof (AudioFileRef))))
    {
      afCloseFile (affd);
      return (NULL);
    }

  /* assign loaded file format vars, so calling function can view them */
  afref->info.frames = frame_count;
  afref->info.rate = samrate;
  afref->info.channels = chan;
  afref->info.width = wdth;
  afref->info.lendian = TRUE;

  /* audiofile reports incorrectly for 8bit UNSIGNED/SIGNED format? */
  if (wdth != 8)
    afref->info.signd = (fmt == AF_SAMPFMT_TWOSCOMP);
  else
    afref->info.signd = FALSE;

  afref->afh = affd;

  return (afref);
}

gint
sam_read_file (AudioFileRef * afref, void *buf, gint count)
{
  gint c, i;

  if ((c = afReadFrames (afref->afh, AF_DEFAULT_TRACK, buf, count)) != count)
    {
      logit (LogFubar, "libaudiofile: %s (%d of %d frames)",
	_("Failed to read sample data"), c, count);
      return (FAIL);
    }

  /* if 8 bit sample data, store in 16 bits (not converted) like libsndfile */
  if (afref->info.width == 8)
    {
      for (i = c - 1; i >= 0; i--)
	((gint16 *) buf)[i] = ((gint8 *) buf)[i];
    }

  return (OK);
}

gint
sam_write_file (AudioFileRef * afref, void *buf, gint count)
{
  gint c;

  if ((c = afWriteFrames (afref->afh, AF_DEFAULT_TRACK, buf, count)) != count)
    {
      logit (LogFubar, "libaudiofile: %s (%d of %d frames)",
	_("Failed to write sample data"), c, count);
      return (FAIL);
    }
  return (OK);
}

/* AUDIOFILE: close an open audio file */
void
sam_close_file (AudioFileRef * afref)
{
  afCloseFile (afref->afh);
  g_free (afref);
}

#endif /* audiofile routines */
