/*==================================================================
 * sequencer.c - Generic sequencer routines (independent of driver)
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include "config.h"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <glib.h>
#include "sequencer.h"
#include "wavetable.h"
#include "util.h"
#include "smurfcfg.h"
#include "i18n.h"

#include "drivers/seq_alsa.h"
#include "drivers/seq_oss.h"

SeqDriverInfo seq_drivers[SEQ_COUNT] = {
  { N_("NONE"), NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL },

#ifdef ALSA_SUPPORT
  { N_("ALSA"),
    seq_alsa_load_config,
    seq_alsa_vkeyb_init,
    seq_alsa_vkeyb_close,
    seq_alsa_set_bank,
    seq_alsa_set_preset,
    seq_alsa_note_on,
    seq_alsa_note_off,
    seq_alsa_pitch_bender,
    seq_alsa_pitch_bend_range,
    seq_alsa_main_volume,
    seq_alsa_chorus,
    seq_alsa_reverb,
  },
#endif

#ifdef OSS_SUPPORT
  { N_("OSS"),
    NULL,
    seq_oss_init,
    NULL,
    seq_oss_set_bank,
    seq_oss_set_preset,
    seq_oss_note_on,
    seq_oss_note_off,
    seq_oss_pitch_bender,
    seq_oss_pitch_bend_range,
    seq_oss_main_volume,
    seq_oss_chorus,
    seq_oss_reverb,
  },
#endif
};

gint seq_driver = SEQ_NONE;	/* active sequencer driver */
gboolean seq_active = FALSE;	/* state of sequencer */


/* update sequencer variables from config and init driver appropriately */
gint
seq_init_from_config (void)
{
  GTokenValue *val;
  gint drvr_id;
  gboolean retval;
  gint i;

  /* load config for each sequencer driver */
  for (i = 0; i < SEQ_COUNT; i++)
    if (seq_drivers[i].config)
      (*seq_drivers[i].config) ();

  val = smurfcfg_get_val (SMURFCFG_SEQ_DRIVER);

  /* if not AUTO and driver name is found, init that driver */
  if (strcmp (val->v_string, "AUTO") != 0
    && (drvr_id = seq_locate_byname (val->v_string)) != -1)
    {
      log_message (_("Initializing sequencer driver \"%s\".."),val->v_string);

      seq_set_driver (drvr_id);
      retval = seq_init ();

      if (retval)
	log_message (_("Sequencer driver initialized"));
      else
	logit (LogFubar, _("Sequencer driver failed"));

      return (retval);
    }
  else return (seq_auto_init ());
}

/* "AUTO" detect and initialize supported sequencer driver */
gint
seq_auto_init (void)
{
  gint i;
  gchar *name;

  log_message (_("Looking for supported sequencer driver.."));

  if (SEQ_COUNT == 1)
    {
      log_message (_("No sequencer drivers were compiled into Smurf!"));
      return (FAIL);
    }

  for (i = 1; i < SEQ_COUNT; i++)
    {
      name = _(seq_drivers[i].name);
      log_message (_("Trying driver %s..."), name);

      seq_set_driver (i);
      if (seq_init ())
	{
	  log_message (_("Initialization of %s driver successful"), name);
	  return (OK);
	}
      else
	log_message (_("Driver %s failed"), name);
    }

  log_message (_("No sequencer support found!"));
  return (FAIL);
}

/* Sets active wavetable driver to one specified by its ID */
void
seq_set_driver (guint drvr_id)
{
  if (drvr_id >= SEQ_COUNT)
    return;

  seq_driver = drvr_id;
}

/* Find a driver by its name, return ID */
gint
seq_locate_byname (gchar * name)
{
  gint i;

  for (i = 0; i < SEQ_COUNT; i++)
    {
      if (strcmp (name, seq_drivers[i].name) == 0)
	return (i);
    }
  return (-1);
}

/* initialize active sequencer driver */
gint
seq_init (void)
{
  gint retval;

  if (seq_active)
    return (OK);		/* return if already open */

  if (!seq_drivers[seq_driver].init)	/* this driver has init function? */
    return (OK);

  /* run the sequencer init routine */
  if ((retval = (*seq_drivers[seq_driver].init) ()))
    seq_active = TRUE;

  return (retval);
}

/* close active sequencer driver */
void
seq_close (void)
{
  if (!seq_active)
    return;

  if (seq_drivers[seq_driver].close)	/* this driver has close function? */
    (*seq_drivers[seq_driver].close) ();

  seq_active = FALSE;
}

void
seq_set_bank (gint chan, gint bank)
{
  if (!seq_active || !seq_drivers[seq_driver].set_bank)
    return;

  (*seq_drivers[seq_driver].set_bank) (chan, bank);
}

void
seq_set_preset (gint chan, gint preset)
{
  if (!seq_active || !seq_drivers[seq_driver].set_preset)
    return;

  (*seq_drivers[seq_driver].set_preset) (chan, preset);
}

void
seq_note_on (gint chan, gint note, gint vel)
{
  if (!seq_active || !seq_drivers[seq_driver].start_note)
    return;

  wtbl_note_on_notify ();

  (*seq_drivers[seq_driver].start_note) (chan, note, vel);
}

void
seq_note_off (gint chan, gint note, gint vel)
{
  if (!seq_active || !seq_drivers[seq_driver].stop_note)
    return;

  (*seq_drivers[seq_driver].stop_note) (chan, note, vel);
}

void
seq_pitch_bender (gint chan, gint bendval)
{
  if (!seq_active || !seq_drivers[seq_driver].pitch_bender)
    return;

  (*seq_drivers[seq_driver].pitch_bender) (chan, bendval);
}

/* bend range val is in semitones (notes) */
void
seq_pitch_bend_range (gint chan, gint val)
{
  if (!seq_active || !seq_drivers[seq_driver].pitch_bend_range)
    return;

  (*seq_drivers[seq_driver].pitch_bend_range) (chan, val);
}

void
seq_main_volume (gint chan, gint val)
{
  if (!seq_active || !seq_drivers[seq_driver].main_volume)
    return;

  (*seq_drivers[seq_driver].main_volume) (chan, val);
}

void
seq_chorus (gint chan, gint val)
{
  if (!seq_active || !seq_drivers[seq_driver].chorus)
    return;

  (*seq_drivers[seq_driver].chorus) (chan, val);
}

void
seq_reverb (gint chan, gint val)
{
  if (!seq_active || !seq_drivers[seq_driver].reverb)
    return;

  (*seq_drivers[seq_driver].reverb) (chan, val);
}
