/*==================================================================
 * sfundo.h - Header file for Sound font undo/redo routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#ifndef __SFUNDO_H__
#define __SFUNDO_H__

#include <glib.h>

/* called at the beginning of all dofunc.save routines, checks to see if undo
   system is active and not currently running (blocks recursive calls to undo
   system) */
#define SFUNDO_ENTER() if (!sfdo_active || sfdo_running) return;

/*
   A "do" tree consists of a tree of SFDoEntries which are
   linked lists of SFDoItems. The SFDoEntries are groups of SFDoItems
   essentially and many operations act on them as a group.
   Each SFDoItem has an enumerated type (generator, sample, sampledata, preset,
   etc) and a pointer to the associated "state" data which can be used to
   restore the state of that type of object.
   Only a new entry (curpos) can be used for grouping. Groups can be nested.
   The same SFDoEntry is used for nested groups. A list of SFDoItems is kept
   in the tree, each is a pointer to the last node of the previous group. This
   list is temporary and exists only while the entry is opened as a group.
   Therefore sub groups are only available while they are open.
   Sub groups are useful for operations that call apon other functions that
   act on many items and group them. Should one of these functions fail, it
   would undo its own group of changes, and return FAIL. The calling routine
   could then undo all the other group data.
*/

typedef struct _SFDoTree
{
  GNode *root;	    /* points to the root (NULL) entry of the tree */
  GNode *curpos;    /* current position in do tree (SFDoEntry GNode) */
  GList *groups;    /* sub group SFDoItem ptrs (most recent first, prepend) */
}
SFDoTree;

typedef struct _SFDoEntry
{
  gchar *descr;	   /* description of this action, or NULL */
  GList *items;    /* list of SFDoItems (most recent item first, prepend) */
}
SFDoEntry;

typedef struct _SFDoItem
{
  guint16 type;			/* item type enumeration */
  gpointer state;		/* item state do data */
}
SFDoItem;

SFDoTree sfdo_tree;

gboolean sfdo_active;
gboolean sfdo_running;

void sfdo_init (void);
void sfdo_debug_dump (GNode *start, gint sibling, gboolean traverse,
		      gboolean detail);
void sfdo_group (gchar *descr);
void sfdo_done (void);
void sfdo_add (guint16 type, gpointer state);
void sfdo_retract (void);
void sfdo_retract_all (void);
void sfdo_undo (void);
void sfdo_redo (GNode *node);
void sfdo_jump (GNode *node);

#endif
