/* SNAC

 * Copyright (C) 1999 the Free Software Foundation

 * Authors : Matias Mutchinick, Amed Muoz Ramos, Jan Struyf

 *         

 * This program is free software; you can redistribute it and/or modify

 * it under the terms of the GNU General Public License as published by

 * the Free Software Foundation; either version 2 of the License, or

 * (at your option) any later version.

 *

 * This program is distributed in the hope that it will be useful,

 * but WITHOUT ANY WARRANTY; without even the implied warranty of

 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the

 * GNU General Public License for more details.

 *

 * You should have received a copy of the GNU General Public License

 * along with this program; if not, write to the Free Software

 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */





#include <glib.h>

#include <stdio.h>

#include <math.h>

#include <stdlib.h>

#include <string.h>

#include <ctype.h>

#include <errno.h>

#include "ag_defs.h"

#include "errorstring.h"

#include "evalstring.h"

#include "ag_math.h"



/**

 * Uncomment for debugging

 *

 * #define PRINT_NUMBER

 * #define PRINT_ATOI

 * #define PRINT_SUBEXPR

 */

 

#define PRINT_NUMBER 

 

static gint angle_mode; 

static gint size_mode;

static gint format_mode;



extern int     errno;

extern AgError errState;



/* Supported functions */

Funtable fun_table[] = {

	{ "sin"    , ag_sin  },

	{ "cos"    , ag_cos  },

	{ "tan"    , ag_tan  },

	{ "asin"   , ag_asin },

	{ "acos"   , ag_acos },

	{ "atan"   , ag_atan },

 	{ "sinh"   , ag_sinh },

	{ "cosh"   , ag_cosh },

	{ "tanh"   , ag_tanh },

	{ "ln"     , ag_ln   },

	{ "log"    , ag_log  },

	{ "exp"    , ag_exp  },

	{ "abs"    , ag_abs  },

	{ "inv"    , ag_inv  },

	{ "neg"    , ag_neg  },

	{ "sq"     , ag_sq   },

	{ "sqrt"   , ag_sqrt },

	{ 0        , 0       }

};



/* Supported operators */

Optable op_table[] = {

	{ "or"  , 1,  OP_OR,    TRUE,  ag_or    },

	{ "|"   , 1,  OP_OR,    FALSE, ag_or    },				

	{ "xor" , 2,  OP_XOR,   TRUE,  ag_xor   },

 	{ "and" , 3,  OP_AND,   TRUE,  ag_and   },

	{ "&"   , 3,  OP_AND,   FALSE, ag_and   },		

	{ "+"   , 4,  OP_PLUS,  FALSE, ag_plus  },

	{ "-"   , 5,  OP_MINUS, FALSE, ag_minus },

	{ ">>"  , 6,  OP_SHR,   FALSE, ag_shr   },

	{ "<<"  , 7,  OP_SHL,   FALSE, ag_shl   },

	{ "*"   , 8,  OP_MULT,  FALSE, ag_mult  },

	{ "/"   , 9,  OP_DIV,   FALSE, ag_div   },

	{ "^"   , 10, OP_POW,   FALSE, ag_pow   },

	{ 0     , 0,  0,        0,     0        }

};



/*

 * keep_double_within_limits

 * @num   : a number

 * @error : pointer to an error state variable

 * Ensure that @num is within the limits of calculation

 * max = 1e300 -> oveflow, min = 1e-300 -> zero.

 */

gdouble

keep_double_within_limits(gdouble  num,

			  gint    *error)

{

  	if (ag_math_overflow(num))

	{

		*error = AG_OVERFLOW;

		return 0;

	}

	

	if (ag_math_underflow(num))

		return 0;

	

	return num;

}



/* str_s_f_cmp

 * @str       : a string

 * @ptr_str   : a string

 * @ptr_start : the start of a substring in @ptr_str

 * @ptr_end   : the end of a substring in @ptr_str

 * Compare the whole string @str with the substring in @ptr_str 

 * limited by @ptr_start and @ptr_end.

 * The return value follows the rules of (strcmp). 

 */

gint 

str_s_f_cmp(gchar      *str,

	    gchar      *ptr_str,

	    gint        ptr_start, 

	    gint        ptr_end)

{

	gint i = 0;

	

	while((*(str+i+1) != 0 ||                     /* exit if end of str */

	       (i+1 < ptr_end-ptr_start)) &&          /* exit if end of ptr_str */

	      *(str+i) == *(ptr_str+i+ptr_start))     /* exit if diff */

		i++;

	

	return *(str+i) - *(ptr_str+i+ptr_start);

}



/*

 * round_double_fix

 * @num    : a number

 * @digits : number of digits

 * round @num for fixed notation at @digit digits.

 */

gdouble 

round_double_fix(gdouble   num,

		 gint      digits)

{

	gint     exp = 0, tmp, sign = 1, use_exp = 0;

	gdouble  int_part ,fract_part, aux;

	

	

	if(num == 0)

		return 0;

	

	if (num < 0){

		num = -num;

		sign = -1;

	}

	aux = num;

	

	/* Do we need to use exp notacion??? */

	if(num <= (1/pow(10,digits)) || num > 1e12){

		use_exp = 1;

		digits++;

	}

	

	/* Convert aux to a floating point 0.nnnnneNN and find out exp */

	if(aux < 1)

		while(aux < .1) {

			aux *= 10;

			exp--;

		}

	else

		while(aux >= 1) {

			aux /= 10;

			exp++;

		}

	

	if(use_exp)

		num = aux;

	else{

		/* Do not round more than 12 digits */

		if( exp + digits > 12)

			digits = 12 - exp;

	}

	

	/* Set the point where we will round n[n,,,n].nnnneNN */	

	for(tmp = 0 ; tmp < digits ; tmp++)

		num *= 10;

	

	fract_part = modf(num,&int_part);

	if(fract_part < .5 )

		fract_part = floor(fract_part);

	else

		fract_part = ceil(fract_part);

	

	num = int_part + fract_part;

	

	/* Set it back to normal or floating point depends of @use_exp */

	for( ; tmp > 0; tmp--)

		num /= 10;

	

	if (use_exp){

		if(exp < 0)

			for( ; exp < 0 ; exp++)

				num /= 10;

		else

			for( ; exp  > 0 ; exp--)

				num *= 10;

	}

	

	num = sign * num;

	return num;

}



/*

 * round_double_std

 * @num    : a number

 * @digits : number of digits

 * round @num the standard way at @digit digits.

 */

gdouble

round_double_std(gdouble  num,

		 gint     digits)

{

	gint     exp = 0, tmp, sign = 1;

	gdouble  int_part ,fract_part;

	

	

	if(num == 0)

		return 0;

	

	if (num < 0) {

		num = -num;

		sign = -1;

	}

	

	/* Convert the number to a floating point 0.nnnnneNN */

	if(num < 1)

		while(num < .1){

			num *= 10;

			exp--;

		} 

	else 

		while(num >= 1){

			num /= 10;

			exp++;

		}

	

	/* Set the point where we will round n[n,,,n].nnnneNN */

	for(tmp = 0; tmp < digits ; tmp++)

		num *= 10;

	

	fract_part = modf(num,&int_part);

	if(fract_part < .5 )

		fract_part = floor(fract_part);

	else

		fract_part = ceil(fract_part);

	

	num = int_part + fract_part;

	

	/* Set it back to floating point */

	for( ; tmp > 0 ; tmp--)

		num /= 10;

	

	/* Set it back to normal */

	if(exp < 0)

		for( ; exp < 0 ; exp++)

			num /= 10;

	else 

		for( ; exp  > 0 ; exp--)

			num *= 10;

	

	num = sign * num;

	return num;

}



/*

 * round_double

 * @num    : a number

 * @format : the format for rounding the number

 * @digits : number of digits

 * rounds a number for some specific needs (number of digits and

 * formating).

 */

gdouble

round_double(gdouble    num,

	     gint       format,

	     gint       digits)

{

	

	switch(format) {

	case STD:

		num = round_double_std(num,digits);

		break;

		

	case FIX:

		num = round_double_fix(num,digits);

		break;

		

	case SCI:

	case ENG:

		num = round_double_std(num,digits+1);

		break;

	case HEX:

	case BIN:

	case OCT:			

		break;

	}

	return num;

}



/*

 * is_base_char

 * @nc : character to test

 * return 1 if the character is a base character.

 */

gint 

is_base_char(gchar nc)

{

	switch(nc) {

	case 'h': /* Hex number */

	case 'b': /* Binary number */

	case 'o': /* Octal number */

		return 1;

	}

	return 0;

}



/*

 * get_base_from_char

 * @nc : the base character

 * return the base given by this character

 */

gdouble

get_base_from_char(gchar nc)

{

	switch(nc) {

	case 'h': return 16.0;  /* Hex number */

	case 'b': return 2.0;   /* Binary number */

	case 'o': return 8.0;   /* Octal number */

	}

	return 10.0;

}



/*

 * is_digit_char

 * @digit : the digit

 * return 1 if digit is a valid digit

 */

gint 

is_digit_char(gchar digit)

{

	if (digit >= '0' && digit <= '9') return 1;

	if (digit >= 'A' && digit <= 'F') return 1;

	if (digit >= 'a' && digit <= 'f') return 1;

	return 0;

}



/*

 * get_digit_value

 * @digit : the digit

 * @base  : the base

 * @value : the result

 * return 1 if digit is a valid digit

 */

gint 

get_digit_value(gchar digit, gdouble base, gdouble* value)

{

	gint num = -1;

	if (digit >= '0' && digit <= '9') num = (gint)digit - '0';

	if (digit >= 'A' && digit <= 'F') num = (gint)digit - 'A' + 10;

	if (digit >= 'a' && digit <= 'f') num = (gint)digit - 'a' + 10;

	if (num == -1) return 0;

	(*value) = (gdouble)num;

	if ((*value) >= base) return 0;

	return 1;

}



/*

 * rel_str_to_double

 * @expr      : string representation of a math expression (or number)

 * @num_start : where in @expr is the first character of the number

 * @num_end   : where in @expr is the end of the number

 * @error     : pointer to error state variable

 * Return the numeric result of a string reperesentation of a number.

 * Checks for errors, allowed syntax is ...

 * [-/+][n][.]nnnn[e[+/-]nn]

 */

gdouble 

rel_str_to_double(gchar     *expr,

		  gint       num_start, 

		  gint       num_end,

		  gint      *error)

{

	gint         exp_status = 0;     /* if 1, there is an exp */

	gint         digit_status = 0;   /* if 1, at least one digit */

	gint         point_status = 0;   /* if 1, there is a '.' */

	gint         pos;                /* Position */

	gint         exp_sign = 1;       /* Sign of the exponent */

        gint         sign = 1;           /* Sign of the number */

	gdouble      num = 0;            /* The number */

	gdouble      dec = 1;            /* # decimals ^ base */  

	gdouble      exp = 0;            /* Exponent */

	gdouble	     base = 10;          /* Base */

	gdouble      digit_value;	 /* Value of single digit */

	

#ifdef PRINT_ATOI

	printf("Evaluate number: ");

	for (pos = num_start; pos < num_end; pos++)

		printf("%c", expr[pos]);

	printf("\n");

#endif

	

	/* Initialize position */

	pos = num_start;

	

	/* Check if range valid */

	if (num_end <= num_start) {

		*error = SYN_ERROR;

		return 0;

	}



	/* First character is sign? */

	if ((pos < num_end) && ((expr[pos] == '-') || (expr[pos] == '+'))) {

		sign = (expr[pos] == '-')? -1 : 1;

		pos++;

	}

	

	/* Last character base [h,b,o]? */

	if (is_base_char(expr[num_end-1])) {

		base = get_base_from_char(expr[num_end-1]);

		num_end--;

	} else {

		/* C syntax hex value 0x?? style */

		if ((pos < num_end-1) && (expr[pos] == '0') && (expr[pos+1] == 'x')) {

			base = 16.0;

			pos += 2;	

		}

	}

	

	/* Parse rest of string */	

	while(pos < num_end) {

	

		if (get_digit_value(expr[pos], base, &digit_value)) {

			if (!digit_status)

				digit_status = 1;

					

			if (exp_status) {

				exp = exp*10 + digit_value;

			} else if(point_status) {

				dec *= base;

				num = num + digit_value / dec;

			} else {

				num = base*num + digit_value;

			}

			

			pos++;			

		} else {	

			switch(expr[pos]) {

			

				case 'e':  /* Exp */

					if(!digit_status || exp_status) {

						*error = SYN_ERROR;

						return 0;

					} 

			

					exp_status = 1;

					pos++;      

										

					if((expr[pos] == '-') || (expr[pos] == '+')){

						exp_sign = (expr[pos] == '-')? -1 : 1;

						pos++; 

					}

			

					/* If the character after the sign of the 'e' is not

					 * a digit: error */

					if(!isdigit(expr[pos])) { 

						*error = SYN_ERROR;

						return 0;

					}

					break;

			

				case '.': /* Point */

					if(exp_status || point_status) { 

						*error = SYN_ERROR;

						return 0;

					}

			

					point_status = 1;

					pos++;

			

					/* do not allow ".eNN" || "." */

					if(!digit_status && !isdigit(expr[pos])) {

						*error = SYN_ERROR;

						return 0;

					}

					break;

				

				case 39: /* ' */

					pos++;

					break;				

					

				default: /* Anything else */

					*error = SYN_ERROR;

					return 0;

			}

		}

	}

			

	for( ; exp > 0 ; exp--){

		if(exp_sign == 1)

			num *= base;

		else

			num /= base;

	}

	

	num = num * sign;

	return num;

}



/* 

 * get_function

 * @str   : string of a math expression

 * @start : the start of the function in the string

 * @end   : the end of the function in the string

 * @error : pointer to an error state variable

 * Return a pointer to a Funtable of the function requested in @str

 * between @start and @end.

 */

Funtable *

get_function(gchar     *str,

	     gint       start, 

	     gint       end,

	     gint      *error)

{

	gint i;

	

	for(i = 0 ; ; i++){

		if(fun_table[i].name == (gchar *) 0){

			*error = SYN_ERROR;

			return NULL;		

		}

		

		if(!str_s_f_cmp(fun_table[i].name,str,start,end)) 

			return &fun_table[i];

	}

}







/*

 * eval_function

 */

gdouble

eval_function(Funtable     *ftable,

	      gdouble       arg,

	      gint         *error)

{

	gdouble result;

	

	if(errState.state != 0) 

		return 0;

	

	result = (ftable->func) (arg,angle_mode,error);

	

	return result;

}





/* 

 * get_operator

 * @str      : string of a math expression

 * @start    : the start of the operator in the string

 * @str_end  : the end of the string 

 * @oper_end : the end of the operator

 * Return a pointer to a Optable of the operator requested in @str

 */

Optable *

get_operator(gchar     *str,

	     gint       start, 

             gint       str_end,	     

	     gint      *oper_end)

{

	gint i, j, o_end;	

	gchar* name;

	

	/* Scan operator table */

        for (i = 0 ; ; i++) {

		name = op_table[i].name;

                if(name == (gchar *)0) return NULL;

		

		/* Scan next operator name */

		j = 0; o_end = start;

		while (name[j] != 0 && name[j] == str[o_end]) {

			o_end++;

			j++;

		}

		

		/* Operator name found? */

		if (name[j] == 0 && o_end <= str_end) {

			/** 

			 * Some operators need a separator character 

			 * xor, or,... do

			 * +, -, *, /,... don't

			 */

			if (op_table[i].need_sep) {

				if (str[o_end] != '(' && str[o_end] != ' ' &&

				    str[o_end] != '.' && !isdigit(str[o_end])) return NULL;

			}

			

			/* Return entry */

		        if (oper_end != NULL) (*oper_end) = o_end;						

                        return &op_table[i];			

		}

        }

}



/* 

 * get_number

 * @str      : string of a math expression

 * @start    : the start of the number in the string

 * @str_end  : the end of the string

 * @num_end : the end of the number

 * Return true if number in @str

 */

gboolean

get_number(gchar     *str,

	   gint       start, 

	   gint       str_end,

	   gint      *num_end)

{

	gint j = 0, i = start;

	gboolean base16 = FALSE;

	gboolean baseok = FALSE;

	gboolean error = FALSE;

	gboolean done = FALSE;

	

	/* Scan complete number */

	while (i < str_end && !done && !error) {

	

		/* Valid character? */

		if (str[i] < '0' || str[i] > '9') {

			switch (str[i]) {

				/* Decimal point and ' */

				case '.': 

				case 39 : 

					break;

					

				/* Exponential - or hex digit */					

				case 'e': 

					if (j == 0) base16 = TRUE;

					break;	

											

				/* Hex/Oct base */					

				case 'h': 

					baseok = TRUE;				

				case 'o': 

					if (i < str_end-1 && is_digit_char(str[i+1])) error = TRUE;				

					if (j == 0) error = TRUE;

					break;

					

				/* Binary base - or hex digit */

				case 'b': 													

					if (i < str_end-1 && is_digit_char(str[i+1])) base16 = TRUE;

					else if (j == 0) error = TRUE;

					break;					

					

				/* Hex base */

				case 'x':

					baseok = TRUE;

					if (j != 1 || str[i-1] != '0') error = TRUE;

					break;

					

				/* Plus/Minus after exponent */			

				case '+':

				case '-':

					if (j <= 0 || str[i-1] != 'e') done = TRUE;

					break;

					

				/* Hex characters */	

				case 'A':

				case 'B':

				case 'C':

				case 'D':

				case 'E':

				case 'F':

				case 'a':

				case 'c':

				case 'd':

				case 'f':

					base16 = TRUE;

					break;

				

				/* Other character is probably end of number */

				default:

					done = TRUE;

					break;

			}		

		}

		

		/* Only increment if not at end or error */	

		if (!done && !error) {	

			i++; 

			j++;

		}

	}

		

	/* Hex mode test */

	if (base16 && !baseok) error = TRUE;

	

	/* Return end position of number */

	if (num_end != NULL)

		(*num_end) = i;

		

	/* No numbers of size zero */	

	if (j == 0) error = TRUE;

	

	return !error;

}



/*

 * operate

 * @num1  : a number

 * @num2  : a number

 * @oper  : character representation of an oper

 * @error : pointer to an error state indicator

 * return the result of the operation that @oper represents between 

 * @num1 and @num2. If an error ocurrs then set @error.

 */

gdouble 

operate(Optable      *otable,

        gdouble       num1, 

	gdouble       num2, 

	gint         *error)

{

	gdouble result = 0;

	

	if(errState.state != 0) 

		return 0;

	

	result = (otable->func) (num1, num2, error);

	

	return result;

}



/*

 * set_ag_error

 * @error  : type of the error

 * @start  : start of the argument tha causes the error in the string

 * @end    : end of the argument tha causes the error in the string

 * Set the error state for the calculator, where starts, and where ends.

 */

gint

set_ag_error(gint     error,

	     gint     start, 

	     gint     end)

{

	errState.state = error;

	errState.start = start;

	errState.end = end;

	return 0;

}



/******************************************************************************/

/*                                                                            */

/* eval_expression                                                            */

/* @expr      : string representation of a mathematical expression            */

/* @str_start : the begining of the expression in the string                  */

/* @str_end   : the end of the expression in the string                       */

/*============================================================================*/

/*                                                                            */

/* The way this function works:                                               */

/*                                                                            */

/* This is a recursive function.                                              */

/*                                                                            */

/* 1) Finds the operandus that should perform last.                           */

/*                                                                            */

/* 2) With that operandus we can separate the string in two pieces.           */

/*    Perform the numeric operation that the operandus rules, between         */

/*    the result of the first and the last part of the string                 */

/*    both send to this same function.                                        */

/*    This function does not cuts the string, it always works                 */

/*    on the same string, this way we dont need to allocate more memory,      */

/*    that is why we need the positions of the relative strings.              */

/*                                                                            */

/* 3)  When the string sent to this function is an argument and not           */

/*     an operation, for example a number, it returns a numeric               */

/*     value. With this value the others operations can be performed          */

/*     and so on, until we get the final result                               */

/*                                                                            */

/******************************************************************************/



gdouble 

eval_expression(gchar    *expr,

		gint      str_start,

		gint      str_end)

{

	Optable*   oper       = NULL;      /* Selected operandus */

	Optable*   new_oper   = NULL;      /* New operandus */

	gint       oper_pos;               /* Selected operandus position */

	gint       oper_end;	           /* Selected operandus end */

        gint       parenth    = 0;         /* Keeps track of parens */

	gint       pos, i;                 /* Current character position */

	gint       ev_error   = 0;         /* Sent to funcs to see if errored */ 

	gint       operFound  = 1;         /* if 0 we can look for a new oper */

	gdouble    result     = 0;         /* The result */

	Funtable*  func;                   /* Pointer to funtable to eval funcs */		

	

	/* If the error is already set we just exit */

	if (errState.state != 0) 

		return 0;

		

	while (isspace(expr[str_start]) && str_end > str_start) 

		str_start++;

	while (isspace(expr[str_end-1]) && str_end > str_start)  

		str_end--;

	

	oper_end = oper_pos = pos = str_start; 

	

	if (str_start >= str_end)

		return set_ag_error(MISSING_ARGUMENT,str_start,str_end);

	

#ifdef PRINT_SUBEXPR

	printf("Expression: ");

	for (i = str_start; i < str_end; i++)

		printf("%c", expr[i]);

	printf("\n");

#endif

	

	/* 

	 * This while loop is the first part of the process.

	 * It finds the last sign of the expression, with the

	 * lowest precedence value. 

	 */

	while (pos < str_end) {      	

	

    		switch(expr[pos]) {

			case '(': /* Open parenth */

				parenth++ ; 

				break;



			case ')': /* Close parenth */

				if(parenth == 0 )

					return set_ag_error(SYN_ERROR,pos,pos+1);

			

				i = pos;

				while(isspace(expr[i]))

					i++;

				

				new_oper = get_operator(expr, i, str_end, NULL);

				if (new_oper == NULL && expr[i] != '\0' && expr[i] != ')')

					return set_ag_error(SYN_ERROR,pos,pos+1);

				

				parenth--;

				break;		

				

	

			default: /* Something else */

				if (parenth != 0)

					break;

				

				if (operFound) {

					/* 

					 * Identify the argument as a number or not 

					 */

					if (get_number(expr, pos, str_end, &i)) {

						pos = i-1;

						operFound = 0;

						break;

					}

				}



				/* 

				 * Find operator

				 */					

				new_oper = get_operator(expr, pos, str_end, &i);

				if (new_oper != NULL) {

					if (pos == str_start) {

						oper = new_oper;

						oper_end = i;

					} else {

						operFound = 1;

						if (oper == NULL || new_oper->prec <= oper->prec) {

							oper = new_oper;

							oper_pos = pos;

							oper_end = i;																	

						}							

					}

					pos = i-1;							

				}

			

				break;				

		} /* switch */	

		

		pos++;

		

	} /* while(pos < str_end ) */

		

	if(parenth > 0)

		return set_ag_error(UNCLOSED_PAREN,str_start,str_end);

		

	/* 

	 * Here starts the second part of the function,

	 * This part makes the operations, strips the original string 

	 * to single numbers and returns a numeric result. 

	 * To do this we operate on the numeric results returned 

	 * by the recursive calls to this same function 

	 */

	if(oper != NULL) { 

		/*

		 * If it's a unaty operandus

		 */

		if(oper_pos == str_start) {		

			

			switch(oper->id) {

				case OP_MINUS:

					result=-eval_expression(expr,oper_end,str_end);

					break;

			

				case OP_PLUS:			

					result=eval_expression(expr,oper_end,str_end);

					break;			

				default:	

					return set_ag_error(SYN_ERROR,oper_pos,oper_end);						

			} 

		}



		/*

		 * Binary operandus

		 */

		else {

			result = operate(oper, 

			                 eval_expression(expr,str_start,oper_pos),

					 eval_expression(expr,oper_end,str_end),

					 &ev_error);

		}

	}	



       /*

 	* If we got here it's because we couldnt find any kind

 	* of operandus

 	*/



	/*

 	 * Factorial 

 	 */

	else if(expr[str_end-1] == '!' ) {

		result = ag_fact(eval_expression(expr,str_start,str_end-1),

		                 angle_mode,&ev_error);  

	}



	/*

 	 * Parenthesis start

 	 */

	else if(expr[str_start] == '(') {

		result = eval_expression(expr,str_start+1,str_end-1);  

	}



	/*

 	 * It's a number

 	 */

	else if (get_number(expr, str_start, str_end, NULL)) {

		result = rel_str_to_double(expr,str_start,str_end,&ev_error);

	}



	/*

  	 * Something else

 	 */

	else { 



		/*

	 	 * Could it be a function?

	 	 */

		while (oper_pos < str_end-1 && expr[oper_pos] != '(')

			oper_pos++;

	

		if(expr[oper_pos] == '(' && expr[str_end-1] == ')') {

			if ((func=get_function(expr,str_start,oper_pos,&ev_error)) != 0) {

				result = eval_expression(expr,oper_pos+1,str_end-1);

				result = eval_function(func,result,&ev_error);

			}

		} else {

			return set_ag_error(SYN_ERROR,str_start,str_end);	

		}

	}	

	

	result = keep_double_within_limits(result,&ev_error);

	

	if(ev_error)

		return set_ag_error(ev_error,str_start,str_end);

	

	result = round_double(result,format_mode,size_mode);	



	return result;

}



/*

 * eval_string

 * @expr   : character string representing a mathematical expression

 * @format : format (FIX, STD, ENG, SCI) to round for

 * @size   : number of digits to round at (1 ... 12)

 * @angle  : angle messure to use (RAD, DEG, GRAD)

 * Given a string reperesenting a math expression, tis function returns

 * a numeric value of the result, rounded specially for @format and @size

 * and in @angle messure

 */

gdouble 

eval_string(gchar      *expr,

	    gint        format,

	    gint        size,

	    gint        angle)

{

	gdouble  result = 0;

	

	format_mode = format;

	size_mode   = size;

	angle_mode  = angle;

	

	result = eval_expression(expr,0,strlen(expr));



	return result;

}









