%{
/**************************************************************************
 * The parser to handle config files.
 *
 * Copyright (C) 2000, Matthew Palmer.  Released under the GPL version 2.
 * You should have received a copy of this licence with this software, if not
 * see http://www.fsf.org/copyleft/gpl.html for a full copy of the licence.
 */

static char rcsid[] = "$Id: fileconf.y,v 1.3 2001/01/02 03:08:53 mjp16 Exp $";

#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <syslog.h>
#include <pthread.h>

#include "fileconf.h"

#define YYPARSE_PARAM confdata

#define YYERROR_VERBOSE
#define YYDEBUG 1
#define YYPRINT(file,type,value) yyprint(file,type,value)

void conf_error(char *);
int yyprint();
extern int conf_lex();
%}

/* Glory be, this introduces all sorts of entertaining things... */
%pure_parser

%union {
	struct conf_item *item;
	char *str;
	conf_section_ll_t *sec_list;
	conf_item_ll_t *item_list;
}

%token TOK_OB TOK_CB
%token TOK_EQUAL
%token <item> VAL_ITEM
%token <str> VAL_TAG VAL_DATA VAL_SECTION

%type <item> item
%type <sec_list> sectionlist section
%type <item_list> itemlist

%start configfile

%%
configfile:	sectionlist
		{
			conf_file_t *cf;
			
			cf = (conf_file_t *) confdata;
			
			cf->headsect = cf->cursect = $1;
		};
		
sectionlist:	section
		{
			$$ = $1;
		}
	|	sectionlist section
		{
			$2->nextsect = $1;
			$$ = $2;
		};
	
section:	TOK_OB VAL_SECTION TOK_CB itemlist
		{
			conf_section_ll_t *sect;
			
			sect = malloc(sizeof(conf_section_ll_t));
			if (!sect) {
				fprintf(stderr, "Error allocating memory");
				YYABORT;
			}
			sect->sectname = $2;
			sect->headitem = sect->curitem = $4;
			sect->nextsect = NULL;
			
			$$ = sect;
		};
		
itemlist:	item
		{
			conf_item_ll_t *itemll;
			
			itemll = malloc(sizeof(conf_item_ll_t));
			if (!itemll) {
				fprintf(stderr, "Error allocating memory");
				YYABORT;
			}
			itemll->nextitem = NULL;
			itemll->item = $1;
			
			$$ = itemll;
		}
	|	itemlist item
		{
			conf_item_ll_t *itemll;
			
			itemll = malloc(sizeof(conf_item_ll_t));
			if (!itemll) {
				fprintf(stderr, "Error allocating memory");
				YYABORT;
			}
			itemll->nextitem = $1;
			itemll->item = $2;
			$$ = itemll;
		};
		
item:		VAL_TAG TOK_EQUAL VAL_DATA
		{
			struct conf_item *item;
			
			item = malloc(sizeof(struct conf_item));
			if (!item) {
				fprintf(stderr, "Error allocating memory");
				YYABORT;
			}
			
			item->tag = $1;
			item->data = $3;
			
			$$ = item;
		};
%%

void conf_error(char *s)
{
	syslog(LOG_ERR, "Parse error in config file: %s", s);
}

int yyprint(FILE *file, int type, YYSTYPE value)
{
	if (type == VAL_SECTION || type == VAL_TAG || type == VAL_DATA) {
		fprintf(file, " %s", value.str);
	} else if (type == VAL_ITEM) {
		fprintf(file, " item");
	} else {
		fprintf(file, " tok %i", type);
	}
	return 0;
}

extern FILE *conf_in;

conf_file_t *conf_read_file(char *filename)
{
	conf_file_t *cfile;
	int rv;

	conf_in = fopen(filename, "r");
	if (!conf_in) {
		return NULL;
	}
	
	cfile = malloc(sizeof(conf_file_t));
	if (!cfile) {
		errno = ENOMEM;
		return NULL;
	}
	cfile->mutex = malloc(sizeof(pthread_mutex_t));
	if (!cfile->mutex) {
		errno = ENOMEM;
		free(cfile);
		return NULL;
	}
	pthread_mutex_init(cfile->mutex, NULL);
	pthread_mutex_lock(cfile->mutex);
	rv = conf_parse((void *) cfile);
	if (rv) {
		free(cfile->mutex);
		free(cfile);
		cfile = NULL;
	}

	if (cfile) {
		pthread_mutex_unlock(cfile->mutex);
	}
	return cfile;
}

void conf_free_file(conf_file_t *conf)
{
	void *cp;
	
	pthread_mutex_lock(conf->mutex);
	conf->cursect = conf->headsect;
	
	while(conf->cursect) {
		conf->cursect->curitem = conf->cursect->headitem;
		while(conf->cursect->curitem) {
			free(conf->cursect->curitem->item->tag);
			free(conf->cursect->curitem->item->data);
			free(conf->cursect->curitem->item);
			cp = conf->cursect->curitem;
			conf->cursect->curitem = conf->cursect->curitem->nextitem;
			free(cp);
		}
		free(conf->cursect->sectname);
		cp = conf->cursect;
		conf->cursect = conf->cursect->nextsect;
		free(cp);
	}

	free(conf->mutex);
	free(conf);
}

char *conf_next_section(conf_file_t *conffile)
{
	char *new_sec;
	
	pthread_mutex_lock(conffile->mutex);
	if (conffile->cursect->nextsect) {
		conffile->cursect = conffile->cursect->nextsect;
		conffile->cursect->curitem = conffile->cursect->headitem;
		new_sec = conffile->cursect->sectname;
	} else {
		new_sec = NULL;
	}

	pthread_mutex_unlock(conffile->mutex);
	return new_sec;
}

int conf_find_section(conf_file_t *conffile, char *sectname)
{
	conf_section_ll_t *cs;
	int rv;
	
	pthread_mutex_lock(conffile->mutex);
	cs = conffile->headsect;
	
	while(cs && strcmp(cs->sectname, sectname)) {
		cs = cs->nextsect;
	}

	if (cs) {
		conffile->cursect = cs;
		cs->curitem = cs->headitem;
		rv = 1;
	} else {
		rv = 0;
	}

	pthread_mutex_unlock(conffile->mutex);
	return rv;
}

int conf_seek_section(conf_file_t *conffile, int n)
{
	int i;
	
	pthread_mutex_lock(conffile->mutex);
	conffile->cursect = conffile->headsect;
	for (i = 0; i < n && conffile->cursect; i++) {
		conffile->cursect = conffile->cursect->nextsect;
	}

	pthread_mutex_unlock(conffile->mutex);
	return (conffile->cursect != NULL);
}

struct conf_item *conf_next_item(conf_file_t *conffile)
{
	struct conf_item *ci;
	
	pthread_mutex_lock(conffile->mutex);
	if (conffile->cursect->curitem) {
		ci = conffile->cursect->curitem->item;
		conffile->cursect->curitem = conffile->cursect->curitem->nextitem;
	} else {
		ci = NULL;
	}
	
	pthread_mutex_unlock(conffile->mutex);
	return ci;
}

char *conf_find_item(conf_file_t *conffile, char *tag)
{
	conf_item_ll_t *ci;
	char *data;
	
	pthread_mutex_lock(conffile->mutex);
	ci = conffile->cursect->headitem;
	data = NULL;

	while(ci) {
		if (!strcmp(ci->item->tag, tag)) {
			data = ci->item->data;
		}
		ci = ci->nextitem;
	}
	
	pthread_mutex_unlock(conffile->mutex);
	return data;
}

int conf_seek_item(conf_file_t *conffile, int n)
{
	int i;
	
	pthread_mutex_lock(conffile->mutex);
	conffile->cursect->curitem = conffile->cursect->headitem;

	for (i = 0; i < n && conffile->cursect->curitem; i++) {
		conffile->cursect->curitem = conffile->cursect->curitem->nextitem;
	}
	
	pthread_mutex_unlock(conffile->mutex);
	return (conffile->cursect->curitem != NULL);
}
