/*
 * ----------------------------------------------------------------
 *
 * Load Binary , SRecord or IntelHex files to target Memory
 * (C) 2004  Lightmaze Solutions AG
 *   Author: Jochen Karrer
 *
 * Status:
 *	Working
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "bus.h"
#include "ihex.h"
#include "srec.h"
#include "configfile.h"
#include "loader.h"

#define FLAG_SWAP32 (2)

/*
 * -----------------------------------------------------
 * Write to Memory devices on CPU main bus
 * If a device is read-only (Flash,Rom) write anyway
 * by using the address in the Read-memory Map. 
 * -----------------------------------------------------
 */
static int 
write_to_bus(uint32_t addr,uint8_t *buf,unsigned int count,int flags) 
{
	while(count) {
		/* This is wrong (outdated) because of smaller mappings */
                uint32_t len=MEM_MAP_BLOCKSIZE-(addr&MEM_MAP_BLOCKMASK);
                uint32_t offs = addr&(MEM_MAP_BLOCKMASK);
                uint8_t *host_mem= Bus_GetHVAWrite(addr & ~MEM_MAP_BLOCKMASK);
                if(!host_mem) {
                	host_mem= Bus_GetHVARead(addr &  ~MEM_MAP_BLOCKMASK);
                }
                if(!host_mem) {
                        fprintf(stderr,"Loader: Cannot write record to memory at 0x%08x\n",addr);
                        return -1;
                }
		if(len>count)
			len=count;
		if(flags & FLAG_SWAP32) {
			int i;	
			for(i=0;i<len;i++) {
				uint8_t *dst = host_mem+((offs+i)^3);	
				*dst = buf[i];
			}
		} else  {
			memcpy(host_mem+offs,buf,len);
		}
		count-=len;
	}
	return 0;
}

typedef struct LoaderInfo {
	int flags;
	uint64_t region_start;
	uint64_t region_end;
} LoaderInfo;
/*
 * --------------------------------------------------------
 * write_srec_to_bus
 *	Callback function for external SRecord parser. 
 *	Is invoked by parser after parsing a data record
 * -------------------------------------------------------
 */
static int 
write_srec_to_bus(uint32_t addr,uint8_t *buf,int count,void *clientData) 
{
	LoaderInfo *li = (LoaderInfo *)clientData;
	if((addr < li->region_start) || ((addr + count - 1) > li->region_end)) {
		fprintf(stderr,"S-Record is outside of memory region\n");
		return -1;
	} else {
		write_to_bus(addr,buf,count,li->flags);
	}
	return 0;
}

static int
Load_SRecords(char *filename,uint32_t startaddr,int flags,uint64_t region_size) 
{
	LoaderInfo li;		
	li.flags = flags;
	li.region_start = startaddr;
	if(region_size > 0) {
		li.region_end = startaddr + region_size -1;
	} else {
		li.region_end = ~0ULL;
	}
	return XY_LoadSRecordFile(filename,write_srec_to_bus,&li);
}
/*
 * --------------------------------------------------
 * write_ihex_to_bus
 *	Callback function for external IHex parser 
 *	is invoked after parsing a data record
 * --------------------------------------------------
 */
static int 
write_ihex_to_bus(uint32_t addr,uint8_t *buf,int count,void *cd) 
{
	LoaderInfo *li = (LoaderInfo *)cd;
	if((addr < li->region_start) || ((addr + count - 1) > li->region_end)) {
		fprintf(stderr,"Ihex: Record at 0x%08x is outside of memory_region\n",addr);
		return -1;
	} else {
		write_to_bus(addr,buf,count,li->flags);
	}
	return 0;
}

static int
Load_IHex(char *filename,uint32_t startaddr,int flags,uint64_t region_size) 
{
	LoaderInfo li;		
	li.flags = flags;
	li.region_start = startaddr;
	if(region_size > 0) {
		li.region_end = startaddr + region_size -1;
	} else {
		li.region_end = ~0ULL;
	}
	fprintf(stderr,"Loading Intel Hex Record file \"%s\"\n",filename);
        return XY_LoadIHexFile(filename,write_ihex_to_bus,&li);;
}

/*
 * -----------------------------------------
 * Load a Binary File to a given address
 * -----------------------------------------
 */
int
Load_Binary(char *filename,uint32_t addr,int flags,uint64_t maxlen) 
{
	int fd=open(filename,O_RDONLY);
        int count;
	int to_big = 0;
        int64_t total=0;
	uint8_t buf[4096];
        if(fd<=0) {
                fprintf(stderr,"Can not open file %s ",filename);
                perror("");
		return -1;
        }
	while(1) {
                count = read(fd,buf,4096);
		if(count==0) {
			close(fd);
			return total;
		} else if(count < 0) {
			perror("error reading binary file");
			return -1;
		}
		if(maxlen && (count + total > maxlen)) {
			count = maxlen - total;
			to_big = 1;
		}
		if(write_to_bus(addr,buf,count,flags) < 0) {
			fprintf(stderr,"Binary loader: Can not write to bus at addr 0x%08x\n",addr);
		}
		total+=count;
		addr+=count;
		if(to_big) {
			fprintf(stderr,"Binary file does not fit into memory region\n");
			return -1;
		}
	}
	close(fd);
	return total;
}

/*
 * --------------------------------------------------------------
 * recognize file type from suffix and then 
 * Loads binary, Intel Hex or Motorola SRecords. 
 * The load address is ignored for srecords
 * --------------------------------------------------------------
 */
int64_t  
Load_AutoType(char *filename,uint32_t load_addr,uint64_t region_size)  {
	uint32_t swap;
	int flags=0;
	int len = strlen(filename);
	if(Config_ReadUInt32(&swap,"loader","swap32") >=0) {
		if(swap) {
			flags=FLAG_SWAP32;
		}
	}
	fprintf(stderr,"Loading %s to 0x%08x flags %d\n",filename,load_addr,flags);
	if((len>=5) && (!strcmp(filename+len-5,".srec"))){
		return Load_SRecords(filename,load_addr,flags,region_size);
	} else if((len>=4) && (!strcmp(filename+len-4,".hex"))){
		return Load_IHex(filename,load_addr,flags,region_size);
	} else {
		return Load_Binary(filename,load_addr,flags,region_size);
	}
}
