/*
 * ----------------------------------------------------------------
 *  Compose the ARMee board from Electronic magazine elektor
 *
 * (C) 2005 Jochen Karrer 
 *   Author: Jochen Karrer
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------------------
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#include <idecode.h>
#include <signode.h>
#include <mmu.h>
#include <fio.h>
#include <bus.h>
#include <configfile.h>
#include <loader.h>
#include <i2c_serdes.h>
#include <m24cxx.h>
#include <boards.h>
#include <lpcflash.h>
#include <sram.h>
#include <pc16550.h>
#include <pl190_irq.h>
#include "clock.h"
#include <lpc2106_timer.h>
#include <lpc2106_gpio.h>
#include <lpc2106_scb.h>
#include "devices/lcd_hd44780/lcd_hd44780.h"

/*
 * ---------------------------------------------------------------
 * Link some Electrical signals in the same way as in schematics
 * This has to be done when all devices are created, because
 * you can not link nonexisting Signals
 * ---------------------------------------------------------------
 */
static void
create_signal_links(void)
{
	SigName_Link("arm.irq","vic.irq");
	SigName_Link("arm.fiq","vic.fiq");
	SigName_Link("timer0.irq","vic.nVICINTSOURCE4");
	SigName_Link("timer1.irq","vic.nVICINTSOURCE5");
	SigName_Link("uart0.irq","vic.nVICINTSOURCE6");
	SigName_Link("uart1.irq","vic.nVICINTSOURCE7");
	/* Connect the LCD */
	SigName_Link("gpio.P0.4","lcd0.D4");
	SigName_Link("gpio.P0.5","lcd0.D5");
	SigName_Link("gpio.P0.6","lcd0.D6");
	SigName_Link("gpio.P0.7","lcd0.D7");

	SigName_Link("gpio.P0.8","lcd0.RS");
	SigName_Link("gpio.P0.9","lcd0.RW");
	SigName_Link("gpio.P0.10","lcd0.E");
	

}

static void
create_clock_links(void)
{
	Clock_Link("arm.clk","scb.cclk");
	Clock_Link("uart0.clk","scb.pclk");
	Clock_Link("uart1.clk","scb.pclk");
	Clock_Link("timer0.pclk","scb.pclk");
	Clock_Link("timer1.pclk","scb.pclk");
}

static int
board_armee_create(void)
{
        ArmCoprocessor *copro;
	BusDevice *dev;
	FbDisplay *display = NULL;
        Keyboard *keyboard = NULL;

	FbDisplay_New("lcd0",&display,&keyboard,NULL,NULL);
        if(!display) {
                fprintf(stderr,"LCD creation failed\n");
		exit(1);
        }

        Bus_Init(MMU_InvalidateTlb,4*1024);
        ARM9_New();
	/* Copro is created but not registered (1:1 translation is bootup default) */
        copro = MMU_Create("mmu",TARGET_BYTEORDER,MMU_ARM926EJS | MMUV_NS9750);

	dev = PL190_New("vic");
	Mem_AreaAddMapping(dev,0xfffff000,0x1000,MEM_FLAG_WRITABLE | MEM_FLAG_READABLE);

	dev = SRam_New("iram");
	Mem_AreaAddMapping(dev,0x40000000,256*1024,MEM_FLAG_WRITABLE | MEM_FLAG_READABLE);

	dev = LPCFlash_New("iflash"); 
	Mem_AreaAddMapping(dev,0x00000000,128*1024,MEM_FLAG_WRITABLE | MEM_FLAG_READABLE);

	dev = LPC2106Timer_New("timer0");
	Mem_AreaAddMapping(dev,0xE0004000,0x20,MEM_FLAG_WRITABLE | MEM_FLAG_READABLE);

	dev = LPC2106Timer_New("timer1");
	Mem_AreaAddMapping(dev,0xE0008000,0x20,MEM_FLAG_WRITABLE | MEM_FLAG_READABLE);

	dev = PC16550_New("uart0",2);
	Mem_AreaAddMapping(dev,0xE000C000,0x40,MEM_FLAG_WRITABLE | MEM_FLAG_READABLE);

	dev = PC16550_New("uart1",2);
	Mem_AreaAddMapping(dev,0xE0010000,0x40,MEM_FLAG_WRITABLE | MEM_FLAG_READABLE);

	LPC2106_GpioNew("gpio.P0");

	dev = LPC2106_ScbNew("scb");
	Mem_AreaAddMapping(dev,0xE01FC000,0x200,MEM_FLAG_WRITABLE | MEM_FLAG_READABLE);

	HD44780_LcdNew("lcd0", display);

	create_signal_links();
	create_clock_links();
	return 0;
}

static void
board_armee_run(Board *bd) {
        ARM9_Run();
}

#define DEFAULTCONFIG \
"[global]\n" \
"start_address: 0\n"\
"cpu_clock: 58982400\n"\
"oscillator: 14745600\n" \
"\n" \
"[iram]\n" \
"size: 64k\n" \
"\n" \
"[lcd0]\n" \
"backend: rfbserver\n" \
"host: 127.0.0.1\n" \
"port: 5901\n" \
"width: 310\n" \
"height: 75\n" \
"start: vncviewer localhost:5901\n" \
"exit_on_close: 1\n" \
"\n" \
"[loader]\n" \
"load_address: 0x0\n"\
"\n" \
"[iflash]\n"\
"size: 128k\n"\
"\n"

Board board_armee = {
        .name = "ARMee",
        .description =  "Elektor ARMee Board",
        .createBoard =  board_armee_create,
        .runBoard =     board_armee_run,
        .defaultconfig = DEFAULTCONFIG
};

void
_init() {
        fprintf(stderr,"Loading ARMee Board module\n");
        Board_Register(&board_armee);
}

