/*
 * -----------------------------------------------------------------------
 * SPI Interface to MMC/SD cards 
 *
 * (C) 2009 Jochen Karrer
 *   Author: Jochen Karrer
 *
 * state: Read blocks is working, known from testing Uzeamp.
 *	  Rest is untested 
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * -----------------------------------------------------------------------
 */

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include "compiler_extensions.h"
#include "sgstring.h" 
#include "signode.h"
#include "mmcard.h"
#include "sd_spi.h"

#if 0
#define dbgprintf(x...) fprintf(stderr,x)
#else
#define dbgprintf(x...)
#endif

struct SD_Spi {
	SigNode *cmd; /* connected to MOSI */
	SigNode *clk; /* connected to SCK */
	SigTrace *clkTrace;
	SigNode *dat3; /* connected to SPI CS */
	SigNode *dat0; /* connected to SPI MISO */
	SigNode *sd_det; /* The switch opened ?? when card is inserted */
	uint8_t shift_in;
	uint8_t shift_out;
	int shiftin_cnt;
	int shiftout_cnt;
	uint8_t inbuf[2048];
	uint8_t outbuf[2048];
	uint8_t delay_fifo[3]; /* Fifo for one byte delay */
	unsigned int delay_fifo_wp;
	/* SPI input state machine variables */
	int istate;
	unsigned int inbuf_wp;

	/* SPI output state machine variables */
	uint64_t outbuf_rp;
	uint64_t outbuf_wp;
	int dataread;
	int cs;
	MMCard *card;
};

#define OUT_WP(sds) ((sds)->outbuf_wp % sizeof(sds->outbuf))
#define OUT_RP(sds) ((sds)->outbuf_rp % sizeof(sds->outbuf))

#define TOKEN_START_BLOCK	(0xfe)
#define TOKEN_MBW_START_BLOCK	(0xfc)
#define TOKEN_STOP_TRAN		(0xfd)

#define DRTOKEN_ACCEPTED	(0x5)
#define DRTOKEN_REJECT_CRC	(0xB)
#define DRTOKEN_REJECT_WRITE	(0xD)

/* SPI input state machine states */
#define ISTATE_IDLE  		(0)
#define ISTATE_CMD   		(1)
#define ISTATE_DATABLOCK   	(2)
#define ISTATE_DATABLOCKS   	(3)

static inline void
put_outbuf(SD_Spi *sds,uint8_t value)
{
	sds->outbuf[OUT_WP(sds)] = value;
	sds->outbuf_wp++;
}

static void 
put_block(SD_Spi *sds,uint8_t *data,int len)
{
	int i;
	dbgprintf("Got transmission of data block\n");
	put_outbuf(sds,TOKEN_START_BLOCK);
	for(i=0;i<len;i++) {
		put_outbuf(sds,data[i]);
	}
	put_outbuf(sds,0); put_outbuf(sds,0); /* Should be crc */
}

static void
send_command(SD_Spi *sds) 
{
	int i;
	uint32_t cmd;
	uint32_t arg;
	int len;
	uint8_t buf[512];
	MMCResponse resp;
	uint8_t *data = sds->inbuf;
	sds->dataread = 0;
	arg = (data[1] << 24) | (data[2] << 16) | (data[3] << 8) | (data[4] << 0);
	cmd = data[0] & 0x3f;
	MMCard_DoCmd(sds->card,cmd,arg,&resp);

	dbgprintf("%02x %02x %02x %02x %02x %02x",
		data[0],data[1],data[2],data[3],data[4],data[5]);
	dbgprintf(", CMD %d, arg 0x%08x, resp len %d d0 0x%02x\n",cmd,arg,resp.len, resp.data[0]);
//	put_outbuf(sds,0xff);

	for(i=0;i<resp.len;i++) {
		put_outbuf(sds,resp.data[i]);
	}
	/* 
 	 ******************************************************
	 * Simply try if cmd has a data phase
 	 ******************************************************
	 */
	if((len=MMCard_Read(sds->card,buf,512)) > 0) {
		put_block(sds,buf,len);
		sds->dataread = 1;
	}

}

/*
 ********************************************
 * Send a datablock to the card
 * currently fixed to 512 Byte blocks
 ********************************************
 */
static void
send_datablock(SD_Spi *sds) 
{
	int rc;
	MMC_DataBlock mdb;
	mdb.data = sds->inbuf;
	mdb.datalen = 512;
	rc = MMCard_Write(sds->card,&mdb);
	if(rc == 512) {
		put_outbuf(sds,DRTOKEN_ACCEPTED);	
	} else {
		put_outbuf(sds,DRTOKEN_REJECT_WRITE);	
	}
}

/*
 ************************************************************
 * Statemachine for bytes comming from SPI
 ************************************************************
 */
static void
spi_byte_in(SD_Spi *sds,uint8_t data) 
{
	if(sds->inbuf_wp >= sizeof(sds->inbuf)) {
		return;
	}
	switch(sds->istate) {
		case ISTATE_IDLE:
			if((data & 0xc0) == 0x40) {
				sds->istate = ISTATE_CMD;
				sds->inbuf[sds->inbuf_wp++] = data; 
			} else if(data == TOKEN_START_BLOCK) {
				/* Forget the byte itself, its only a token */
				sds->istate = ISTATE_DATABLOCK;
			} else if(data == TOKEN_MBW_START_BLOCK) {
				/* Forget the byte itself, its only a token */
				sds->istate = ISTATE_DATABLOCKS;
			} else if(data == TOKEN_STOP_TRAN) {
				/* Don't know for what i should need this token */
			}
			break;

		case ISTATE_CMD:
			sds->inbuf[sds->inbuf_wp++] = data; 
			if(sds->inbuf_wp == 6) {
				send_command(sds);
				sds->inbuf_wp = 0;
				sds->istate = ISTATE_IDLE;
			}	
			break;

		case ISTATE_DATABLOCK:
			sds->inbuf[sds->inbuf_wp++] = data; 
			/* Fixed block size of 512 + 2 byte CRC for now */
			if(sds->inbuf_wp == 514) {
				send_datablock(sds);
				sds->inbuf_wp = 0;
				sds->istate = ISTATE_IDLE;
			}
			break;

		case ISTATE_DATABLOCKS:
			sds->inbuf[sds->inbuf_wp++] = data; 
			/* Fixed block size of 512 + 2 byte CRC for now */
			if(sds->inbuf_wp == 514) {
				send_datablock(sds);
				sds->inbuf_wp = 0;
				sds->istate = ISTATE_IDLE;
			}
			break;
			
	}
}

static inline uint8_t 
spi_fetch_next_byte(SD_Spi *sds)
{
	uint8_t next;
	uint8_t buf[512];
	if(likely(sds->outbuf_rp < sds->outbuf_wp)) {
		next = sds->outbuf[OUT_RP(sds)];
		sds->outbuf_rp++;
	} else {
		if(sds->dataread) {
			int len;
			if((len=MMCard_Read(sds->card,buf,512)) > 0) {
				dbgprintf("multi-Read success %d\n",len);
				put_block(sds,buf,len);
			} else {
				dbgprintf("multi-Read fail %d\n",len);
				sds->dataread = 0;
			}
		}
		next = 0xff;
	}
	//fprintf(stderr,"fetched %02x\n",next);
	return next;
}
/*
 ********************************************************************
 * spi_clk_change
 * 	Signal Trace of the SPI Clock line
 * 	Shifts the bits.	
 ********************************************************************
 */
static void 
spi_clk_change(SigNode *node,int value,void *clientData)
{
	SD_Spi *sds = (SD_Spi *) clientData;
	if(sds->cs) {
		return;
	}
	if(value == SIG_HIGH) {
		if(SigNode_Val(sds->cmd) == SIG_HIGH) {
			sds->shift_in = (sds->shift_in << 1) | 1;
		} else {	
			sds->shift_in = (sds->shift_in << 1);
		}
		sds->shiftin_cnt++;
		if(sds->shiftin_cnt == 8) {
			spi_byte_in(sds,sds->shift_in);
			sds->shiftin_cnt = 0;
		}
	}  else if(value == SIG_LOW) {
		if(sds->shift_out & 0x80) {
			SigNode_Set(sds->dat0,SIG_HIGH);
		} else {
			SigNode_Set(sds->dat0,SIG_LOW);
		}
		sds->shift_out <<= 1;
		sds->shiftout_cnt++;
		if(sds->shiftout_cnt == 8) {
			sds->shiftout_cnt = 0;
			sds->shift_out = spi_fetch_next_byte(sds);
		}
	}
	return;
}


/*
 *******************************************************************
 * spi_cs_change
 * 	Signal trace of the chip select line
 *******************************************************************
 */


static void 
spi_cs_change(SigNode *node,int value,void *clientData)
{
	SD_Spi *sds = (SD_Spi *) clientData;
	sds->cs = (value != SIG_LOW);
	if(value == SIG_LOW) {
		sds->shift_in = 0;	
		sds->shiftin_cnt = 0;	
		sds->inbuf_wp = 0;	
		sds->istate = ISTATE_IDLE;
		sds->dataread = 0;
		sds->outbuf_wp = sds->outbuf_rp = 0;

		sds->shift_out= 0xff;	
		if(sds->shift_out & 0x80) {
			SigNode_Set(sds->dat0,SIG_HIGH);
		} else {
			SigNode_Set(sds->dat0,SIG_LOW);
		}
		sds->shiftout_cnt = 1;	
		sds->shift_out <<= 1;
	}
	dbgprintf("CS %d\n",value);
	return;
}

uint8_t 
SDSpi_ByteExchange(void *clientData,uint8_t data)
{
	SD_Spi *sds = (SD_Spi*) clientData;
	uint8_t retval;
	if(unlikely(!sds)) {
		fprintf(stderr,"Bug: SDCard Spi interface called with NULL pointer\n");
		exit(1);
	}
	spi_byte_in(sds,data); 
	sds->delay_fifo[sds->delay_fifo_wp] = spi_fetch_next_byte(sds);
	sds->delay_fifo_wp = (sds->delay_fifo_wp + 1) % 3;
	retval = sds->delay_fifo[sds->delay_fifo_wp];
//	fprintf(stderr,"exch %02x, %02x\n",data,retval);
//	usleep(100000);
	return retval;
}

SD_Spi *
SDSpi_New(const char *name, MMCard *card)
{
	SD_Spi *sds;
	if(!card) {
		fprintf(stderr,"No mmcard given for SPI-SD Interface\n");
		return NULL;
	}
	sds = sg_new(SD_Spi);	
	sds->cmd = SigNode_New("%s.cmd",name);
	sds->clk = SigNode_New("%s.clk",name);
	sds->dat3 = SigNode_New("%s.dat3",name);
	sds->dat0 = SigNode_New("%s.dat0",name);
	sds->sd_det = SigNode_New("%s.sd_det",name);
	if(!sds->cmd || !sds->clk || !sds->dat3 || !sds->dat0 || !sds->sd_det) {
		fprintf(stderr,"Can not create Signal lines for SD-Card SPI interface\n");
		exit(1);
	}
	sds->card = card;
	sds->delay_fifo[0] = sds->delay_fifo[1] = 0xff;
	sds->delay_fifo_wp = 0;
	SigNode_Trace(sds->clk,spi_clk_change,sds);
	SigNode_Trace(sds->dat3,spi_cs_change,sds);
	MMCard_GotoSpi(card);
	fprintf(stderr,"Created SPI interface \"%s\" to MMCard\n",name);
	return sds;
}
