#include <controller/somaplayer.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <termios.h>

#define NAME_EXAMPLE "somaplayer_controller"

static struct termios t1, t2;

void
echo_off (void)
{
  tcgetattr (STDIN_FILENO, &t1);
  t2 = t1;

  t2.c_lflag &= ~(ICANON | ECHO);
  tcsetattr (STDIN_FILENO, TCSADRAIN, &t2);
}

void
echo_on (void)
{
  tcsetattr (STDIN_FILENO, TCSADRAIN, &t1);
}

void
quit (void)
{
  puts ("Error: server connection.");
  echo_on ();
  exit (1);
}

int
usage (void)
{
  fprintf (stderr,
	   "Usage:\n\n" "\t%s [unix://file] or [tcp://server[:port]]\n\n"
	   "You can run:\n\n" "\t%s [unix] or [tcp]\n"
	   "for generic connections\n\n", NAME_EXAMPLE, NAME_EXAMPLE);
  return 1;
}

somaplayer_controller *
parse (char *a)
{

  if (!strcmp (a, "unix"))
    return somaplayer_open_unix (NULL);

  if (!strcmp (a, "tcp"))
    return somaplayer_open_tcp (NULL, 0);

  if (!strncmp (a, "unix://", 7))
    {
      char *file;
      file = a + 7;

      return somaplayer_open_unix (file);
    }

  if (!strncmp (a, "tcp://", 6))
    {
      char *server;
      int i, port = 0;

      server = a + 6;

      for (i = 0; i < strlen (server); i++)
	{
	  if (server[i] == ':')
	    {

	      if (i + 1 < strlen (server))
		{
		  port = atoi (&server[i + 1]);
		}

	      server[i] = 0;
	    }
	}

      return somaplayer_open_tcp (server, port);
    }

  return NULL;
}

int
main (int argc, char *argv[])
{
  somaplayer_controller *c;
  int v, b, q = 0;
  char *file;
  int flag;
  fd_set fd_read;
  struct timeval tv;
  char ch;
  char buf[80];

  flag = fcntl (1, F_GETFL, 0);
  fcntl (1, F_SETFL, flag | O_NONBLOCK);

  if (argc != 2 || (!strcmp (argv[1], "-h") || !strcmp (argv[1], "--help")))
    return usage ();

  echo_off ();

  if (!(c = parse (argv[1])))
    {
      echo_on ();
      puts ("Argoment error!");
      return 1;
    }

  v = snprintf (buf, 80, NAME_EXAMPLE);
  v = (80 - v) ? (80 - v) / 2 : 0;
  for (b = 0; b < v; b++)
    putchar (' ');
  fprintf (stdout, "%s", buf);
  for (b = 0; b < v; b++)
    putchar (' ');
  putchar ('\n');

  for (b = 0; b < 80; b++)
    putchar ('-');
  putchar ('\n');

  v =
    snprintf (buf, 80,
	      "Enter = exit | Space = pause | q = quit | n = next | p = prev");
  v = (80 - v) ? (80 - v) / 2 : 0;
  for (b = 0; b < v; b++)
    putchar (' ');
  fprintf (stdout, "%s", buf);
  for (b = 0; b < v; b++)
    putchar (' ');
  putchar ('\n');

  v =
    snprintf (buf, 80,
	      "v <volume | V>volume | b<balance | B>balance | c = center");
  v = (80 - v) ? (80 - v) / 2 : 0;
  for (b = 0; b < v; b++)
    putchar (' ');
  fprintf (stdout, "%s", buf);
  for (b = 0; b < v; b++)
    putchar (' ');
  putchar ('\n');

  for (b = 0; b < 80; b++)
    putchar ('-');
  putchar ('\n');

  while (!q)
    {
      if (somaplayer_get_volume (c, &v))
	quit ();
      if (somaplayer_get_balance (c, &b))
	quit ();

      file = somaplayer_this (c);

      snprintf (buf, 79, "[ V: %3d - B: %4d ] - %s", v, b,
		file ? file : "no playing");

      buf[79] = '\r';

      write (1, buf, 80);

      free (file);

      FD_ZERO (&fd_read);
      FD_SET (0, &fd_read);

      tv.tv_sec = 0;
      tv.tv_usec = 500000;

      select (1, &fd_read, NULL, NULL, &tv);

      if (read (0, &ch, 1) != 1)
	continue;

      switch (ch)
	{
	case '\n':
	  q = 1;
	  break;

	case 'q':
	case 'Q':
	  somaplayer_quit (c);
	  q = 1;
	  break;

	case ' ':
	  somaplayer_pause (c);
	  break;

	case 'p':
	case 'P':
	  somaplayer_prev (c);
	  break;

	case 'n':
	case 'N':
	  somaplayer_next (c);
	  break;

	case 'v':
	  somaplayer_set_volume (c, v - 5);
	  break;

	case 'V':
	  somaplayer_set_volume (c, v + 5);
	  break;

	case 'b':
	  somaplayer_set_balance (c, b - 5);
	  break;

	case 'B':
	  somaplayer_set_balance (c, b + 5);
	  break;

	case 'c':
	case 'C':
	  somaplayer_set_balance (c, 0);
	  break;
	}
    }

  fcntl (1, F_SETFL, flag);

  if (somaplayer_close (c))
    quit ();

  echo_on ();

  return 0;
}
