/* Somaplayer - Copyright (C) 2003-4 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include "player.h"
#include "daemon.h"
#include "sds/sds.h"
#include "audio.h"
#include "output.h"
#include "other.h"
#include "util.h"
#include "output/monitor/monitor.h"
#include "ftime.h"
#include "mix.h"
#include "sock.h"

#ifdef ENABLE_GTK
#include "gtk/graphic.h"
#include "gdaemon/gdaemon.h"
#endif

#define DAEMON_MAX_TIMER 2.00

/* Questions:
 * 1 - Now there is a static array of clients. It is better a list ?
 *     I thinks: an array is more fast and 256 clients are too :)
 *
 * 2 - The output must be pesistented ?
 */

pthread_t th_graphic;

void daemon_get_client (int);
int daemon_socket (void);
void *daemon_play (void *);
void *daemon_client_read (void *);
void daemon_mix_audio (char *, int, int, int, int, char *);
void daemon_check_bigendian (daemon_client *, char *, int);
int daemon_check_autofader (void);

/* Check the cli options */
int
daemon_option (char *opt)
{

  int ch = 0;
  int len;
  int ok = 0;

  play->daemon->interface = NULL;
  play->daemon->port = SDS_PORT;
  play->daemon->listen = SDS_LISTEN;
#ifdef ENABLE_IPV6
  play->daemon->ipv6 = 0;
#endif
  play->daemon->socket = SOCKET_TCP;

  len = strlen (opt);

  while (ch < len)
    {

      switch (*(opt + ch))
	{

	case 'l':
	  if (!strncmp (opt + ch, "listen=", 7))
	    {
	      ch += 7;
	      play->daemon->listen = audio_parse_int (opt, &ch, len);
	      ok = 1;
	    }
	  else if (ok)
	    fatal (_("Error in daemon parameter: %s"), opt);

	  break;

	case 'i':
	  if (!strncmp (opt + ch, "interface=", 10))
	    {
	      ch += 10;
	      play->daemon->interface = audio_parse_str (opt, &ch, len);
	      ok = 1;
	    }

#ifdef ENABLE_IPV6
	  else if (!strncmp (opt + ch, "ipv6=", 5))
	    {
	      ch += 5;
	      play->daemon->ipv6 = audio_parse_int (opt, &ch, len);
	      ok = 1;
	    }
#endif

	  else if (ok)
	    fatal (_("Error in daemon parameter: %s"), opt);

	  break;

	case 'p':
	  if (!strncmp (opt + ch, "port=", 5))
	    {
	      ch += 5;
	      play->daemon->port = audio_parse_int (opt, &ch, len);
	      ok = 1;
	    }
	  else if (ok)
	    fatal (_("Error in daemon parameter: %s"), opt);
	  
	  break;

	case 't':
	  if (!strncmp (opt + ch, "type=", 5))
	    {
	      char *b;
	      ch += 5;
	      b = audio_parse_str (opt, &ch, len);
	      if (!b || !strcasecmp (b, "tcp") || !strcasecmp (b, "t"))
		play->daemon->socket = SOCKET_TCP;
	      else
		play->daemon->socket = SOCKET_UNIX;

	      ok = 1;
	    }
	  else if (ok)
	    fatal (_("Error in daemon parameter: %s"), opt);

	  break;

	default:
	  if (ok)
	    fatal (_("Error in daemon parameter: %s"), opt);

	  else
	    return 1;
	}
      ch++;
    }

  if (ok)
    return 0;
  else
    return 1;
}

/* This function is the controller */
void
daemon_start (void)
{

  int fd;
  register int i;

  if (pthread_mutex_init (&play->daemon->mutex, NULL))
    fatal (_("Error: Mutex init."));

#ifdef ENABLE_GTK
  /* Graphic interface: */
  if (play->graphic)
    if (pthread_create (&th_graphic, NULL, daemon_graphic, NULL))
      fatal (_("Thread error."));
#endif

  /* Clear the list of client. */
  for (i = 0; i < DAEMON_MAX_CLIENTS; i++)
    play->daemon->client[i] = NULL;

  fd = daemon_socket ();

  msg (_("soma distribuited sound start in %s mode."),
       play->daemon->socket == SOCKET_TCP ? "tcp" : "unix");

  /* The thread of mix */
  if (pthread_create (&th1, NULL, daemon_play, NULL))
    fatal (_("Thread error."));

  /* Get the client... */
  while (!events.quit)
    {
      sched_yield ();

      daemon_get_client (fd);
    }

  /* Exit if events.quit is 1 */
  close (fd);

  if (play->daemon->socket == SOCKET_UNIX)
    {
      char a[SIZE_BUFFER];
      snprintf (a, SIZE_BUFFER, "%s/sds-%d.sock", get_tmp_dir (), getpid ());

      unlink (a);
    }

  pthread_cond_signal (&play->daemon->daemon);
  pthread_join (th1, NULL);

  pthread_mutex_destroy (&play->daemon->mutex);

#ifdef ENABLE_GTK
  pthread_join (th_graphic, NULL);
#endif

}

/* Open the socket */
int
daemon_socket (void)
{

  int fd, yes = 1;
  struct sockaddr_in sock;
  struct hostent *hp;
  struct linger lin;
#ifdef ENABLE_IPV6
  struct addrinfo hints, *res, *result = NULL;
#endif

  if (!play->daemon->port)
    play->daemon->port = SDS_PORT;

  if (!play->daemon->listen)
    play->daemon->listen = SDS_LISTEN;

  if (play->daemon->socket == SOCKET_TCP)
    {
#ifdef ENABLE_IPV6
      if (play->daemon->ipv6 && ipv6_check ())
	{
	  memset (&hints, 0, sizeof (hints));
	  hints.ai_socktype = SOCK_STREAM;

	  if (play->daemon->interface)
	    {
	      if (getaddrinfo (play->daemon->interface, NULL, &hints, &result)
		  != 0)
		fatal (_("Host error. %s"), play->daemon->interface);

	      for (res = result; res; res = res->ai_next)
		if (res->ai_family != AF_INET || res->ai_family != AF_INET6)
		  break;

	      if (res->ai_family == AF_INET)
		{
		  ((struct sockaddr_in *) res->ai_addr)->sin_port =
		    htons (play->daemon->port);
		  play->daemon->ipv6 = 0;

		  if (!play->noverbose)
		    msg (_("No ipv6 server."));
		}

	      if (res->ai_family == AF_INET6)
		{
		  ((struct sockaddr_in6 *) res->ai_addr)->sin6_port =
		    htons (play->daemon->port);
		}

	      fd = socket (res->ai_family, SOCK_STREAM, 0);
	    }
	  else
	    {
	      struct sockaddr_in6 socket6_addr;

	      memset (&socket6_addr, 0, sizeof (struct sockaddr_in6));
	      socket6_addr.sin6_family = AF_INET6;
	      socket6_addr.sin6_port = htons (play->daemon->port);

	      socket6_addr.sin6_addr = in6addr_any;
	      fd = socket (AF_INET6, SOCK_STREAM, 0);
	    }
	}
      else
	{
#endif

	  memset ((void *) &sock, 0, sizeof (sock));
	  sock.sin_family = AF_INET;
	  sock.sin_port = htons (play->daemon->port);

	  if (!play->daemon->interface)
	    sock.sin_addr.s_addr = htonl (INADDR_ANY);
	  else
	    {

	      if (!(hp = gethostbyname (play->daemon->interface)))
		fatal (_("Binding error: %s."), play->daemon->interface);

	      sock.sin_family = hp->h_addrtype;
	      memcpy ((caddr_t) & sock.sin_addr, hp->h_addr, hp->h_length);
	    }

	  if ((fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	    fatal (_("Socket error."));

#if ENABLE_IPV6
	}
#endif

      /* UNIX SOCKET */
    }
  else
    {
      if ((fd = socket (AF_UNIX, SOCK_STREAM, 0)) < 0)
	{
	  msg_error (_("Socket error."));
	  return 1;
	}
    }

  if (fcntl (fd, F_SETFL, O_NONBLOCK) < 0)
    fatal (_("Non-blocking error."));

  if (play->daemon->socket == SOCKET_TCP)
    {
      lin.l_onoff = 1;
      lin.l_linger = 100;
      if (setsockopt (fd, SOL_SOCKET, SO_LINGER, &lin, sizeof (struct linger))
	  < 0)
	fatal (_("Linger error."));

      if (setsockopt (fd, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof (int)))
	fatal (_("Setsockopt error."));

      if (bind (fd, (struct sockaddr *) &sock, sizeof (sock)) < 0)
	fatal (_("Binding error."));

#ifdef ENABLE_IPV6
      if (result)
	freeaddrinfo (result);
#endif
    }
  else
    {
      struct sockaddr_un saddr;
      int d = 0;
      struct stat st;

      saddr.sun_family = AF_UNIX;

      snprintf (saddr.sun_path, 108, "%s/sds-%d.sock", get_tmp_dir (),
		getpid ());

      if (!lstat (saddr.sun_path, &st))
	{
	  d = 1;

	  while (d < 1000)
	    {
	      snprintf (saddr.sun_path, 108, "%s/sds-%d.%.2d.sock",
			get_tmp_dir (), getpid (), d);

	      if (lstat (saddr.sun_path, &st))
		{
		  d = 0;
		  break;
		}

	      d++;
	    }

	  if (d)
	    {
	      msg_error (_("You have 1000 somaplayer runs?"));
	      return 1;
	    }
	}

      if (bind (fd, (struct sockaddr *) &saddr, sizeof (saddr)) < 0)
	{
	  msg_error (_("Binding error."));
	  return 1;
	}
    }

  if (listen (fd, play->daemon->listen) < 0)
    fatal (_("Listen error."));

  return fd;
}

/* Get a client */
void
daemon_get_client (int fd)
{

  daemon_client *c;
  int port, addr;

#ifdef ENABLE_GTK
  char s[SIZE_BUFFER];
#endif

  if (!(c = (daemon_client *) malloc (sizeof (daemon_client))))
    fatal (_("Error: memory."));

  while (!events.quit)
    {
      char buffer[SIZE_BUFFER];
      int ret;
      char ch;

      usleep (100);

      if (play->daemon->socket == SOCKET_TCP)
	{
#ifdef ENABLE_IPV6
	  if (play->daemon->ipv6)
	    {
	      struct sockaddr_in6 incoming6;
	      size_t size_in6 = sizeof (struct sockaddr_in6);

	      if ((c->fd =
		   accept (fd, (struct sockaddr *) &incoming6,
			   &size_in6)) < 0)
		continue;

	      port = ntohs (incoming6.sin6_port);
	      addr = ntohl (incoming6.sin6_addr.s6_addr32);

	      if (!play->noverbose)
		msg (_("New client request (%d - %d.%d.%d.%d:%5d)"),
		     c->fd, (unsigned int) addr >> 24,
		     (unsigned int) (addr >> 16) % 256,
		     (unsigned int) (addr >> 8) % 256,
		     (unsigned int) addr % 256, port);

#ifdef ENABLE_GTK
	      if (!play->graphic)
		snprintf (s, SIZE_BUFFER, "%d.%d.%d.%d",
			  (unsigned int) addr >> 24,
			  (unsigned int) (addr >> 16) % 256,
			  (unsigned int) (addr >> 8) % 256,
			  (unsigned int) addr % 256);
#endif
	    }
	  else
	    {
#endif
	      struct sockaddr_in incoming;
	      size_t size_in = sizeof (struct sockaddr_in);

	      if ((c->fd =
		   accept (fd, (struct sockaddr *) &incoming, &size_in)) < 0)
		continue;

	      port = ntohs (incoming.sin_port);
	      addr = ntohl (incoming.sin_addr.s_addr);

	      if (!play->noverbose)
		msg (_("New client request (%d - %d.%d.%d.%d:%5d)"),
		     c->fd, (unsigned int) addr >> 24,
		     (unsigned int) (addr >> 16) % 256,
		     (unsigned int) (addr >> 8) % 256,
		     (unsigned int) addr % 256, port);

#ifdef ENABLE_GTK
	      if (!play->graphic)
		snprintf (s, SIZE_BUFFER, "%d.%d.%d.%d",
			  (unsigned int) addr >> 24,
			  (unsigned int) (addr >> 16) % 256,
			  (unsigned int) (addr >> 8) % 256,
			  (unsigned int) addr % 256);
#endif

#ifdef ENABLE_IPV6
	    }
#endif
	}

      /* UNIX SOCKET */
      else
	{
	  struct sockaddr_un incoming;
	  size_t size_un = sizeof (struct sockaddr_un);

	  if ((c->fd =
	       accept (fd, (struct sockaddr *) &incoming, &size_un)) < 0)
	    continue;

	  if (!play->noverbose)
	    msg (_("New client request"));

#ifdef ENABLE_GTK
	  if (!play->graphic)
	    snprintf (s, SIZE_BUFFER, "Unix Socket");
#endif
	  strcpy (s, "socket");
	}

      /* In the first line there is the type of clients. 
       * MONITOR if it is a MONITOR :)
       * rate, channel and bitrate if is a client
       *
       * There are some functions in libsds and 1 example.
       */

      ret = 0;
      while (read (c->fd, &ch, 1) > 0)
	{
	  if (ch == '\n')
	    break;

	  buffer[ret++] = ch;

	  if (ret == SIZE_BUFFER - 1)
	    break;
	}

      buffer[ret] = 0;

      if (ch != '\n' && ret != SIZE_BUFFER - 1)
	{
	  close (c->fd);

	  if (!play->noverbose)
	    msg (_("Error protocol."));

	  continue;
	}

      /* For test check */
      if (!strncmp (buffer, "TEST", 4))
	{
	  close (c->fd);

	  if (!play->noverbose)
	    msg (_("Connection test"));

	  continue;
	}

      /* For me a monitor is ONLY another output. So I add in
       * the list of outputs this new element. */
      else if (!strncmp (buffer, "MONITOR ", 8))
	{
	  audio *tmp;

	  sscanf (buffer + 8, "%d", &c->bigendian);

	  if (!(tmp = (audio *) malloc (sizeof (audio))))
	    fatal (_("Error: memory."));

	  tmp->audio_type = USE_MONITOR;

	  tmp->realtime = 0;

	  tmp->open = monitor_init;
	  tmp->write = monitor_write;
	  tmp->close = monitor_close;
	  tmp->check = NULL;
	  tmp->info = monitor_show;
	  tmp->data = (void *) c;
	  tmp->next = NULL;
	  tmp->status = OUTPUT_DONE;

	  if (!(tmp->audio_dev = strdup (s)))
	    fatal ("Error: memory.");

#ifdef ENABLE_GTK
	  tmp->stop = 0;
#endif

	  if (pthread_cond_init (&tmp->cond, NULL))
	    fatal (_("Error: Cond init."));
	  if (pthread_mutex_init (&tmp->mutex, NULL))
	    fatal (_("Error: Mutex init."));
	  if (pthread_create (&tmp->th, NULL, output_thread, tmp))
	    fatal (_("Thread error."));

	  if (play->ao_rate && play->ao_channels && play->ao_bitrate)
	    {
	      if (tmp->
		  open (tmp, play->ao_rate, play->ao_channels,
			play->ao_bitrate))
		{
		  free (tmp->audio_dev);
		  free (tmp);
		}
	      else
		{
		  tmp->info (tmp);

		  output_lock ();

		  tmp->next = play->output;
		  play->output = tmp;

		  output_unlock ();

#ifdef ENABLE_GTK
		  if (play->graphic)
		    output_window_add (tmp, PREPEND);
#endif
		}
	    }
	  else
	    {
	      output_lock ();

	      tmp->next = play->output;
	      play->output = tmp;

	      output_unlock ();

#ifdef ENABLE_GTK
	      if (play->graphic)
		output_window_add (tmp, PREPEND);
#endif
	    }

	  /* I now exit, because the new output element is inside
	   * of the list and every is ok... mmm.... I hope */
	  return;
	}

      /* If this client is a audio input... */
#if ENABLE_GTK
      ret =
	sscanf (buffer, "%d %d %d %d %d %d %d %d", &c->rate, &c->channels,
		&c->bitrate, &c->volume, &c->balance, &c->solo, &c->mute,
		&c->bigendian);
#else
      ret = sscanf (buffer, "%d %d %d", &c->rate, &c->channels, &c->bitrate);
#endif

      if (ret < 3)
	{
	  write (c->fd, "Syntax error.\n", 14);
	  close (c->fd);

	  if (!play->noverbose)
	    msg (_("Error protocol."));

	  continue;
	}

#if ENABLE_GTK
      c->autofader = 0;

      switch (ret)
	{
	case 3:
	  c->volume = 100;
	  c->balance = 50;
	  c->solo = 0;
	  c->mute = 0;
	  break;

	case 4:
	  c->balance = 50;
	  c->solo = 0;
	  c->mute = 0;
	  break;

	case 5:
	  c->solo = 0;
	  c->mute = 0;
	  break;

	case 6:
	  c->mute = 0;
	  break;
	}

#endif

      if (!play->noverbose)
	{
	  msg (_("Channels: %d"), c->channels);
	  msg (_("Rate: %d"), c->rate);
	  msg (_("Bitrate: %d\n"), c->bitrate);
	}

      if (c->channels > 2 || c->channels < 1)
	{
	  write (c->fd, "Channels error.\n", 16);
	  close (c->fd);
	  continue;
	}

      if (c->bitrate > 16 || c->bitrate < 8)
	{
	  write (c->fd, "Channels error.\n", 16);
	  close (c->fd);
	  continue;
	}

      c->done = 0;
      c->id_w = c->id_r = 0;
      if (pthread_mutex_init (&c->mutex, NULL))
	fatal (_("Error: Mutex init."));
      if (pthread_cond_init (&c->wait, NULL))
	fatal (_("Error: Cond init."));

      pthread_mutex_lock (&play->daemon->mutex);

      ret = 0;
      while (ret < DAEMON_MAX_CLIENTS)
	{
	  if (play->daemon->client[ret] == NULL)
	    {
	      play->daemon->client[ret] = c;
	      break;
	    }
	  ret++;
	}

      if (ret == DAEMON_MAX_CLIENTS)
	{
	  pthread_mutex_unlock (&play->daemon->mutex);

	  if (!play->noverbose)
	    msg (_("Too clients."));

	  close (c->fd);
	  pthread_mutex_destroy (&c->mutex);
	  pthread_cond_destroy (&c->wait);

	  continue;
	}

#ifdef ENABLE_GTK
      if (play->graphic)
	{
	  c->widget = gdaemon_add_mix (s, c);
	  msg_gstatusbar (_("Connection from %s"), s);
	}
#endif

      /* A new input is a new thread of input... */
      if (pthread_create (&c->th, NULL, daemon_client_read, c))
	fatal (_("Thread error."));

      pthread_mutex_unlock (&play->daemon->mutex);

      return;
    }
}

/* This function MIX the client's audios... 
 * REMEMBER: It is a separated thread! */
void *
daemon_play (void *a)
{
  register int i, n_inputs;
  int size = SIZE_BUFFER / sizeof (signed short);
  signed short buffer[size];
  struct timeval *timer = NULL;

  pthread_mutex_lock (&play->mutex);
  pthread_cond_wait (&play->daemon->daemon, &play->mutex);
  pthread_mutex_unlock (&play->mutex);

  /* LOOP */
  while (!events.quit)
    {

      /* New simple: */
      memset (buffer, 0, SIZE_BUFFER);

      /* Check all clients */
      for (i = n_inputs = 0; i < DAEMON_MAX_CLIENTS; i++)
	{

	  /*
	   * This is an array, so when a client exit, 
	   * I don't shift ALL other client and the 
	   * element of ex client will be NULL.
	   *
	   * Here: If the client exist...
	   */
	  if (play->daemon->client[i] != NULL)
	    {

	      /* If the client exist and the buffer is empty, it is exist? */
	      if (play->daemon->client[i]->done == 1)
		{
		  pthread_mutex_lock (&play->daemon->mutex);

		  pthread_join (play->daemon->client[i]->th, NULL);
		  pthread_mutex_destroy (&play->daemon->client[i]->mutex);
		  pthread_cond_destroy (&play->daemon->client[i]->wait);

#ifdef ENABLE_GTK
		  if (play->graphic)
		    {
		      gdaemon_del_mix (play->daemon->client[i]);
		      msg_gstatusbar (_("Remove one mixer."));
		    }
#endif

		  free (play->daemon->client[i]);
		  play->daemon->client[i] = NULL;

		  pthread_mutex_unlock (&play->daemon->mutex);
		}

	      /* ... and there is some thinks in his buffer: */
	      else if (play->daemon->client[i]->id_r !=
		       play->daemon->client[i]->id_w)
		{
		  int k;
		  signed short *p;

		  /* The buffer of client can be no empty, but it can write when
		   * i want read it... so I must check if a can LOCK his buffer:
		   */
		  if (pthread_mutex_trylock
		      (&play->daemon->client[i]->mutex) != EBUSY)
		    {

		      n_inputs++;

#ifdef ENABLE_GTK
		      /* In graphic i can change the single volumes, so i change
		       * it here: */
		      if (play->graphic)
			daemon_set_audio (play->daemon->client[i],
					  play->daemon->client[i]->
					  buffer[play->daemon->client[i]->
						 id_r]);
#endif

#ifdef ENABLE_GTK
		      /* If no mute: */
		      if (!play->daemon->client[i]->mute)
			{
#endif
			  /* Update the output buffer */
			  p =
			    (signed short *) play->daemon->client[i]->
			    buffer[play->daemon->client[i]->id_r];

			  for (k = 0; k < size; k++)
			    buffer[k] += p[k];
#ifdef ENABLE_GTK
			}
		      else if (n_inputs == 1)
			{
			  /* if all is mute ... */
			  memset (buffer, 0, SIZE_BUFFER);
			}
#endif
		      /* Ok, now i must erase one element of this client: */
		      if (play->daemon->client[i]->id_r ==
			  D_C_MAX_ELEMENT - 1)
			play->daemon->client[i]->id_r = 0;
		      else
			play->daemon->client[i]->id_r++;

		      pthread_mutex_unlock (&play->daemon->client[i]->mutex);
		      pthread_cond_signal (&play->daemon->client[i]->wait);
		    }
		}
	    }
	}

      /* If something exist, I put it out */
      if (n_inputs)
	{

	  if (timer)
	    {
	      timer_clear (timer);
	      timer = NULL;
	    }

	  if (!play->play)
	    {
	      output_open (44100, 2, 16);
	      output_info ();

	      play->play = 1;
	    }

#ifdef ENABLE_GTK
	  if (play->mute)
	    memset (buffer, 0, SIZE_BUFFER);
#endif

	  output_write (buffer, SIZE_BUFFER);
	}

      /* Else... no output. Close the output only if timer is > 
       * DAEMON_MAX_TIMER. The reason is: if there are inputs, but
       * all struct are locked ? Aspect...

       * I can use the number of client, but... if 3 client don't send data
       * for too seconds? I close the outputs.
       */
      else
	{
	  if (!timer)
	    timer = timer_start ();

	  else
	    {

	      double t = timer_time (timer);

	      if ((t - 60.0f * ((long) t / (long) 60)) > DAEMON_MAX_TIMER)
		{
		  if (play->play)
		    {
		      play->play = 0;


		      output_forced_close ();

		      pthread_mutex_lock (&play->mutex);
		      pthread_cond_wait (&play->daemon->daemon, &play->mutex);
		      pthread_mutex_unlock (&play->mutex);
		    }
		}
	    }
	}
    }

  /* Close the output, and exit */
  if (play->play)
    {
      play->play = 0;

      output_close ();
    }

  pthread_exit (NULL);
}

/* This read from the client */
void *
daemon_client_read (void *a)
{
  fd_set fd_read;
  int ret = 0;
  daemon_client *c = (daemon_client *) a;
  int size = SIZE_BUFFER;
  struct timeval tv;

  if (!c)
    fatal (_("Internal error."));

  if (c->channels == 1)
    size /= 2;

  if (c->bitrate == 8)
    size /= 2;

  size /= 44100 / c->rate;

  FD_ZERO (&fd_read);
  FD_SET (c->fd, &fd_read);

  while (!c->done)
    {
      char buf[size];

      c->len = 0;

      /* Read a internal buffer */
      while (c->len < size)
	{

	  tv.tv_sec = 5;
	  tv.tv_usec = 0;

	  if (select (c->fd + 1, &fd_read, NULL, NULL, &tv) < 0)
	    {
	      c->done = 1;
	      break;
	    }

	  ret = read (c->fd, buf + c->len, size - c->len);

	  if (ret <= 0 && errno != EAGAIN)
	    {
	      c->done = 1;
	      break;
	    }

	  c->len += ret;
	}

      if (c->done == 1)
	break;

      daemon_check_bigendian (c, buf, c->len);

      /* Insert the internal buffer inside of data array */
      pthread_mutex_lock (&c->mutex);

      if (c->id_r == (c->id_w + 1)
	  || (!c->id_r && c->id_w == D_C_MAX_ELEMENT - 1))
	{
	  pthread_cond_signal (&play->daemon->daemon);
	  pthread_cond_wait (&c->wait, &c->mutex);
	}

      /* Convert the buffer in rate=44100 bitrate=16 and channels=2 */
      daemon_mix_audio (buf, size, c->rate, c->channels, c->bitrate,
			c->buffer[c->id_w]);

      if (c->id_w == D_C_MAX_ELEMENT - 1)
	c->id_w = 0;
      else
	c->id_w++;

      pthread_mutex_unlock (&c->mutex);
      pthread_cond_signal (&play->daemon->daemon);

    }

  /* When done == 1 I exit */
  if (!play->noverbose)
    msg (_("Client exit (%d)"), c->fd);

  pthread_exit (NULL);
}

#ifdef ENABLE_GTK
/* Set the volume from the graphic interface */
void
daemon_set_audio (daemon_client * c, char *b)
{
  signed short *ptr = (signed short *) b;
  int len;
  double v;

  if (c->channels == 2)
    {
      len = SIZE_BUFFER / 4;

      if (play->daemon->autofader && !c->autofader
	  && daemon_check_autofader ())
	{
	  while (len--)
	    {
	      v = gdaemon_volume_l_get (c) - (double) play->daemon->autofader;
	      if (v < 0)
		v = 0;

	      *ptr++ = *ptr * (v / 100);

	      v = gdaemon_volume_r_get (c) - (double) play->daemon->autofader;
	      if (v < 0)
		v = 0;

	      *ptr++ = *ptr * (v / 100);
	    }
	}
      else
	while (len--)
	  {
	    *ptr++ = *ptr * (gdaemon_volume_l_get (c) / 100);
	    *ptr++ = *ptr * (gdaemon_volume_r_get (c) / 100);
	  }
    }

  else if (c->channels == 1)
    {
      len = SIZE_BUFFER / 2;

      if (play->daemon->autofader && !c->autofader
	  && daemon_check_autofader ())
	{
	  while (len--)
	    {
	      v = (int) gdaemon_volume_get (c) - play->daemon->autofader;
	      if (v < 0)
		v = 0;

	      *ptr++ = *ptr * (v / 100);
	    }
	}
      else
	while (len--)
	  {
	    *ptr++ = *ptr * ((int) gdaemon_volume_get (c) / 100);
	  }
    }
}

#endif

/* Convert rate, bitrate and channels */
void
daemon_mix_audio (char *buf, int len, int rate, int channels, int bitrate,
		  char *dst)
{
  if (bitrate == 8)
    {
      if (channels == 1)
	mix_8_mono_16_stereo (buf, len, rate, dst);
      else
	mix_8_stereo_16_stereo (buf, len, rate, dst);
    }
  else if (bitrate == 16)
    {
      if (channels == 1)
	mix_16_mono_16_stereo (buf, len, rate, dst);
      else
	mix_16_stereo_16_stereo (buf, len, rate, dst);
    }
}

void
daemon_check_bigendian (daemon_client * c, char *bug, int len)
{
  int i;
  char t;

  if (c->bigendian == big_endian ())
    return;

  for (i = 0; i < len / 2; i += 2)
    {
      t = bug[i];
      bug[i] = bug[i + 1];
      bug[i + 1] = t;
    }
}

#ifdef ENABLE_GTK
int
daemon_check_autofader (void)
{
  register int i, j;
  int ret = 0;
  signed short *p;
  int size = SIZE_BUFFER / sizeof (signed short);

  for (i = 0; i < DAEMON_MAX_CLIENTS; i++)
    {
      if (play->daemon->client[i] != NULL
	  && play->daemon->client[i]->autofader)
	{
	  p =
	    (signed short *) play->daemon->client[i]->buffer[play->daemon->
							     client[i]->id_r];

	  for (j = 0; j < size; j++)
	    {
	      /* 0 is stupid! 200 ? 300 ? Audio test... */
	      if (p[j] > 200)
		{
		  ret = 1;
		  break;
		}
	    }
	}
    }

  return ret;
}
#endif

/* EOF */
