/* Somaplayer - Copyright (C) 2003-4 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include "player.h"
#include "file.h"
#include "other.h"

/* It is a remote file ? */
int
remote_file (char *f)
{
  if (!strncmp (f, "https://", 8))
    return HTTPS;

  if (!strncmp (f, "http://", 7))
    return HTTP;

  return 0;
}

/* Return the file type. */
int
type_file (char *f)
{

  if (!strcasecmp (".pls", f + strlen (f) - 4))
    return TYPE_PLAYLIST;

  if (!strcasecmp (".m3u", f + strlen (f) - 4))
    return TYPE_PLAYLIST;

#ifdef ENABLE_MP3
  if (!strcasecmp (".mp3", f + strlen (f) - 4))
    return TYPE_MP3;
#endif

#ifdef ENABLE_OGG
  if (!strcasecmp (".ogg", f + strlen (f) - 4))
    return TYPE_OGG;
#endif

#ifdef ENABLE_SNDFILE
  if (!strcasecmp (".wav", f + strlen (f) - 4))
    return TYPE_SNDFILE;

  if (!strcasecmp (".aif", f + strlen (f) - 4))
    return TYPE_SNDFILE;

  if (!strcasecmp (".aiff", f + strlen (f) - 5))
    return TYPE_SNDFILE;

  if (!strcasecmp (".au", f + strlen (f) - 3))
    return TYPE_SNDFILE;

  if (!strcasecmp (".raw", f + strlen (f) - 4))
    return TYPE_SNDFILE;

  if (!strcasecmp (".cdr", f + strlen (f) - 4))
    return TYPE_SNDFILE;

  if (!strcasecmp (".paf", f + strlen (f) - 4))
    return TYPE_SNDFILE;

  if (!strcasecmp (".svx", f + strlen (f) - 4))
    return TYPE_SNDFILE;

  if (!strcasecmp (".nist", f + strlen (f) - 5))
    return TYPE_SNDFILE;

  if (!strcasecmp (".voc", f + strlen (f) - 4))
    return TYPE_SNDFILE;

  if (!strcasecmp (".ircam", f + strlen (f) - 6))
    return TYPE_SNDFILE;

  if (!strcasecmp (".w64", f + strlen (f) - 4))
    return TYPE_SNDFILE;

  if (!strcasecmp (".mat4", f + strlen (f) - 5))
    return TYPE_SNDFILE;

  if (!strcasecmp (".mat5", f + strlen (f) - 5))
    return TYPE_SNDFILE;

  if (!strcasecmp (".pvf", f + strlen (f) - 4))
    return TYPE_SNDFILE;

  if (!strcasecmp (".xi", f + strlen (f) - 4))
    return TYPE_SNDFILE;

  if (!strcasecmp (".htk", f + strlen (f) - 4))
    return TYPE_SNDFILE;

#ifdef MIDI
  if (!strcasecmp (".sds", f + strlen (f) - 4))
    return TYPE_SNDFILE;
#endif

#ifdef USE_AVR
  if (!strcasecmp (".avr", f + strlen (f) - 4))
    return TYPE_SNDFILE;
#endif

#ifdef USE_WAVEX
  if (!strcasecmp (".wavex", f + strlen (f) - 6))
    return TYPE_SNDFILE;
#endif

#endif

#ifdef ENABLE_CDAUDIO
  if (!strncasecmp ("cdrom://", f, 8))
    return TYPE_CDAUDIO;
#endif

#ifdef ENABLE_MIC
  if (!strcasecmp ("mic://*", f))
    return TYPE_MIC;
#endif

#ifdef ENABLE_DAEMON
  if (!strncmp ("sds://", f, 6))
    return TYPE_DAEMON_INPUT;
#endif

#ifdef ENABLE_MP3
  if (remote_file (f))
    return TYPE_MP3;
#endif

#ifdef ENABLE_SNDFILE
  return TYPE_SNDFILE;
#else
  return 0;
#endif
}

/* My OPEN! 
 * Open streams, files, ipv6, ssl... */
int
my_open (char *fl, int type)
{
  char *filename = NULL;
  char *filename_m = NULL;
  char *server = NULL;
  char *port = NULL;
  char *file = NULL;
  char buffer[SIZE_BUFFER];
  int len;
#ifdef ENABLE_OPENSSL
  SSL *ssl;
  SSL_CTX *ctx;
#endif
  struct sockaddr_in sock;
  struct hostent *hp;
  int error, errlen;
  struct timeval tv;
  fd_set set;

  if (!type)
    {
      struct stat st;

      if (lstat (fl, &st))
	{
	  msg_error (_("Error Opening file %s"), fl);
	  return 1;
	}

      play->length = st.st_size;

      if ((play->fd = open (fl, O_RDONLY)) < 0)
	return 1;

      return 0;
    }
  else
    {
      play->length = 0;
    }

  filename_m = strdup (fl);
  filename = filename_m;

  if (type == HTTPS)
    filename += 8;
  else
    filename += 7;

  server = filename;

  while (*filename++)
    {
      if (*filename == ':')
	{
	  *filename = 0;
	  port = ++filename;
	}
      else if (*filename == '/')
	{
	  *filename = 0;
	  file = ++filename;
	}
    }

  if (!port)
    port = "80";

  if (!file)
    file = "";


  if (type == HTTPS)
    {
#ifdef ENABLE_OPENSSL
      char *tmp;

      msg
	(_
	 ("Connect to streaming:\n\tServer %s\n\tPort %s\n\tFile %s\n\tProtocol https"),
	 server, port, file);

      play->secure = 1;

      SSLeay_add_ssl_algorithms ();
      ctx = SSL_CTX_new (SSLv23_client_method ());
      play->bio = BIO_new_ssl_connect (ctx);
      BIO_get_ssl (play->bio, &ssl);

      if (!ssl)
	{
	  msg_error (_("Ssl connection error."));
	  return 1;
	}

      SSL_set_mode (ssl, SSL_MODE_AUTO_RETRY);

      if (!(tmp = (char *) malloc ((strlen (server) + 7) * sizeof (char))))
	fatal (_("Error: memory."));

      sprintf (tmp, "%s:%s", server, port);

      if (BIO_set_conn_hostname (play->bio, tmp) < 0)
	{
	  msg_error (_("Ssl connection error."));
	  free (tmp);
	  return 1;
	}

      if (BIO_do_connect (play->bio) < 0)
	{
	  msg_error (_("Ssl connection error."));
	  free (tmp);
	  return 1;
	}

      if (BIO_do_handshake (play->bio) < 0)
	{
	  msg_error (_("Ssl connection error."));
	  free (tmp);
	  return 1;
	}

      free (tmp);

#else
      msg_error (_("No support for ssl connection. Compile with SSL."));
      return 1;
#endif
    }
  else
    {

#ifdef ENABLE_OPENSSL
      play->secure = 0;
#endif

      msg
	(_
	 ("Connect to streaming:\n\tServer %s\n\tPort %s\n\tFile %s\n\tProtocol http"),
	 server, port, file);

      if ((play->fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	{
	  msg_error (_("Socket error."));
	  return 1;
	}

      if (fcntl (play->fd, F_SETFL, O_NONBLOCK) < 0)
	{
	  msg_error (_("Socket error."));
	  return 1;
	}

      if (!(hp = gethostbyname (server)))
	{
	  msg_error (_("Error: unknown host."));
	  return 1;
	}

      memset ((void *) &sock, 0, sizeof (sock));
      sock.sin_family = AF_INET;
      sock.sin_addr.s_addr = ((struct in_addr *) (hp->h_addr))->s_addr;
      sock.sin_port = htons (atoi (port));

      if (connect (play->fd, (struct sockaddr *) &sock, sizeof (sock)) < 0)
	{
	  if (errno != EINPROGRESS)
	    {
	      msg_error (_("Connect error."));
	      return 1;
	    }
	}

      while (1)
	{

	  if (events.skip || events.quit)
	    return 1;

	  tv.tv_sec = 0;
	  tv.tv_usec = 10000;
	  FD_ZERO (&set);
	  FD_SET (play->fd, &set);

	  if (select (play->fd + 1, NULL, &set, NULL, &tv) > 0)
	    {
	      errlen = sizeof (error);
	      getsockopt (play->fd, SOL_SOCKET, SO_ERROR, &error, (socklen_t *)&errlen);

	      if (error)
		{
		  msg_error (_("Connect error."));
		  return 1;
		}

	      break;

	    }
	}
    }

  len =
    snprintf (buffer, SIZE_BUFFER,
	      "GET /%s HTTP/1.0\r\n" "Host: %s\r\n" "User-Agent: %s/%s\r\n"
	      "Accept: */*\r\n\r\n", file, server, NAME, VERSION);
  my_write (buffer, len);

  free (filename_m);

  do
    {

      if ((len = get_line (buffer)) < 0)
	{
	  msg_error (_("http error."));
	  return 1;
	}

      buffer[len] = 0;

      if (!strncmp (buffer, "Location:", 9))
	{
	  my_close ();

	  free (filename);

	  return my_open (buffer + 10, remote_file (buffer + 10));
	}

      if (!strncmp (buffer, "ICY ", 4))
	{
	  if (!strncmp (buffer + 4, "200 ", 4))
	    msg (_("Connection done."));

	  else if (!strncmp (buffer + 4, "404 ", 4))
	    {
	      msg_error (_("File not found."));
	      return 1;
	    }
	  else
	    {
	      msg_error (_("Error in icecast protocol."));
	      return 1;
	    }
	}

    }
  while (!strcmp (buffer, "\n") || !strcmp (buffer, "\r\n"));

  return 0;
}


/* My Close */
void
my_close (void)
{

#ifdef ENABLE_OPENSSL
  if (play->secure)
    {
      BIO_ssl_shutdown (play->bio);
      BIO_free_all (play->bio);
    }
  else
#endif

    close (play->fd);
}

/* My Write :) */
size_t
my_write (void *a, size_t b)
{
#ifdef ENABLE_OPENSSL
  if (play->secure)
    return BIO_write (play->bio, a, b);
#endif

  return write (play->fd, a, b);
}

/* And my Read */
size_t
my_read (void *a, size_t b)
{
  int ret;

#ifdef ENABLE_OPENSSL
  if (play->secure)
    return BIO_read (play->bio, a, b);

#endif
  ret = read (play->fd, a, b);

  if (ret < 0 && errno == EAGAIN)
    {
      fd_set set;
      struct timeval tv;

      tv.tv_sec = 0;
      tv.tv_usec = 50000;
      FD_ZERO (&set);
      FD_SET (play->fd, &set);
      ret = select (play->fd + 1, &set, NULL, NULL, &tv);

      if (events.skip || events.quit)
	return -1;

      return my_read (a, b);
    }

  return ret;
}

/* For remote header, I must analize strings with 1 line only. 
 * This functions, with my_read gets this lines. */
size_t
get_line (char *buff)
{
  register int b;
  static char line[SIZE_BUFFER];
  static int fill = 0;

  while (fill <= SIZE_BUFFER)
    {

      for (b = 0; b < fill; b++)
	{

	  if (*(line + b) == '\n')
	    {
	      memcpy (buff, line, b);
	      memmove (line, line + b + 1, SIZE_BUFFER - (b + 1));

	      fill -= b + 1;
	      *(buff + b) = 0;

	      return b;
	    }
	}

      if (!(b = my_read (line + fill, SIZE_BUFFER - fill)))
	{
	  fill = 0;
	  return 0;
	}
      else if (b < 0)
	return -1;

      fill += b;
    }

  memcpy (buff, line, SIZE_BUFFER);
  fill = 0;

  return SIZE_BUFFER;
}

/* EOF */
