/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifndef PLAYER_H
#define PLAYER_H

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/time.h>
#include <sys/wait.h>

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#ifdef ENABLE_AO
#include <ao/ao.h>
#endif

#ifdef ENABLE_SNDFILE
#include <sndfile.h>
#endif

#ifdef ENABLE_LAME
#include <lame/lame.h>
#endif

#ifdef ENABLE_OPENSSL
#include <openssl/ssl.h>
#include <openssl/bio.h>
#endif

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <pthread.h>
#include <termios.h>
#include <signal.h>
#include <dirent.h>

#ifdef ENABLE_CDAUDIO
#include <cdda_interface.h>
#include <cdda_paranoia.h>
#endif

#ifdef ENABLE_MP3
#ifdef ENABLE_ID3
#include <id3.h>
#include <id3tag.h>
#endif
#include <mad.h>
#endif

#ifdef ENABLE_OGG
#include <ogg/ogg.h>
#include <vorbis/codec.h>
#include <vorbis/vorbisfile.h>
#include <vorbis/vorbisenc.h>
#endif

#ifdef ENABLE_MIC
#ifdef HAVE_MACHINE_SOUNDCARD_H
#include <machine/soundcard.h>
#endif
#if HAVE_LINUX_SOUNDCARD_H
#include <linux/soundcard.h>
#endif
#if HAVE_SYS_SOUNDCARD_H
#include <sys/soundcard.h>
#endif
#if HAVE_SOUNDCARD_H
#include <soundcard.h>
#endif
#if HAVE_SYS_AUDIO_H
#include <sys/audio.h>
#endif
#if HAVE_ALSA_SOUNDLIB_H
#include <alsa/soundlib.h>
#endif
#if HAVE_SYS_SOUNDLIB_H
#include <alsa/asoundlib.h>
#endif
#endif

#include <sys/ioctl.h>
#include <errno.h>

#include <confuse.h>

#define AUTHOR_COPYRIGHT "Bakunin - <bakunin@autistici.org> Copyright GPL 2003-5"
#define NAME "somaplayer"

#ifdef ENABLE_NLS
#  include <locale.h>
#  include <libintl.h>
#  define _(string) gettext (string)
#  ifdef gettext_noop
#    define N_(string) gettext_noop (string)
#  else
#    define N_(string) (string)
#  endif
#else
#  define textdomain(string) (string)
#  define gettext(string) (string)
#  define dgettext(domain, message) (message)
#  define dcgettext(domain,message,type) (message)
#  define bindtextdomain(domain,directory) (domain)
#  define _(string) (string)
#  define N_(string) (string)
#endif

#define SIZE_BUFFER 2048

#ifdef ENABLE_CDAUDIO
#  ifndef CDROM_DEVICE
#    define CDROM_DEVICE "/dev/cdrom"
#  endif
#endif

#ifdef ENABLE_GTK
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#endif

#define FORMATS_NUMBER	6
#define DAEMON_MAX_CLIENTS 100

#ifndef ENABLE_LAME
typedef int MPEG_mode;
#endif

#define OUT_MONO 1
#define OUT_STEREO 2

/* Struct for mp3 dither */
#ifdef ENABLE_MP3
struct audio_dither
{
  mad_fixed_t error[3];
  mad_fixed_t random;
};
#endif

/* Socket struct */
typedef struct
{
  int type;
  char *server;
  int port;
} sock_struct;

/* Output struct. */
typedef struct _audio
{
  int (*open) (struct _audio *, long, int, int);
  void (*info) (struct _audio *);
  int (*write) (struct _audio *, int, int, void *, size_t);
  void (*close) (struct _audio *);
  int (*check) (struct _audio *);
  void (*update) (struct _audio *);

  void *data;

#ifdef ENABLE_GTK
  int stop;
#endif

  int status;

  int audio_type;
  char *audio_dev;

  pthread_t th;
  pthread_cond_t cond;
  pthread_mutex_t mutex;

  int realtime;
  int bad;

  struct _audio *next;

} audio;

/* Format struct */
typedef struct
{
  int type;
  void (*run) (void);
  char *(*get_time) (void);
  int (*get_buffer) (void);
  char *(*get_name) (char *);

  void *data;
} format;

/* Event mask */
typedef struct
{
  int quit;
  int skip;
  int next;
  int pause;
#ifdef ENABLE_GTK
  int restart;
  int stop;
#endif
} event;

/* List of files */
typedef struct _list_
{
  char *filename;
  char *name;

  int n;			/* <-- for repeat element and random playlist */
  int queue;			/* Number of queue list */

  struct _list_ *prev;
  struct _list_ *next;
} list;

#ifdef ENABLE_GTK
/* Queue */
typedef struct _queue_
{
  list *item;
  struct _queue_ *next;
} queue;
#endif

/* Daemon_Client struct */
typedef struct
{
  int fd;
  int rate;
  int channels;
  int bitrate;

#define D_C_MAX_ELEMENT 10

  char buffer[D_C_MAX_ELEMENT][SIZE_BUFFER];	/* The buffer */

  int len;

  int id_r;
  int id_w;

  int done;

  pthread_cond_t wait;
  pthread_mutex_t mutex;
  pthread_t th;

  /* Graphic elements: */
#ifdef ENABLE_GTK
  char *addr;

  int mute;
  int solo;

  GtkWidget *widget;

  GtkWidget *w_v_up;
  int v_up;

  GtkWidget *w_v_down;
  int v_down;

  GtkWidget *w_b_up;
  int b_up;

  GtkWidget *w_b_down;
  int b_down;

  GtkWidget *w_volume;
  int volume;

  GtkWidget *w_balance;
  int balance;

  GtkWidget *w_autofader;
  int autofader;

  char *socket;
  GdkColor *color;

#endif

  int bigendian;

} daemon_client;

/* Daemon struct */
typedef struct
{
  int socket;

  char *interface;
  int port;
  int listen;
#ifdef ENABLE_IPV6
  int ipv6;
#endif

#ifdef ENABLE_GTK
  int autofader;
#endif

  daemon_client *client[DAEMON_MAX_CLIENTS];
  pthread_cond_t daemon;
  pthread_mutex_t mutex;

} daemon_data;

/* Admin struct */
typedef struct
{
  char *socket;

  char *interface;
  int port;
  int listen;

  int type;

  int fd;

  pthread_t th;

} admin_data;

/* The main struct */
typedef struct
{
  void *realbuf;		/* Real buffer */
  int realsize;			/* Size of real buffer */
  size_t start;			/* pointer of start */
  size_t stop;			/* pointer to stop */
  int done;			/* Check variable */
  int play;			/* I'm playing or not? */

  int remote;

  list *this;			/* This input. */
  list *first;			/* The first input */
  list *last;			/* The last input */
  int number;			/* Number of inputs */

  pthread_mutex_t m_list;
  pthread_cond_t c_list;

  int repeat;			/* Repeat ? */

  int fd;			/* FileDescriptor of local or remote file */
#ifdef ENABLE_OPENSSL
  BIO *bio;			/* BIO ssl connect of remote ssl file */
  int secure;			/* SSL ? */
#endif

  size_t length;		/* Length of file. */

#ifdef ENABLE_MIC
  /* mic data */
  int microphone;
  char *devdsp;
  char *mic_option;
#endif

#ifdef ENABLE_DAEMON
  int daemon_input;
  char *daemon_input_opt;
#endif

  char *config_audio;		/* Audio output from config file */

  int volume;			/* Volume */
  int balance;			/* Bilance */

  int random;			/* Random */

  int noverbose;		/* Verbose ? */
  int nobuffer;			/* Buffer ? */

#ifdef ENABLE_GTK
  int graphic;			/* Graphic interface */
  int go_graphic;		/* Graphis is started ? */
  int mute;			/* gsds Mute */

  int size_x;			/* X and Y of graphic interface */
  int size_y;
#endif

  int trimming;			/* Silence Trimming */

  pthread_mutex_t mutex;
  pthread_cond_t p_push;
  pthread_cond_t p_pop;

  format *formats[FORMATS_NUMBER];	/* Formats */
  format *format_current;	/* Current format */

  /* Output data */
  long ao_rate;
  int ao_channels;
  int ao_bitrate;
  char *ao_buffer;
  size_t ao_size;

  /* ledbars: */
  float ao_left;
  float ao_right;

  pthread_mutex_t m_output;
  audio *output;

  int nooutput;
  int norealtime;
  int timeforced;

  int splitoutput;

  int badlist_timer;
  int badlist_max_retry;

  audio *bad_list;
  int bad_list_status;
  pthread_t bad_list_th;
  pthread_mutex_t bad_list_mutex;

#ifdef ENABLE_CDAUDIO
  /* CDROM */
  char *cd;			/* cdrom device */
#endif

  daemon_data *daemon;		/* daemon ? */

  admin_data *admin;		/* admin interface? */

#ifdef ENABLE_GTK
  int admin_active;		/* Active and disactive the admin interface */
#endif

#ifdef ENABLE_GTK
  pthread_mutex_t queue_mutex;
  queue *queue_list;
#endif

  list *playlist_selected;

} somaplayer;

#ifdef ENABLE_MP3
/* Mp3 data format */
typedef struct
{
  mad_timer_t mad_time;		/* time of playing */
  mad_timer_t mad_duration;	/* total time */

  int frame;			/* Current frame */
  int n_frames;			/* Number of frames */

} format_data_mp3;
#endif

#ifdef ENABLE_OGG
/* Ogg data format */
typedef struct
{
  OggVorbis_File ogg;		/* Ogg element */
  double time;			/* Time */
  double duration;		/* Duration */

  int bytes;			/* Bytes */
} format_data_ogg;
#endif

#ifdef ENABLE_SNDFILE
/* Sndfile data format */
typedef struct
{
  SNDFILE *sndfile;		/* FileDescriptor */
  SF_INFO sfinfo;		/* FileInfo */
  struct timeval *start;

} format_data_sndfile;
#endif

#ifdef ENABLE_CDAUDIO

#define CDAUDIO_CODE_MAX 512
/* Cdaudio data format */
typedef struct
{
  char buffer[CD_FRAMESIZE_RAW * CDAUDIO_CODE_MAX];	/* The buffer */
  int id_head;
  int id_tail;
  int go;
  pthread_mutex_t mutex;
  int channels;

  long first, last, current;

} format_data_cdaudio;
#endif

#ifdef ENABLE_MIC
/* Mic data format */
typedef struct
{
  struct timeval *start;	/* Start time */
  int duration;			/* Duration */
  int rate;
  int bitrate;
  int channels;

} format_data_mic;
#endif

#ifdef ENABLE_DAEMON
/* Daemon data format */
typedef struct
{
  struct timeval *start;	/* Start time */
  sock_struct *sock;

} format_data_daemon_input;
#endif

#ifdef ENABLE_LAME
/* Lame data output */
typedef struct
{

  char *file;
  FILE *fd;

  int rate;
  int bitrate;
  int quality;
  int channels;

  int lowpass;
  int highpass;

  lame_global_flags *lame;

} audio_data_lame;
#endif

#ifdef ENABLE_SNDFILE
/* Sndfile data output */
typedef struct
{
  SF_INFO info;
  SNDFILE *fd;

} audio_data_sndfile;
#endif

#ifdef ENABLE_OGG
/* Vorbis encode data output */
typedef struct
{
  vorbis_info vi;
  vorbis_dsp_state dsp;
  vorbis_block block;
  ogg_stream_state ogg;
  vorbis_comment comment;

} audio_data_vorbis_encode;

/* Vorbis data output (like lame) */
typedef struct
{
  char *file;
  FILE *fd;

  int rate;
  int bitrate;
  int quality;
  int channels;

  int lowpass;
  int highpass;

  audio_data_vorbis_encode *vorbis;

} audio_data_vorbis;
#endif

#ifdef ENABLE_STREAMING
/* streaming data output */
typedef struct
{
  int type;			/* icecast, icecast2, shoutcat */

  int rate;
  int bitrate;
  int quality;

  int lowpass;
  int highpass;

  char *genre;
  char *name;
  char *url;
  char *description;

  char *mount;
  char *server;
  int port;
  char *password;
  int connect;

  int public;

  char *aim;
  char *icq;
  char *irc;

  FILE *dumpfile;
  char *dumpfilelocal;
  char *dumpfileremote;

  /* CONNECTION */
#ifdef ENABLE_OPENSSL
  BIO *bio;
#endif
  int fd;

  /* Lame data */
#ifdef ENABLE_LAME
  lame_global_flags *lame;
#endif
  MPEG_mode channels;

  /* Ogg data */
#ifdef ENABLE_OGG
  audio_data_vorbis_encode *vorbis;
#endif

  struct timeval *timer;

} audio_data_streaming;

#endif

extern pthread_t th1, th2;
#ifdef ENABLE_GTK
extern pthread_t th_graphic;
#endif
extern struct termios t1, t2;
extern int idle;

extern char **arg;
extern char **env;

extern somaplayer *play;
extern event events;

void *player (void *);
int playing (void);

/* Control function with struct locking */
char *playing_file (void);
char *playing_name (void);
int playing_list (void);

#ifdef ENABLE_GTK
void *daemon_graphic (void *);

void gtk_read (void);
#endif

#endif

/* EOF */
