<?php
/**
 * $Horde: forwards/lib/Driver/mdaemon.php,v 1.3.2.3 2003/01/20 05:17:56 ericr Exp $
 *
 * Copyright 2001-2003 Mike Cochrane <mike@graftonhall.co.nz>
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/bsdl.php.
 *
 * Forwards_Driver_mdaemon:: implements the Forwards_Driver API for the
 * Mdaemon mail servers.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @version $Revision: 1.3.2.3 $
 * @since   Forwards 2.1
 * @package forwards
 */

class Forwards_Driver_mdaemon extends Forwards_Driver {

    /** Hash containing mdaemon server location. */
    var $params;

    /**
     * Constructs a new Mdaemon Forwards_Driber object.
     *
     * @param array  $params    A hash containing connection parameters.
     */
    function Forwards_Driver_mdaemon($params = array())
    {
        $this->params = $params;
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string    $user       The username to enable forwarding for.
     * @param string    $realm      The realm of the user.
     * @param string    $pass       The password of the user.
     *
     * @param string    $target     The email address that mail should be forwarded to.
     *
     * @param optional boolean $keeplocal A flag that if true causes a copy of
     *                                    forwarded email to be kept in the
     *                                    local mailbox.
     *
     * @return boolean  Returns true on success, false on error.
     */
    function enableForwarding($user, $realm = 'default', $pass = "", $target, $keeplocal = false)
    {
        if (!is_dir($this->params[$realm]['location'])) {
           $this->err_str = _("Mdaemon path not found");
           return false;
        }

        // update forward target list
        if ($fp = fopen($this->params[$realm]['location'] . '/forward.dat', "rb")) {
            $current = fread ($fp, filesize ($this->params[$realm]['location'] . '/forward.dat'));
            fclose ($fp);
            if ($fp = fopen($this->params[$realm]['location'] . '/forward.dat', "wb")) {
                fwrite( $fp, "[$user@$realm]\nAddress=$target\n$current");
                fclose($fp);
            } else {
                return error;
            }
        } else {
            return error;
        }

        // create lock file
        $fp = fopen($this->params[$realm]['location'] . '/edituser.lck', "w");
        if (!$fp) {
           //$this->err_str = _("Not able to create lock file");
           return false;
        } else {
            fclose($fp);
        }

        // get current details
        $current_details = $this->_getUserDetails($user, $realm);
        if ($current_details === false) {
           //$this->err_str = _("Not able to retrieve current details.");
           return false;
        }

        // set forwarding flag
        $new_details = substr_replace($current_details, "Y", 216, 1);

        // set retain copy flag
        if ($keeplocal) {
            $new_details = substr_replace($new_details, "Y", 219, 1);
        } else {
            $new_details = substr_replace($new_details, "N", 219, 1);
        }

        // create Semaphore file
        $fp = fopen($this->params[$realm]['location'] . '/edituser.sem', "wb");
        if (!$fp) {
           $this->err_str = _("Not able to create semaphore file");
           return false;
        } else {
            fwrite($fp, "$user@$realm, " . $new_details);
            fclose($fp);
        }

        // remove lock file
        @unlink($this->params[$realm]['location'] . '/edituser.lck');

        return true;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string    $user       The username of the user.
     * @param string    $realm      The realm of the user.
     * @param string    $pass       The password of the user.
     *
     * @return boolean  Returns true on success, false on error.
     */
    function disableForwarding($user, $realm = 'default', $pass = "")
    {
        if (!is_dir($this->params[$realm]['location'])) {
           $this->err_str = _("Mdaemon path not found");
           return false;
        }

        // create lock file
        $fp = fopen($this->params[$realm]['location'] . '/edituser.lck', "w");
        if (!$fp) {
           //$this->err_str = _("Not able to create lock file");
           return false;
        } else {
            fclose($fp);
        }

        // get current details
        $current_details = $this->_getUserDetails($user, $realm);
        if ($current_details === false) {
           //$this->err_str = _("Not able to retrieve current details.");
           return false;
        }

        // set forwarding flag
        $new_details = substr_replace($current_details, "N", 216, 1);

        // create Semaphore file
        $fp = fopen($this->params[$realm]['location'] . '/edituser.sem', "wb");
        if (!$fp) {
           $this->err_str = _("Not able to create semaphore file");
           return false;
        } else {
            fwrite($fp, "$user@$realm, " . $new_details);
            fclose($fp);
        }

        // remove lock file
        @unlink($this->params[$realm]['location'] . '/edituser.lck');

        return true;
    }

    /**
     * Retrieves status of mail redirection for a user
     *
     * @param string    $user       The username of the user to check.
     *
     * @param string    $realm      The realm of the user to check.
     *
     * @return boolean    Returns true if forwarding is enabled for the user or false if
     *                  forwarding is currently disabled.
     */
    function isEnabledForwarding($user, $realm = 'default', $password = "")
    {
        // get current details
        $current_details = $this->_getUserDetails($user, $realm);
        if ($current_details === false) {
           //$this->err_str = _("Not able to retrieve current details.");
           return false;
        }

        // check forwarding flag
        if (substr($current_details, 216, 1) == "Y") {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Returns true if a local copy of forwarded messages is being kept
     *
     * @param string    $user       The username of the user to check.
     *
     * @param string    $realm      The realm of the user to check.
     *
     * @return boolean    Returns true if retain local copy is enabled else false.
     *
     */
    function isKeepLocal($user, $realm = 'default', $password = "")
    {
        // get current details
        $current_details = $this->_getUserDetails($user, $realm);
        if ($current_details === false) {
           //$this->err_str = _("Not able to retrieve current details.");
           return false;
        }

        // check retain copy flag
        if (substr($current_details, 219, 1) == "Y") {
            return true;
        } else {
            return false;
        }
    }


    /**
     * Retrieves current target of mail redirection
     *
     * @param string    $user       The username of the user.
     * @param string    $realm      The realm of the user.
     *
     * @return string   A string of current forwarding mail address or false.
     */
    function currentTarget($user, $realm = 'default', $password = "")
    {
        $searchString = "[$user@$realm]";

        $fp = fopen($this->params[$realm]['location'] . '/forward.dat', "rb");
        if (!$fp) {
           return false;
        }

        while (!feof ($fp)) {
            $line = fgets($fp, 4096);
            if (substr($line,0,strlen($searchString)) == $searchString) {
                $line = fgets($fp, 4096);
                fclose($fp);
                return substr($line,8);
            }
        }
       fclose($fp);

       return false;
    }

    /**
     * Retreive relevant line from userlist.dat
     *
     * @param   $user         The username for which to retrieve detals..
     * @param   $realm        The realm (domain) for the user.
     * @param   $location     The location of the mdaemon directory.
     *
     * @return  Mixed         (String) Line from userlist.dat or (boolean) False
     */
    function _getUserDetails($user, $realm = 'default') {

        $searchString = str_pad($realm, 45) . str_pad($user, 30);

        $fp = fopen($this->params[$realm]['location'] . '/userlist.dat', "rb");
        if (!$fp) {
//           $this->err_str = _("Mdaemon userlist file not accessable or not found.");
           return false;
        }

        while (!feof ($fp)) {
            $line = fgets($fp, 4096);
            if (substr($line,0,strlen($searchString)) == $searchString) {
                return $line;
            }
        }

       fclose($fp);
//       $this->err_str = sprintf(_("User %s@%s not found"), $user_name, $realm);
       return false;
    }
}
